/*
 *  Copyright 2001-2005 Internet2
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/* SAMLAttributeQuery.cpp - SAML attribute query implementation

   Scott Cantor
   5/21/02

   $History:$
*/

#include "internal.h"

using namespace saml;
using namespace std;


SAMLAttributeQuery::SAMLAttributeQuery(
    SAMLSubject* subject, const XMLCh* resource, const Iterator<SAMLAttributeDesignator*>& designators
    ) : SAMLSubjectQuery(subject), m_resource(XML::assign(resource))
{
    RTTI(SAMLAttributeQuery);
    while (designators.hasNext())
        m_designators.push_back(static_cast<SAMLAttributeDesignator*>(designators.next()->setParent(this)));
}

SAMLAttributeQuery::SAMLAttributeQuery(DOMElement* e) : SAMLSubjectQuery(e), m_resource(NULL)
{
    RTTI(SAMLAttributeQuery);
    fromDOM(e);
}

SAMLAttributeQuery::SAMLAttributeQuery(istream& in) : SAMLSubjectQuery(in), m_resource(NULL)
{
    RTTI(SAMLAttributeQuery);
    fromDOM(m_document->getDocumentElement());
}

SAMLAttributeQuery::~SAMLAttributeQuery()
{
    if (m_bOwnStrings)
        XMLString::release(&m_resource);
    for (vector<SAMLAttributeDesignator*>::const_iterator i=m_designators.begin(); i!=m_designators.end(); i++)
        delete (*i);
}

void SAMLAttributeQuery::ownStrings()
{
    if (!m_bOwnStrings) {
        m_resource=XML::assign(m_resource);
        m_bOwnStrings=true;
    }
}

void SAMLAttributeQuery::fromDOM(DOMElement* e)
{
    if (SAMLConfig::getConfig().strict_dom_checking) {
        if (XMLString::compareString(XML::SAMLP_NS,e->getNamespaceURI()))
            throw MalformedException(SAMLException::REQUESTER,"SAMLAttributeQuery::fromDOM() root element isn't in samlp namespace");
        if (XMLString::compareString(L(AttributeQuery),e->getLocalName())) {
            auto_ptr<saml::QName> type(saml::QName::getQNameAttribute(e,XML::XSI_NS,L(type)));
            if ((XMLString::compareString(L(Query),e->getLocalName()) && XMLString::compareString(L(SubjectQuery),e->getLocalName())) ||
                    !type.get() || XMLString::compareString(XML::SAMLP_NS,type->getNamespaceURI()) ||
                    XMLString::compareString(L(AttributeQueryType),type->getLocalName()))
                throw MalformedException(SAMLException::REQUESTER,"SAMLAttributeQuery::fromDOM() missing samlp:AttributeQuery element at root");
        }
    }

    m_resource = const_cast<XMLCh*>(e->getAttributeNS(NULL,L(Resource)));

    DOMElement* de=XML::getFirstChildElement(e,XML::SAML_NS,L(AttributeDesignator));
    while (de) {
        SAMLAttributeDesignator* d=new SAMLAttributeDesignator(de);
        m_designators.push_back(static_cast<SAMLAttributeDesignator*>(d->setParent(this)));
        de=XML::getNextSiblingElement(de,XML::SAML_NS,L(AttributeDesignator));
    }
    checkValidity();
}

void SAMLAttributeQuery::setResource(const XMLCh* resource)
{
    if (m_bOwnStrings)
        XMLString::release(&m_resource);
    else {
        m_resource=NULL;
        ownStrings();
    }
    m_resource=XML::assign(resource);
    setDirty();
}

void SAMLAttributeQuery::setDesignators(const Iterator<SAMLAttributeDesignator*>& designators)
{
    while (m_designators.size())
        removeDesignator(0);
    while (designators.hasNext())
        addDesignator(designators.next());
}

void SAMLAttributeQuery::addDesignator(SAMLAttributeDesignator* designator)
{
    if (designator) {
        designator->setParent(this);
        m_designators.push_back(designator);
        ownStrings();
        setDirty();
    }
    else
        throw MalformedException("designator cannot be null");
}

void SAMLAttributeQuery::removeDesignator(unsigned long index)
{
    SAMLAttributeDesignator* kill=m_designators[index];
    m_designators.erase(m_designators.begin()+index);
    delete kill;
    ownStrings();
    setDirty();
}

DOMElement* SAMLAttributeQuery::buildRoot(DOMDocument* doc, bool xmlns) const
{
    DOMElement* q=doc->createElementNS(XML::SAMLP_NS,L(AttributeQuery));
    if (xmlns)
        q->setAttributeNS(XML::XMLNS_NS,L(xmlns),XML::SAMLP_NS);
    return q;
}

DOMNode* SAMLAttributeQuery::toDOM(DOMDocument* doc, bool xmlns) const
{
    SAMLSubjectQuery::toDOM(doc,xmlns);
    DOMElement* q=static_cast<DOMElement*>(m_root);
    doc=q->getOwnerDocument();
    
    if (m_bDirty) {
        if (!XML::isEmpty(m_resource))
            q->setAttributeNS(NULL,L(Resource),m_resource);
    
        for (vector<SAMLAttributeDesignator*>::const_iterator i=m_designators.begin(); i!=m_designators.end(); i++)
            q->appendChild((*i)->toDOM(doc));
    
        setClean();
    }
    else if (xmlns) {
        DECLARE_DEF_NAMESPACE(q,XML::SAMLP_NS);
    }

    return q;
}

SAMLObject* SAMLAttributeQuery::clone() const
{
    return new SAMLAttributeQuery(static_cast<SAMLSubject*>(m_subject->clone()),m_resource,getDesignators().clone());
}
