//                                               -*- C++ -*-
/**
 *  @file  Log.hxx
 *  @brief Log records all user information to a file or tty
 *
 *  (C) Copyright 2005-2010 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2010-02-04 16:44:49 +0100 (jeu. 04 févr. 2010) $
 *  Id:      $Id: Log.hxx 1473 2010-02-04 15:44:49Z dutka $
 */
#ifndef OPENTURNS_LOG_HXX
#define OPENTURNS_LOG_HXX

#include <fstream>
#include <map>
#include "OTprivate.hxx"

#ifdef WIN32
#undef ERROR
#endif


namespace OpenTURNS
{

  namespace Base
  {

    namespace Common
    {


      /**
       * @class log
       * @brief Records all user information to a file or tty
       * @internal
       */

      class Log
      {
      public:

	typedef unsigned long Severity;

	/** Those flags should be ORed */
	static const Severity DBG;
	static const Severity WRAPPER;
	static const Severity INFO;
	static const Severity USER;
	static const Severity WARN;
	static const Severity ERROR;

	static const Severity NONE;
	static const Severity ALL;

	/** Log messages according to the DBG level
	 * @param msg The message to be logged
	 */
	static void Debug(const String & msg);

	/** Log messages according to the WRAPPER level
	 * @param msg The message to be logged
	 */
	static void Wrapper(const String & msg);

	/** Log messages according to the INFO level
	 * @param msg The message to be logged
	 */
	static void Info(const String & msg);

	/** Log messages according to the USER level
	 * @param msg The message to be logged
	 */
	static void User(const String & msg);

	/** Log messages according to the WARN level
	 * @param msg The message to be logged
	 */
	static void Warn(const String & msg);

	/** Log messages according to the ERROR level
	 * @param msg The message to be logged
	 */
	static void Error(const String & msg);

	/** Divert the output to a file
	 * @param file The filename where the log will be written
	 */
	static void SetFile(const FileName & file);

	/** Set the level flags for the messages logged to the file
	 * @param flags An integer built from ORed level flags
	 */
	static void Show(Severity flags);

	/** Get the current level flags
	 * @return An integer built from ORed level flags
	 */
	static Severity Flags();

	/** Flush pending messages
	 * @internal
	 */
	static void Flush();


      private:
	static Log & GetInstance();
	static void Release();

      private:
	/** The unique instance of the Singleton */
	static Log * P_instance_;

	/** Minimum severity logged to the file */
	static Severity Severity_;

      public:
#ifndef SWIG	
	struct Entry
	{
	  Severity sev_;
	  String msg_;
	  Entry(Severity sev, String msg) : sev_(sev), msg_(msg) {}
	  Entry() : sev_(0), msg_() {}
	  Bool operator== (const Entry & other) const { return (this->sev_ == other.sev_) && (this->msg_ == other.msg_); }
	}; /* end struct Entry */
#endif
	~Log();

      private:
	Log();
	void push(const Entry & entry);
	void printRepeatedMessage(const Entry & entry);
	void flush();

	/** Set the name of the log file */
	void setFile(const FileName & file);

      private:
        /** human readable log */
	std::map<Severity, String> logName_;

        /** The environment variable name */
        const char * openturnsLogSeverityVariableName_;

        /** Set Severity according to Openturns LogSeverity Variable */
	void initSeverityFromEnvironment();

      private:
	/** The file where to write messages */
	std::ostream * p_file_;

	/** Remember the previous message */
	mutable Entry previousMessage_;
	mutable UnsignedLong count_;

      }; /* end class Log */






    } /* namespace Common */
  } /* namespace Base */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_LOG_HXX */
