//                                               -*- C++ -*-
/**
 *  @file  Epanechnikov.cxx
 *  @brief The Epanechnikov distribution
 *
 *  (C) Copyright 2005-2010 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2010-02-04 16:44:49 +0100 (jeu. 04 févr. 2010) $
 *  Id:      $Id: Epanechnikov.cxx 1473 2010-02-04 15:44:49Z dutka $
 */
#include <cmath>
#include "Epanechnikov.hxx"
#include "RandomGenerator.hxx"
#include "SpecFunc.hxx"
#include "DistFunc.hxx"
#include "PersistentObjectFactory.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Distribution {

      CLASSNAMEINIT(Epanechnikov);

      static Base::Common::Factory<Epanechnikov> RegisteredFactory("Epanechnikov");

      typedef Base::Stat::RandomGenerator RandomGenerator;

      /* Default constructor */
      Epanechnikov::Epanechnikov()
        : NonEllipticalDistribution("Epanechnikov")
      {
        setDimension(1);
        setRange(Interval(-1.0, 1.0));
      }

      /* Comparison operator */
      Bool Epanechnikov::operator ==(const Epanechnikov & other) const
      {
        return true;
      }

      /* String converter */
      String Epanechnikov::__repr__() const {
        OSS oss;
        oss << "class=" << Epanechnikov::GetClassName()
            << " name=" << getName()
            << " dimension=" << getDimension();
        return oss;
      }

      /* Virtual constructor */
      Epanechnikov * Epanechnikov::clone() const
      {
        return new Epanechnikov(*this);
      }

      /* Get the DDF of the distribution */
      Epanechnikov::NumericalPoint Epanechnikov::computeDDF(const NumericalPoint & point) const
      {
        NumericalScalar x(point[0]);
        if ((x <= -1.0) || (x > 1.0)) return NumericalPoint(1, 0.0);
        return NumericalPoint(1, -1.5 * x);
      }


      /* Get the PDF of the distribution */
      NumericalScalar Epanechnikov::computePDF(const NumericalPoint & point) const
      {
        NumericalScalar x(point[0]);
        if ((x <= -1.0) || (x > 1.0)) return 0.0;
        return 0.75 * (1.0 + x) * (1.0 - x);
      }


      /* Get the CDF of the distribution */
      NumericalScalar Epanechnikov::computeCDF(const NumericalPoint & point, const Bool tail) const
      {
        NumericalScalar x(point[0]);
        if (x <= -1.0) return (tail ? 1.0 : 0.0);
        if (x > 1.0) return (tail ? 0.0 : 1.0);
        if (tail) return 0.5 - x * (0.75 - 0.25 * x * x);
        return 0.5 + x * (0.75 - 0.25 * x * x);
      }

      /** Get the PDFGradient of the distribution */
      Epanechnikov::NumericalPoint Epanechnikov::computePDFGradient(const NumericalPoint & point) const
      {
        return NumericalPoint(0);
      }

      /** Get the CDFGradient of the distribution */
      Epanechnikov::NumericalPoint Epanechnikov::computeCDFGradient(const NumericalPoint & point) const
      {
        return NumericalPoint(0);
      }

      /* Get the quantile of the distribution */
      NumericalScalar Epanechnikov::computeScalarQuantile(const NumericalScalar prob,
							  const Bool tail,
                                                          const NumericalScalar precision) const
      {
        // 2.094395102393195492308429 = 2 * Pi / 3
	if (tail) return 2.0 * cos(0.3333333333333333333333333 * acos(2.0 * prob - 1.0) - 2.094395102393195492308429);
        return 2.0 * cos(0.3333333333333333333333333 * acos(1.0 - 2.0 * prob) - 2.094395102393195492308429);
      }

      /* Get the roughness, i.e. the L2-norm of the PDF */
      NumericalScalar Epanechnikov::getRoughness() const
      {
        return 0.6;
      }

      /* Get the mean of the distribution */
      Epanechnikov::NumericalPoint Epanechnikov::getMean() const /* throw(NotDefinedException) */
      {
        return NumericalPoint(1, 0.0);
      }

      /* Get the standard deviation of the distribution */
      Epanechnikov::NumericalPoint Epanechnikov::getStandardDeviation() const /* throw(NotDefinedException) */
      {
        // 0.4472135954999579392818348 = 1 / sqrt(5)
        return NumericalPoint(1, 0.4472135954999579392818348);
      }

      /* Get the skewness of the distribution */
      Epanechnikov::NumericalPoint Epanechnikov::getSkewness() const /* throw(NotDefinedException) */
      {
        return NumericalPoint(1, 0.0);
      }

      /* Get the kurtosis of the distribution */
      Epanechnikov::NumericalPoint Epanechnikov::getKurtosis() const /* throw(NotDefinedException) */
      {
        // 2.142857142857142857142857 = 15 / 7
        return NumericalPoint(1, 2.142857142857142857142857);
      }

      /* Get the covariance of the distribution */
      Epanechnikov::CovarianceMatrix Epanechnikov::getCovariance() const /* throw(NotDefinedException) */
      {
        CovarianceMatrix covariance(1);
        covariance(0, 0) = 0.2;
        return covariance;
      }

      /* Parameters value and description accessor */
      Epanechnikov::NumericalPointWithDescriptionCollection Epanechnikov::getParametersCollection() const
      {
        // No parameter, return a collection with an empty NumericalPointWithDescription
        return NumericalPointWithDescriptionCollection(1, NumericalPointWithDescription(0));
      }

      /* Method save() stores the object through the StorageManager */
      void Epanechnikov::save(StorageManager::Advocate & adv) const
      {
        NonEllipticalDistribution::save(adv);
      }

      /* Method load() reloads the object from the StorageManager */
      void Epanechnikov::load(StorageManager::Advocate & adv)
      {
        NonEllipticalDistribution::load(adv);
      }

    } /* namespace Distribution */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
