//                                               -*- C++ -*-
/**
 *  @file  PoissonFactory.cxx
 *  @brief Factory for Poisson distribution
 *
 *  (C) Copyright 2005-2012 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: schueller $
 *  @date:   $LastChangedDate: 2012-02-17 19:35:43 +0100 (Fri, 17 Feb 2012) $
 *  Id:      $Id: PoissonFactory.cxx 2392 2012-02-17 18:35:43Z schueller $
 */
#include <cmath>
#include "PoissonFactory.hxx"

BEGIN_NAMESPACE_OPENTURNS



/* Default constructor */
PoissonFactory::PoissonFactory():
  DistributionImplementationFactory()
{
  // Nothing to do
}

/* Virtual constructor */
PoissonFactory * PoissonFactory::clone() const
{
  return new PoissonFactory(*this);
}

/* Here is the interface that all derived class must implement */

Poisson * PoissonFactory::build(const NumericalSample & sample) const
{
  if (sample.getSize() == 0) throw InvalidArgumentException(HERE) << "Error: cannot build a Poisson distribution from an empty sample";
  if (sample.getDimension() != 1) throw InvalidArgumentException(HERE) << "Error: can build a Poisson distribution only from a sample of dimension 1, here dimension=" << sample.getDimension();
  // Check if all the components of the sample are integers > 0
  const UnsignedLong size(sample.getSize());
  NumericalScalar lambda(0.0);
  for (UnsignedLong i = 0; i < size; ++i)
    {
      NumericalScalar x(sample[i][0]);
      if ((x != trunc(x)) || (x < 0.0)) throw InternalException(HERE) << "Error: can build a Poisson distribution only from a sample with integer components >= 0, here sample[" << i << "][0]=" << x;
      lambda += x;
    }
  if (lambda <= 0.0) throw InternalException(HERE) << "Error: can build a poisson distribution only if lambda > 0, here lambda=" << lambda;
  return Poisson(lambda / size).clone();
}

Poisson * PoissonFactory::build(const NumericalPointCollection & parameters) const
{
  try {
    Poisson distribution;
    distribution.setParametersCollection(parameters);
    return distribution.clone();
  }
  catch (InvalidArgumentException & ex)
    {
      throw InternalException(HERE) << "Error: cannot build a Poisson distribution from the given parameters";
    }
}

Poisson * PoissonFactory::build() const
{
  return Poisson().clone();
}

END_NAMESPACE_OPENTURNS
