//                                               -*- C++ -*-
/**
 *  @file  t_ExponentialCauchy_std.cxx
 *  @brief The test file of class ExponentialCauchy
 *
 *  (C) Copyright 2005-2012 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: lebrun $
 *  @date:   $LastChangedDate: 2012-03-20 06:42:21 +0100 (Tue, 20 Mar 2012) $
 *  Id:      $Id: t_ExponentialCauchy_std.cxx 2428 2012-03-20 05:42:21Z lebrun $
 */
#include <iostream>
#include "OT.hxx"
#include "OTtestcode.hxx"

using namespace OT;
using namespace OT::Test;


int main(int argc, char *argv[])
{
  TESTPREAMBLE;
  OStream fullprint(std::cout);

  try {

    /* Default dimension parameter to evaluate the model */
    const UnsignedLong defaultDimension(1);

    /* Amplitude values */
    NumericalPoint amplitude(defaultDimension, 1.0);
    /* Scale values */
    NumericalPoint scale(defaultDimension, 1.0);


    /* Default constructor */
    ExponentialCauchy myDefautModel;
    fullprint << "myDefautModel = " << myDefautModel << std::endl;

    /* Second order model with parameters */
    ExponentialCauchy myModel(amplitude, scale);
    fullprint << "myModel = " << myModel << std::endl;

    const NumericalScalar frequencyValueOne(1.0);
    fullprint << "spectral density matrix at f = " << frequencyValueOne << " : " << myModel.computeSpectralDensity(frequencyValueOne) << std::endl;
    fullprint << "spectral density matrix at f = " << -frequencyValueOne << " : " << myModel.computeSpectralDensity(-frequencyValueOne) << std::endl;

    const NumericalScalar frequencyValueHigh(4.0);
    fullprint << "spectral density matrix at f = " << frequencyValueHigh << " : " << myModel.computeSpectralDensity(frequencyValueHigh) << std::endl;
    fullprint << "spectral density matrix at f = " << -frequencyValueHigh << " : " << myModel.computeSpectralDensity(-frequencyValueHigh) << std::endl;

    const NumericalScalar timeValueOne(1.0);
    fullprint << "covariance matrix at t = " << timeValueOne << " : " << myModel.computeCovariance(timeValueOne) << std::endl;
    fullprint << "covariance matrix at t = " << -timeValueOne << " : " << myModel.computeCovariance(-timeValueOne) << std::endl;

    /* Evaluation at time higher to check the decrease of the exponential values */
    const NumericalScalar timeValueHigh(4.0);
    fullprint << "covariance matrix at t = "  << timeValueHigh << " : " << myModel.computeCovariance(timeValueHigh) << std::endl;

    /* Discretize the process on a small time grid */
    RegularGrid timeGrid(0.0, 1.0 / 3.0, 4);
    fullprint << "discretized covariance over the time grid=" << timeGrid << " is" << std::endl;
    fullprint << myModel.discretizeCovariance(timeGrid) << std::endl;
    /* Default dimension parameter to evaluate the model */
    const UnsignedLong highDimension(3);

    /* Reallocation of adequate sizes*/
    amplitude.resize(highDimension);
    scale.resize(highDimension);

    CorrelationMatrix spatialCorrelation(highDimension);

    for (UnsignedLong index = 0 ; index < highDimension; ++index)
      {
        // constant amplitude
        amplitude[index] = 1.0 ;
        scale[index] = (index + 1.0) / defaultDimension ;
        for (UnsignedLong k = 0; k < index; ++k) spatialCorrelation(index, k) = 1.0 / pow(1.0 + index + k, 2);
      }
    //fullprint << "spatialCorrelation=" << spatialCorrelation << std::endl;
    /* checking the cast*/
    SecondOrderModel mySecondOrderModel(ExponentialCauchy(amplitude, scale, spatialCorrelation));
    fullprint << "mySecondOrderModel = " << mySecondOrderModel << std::endl;

    /* Second order model - high dimension */
    ExponentialCauchy myHighModel(amplitude, scale, spatialCorrelation);
    fullprint << "myHighModel = " << myHighModel << std::endl;

    fullprint << "spectral density matrix at f = " << frequencyValueOne << " : " << myModel.computeSpectralDensity(frequencyValueOne) << std::endl;
    fullprint << "spectral density matrix at f = " << -frequencyValueOne << " : " << myModel.computeSpectralDensity(-frequencyValueOne) << std::endl;
    fullprint << "covariance matrix at t = " << timeValueOne << " : " << myHighModel.computeCovariance(0, timeValueOne) << std::endl;
    fullprint << "covariance matrix at t = " << -1.0 * timeValueOne << " : " << myHighModel.computeCovariance(0, -1.0 * timeValueOne) << std::endl;
    fullprint << "spectral density matrix at f = " << frequencyValueHigh << " : " << myModel.computeSpectralDensity(frequencyValueHigh) << std::endl;
    fullprint << "spectral density matrix at f = " << -frequencyValueHigh << " : " << myModel.computeSpectralDensity(-frequencyValueHigh) << std::endl;
    fullprint << "covariance matrix at t = "  << timeValueHigh << " : " << myHighModel.computeCovariance(0, timeValueHigh) << std::endl;

    fullprint << "discretized covariance over the time grid=" << timeGrid << " is" << std::endl;
    fullprint << myHighModel.discretizeCovariance(timeGrid) << std::endl;

    /* Test save / load */
    Study myStudy;
    myStudy.setStorageManager(XMLStorageManager("testStudy.ot"));
    myStudy.add("myModel", myHighModel);
    myStudy.save();

    Study loadedStudy;
    loadedStudy.setStorageManager(XMLStorageManager("testStudy.ot"));
    loadedStudy.load();
    ExponentialCauchy myModelLoaded;
    loadedStudy.fillObject("myModel", myModelLoaded);
    fullprint << "loaded model =" << myModelLoaded << std::endl;

  }
  catch (TestFailed & ex) {
    std::cerr << ex << std::endl;
    return ExitCode::Error;
  }


  return ExitCode::Success;
}
