//                                               -*- C++ -*-
/**
 *  @brief The class that implements randomMixtures
 *
 *  Copyright 2005-2015 Airbus-EDF-IMACS-Phimeca
 *
 *  This library is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  along with this library.  If not, see <http://www.gnu.org/licenses/>.
 *
 */
#ifndef OPENTURNS_RANDOMMIXTURE_HXX
#define OPENTURNS_RANDOMMIXTURE_HXX

#include "Distribution.hxx"
#include "DistributionFactory.hxx"
#include "DistributionImplementation.hxx"
#include "Collection.hxx"
#include "PersistentCollection.hxx"
#include "SpecFunc.hxx"
#include "Normal.hxx"
#include "ResourceMap.hxx"
#include "SphereUniformNorm.hxx"
#include "Indices.hxx"
#include "FFT.hxx"
#include <limits>       // std::numeric_limits

BEGIN_NAMESPACE_OPENTURNS

/**
 * @class RandomMixture
 *
 * The class describes the probabilistic concept of RandomMixture.
 */
class OT_API RandomMixture
  : public DistributionImplementation
{
  CLASSNAME;
public:

  typedef Collection<Distribution>               DistributionCollection;
  typedef PersistentCollection<Distribution>     DistributionPersistentCollection;
  typedef PersistentCollection<NumericalComplex> NumericalComplexPersistentCollection;
  typedef Collection<DistributionFactory>        DistributionFactoryCollection;


  /** Parameter constructor - 1D */
  explicit RandomMixture(const DistributionCollection & coll,
                         const NumericalScalar constant = 0.0);

  /** Parameter constructor - 1D */
  explicit RandomMixture(const DistributionCollection & coll,
                         const NumericalPoint & weights,
                         const NumericalScalar constant = 0.0);

  /** Parameter constructor - nD */
  RandomMixture(const DistributionCollection & coll,
                const Matrix & weights,
                const NumericalPoint constant);

  /** Parameter constructor - nD */
  RandomMixture(const DistributionCollection & coll,
                const Matrix & weights);

  /** Parameter constructor - nD */
  RandomMixture(const DistributionCollection & coll,
                const NumericalSample & weights,
                const NumericalPoint constant);

  /** Parameter constructor - nD */
  RandomMixture(const DistributionCollection & coll,
                const NumericalSample & weights);

  /** Comparison operator */
  Bool operator ==(const RandomMixture & other) const;

  /** String converter */
  String __repr__() const;
  String __str__(const String & offset = "") const;


  /** Distribution collection accessor */
  const DistributionCollection & getDistributionCollection() const;

  /** FFT algorithm accessor */
  FFT getFFTAlgorithm() const;
  void setFFTAlgorithm(const FFT & fft);

  /** Constant accessor */
  void setConstant(const NumericalPoint & constant);
  NumericalPoint getConstant() const;



  /* Here is the interface that all derived class must implement */

  /** Virtual constructor */
  virtual RandomMixture * clone() const;

  /** Get one realization of the RandomMixture */
  NumericalPoint getRealization() const;

  /** Get the DDF of the RandomMixture */
  using DistributionImplementation::computeDDF;
  NumericalPoint computeDDF(const NumericalPoint & point) const;

  /** Get the PDF of the RandomMixture */
  using DistributionImplementation::computePDF;
  NumericalScalar computePDF(const NumericalPoint & point) const;

  /** Compute the PDF over a regular grid */
  NumericalSample computePDF(const NumericalScalar xMin,
                             const NumericalScalar xMax,
                             const UnsignedInteger pointNumber,
                             NumericalSample & grid) const;

  /* Compute the PDF of over a regular grid */
  NumericalSample computePDF(const NumericalPoint & xMin,
                             const NumericalPoint & xMax,
                             const Indices & pointNumber,
                             NumericalSample & grid) const;

  /** Get the i-th marginal distribution */
  Implementation getMarginal(const UnsignedInteger i) const;

  /** Get the distribution of the marginal distribution corresponding to indices dimensions */
  Implementation getMarginal(const Indices & indices) const;

protected:

private:

  /** Quantile computation for dimension=1 */
  NumericalScalar computeScalarQuantile(const NumericalScalar prob,
                                        const Bool tail = false) const;

  /** Compute the characteristic function of 1D distributions by difference to a reference Normal distribution with the same mean and the same standard deviation in a regular pattern with cache */
  NumericalComplex computeDeltaCharacteristicFunction(const UnsignedInteger index) const;

  /** Compute the characteristic function of nD distributions by difference to a reference Normal distribution with the same mean and the same covariance */
  friend struct AddPDFOn1DGridPolicy;
  friend struct AddPDFOn2DGridPolicy;
  friend struct AddPDFOn3DGridPolicy;
  NumericalComplex computeDeltaCharacteristicFunction(const NumericalPoint & x) const;

  /** Update cache of the characteristic function */
  void updateCacheDeltaCharacteristicFunction(const NumericalSample & points) const;

  /** Contribution to computePDF on a 1D grid */
  void addPDFOn1DGrid(const Indices & pointNumber, const NumericalPoint & h, const NumericalPoint & tau, NumericalSample & result) const;

  /** Contribution to computePDF on a 2D grid */
  void addPDFOn2DGrid(const Indices & pointNumber, const NumericalPoint & h, const NumericalPoint & tau, NumericalSample & result) const;

  /** Contribution to computePDF on a 3D grid */
  void addPDFOn3DGrid(const Indices & pointNumber, const NumericalPoint & h, const NumericalPoint & tau, NumericalSample & result) const;

public:
  /** Get the CDF of the RandomMixture */
  using DistributionImplementation::computeCDF;
  NumericalScalar computeCDF(const NumericalPoint & point) const;
  using DistributionImplementation::computeComplementaryCDF;
  NumericalScalar computeComplementaryCDF(const NumericalPoint & point) const;

  /** Compute the CDF over a regular grid */
  NumericalSample computeCDF(const NumericalScalar xMin,
                             const NumericalScalar xMax,
                             const UnsignedInteger pointNumber,
                             NumericalSample & grid) const;

  /** Get the probability content of an interval */
  NumericalScalar computeProbability(const Interval & interval) const;

  /** Compute the quantile over a regular grid */
  using DistributionImplementation::computeQuantile;
  NumericalSample computeQuantile(const NumericalScalar qMin,
                                  const NumericalScalar qMax,
                                  const UnsignedInteger pointNumber,
                                  const Bool tail = false) const;

  /** Get the characteristic function of the distribution, i.e. phi(u) = E(exp(I*u*X)) */
  using DistributionImplementation::computeCharacteristicFunction;
  NumericalComplex computeCharacteristicFunction(const NumericalScalar x) const;
  NumericalComplex computeCharacteristicFunction(const NumericalPoint & x) const;
  using DistributionImplementation::computeLogCharacteristicFunction;
  NumericalComplex computeLogCharacteristicFunction(const NumericalScalar x) const;
  NumericalComplex computeLogCharacteristicFunction(const NumericalPoint & x) const;

  /** Get the PDF gradient of the distribution */
  using DistributionImplementation::computePDFGradient;
  NumericalPoint computePDFGradient(const NumericalPoint & point) const;

  /** Get the CDF gradient of the distribution */
  using DistributionImplementation::computeCDFGradient;
  NumericalPoint computeCDFGradient(const NumericalPoint & point) const;

  /** Parameters value and description accessor */
  NumericalPointWithDescriptionCollection getParametersCollection() const;

  /** Weights distribution accessor */
protected:
  void setWeights(const Matrix & weights);
  void setDistributionCollection(const DistributionCollection & coll);

public:
  Matrix getWeights() const;

  /** Get a positon indicator for a 1D distribution */
  NumericalScalar getPositionIndicator() const;

  /** Get a dispersion indicator for a 1D distribution */
  NumericalScalar getDispersionIndicator() const;

  /** BlockMin accessor */
  void setBlockMin(const UnsignedInteger blockMin);
  UnsignedInteger getBlockMin() const;

  /** BlockMax accessor */
  void setBlockMax(const UnsignedInteger blockMax);
  UnsignedInteger getBlockMax() const;

  /** MaxSize accessor */
  void setMaxSize(const UnsignedInteger maxSize);
  UnsignedInteger getMaxSize() const;

  /** Alpha accessor */
  void setAlpha(const NumericalScalar alpha);
  NumericalScalar getAlpha() const;

  /** Beta accessor */
  void setBeta(const NumericalScalar beta);
  NumericalScalar getBeta() const;

  /** Reference bandwidth accessor */
  void setReferenceBandwidth(const NumericalPoint & bandwidth);
  NumericalPoint getReferenceBandwidth() const;

  /** PDF epsilon accessor. For other distributions, it is a read-only attribute. */
  void setPDFPrecision(const NumericalScalar pdfPrecision);

  /** CDF epsilon accessor. For other distributions, it is a read-only attribute. */
  void setCDFPrecision(const NumericalScalar cdfPrecision);

  /** Project a RandomMixture distribution over a collection of DistributionFactory by using sampling and Kolmogorov distance. */
  DistributionCollection project(const DistributionFactoryCollection & factoryCollection,
                                 NumericalPoint & kolmogorovNorm,
                                 const UnsignedInteger size = ResourceMap::GetAsUnsignedInteger( "RandomMixture-ProjectionDefaultSize" )) const;

  /** Tell if the distribution has independent copula */
  Bool hasIndependentCopula() const;

  /** Tell if the distribution has elliptical copula */
  Bool hasEllipticalCopula() const;

  /** Check if the distribution is elliptical */
  Bool isElliptical() const;

  /** Check if the distribution is continuous */
  Bool isContinuous() const;

  /** Check if the distribution is discrete */
  Bool isDiscrete() const;

  /** Tell if the distribution is integer valued */
  Bool isIntegral() const;

  /** Method save() stores the object through the StorageManager */
  void save(Advocate & adv) const;

  /** Method load() reloads the object from the StorageManager */
  void load(Advocate & adv);

private:

  class KolmogorovProjection
  {
  public:
    /** Constructor from a distribution and a data set */
    KolmogorovProjection(const NumericalSample & dataX,
                         const NumericalSample & dataY,
                         const DistributionFactory & factory):
      dataX_(dataX),
      dataY_(dataY),
      factory_(factory) {};

    /** Compute the Kolmogorov distance based on the given data, for a given parameter set */
    NumericalPoint computeNorm(const NumericalPoint & parameters) const
    {
      NumericalScalar norm(0.0);
      try
      {
        const Distribution candidate(factory_.build(NumericalPointCollection(1, parameters)));
        for (UnsignedInteger i = 0; i < dataX_.getSize(); ++i)
          norm += std::pow(candidate.computeCDF(dataX_[i][0]) - dataY_[i][0], 2);
        return NumericalPoint(1, norm);
      }
      catch(...)
      {
        return NumericalPoint(1, SpecFunc::MaxNumericalScalar);
      }
    }

    /** factory accessor */
    void setDistributionFactory(const DistributionFactory & factory)
    {
      factory_ = factory;
    }
  private:
    NumericalSample dataX_;
    NumericalSample dataY_;
    DistributionFactory factory_;
  };

protected:
  /** Compute the numerical range of the distribution given the parameters values */
  void computeRange();

  /** Default constructor for save/load mechanism */
  RandomMixture() {};
  friend class Factory<RandomMixture>;

  /** Get the mean of a randomMixture */
  void computeMean() const;

  /** Get the covariance of a randomMixture */
  void computeCovariance() const;

private:
  /** Compute the left-hand sum in Poisson's summation formula for the equivalent normal */
  NumericalScalar computeEquivalentNormalPDFSum(const NumericalScalar x) const;
  NumericalScalar computeEquivalentNormalCDFSum(const NumericalScalar s, const NumericalScalar t) const;

  friend struct EquivalentNormalPDFSumPolicy;
  NumericalScalar computeEquivalentNormalPDFSum(const NumericalPoint & y, const NumericalPoint & gridStep,
      UnsignedInteger imax, UnsignedInteger & levelMax) const;
public:
  /** Get the standard deviation of the distribution */
  NumericalPoint getStandardDeviation() const;

  /** Get the skewness of the distribution */
  NumericalPoint getSkewness() const;

  /** Get the kurtosis of the distribution */
  NumericalPoint getKurtosis() const;

private:
  /** Compute the position indicator */
  void computePositionIndicator() const;

  /** Compute the dispersion indicator */
  void computeDispersionIndicator() const;

  /** Compute the reference bandwidth. It is defined as the maximum bandwidth
      that allow a precise computation of the PDF over the range
      [positionIndicator_ +/- beta * dispersionIndicator_] */
  void computeReferenceBandwidth();

  /** Compute the equivalent normal distribution, i.e. with the same mean and
      the same standard deviation */
  void computeEquivalentNormal();

  /** The collection of distribution of the randomMixture */
  DistributionPersistentCollection distributionCollection_;

  /** The constant term of the mixture */
  NumericalPoint constant_;

  /** The Weight matrix */
  Matrix weights_;

  /** inverse weight matrix if defined */
  Matrix inverseWeights_;

  /** Determinant of inverse weights */
  NumericalScalar detWeightsInverse_;

  /** FFT algorithm */
  FFT fftAlgorithm_;

  /** The RandomMixture is analytic if size of collection = dimension */
  Bool isAnalytical_;

  /** Position indicator */
  mutable NumericalScalar positionIndicator_;
  mutable Bool isAlreadyComputedPositionIndicator_;

  /** Dispersion indicator */
  mutable NumericalScalar dispersionIndicator_;
  mutable Bool isAlreadyComputedDispersionIndicator_;

  /** Minimum number of blocks to consider for PDF and CDF computation */
  UnsignedInteger blockMin_;

  /** Maximum number of blocks to consider for PDF and CDF computation */
  UnsignedInteger blockMax_;

  /** Reference bandwidth */
  NumericalPoint referenceBandwidth_;

  /** Reference bandwidth factor */
  NumericalScalar referenceBandwidthFactor_;

  /** Maximum size of the cache for the CharacteristicFunction values */
  UnsignedInteger maxSize_;

  /** Index of the top of the cache */
  mutable UnsignedInteger storedSize_;

  /** Cache for the characteristic function values */
  mutable NumericalComplexPersistentCollection characteristicValuesCache_;

  /** A priori range of PDF and CDF argument expressed in dispersionIndicator units */
  NumericalScalar alpha_;

  /** Distance from the boundary of the a priori range at which the PDF is negligible */
  NumericalScalar beta_;

  /** Requested precision for PDF computation */
  mutable NumericalScalar pdfPrecision_;

  /** Requested precision for CDF computation */
  mutable NumericalScalar cdfPrecision_;

  /** Normal distribution with the same mean and standard deviation than the RandomMixture */
  Normal equivalentNormal_;

  /** Helper object to retrieve points on a regular grid */
  SphereUniformNorm gridMesher_;

}; /* class RandomMixture */


END_NAMESPACE_OPENTURNS

#endif /* OPENTURNS_RANDOMMIXTURE_HXX */
