/*
 *
 * Copyright (C) 2005, 2006 Mekensleep
 *
 *	Mekensleep
 *	24 rue vieille du temple
 *	75004 Paris
 *       licensing@mekensleep.com
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * Authors:
 *  Johan Euphrosine <proppy@gmail.com>
 *  Cedric Pinson <cpinson@freesheep.org>
 *
 */
#include "osgviewer.h"

#include <assert.h>
#include <sys/time.h>
#include <assert.h>
#include <ctime>
#include <string>
#include <iostream>
#include <SDL.h>
#include <osgUtil/SceneView>
#include <osg/FrameStamp>
#include <osg/Timer>

class FrameStampExtended : public osg::FrameStamp
{
protected:

	double _deltaFrame;

public:

	void setDeltaFrame(double delta) { _deltaFrame = delta; }
	double getDeltaFrame() const { return _deltaFrame; }

};



void Assert::AssertMsg(bool condition, const std::string& msg, const std::string& function, const std::string& file, int line)
{
	if (condition)
		return;
	std::cout << "assert " << msg << " in function " << function << " in file " << file << " at line " << line << std::endl;
	assert(condition);
}

osgViewer::osgViewer() : mRunning(false), mDeltaTime(0.0), mCurrentTime(0.0), mPreviousTime(0.0), mSurface(NULL), mSceneView(NULL), mSceneRoot(NULL)
{
}
osgViewer::~osgViewer()
{
  ASSERT(false == mRunning);
  ASSERT(NULL == mSurface);
  ASSERT(NULL == mSceneView.get());
  ASSERT(NULL == mSceneRoot.get());
}
void osgViewer::Create(int width, int height, int bpp, bool fullscreen)
{
  ASSERT(false == mRunning);
  ASSERT(NULL == mSurface);
  ASSERT(NULL == mSceneView.get());
  ASSERT(NULL == mSceneRoot.get());
  SDL_Init(SDL_INIT_VIDEO);
  int FLAGS = SDL_OPENGL|SDL_DOUBLEBUF;
  if (fullscreen)
    FLAGS |= SDL_FULLSCREEN;
  mSurface = SDL_SetVideoMode(width, height, BPP, FLAGS);
  assert(mSurface && "SDL_SetVideoMode()");
  mSceneView = new osgUtil::SceneView();
  mSceneView->setDefaults();
	mSceneView->setFrameStamp(new FrameStampExtended());
  mSceneRoot = new osg::Group();
  mSceneView->setSceneData(mSceneRoot.get());
  mSceneView->setViewport(0, 0, WIDTH, HEIGHT);
  mRunning = true;
  UpdateCurrentTime();
  mPreviousTime = mCurrentTime;
  mDeltaTime = 0.0;
}
void osgViewer::Destroy()
{
  ASSERT(NULL != mSurface);
  ASSERT(NULL != mSceneView.get());
  ASSERT(NULL != mSceneRoot.get());    
  mSceneRoot = NULL;
  mSceneView = NULL;
  mSurface = NULL;
  mRunning = false;
  SDL_Quit();
}


EventList& osgViewer::GetEventList()
{
	return mEvents;
}


void osgViewer::Update()
{
	static unsigned int frameNumer = 0;
  mPreviousTime = mCurrentTime;
  UpdateCurrentTime();  
  mDeltaTime = mCurrentTime - mPreviousTime;

	mEvents.clear();
	/* Check for events */
	SDL_Event event; /* Event structure */
	while(SDL_PollEvent(&event)){  /* Loop until there are no events left on the queue */
		switch(event.type){  /* Process the appropiate event type */
		case SDL_KEYUP:  /* Handle a KEYDOWN event */
			if (event.key.keysym.sym == SDLK_ESCAPE)
				mRunning = false;
			break;
		}
		mEvents.push_back(event);
	}

	osg::FrameStamp* fss = const_cast<osg::FrameStamp*>(mSceneView->getFrameStamp());
	fss->setReferenceTime(mCurrentTime);
	fss->setFrameNumber(frameNumer);
	FrameStampExtended* fs = dynamic_cast<FrameStampExtended*>(fss);
	fs->setDeltaFrame(mDeltaTime);
	frameNumer++;
}

void osgViewer::UpdateCurrentTime()
{
  mCurrentTime = osg::Timer::instance()->tick() * osg::Timer::instance()->getSecondsPerTick();
}

void osgViewer::Render()
{
	Cull();
	Draw();
	SwapBuffer();
}

void osgViewer::Cull()
{
  mSceneView->update();
  mSceneView->cull();
}

void osgViewer::Draw()
{
  mSceneView->draw();
}

void osgViewer::SwapBuffer()
{
  SDL_GL_SwapBuffers();   
} 

bool osgViewer::GetRunning() const
{
  return mRunning;
}

osgUtil::SceneView* osgViewer::GetSceneView()
{
  ASSERT(mSceneView.get());
	return mSceneView.get();
}

osg::Group* osgViewer::GetRoot()
{
  ASSERT(mSceneRoot.get());
  return mSceneRoot.get();
}
float osgViewer::GetDeltaTime() const
{
  return mDeltaTime;
}
float osgViewer::GetCurrentTime() const
{
  return mCurrentTime;
}
float osgViewer::GetPreviousTime() const
{
  return mPreviousTime;
}
