///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#ifndef __CORE_PREVIEW_RENDERER_H
#define __CORE_PREVIEW_RENDERER_H

#include <core/Core.h>
#include <core/rendering/PluginRenderer.h>
#include <core/scene/animation/controller/Controller.h>

namespace Core {

/**
 * \brief A quick scene renderer that uses hardware accelerated OpenGL routines.
 *
 * \author Alexander Stukowski
 */
class CORE_DLLEXPORT PreviewRenderer : public PluginRenderer
{
public:
	/// Default constructor.
	PreviewRenderer(bool isLoading = false);

	///	Prepares the renderer for rendering of the given scene.
	/// Throws an exception on error. Returns false when the operation has been aborted by the user.
	virtual bool startRender(DataSet* dataset);

	/// Renders a single animation frame into the given frame buffer.
	/// Throws an exception on error. Returns false when the operation has been aborted by the user.
	virtual bool renderFrame(TimeTicks time, CameraViewDescription view, FrameBuffer* frameBuffer);

	///	Finishes the rendering pass. This is called after all animation frames have been rendered
	/// or when the rendering operation has been aborted.
	virtual void endRender();

	int antialiasingLevel() const { return _antialiasingLevel; }
	void setAntialiasingLevel(int newLevel) { _antialiasingLevel = newLevel; }

public:

	Q_PROPERTY(int antialiasingLevel READ antialiasingLevel WRITE setAntialiasingLevel)

private:

	/// The data set to be rendered.
	/// This field is only used during a render pass.
	DataSet* dataset;

	/// Controls the number of sub-pixels to render.
	PropertyField<int> _antialiasingLevel;

	/// Renders the scene objects.
	/// \return \c false if the rendering has been aborted by the user; \c true otherwise.
	bool renderObjects(TimeTicks time, const CameraViewDescription& view, int imageWidth, int imageHeight, Window3D* glcontext);

	Q_OBJECT
	DECLARE_SERIALIZABLE_PLUGIN_CLASS(PreviewRenderer)
	DECLARE_PROPERTY_FIELD(_antialiasingLevel)
};

};	// End of namespace Core

#endif // __CORE_PREVIEW_RENDERER_H
