
/*
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: menu_playback.c 2472 2007-07-09 11:20:08Z mschwerin $
 *
 */
#include "config.h"

#include <assert.h>
#include <ctype.h>
#include <math.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>

#include "codeset.h"
#include "download.h"
#include "environment.h"
#include "heap.h"
#include "i18n.h"
#include "lang.h"
#include "logger.h"
#include "meta_info.h"
#include "oxine.h"
#include "menu_base.h"
#include "menu_dvd.h"
#include "menu_filelist.h"
#include "menu_main.h"
#include "menu_playback.h"
#include "menu_playlist.h"
#include "utils.h"
#include "scheduler.h"

extern oxine_t *oxine;

#define B_FOCUS_MODE    1
#define B_FOCUS_PLAY    2
#define B_FOCUS_VOL     3
#define B_FOCUS_PARAM   4
static int playback_menu_focus_buttons = B_FOCUS_PLAY;
static bool playback_menu_show_volume = false;

/*
 * **************************************************************************
 * Callbacks of the stream parameters menu
 * **************************************************************************
 */
static int
volume_set (void *p, int value)
{
    return odk_set_stream_param (oxine->odk, ODK_PARAM_AO_VOLUME, value);
}


static int
volume_get (void *p)
{
    return odk_get_stream_param (oxine->odk, ODK_PARAM_AO_VOLUME);
}


static int
volume_mute_change (void *p, int pos)
{
    odk_set_stream_param (oxine->odk, ODK_PARAM_AO_VOLUME_MUTE, pos);

    return pos;
}


static int
audio_channel_change (void *p, int pos)
{
    odk_set_stream_param (oxine->odk, ODK_PARAM_AUDIO_CHANNEL, pos - 1);

    return pos;
}


static int
audio_offset_set (void *p, int value)
{
    return odk_set_stream_param (oxine->odk, ODK_PARAM_AUDIO_OFFSET, value);
}


static int
audio_offset_get (void *p)
{
    return odk_get_stream_param (oxine->odk, ODK_PARAM_AUDIO_OFFSET);
}


static int
spu_channel_change (void *p, int pos)
{
    odk_set_stream_param (oxine->odk, ODK_PARAM_SPU_CHANNEL, pos - 1);

    return pos;
}


static int
spu_offset_set (void *p, int value)
{
    return odk_set_stream_param (oxine->odk, ODK_PARAM_SPU_OFFSET, value);
}


static int
spu_offset_get (void *p)
{
    return odk_get_stream_param (oxine->odk, ODK_PARAM_SPU_OFFSET);
}


static int
deinterlace_change (void *p, int pos)
{
    return odk_set_stream_param (oxine->odk, ODK_PARAM_VO_DEINTERLACE, pos);
}


static int
aspect_ratio_change (void *p, int pos)
{
    int value = ODK_VO_ASPECT_RATIO_AUTO;
    switch (pos) {
    case 1:
        value = ODK_VO_ASPECT_RATIO_SQUARE;
        break;
    case 2:
        value = ODK_VO_ASPECT_RATIO_4_3;
        break;
    case 3:
        value = ODK_VO_ASPECT_RATIO_ANAMORPHIC;
        break;
    case 4:
        value = ODK_VO_ASPECT_RATIO_DVB;
        break;
    default:
        break;
    }
    odk_set_stream_param (oxine->odk, ODK_PARAM_VO_ASPECT_RATIO, value);

    return pos;
}


static int
saturation_get (void *p)
{
    int value = odk_get_stream_param (oxine->odk, ODK_PARAM_VO_SATURATION);
    return round ((double) value / (double) ODK_VO_SATURATION_STEP);
}


static int
saturation_set (void *p, int value)
{
    odk_set_stream_param (oxine->odk, ODK_PARAM_VO_SATURATION,
                          value * ODK_VO_SATURATION_STEP);
    return saturation_get (p);
}


static int
hue_get (void *p)
{
    int value = odk_get_stream_param (oxine->odk, ODK_PARAM_VO_HUE);
    return round ((double) value / (double) ODK_VO_HUE_STEP);
}


static int
hue_set (void *p, int value)
{
    return odk_set_stream_param (oxine->odk, ODK_PARAM_VO_HUE,
                                 value * ODK_VO_HUE_STEP);
    return hue_get (p);
}


static int
contrast_get (void *p)
{
    int value = odk_get_stream_param (oxine->odk, ODK_PARAM_VO_CONTRAST);
    return round ((double) value / (double) ODK_VO_CONTRAST_STEP);
}


static int
contrast_set (void *p, int value)
{
    return odk_set_stream_param (oxine->odk, ODK_PARAM_VO_CONTRAST,
                                 value * ODK_VO_CONTRAST_STEP);
    return contrast_get (p);
}


static int
brightness_get (void *p)
{
    int value = odk_get_stream_param (oxine->odk, ODK_PARAM_VO_BRIGHTNESS);
    return round ((double) value / (double) ODK_VO_BRIGHTNESS_STEP);
}


static int
brightness_set (void *p, int value)
{
    return odk_set_stream_param (oxine->odk, ODK_PARAM_VO_BRIGHTNESS,
                                 value * ODK_VO_BRIGHTNESS_STEP);
    return brightness_get (p);
}


static int
zoom_get (void *p)
{
    return odk_get_stream_param (oxine->odk, ODK_PARAM_VO_ZOOM_X);
}


static int
zoom_set (void *p, int value)
{
    odk_set_stream_param (oxine->odk, ODK_PARAM_VO_ZOOM_Y, value);
    odk_set_stream_param (oxine->odk, ODK_PARAM_VO_ZOOM_X, value);

    return zoom_get (p);
}


/*
 * **************************************************************************
 * GUI of the stream parameter menu
 * **************************************************************************
 */
#define BUTTON_Y                (odk_osd_get_height (oxine->odk) - 55)

#define LABEL_X                 (20)
#define LABEL_W                 ((odk_osd_get_width (oxine->odk) - 40) * 60 / 100)

#define EDIT_X                  (LABEL_X + LABEL_W + 10)
#define EDIT_W                  (odk_osd_get_width (oxine->odk) - 20 - EDIT_X)
#define EDIT_H                  (40)


static void
show_stream_param_slider (oxine_t * oxine, odk_stream_param_t param,
                          otk_int_get_cb_t get_cb,
                          otk_int_set_cb_t set_cb,
                          bool with_reset_button, int y,
                          char *value_format, double value_scale,
                          int def_value, int min_step,
                          int min_value, int max_value)
{
    const int al = OTK_ALIGN_LEFT | OTK_ALIGN_VCENTER;
    otk_widget_t *w;

    char *param_name = odk_get_stream_param_name (oxine->odk, param);
    w = otk_label_new (oxine->otk, LABEL_X, y + EDIT_H / 2, LABEL_W,
                       al, param_name);
    ho_free (param_name);

    w = otk_slider_new (oxine->otk, EDIT_X, y, EDIT_W, EDIT_H, 8, 36, true,
                        OTK_SLIDER_HORIZONTAL, value_format, value_scale,
                        true, with_reset_button,
                        def_value, min_step, min_value, max_value,
                        get_cb, oxine, set_cb, oxine);
    otk_widget_set_updated (w, true);
}


static void
show_stream_param_selector (oxine_t * oxine, odk_stream_param_t param,
                            otk_int_set_cb_t cb,
                            int min_value, int max_value, int y)
{
    const int al = OTK_ALIGN_LEFT | OTK_ALIGN_VCENTER;
    otk_widget_t *w;

    char *param_name = odk_get_stream_param_name (oxine->odk, param);
    w = otk_label_new (oxine->otk, LABEL_X, y + EDIT_H / 2, LABEL_W,
                       al, param_name);
    ho_free (param_name);

    w = otk_selector_new (oxine->otk, EDIT_X, y, EDIT_W, cb, oxine);
    int i = min_value;
    for (; i <= max_value; i++) {
        char *t = odk_get_stream_param_value_as_string (oxine->odk, param, i);
        otk_listentry_new (w, t, NULL, NULL, NULL, NULL, NULL, NULL);
        ho_free (t);
    }

    int position = odk_get_stream_param (oxine->odk, param) - min_value;
    otk_list_set_pos (w, position);
}


static void
stream_parameter_menu_event_handler (void *p, oxine_event_t * event)
{
    if (!is_current_menu (show_playback_settings)
        && !is_current_menu (show_playback_settings_two)) {
        return;
    }
    if (event->type != OXINE_EVENT_KEY) {
        return;
    }

    switch (event->source.key) {
    case OXINE_KEY_BACK:
        show_menu_playback (oxine);
        event->source.key = OXINE_KEY_NULL;
        break;
    default:
        break;
    }
}


void
show_playback_settings_two (void *p)
{
    /* Just to make sure that we never enter the playback menu when not in
     * playback mode just because the developer fucked up. */
    if (odk_current_is_logo_mode (oxine->odk)) {
        debug ("HIT THE PROGRAMMER ON THE HEAD!");
        show_menu_main (NULL);
        return;
    }
    else if (!odk_current_is_playback_mode (oxine->odk)) {
        return;
    }
    else if (is_current_menu (show_playback_settings_two)) {
        return;
    }

    playback_menu_focus_buttons = B_FOCUS_PARAM;
    odk_add_event_handler (oxine->odk, stream_parameter_menu_event_handler,
                           oxine, EVENT_HANDLER_PRIORITY_NORMAL);

    create_new_window (false, true);
    int y = 100;

    int sc = odk_get_stream_info (oxine->odk,
                                  ODK_STREAM_INFO_MAX_SPU_CHANNEL);

    /* Subtitle Channel */
    if (sc > 1) {
        show_stream_param_selector (oxine, ODK_PARAM_SPU_CHANNEL,
                                    spu_channel_change,
                                    SPU_CHANNEL_AUTO, sc - 1, y);
        y += 50;
    }
    /* Subtitle Offset */
    if (sc > 0) {
        show_stream_param_slider (oxine, ODK_PARAM_SPU_OFFSET,
                                  spu_offset_get, spu_offset_set,
                                  true, y, "%4.2f ms", 1.0 / 90.0,
                                  ODK_SPU_OFFSET_DEF, ODK_SPU_OFFSET_STEP,
                                  ODK_SPU_OFFSET_MIN, ODK_SPU_OFFSET_MAX);
        y += 50;
    }
    /* Zoom */
    {
        show_stream_param_slider (oxine, ODK_PARAM_VO_ZOOM_X,
                                  zoom_get, zoom_set,
                                  true, y, "%2.0f %%", 1,
                                  ODK_VO_ZOOM_DEF, ODK_VO_ZOOM_STEP,
                                  ODK_VO_ZOOM_MIN, ODK_VO_ZOOM_MAX);
        y += 50;
    }
    /* Aspect Ratio */
    {
        show_stream_param_selector (oxine, ODK_PARAM_VO_ASPECT_RATIO,
                                    aspect_ratio_change,
                                    ODK_VO_ASPECT_RATIO_AUTO,
                                    ODK_VO_ASPECT_RATIO_DVB, y);
        y += 50;
    }
    /* Deinterlace */
    {
        show_stream_param_selector (oxine, ODK_PARAM_VO_DEINTERLACE,
                                    deinterlace_change,
                                    ODK_VO_DEINTERLACE_OFF,
                                    ODK_VO_DEINTERLACE_ON, y);
        y += 50;
    }

    otk_widget_t *b;
    b = otk_text_button_new (oxine->otk, 20, BUTTON_Y, 180, 35,
                             _("Next Page"), show_playback_settings, oxine);
    otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
    otk_widget_set_focused (b, true);

    b = otk_text_button_new (oxine->otk, 210, BUTTON_Y, 180, 35,
                             _("Back"), show_menu_playback, oxine);
    otk_widget_set_alignment (b, OTK_ALIGN_CENTER);

    set_backto_menu (show_playback_settings_two, NULL);
    set_current_menu (show_playback_settings_two, NULL);

    show_user_interface (NULL);
    show_menu_background (NULL);
}


void
show_playback_settings (void *p)
{
    /* Just to make sure that we never enter the playback menu when not in
     * playback mode just because the developer fucked up. */
    if (odk_current_is_logo_mode (oxine->odk)) {
        debug ("HIT THE PROGRAMMER ON THE HEAD!");
        show_menu_main (NULL);
        return;
    }
    else if (!odk_current_is_playback_mode (oxine->odk)) {
        return;
    }
    else if (is_current_menu (show_playback_settings)) {
        return;
    }

    playback_menu_focus_buttons = B_FOCUS_PARAM;
    odk_add_event_handler (oxine->odk, stream_parameter_menu_event_handler,
                           oxine, EVENT_HANDLER_PRIORITY_NORMAL);

    create_new_window (false, true);
    int y = 100;

    int ac = odk_get_stream_info (oxine->odk,
                                  ODK_STREAM_INFO_MAX_AUDIO_CHANNEL);

    /* Volume */
    otk_widget_t *b;
    if (odk_current_has_audio (oxine->odk)) {
        show_stream_param_selector (oxine, ODK_PARAM_AO_VOLUME_MUTE,
                                    volume_mute_change, 0, 1, y);
        y += 50;
        show_stream_param_slider (oxine, ODK_PARAM_AO_VOLUME,
                                  volume_get, volume_set,
                                  true, y, "%2.0f %%", 1, 50, 5, 0, 100);
        y += 50;
    }
    /* Audio Channel */
    if (odk_current_has_audio (oxine->odk) && (ac > 1)) {
        show_stream_param_selector (oxine, ODK_PARAM_AUDIO_CHANNEL,
                                    audio_channel_change,
                                    AUDIO_CHANNEL_AUTO, ac - 1, y);
        y += 50;
    }
    /* Audio-Video Offset */
    if (odk_current_has_audio (oxine->odk) && (ac > 0)
        && odk_current_has_video (oxine->odk)) {
        show_stream_param_slider (oxine, ODK_PARAM_AUDIO_OFFSET,
                                  audio_offset_get, audio_offset_set,
                                  true, y, "%4.2f ms", 1.0 / 90.0,
                                  ODK_AUDIO_OFFSET_DEF, ODK_AUDIO_OFFSET_STEP,
                                  ODK_AUDIO_OFFSET_MIN, ODK_AUDIO_OFFSET_MAX);
        y += 50;
    }
    if ((odk_current_has_audio (oxine->odk)) && (ac > 0)) {
        y += 10;
    }
    {
        show_stream_param_slider (oxine, ODK_PARAM_VO_BRIGHTNESS,
                                  brightness_get, brightness_set,
                                  true, y, "%2.0f %%", 1, 50, 2, 0, 100);
        y += 50;
        show_stream_param_slider (oxine, ODK_PARAM_VO_CONTRAST,
                                  contrast_get, contrast_set,
                                  true, y, "%2.0f %%", 1, 50, 2, 0, 100);
        y += 50;
        show_stream_param_slider (oxine, ODK_PARAM_VO_SATURATION,
                                  saturation_get, saturation_set,
                                  true, y, "%2.0f %%", 1, 50, 2, 0, 100);
        y += 50;
        show_stream_param_slider (oxine, ODK_PARAM_VO_HUE,
                                  hue_get, hue_set,
                                  true, y, "%2.0f %%", 1, 50, 1, 0, 100);
        y += 50;
    }

    if (odk_current_has_video (oxine->odk)) {
        b = otk_text_button_new (oxine->otk, 20, BUTTON_Y, 180, 35,
                                 _("Next Page"), show_playback_settings_two,
                                 oxine);
        otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
        otk_widget_set_focused (b, true);
        b = otk_text_button_new (oxine->otk, 210, BUTTON_Y, 180, 35,
                                 _("Back"), show_menu_playback, oxine);
        otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
    }
    else {
        b = otk_text_button_new (oxine->otk, 20, BUTTON_Y, 180, 35, _("Back"),
                                 show_menu_playback, oxine);
        otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
        otk_widget_set_focused (b, true);
    }

    set_backto_menu (show_playback_settings, NULL);
    set_current_menu (show_playback_settings, NULL);

    show_user_interface (NULL);
    show_menu_background (NULL);
}

/*
 * **************************************************************************
 * Callbacks of the playing menu
 * **************************************************************************
 */

static void
hide_volume_slider_cb (void *p)
{
    playback_menu_focus_buttons = B_FOCUS_VOL;
    playback_menu_show_volume = false;
    show_menu_playback (p);
}


static void
show_volume_slider_cb (void *p)
{
    playback_menu_focus_buttons = B_FOCUS_VOL;
    playback_menu_show_volume = true;
    show_menu_playback (p);
}


static void
change_playlist_mode_cb (void *p)
{
    oxine_event_t event;
    event.type = OXINE_EVENT_KEY;
    event.source.key = OXINE_KEY_PLAYMODE;

    playback_menu_focus_buttons = B_FOCUS_MODE;

    odk_oxine_event_send (oxine->odk, &event);
}


static void
track_select_cb (void *entry_cb_data)
{
    playitem_t *item = (playitem_t *) entry_cb_data;

    playlist_play_item (oxine->current_playlist, item);

    /* Save current channel to config file. */
    if (odk_current_is_dvb (oxine->odk)) {
        xine_config_save (oxine->xine, get_file_config ());
    }
}


static void
playlist_prev_cb (void *p)
{
    oxine_event_t event;
    event.type = OXINE_EVENT_KEY;
    event.source.key = OXINE_KEY_PREV;

    odk_oxine_event_send (oxine->odk, &event);
}


static void
playlist_next_cb (void *p)
{
    oxine_event_t event;
    event.type = OXINE_EVENT_KEY;
    event.source.key = OXINE_KEY_NEXT;

    odk_oxine_event_send (oxine->odk, &event);
}


static void
media_pplay_cb (void *p)
{
    oxine_event_t event;
    event.type = OXINE_EVENT_KEY;
    event.source.key = OXINE_KEY_PPLAY;

    odk_oxine_event_send (oxine->odk, &event);
}


static void
media_stop_cb (void *p)
{
    oxine_event_t event;
    event.type = OXINE_EVENT_KEY;
    event.source.key = OXINE_KEY_STOP;

    odk_oxine_event_send (oxine->odk, &event);
}


static int
set_stream_progress (void *p, int position)
{
    odk_set_stream_param (oxine->odk, ODK_PARAM_POSITION, position);

    return position;
}


static int
get_stream_progress (void *p)
{
    return odk_get_stream_param (oxine->odk, ODK_PARAM_POSITION);
}


#ifdef HAVE_IMAGE_ROTATION
static void
rotate_right_cb (void *p)
{
    oxine_event_t event;
    event.type = OXINE_EVENT_KEY;
    event.source.key = OXINE_KEY_ROTATE_RIGHT;
    event.how = +90;

    odk_oxine_event_send (oxine->odk, &event);
}


static void
rotate_left_cb (void *p)
{
    oxine_event_t event;
    event.type = OXINE_EVENT_KEY;
    event.source.key = OXINE_KEY_ROTATE_LEFT;
    event.how = -90;

    odk_oxine_event_send (oxine->odk, &event);
}
#endif /* HAVE_IMAGE_ROTATION */


/* 
 * **************************************************************************
 * GUI-Layout of the playing menu
 * **************************************************************************
 */
otk_widget_t *label_elapsed = NULL;
otk_widget_t *label_length = NULL;

static void
set_time_string (void *p, otk_widget_t * widget)
{
    int elapsed;
    int length;

    if (odk_get_pos_length (oxine->odk, NULL, &elapsed, &length)) {
        if (length == 0) {
            length = elapsed;
        }

        elapsed /= 1000;
        length /= 1000;

        int el_h = (elapsed / 3600);
        int el_m = ((elapsed % 3600) / 60);
        int el_s = ((elapsed % 3600) % 60);

        int le_h = (length / 3600);
        int le_m = ((length % 3600) / 60);
        int le_s = ((length % 3600) % 60);

        char elapsed_str[64];
        if (le_h == 0) {
            snprintf (elapsed_str, 64, "%02d:%02d", el_m, el_s);
        }
        else {
            snprintf (elapsed_str, 64, "%d:%02d:%02d", el_h, el_m, el_s);
        }
        otk_label_set_text (label_elapsed, elapsed_str);

        char length_str[64];
        if (le_h == 0) {
            snprintf (length_str, 64, "%02d:%02d", le_m, le_s);
        }
        else {
            snprintf (length_str, 64, "%d:%02d:%02d", le_h, le_m, le_s);
        }
        otk_label_set_text (label_length, length_str);
    }
}


static void
set_pause_button (void *p, otk_widget_t * button)
{
    int speed = odk_get_stream_param (oxine->odk, ODK_PARAM_SPEED);

    if (speed == ODK_SPEED_PAUSE) {
        otk_button_set_text (button, ">");
    }
    else {
        otk_button_set_text (button, "<");
    }
}


static void
set_mode_button (void *p, otk_widget_t * button)
{
    playback_mode_t mode = playlist_get_playmode (oxine->current_playlist);

    switch (mode) {
    case PLAYBACK_MODE_REPEAT:
        otk_button_set_text (button, _("Loop Mode"));
        break;
    case PLAYBACK_MODE_RANDOM:
        otk_button_set_text (button, _("Shuffle Mode"));
        break;
    default:
        otk_button_set_text (button, _("Normal Mode"));
        break;
    }
}


static void
set_volume_button (void *p, otk_widget_t * button)
{
    bool mute = odk_get_stream_param (oxine->odk, ODK_PARAM_AO_VOLUME_MUTE);

    if (mute) {
        otk_button_set_vector (button, OSD_VECTOR_VOLUME_MUTE);
    }
    else {
        otk_button_set_vector (button, OSD_VECTOR_VOLUME);
    }
}


static void
set_title_string (void *p, otk_widget_t * widget)
{
    if (!odk_current_is_dvd (oxine->odk)) {
        return;
    }

    uint32_t cc = odk_get_stream_info (oxine->odk,
                                       ODK_STREAM_INFO_DVD_TITLE_COUNT);
    uint32_t cn = odk_get_stream_info (oxine->odk,
                                       ODK_STREAM_INFO_DVD_TITLE_NUMBER);

    char *title = ho_strdup_printf ("%02d/%02d", cn, cc);
    otk_label_set_text (widget, title);
    ho_free (title);
}


static void
set_chapter_string (void *p, otk_widget_t * widget)
{
    if (!odk_current_is_dvd (oxine->odk)
        || !odk_current_has_chapters (oxine->odk)) {
        return;
    }

    uint32_t cc = odk_get_stream_info (oxine->odk,
                                       ODK_STREAM_INFO_DVD_CHAPTER_COUNT);
    uint32_t cn = odk_get_stream_info (oxine->odk,
                                       ODK_STREAM_INFO_DVD_CHAPTER_NUMBER);

    char *chapter = ho_strdup_printf ("%02d/%02d", cn, cc);
    otk_label_set_text (widget, chapter);
    ho_free (chapter);
}


static void
show_playback_menu_file_info (oxine_t * oxine)
{
    const int a0 = OTK_ALIGN_LEFT | OTK_ALIGN_VCENTER;
    const int al = OTK_ALIGN_LEFT | OTK_ALIGN_TOP;
    const int ar = OTK_ALIGN_RIGHT | OTK_ALIGN_TOP;

    otk_widget_t *w;

    static int current_year = 0;
    if (!current_year) {
        time_t current = time (NULL);
        struct tm *brokentime = localtime (&current);
        current_year = brokentime->tm_year + 1900;
    }

    char *title = odk_get_meta_info (oxine->odk, META_INFO_TITLE);
    if (!title) {
        if (odk_current_get_title (oxine->odk)) {
            title = ho_strdup (odk_current_get_title (oxine->odk));
        }
        else {
            title = ho_strdup (_("Unknown Title"));
        }
    }

    char *artist = odk_get_meta_info (oxine->odk, META_INFO_ARTIST);
    char *album = odk_get_meta_info (oxine->odk, META_INFO_ALBUM);
    char *year = odk_get_meta_info (oxine->odk, META_INFO_YEAR);
    char *genre = odk_get_meta_info (oxine->odk, META_INFO_GENRE);
    char *track = odk_get_meta_info (oxine->odk, XINE_META_INFO_TRACK_NUMBER);

#ifdef HAVE_LIBEXIF
    const char *mrl = odk_current_get_mrl (oxine->odk);
    char *date = NULL;
    char *time = NULL;
    if (is_file_image (mrl)) {
        date = exif_info_get (EXIF_INFO_DATE, mrl);
        time = exif_info_get (EXIF_INFO_TIME, mrl);
    }
#endif /* HAVE_LIBEXIF */

    int y0 = 117;
    int x0 = (odk_osd_get_width (oxine->odk) / 2) - 180;
    int w0 = odk_osd_get_width (oxine->odk) - x0 - 20;

    {
        w = otk_label_new (oxine->otk, x0, y0, w0, a0, title);
        otk_widget_set_font (w, "sans", 40);
        otk_label_set_scrolling (w, true);
    }

    int y1 = 150;
    int x2 = (odk_osd_get_width (oxine->odk) / 2);
    int x1 = x2 - 10;
    int w1 = x1 - 220;
    int w2 = odk_osd_get_width (oxine->odk) - 20 - x2;

    if (album && (strlen (album) > 0)) {
        otk_label_new (oxine->otk, x1, y1, w1, ar, _("Album:"));
        otk_label_new (oxine->otk, x2, y1, w2, al, album);
        y1 += 40;
    }

    if (artist && (strlen (artist) > 0)) {
        otk_label_new (oxine->otk, x1, y1, w1, ar, _("Artist:"));
        otk_label_new (oxine->otk, x2, y1, w2, al, artist);
        y1 += 40;
    }

    if (year && (atoi (year) > 0) && (atoi (year) <= current_year)) {
        otk_label_new (oxine->otk, x1, y1, w1, ar, _("Year:"));
        otk_label_new (oxine->otk, x2, y1, w2, al, year);
        y1 += 40;
    }

    if (track && (strlen (track) > 0)) {
        otk_label_new (oxine->otk, x1, y1, w1, ar, _("Track:"));
        otk_label_new (oxine->otk, x2, y1, w2, al, track);
        y1 += 40;
    }

    if (genre && (strlen (genre) > 0)) {
        genre[0] = toupper (genre[0]);
        otk_label_new (oxine->otk, x1, y1, w1, ar, _("Genre:"));
        otk_label_new (oxine->otk, x2, y1, w2, al, genre);
        y1 += 40;
    }

#ifdef HAVE_LIBEXIF
    if (date && strlen (date) > 0) {
        date[0] = toupper (date[0]);
        otk_label_new (oxine->otk, x1, y1, w1, ar, _("Date:"));
        otk_label_new (oxine->otk, x2, y1, w2, al, date);
        y1 += 40;
    }

    if (time && strlen (time) > 0) {
        time[0] = toupper (time[0]);
        otk_label_new (oxine->otk, x1, y1, w1, ar, _("Time:"));
        otk_label_new (oxine->otk, x2, y1, w2, al, time);
        y1 += 40;
    }
#endif /* HAVE_LIBEXIF */

    if (odk_current_is_image (oxine->odk)) {
        int len = playlist_length (oxine->current_playlist);
        int cur = playlist_get_current_pos (oxine->current_playlist);
        char *num = ho_strdup_printf ("%02d/%02d", cur + 1, len);
        w = otk_label_new (oxine->otk, x1, y1, w1, ar, _("Image:"));
        w = otk_label_new (oxine->otk, x2, y1, w2, al, num);
        ho_free (num);
        y1 += 40;
    }

    if (odk_current_is_dvd (oxine->odk)) {
        w = otk_label_new (oxine->otk, x1, y1, w1, ar, _("Title:"));
        w = otk_label_new (oxine->otk, x2, y1, w2, al, "00/00");
        set_title_string (oxine, w);
        otk_label_set_update_cb (w, set_title_string, oxine);
        otk_widget_set_updated (w, true);
        y1 += 40;
    }

    if (odk_current_is_dvd (oxine->odk)
        && odk_current_has_chapters (oxine->odk)) {
        w = otk_label_new (oxine->otk, x1, y1, w1, ar, _("Chapter:"));
        w = otk_label_new (oxine->otk, x2, y1, w2, al, "00/00");
        set_chapter_string (oxine, w);
        otk_label_set_update_cb (w, set_chapter_string, oxine);
        otk_widget_set_updated (w, true);
        y1 += 40;
    }

    {
        y1 += 10;
        w = otk_label_new (oxine->otk, x1, y1, w1, ar, _("Length:"));
        w = otk_label_new (oxine->otk, x2, y1, w2, al, "00:00");
        label_length = w;
        y1 += 40;
    }

    {
        w = otk_label_new (oxine->otk, x1, y1, w1, ar, _("Elapsed:"));
        w = otk_label_new (oxine->otk, x2, y1, w2, al, "00:00");
        label_elapsed = w;
        set_time_string (oxine, w);
        otk_label_set_update_cb (w, set_time_string, oxine);
        otk_widget_set_updated (w, true);
        y1 += 40;
    }

    ho_free (title);
    ho_free (artist);
    ho_free (album);
    ho_free (year);
    ho_free (genre);
    ho_free (track);

#ifdef HAVE_LIBEXIF
    ho_free (date);
    ho_free (time);
#endif /* HAVE_LIBEXIF */
}


static void
show_playback_menu_channel_list (oxine_t * oxine)
{
    if (!odk_current_is_dvb (oxine->odk)) {
        return;
    }

    otk_widget_t *l = otk_list_new (oxine->otk, 220, 100, 560, 470, 30, 33,
                                    true, true, OTK_LIST_SELECTION_NONE,
                                    oxine);

    playitem_t *channel = playlist_first (oxine->current_playlist);
    while (channel) {
        otk_listentry_new (l, channel->title, track_select_cb, channel,
                           NULL, NULL, NULL, NULL);
        channel = playlist_next (oxine->current_playlist, channel);
    }

    int pos = playlist_get_current_pos (oxine->current_playlist);
    otk_list_set_selected (l, pos, 1);
    otk_list_set_pos (l, pos);
    otk_list_set_focus (l, pos);
}


static void
show_controls (bool with_volume)
{
    if (odk_current_is_dvb (oxine->odk)
        || odk_current_is_v4l (oxine->odk)) {
        return;
    }

    if (odk_current_is_image (oxine->odk)) {
        with_volume = false;
    }

    /* If we're playing a playlist and the playlist has more than one entry, a
     * lot of buttons have to be shown. */
    int length = playlist_length (oxine->current_playlist);
    bool playlist_with_more_than_one_entry = (length > 1);

    int x = 20;
    int y = odk_osd_get_height (oxine->odk) - 70;

    /* Show the play/ pause, stop, next/ prev buttons. */
    otk_widget_t *b;
    {
        b = otk_text_button_new (oxine->otk, x, y, 50, 50,
                                 ">", media_pplay_cb, oxine);
        otk_widget_set_font (b, "cetus", 20);
        otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
        set_pause_button (oxine, b);
        otk_button_set_update_cb (b, set_pause_button, oxine);
        otk_widget_set_updated (b, true);
        if (playback_menu_focus_buttons == B_FOCUS_PLAY) {
            otk_widget_set_focused (b, true);
        }
        x += 60;
    }

    y = odk_osd_get_height (oxine->odk) - 65;
    {
        b = otk_text_button_new (oxine->otk, x, y, 40, 40,
                                 "}", media_stop_cb, oxine);
        otk_widget_set_font (b, "cetus", 20);
        otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
        x += 50;
    }

    if ((playlist_with_more_than_one_entry)
        || (odk_current_is_dvd (oxine->odk)
            && odk_current_has_chapters (oxine->odk))) {
        b = otk_text_button_new (oxine->otk, x, y, 40, 40, "[",
                                 playlist_prev_cb, oxine);
        otk_widget_set_font (b, "cetus", 20);
        otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
        x += 50;

        b = otk_text_button_new (oxine->otk, x, y, 40, 40,
                                 "]", playlist_next_cb, oxine);
        otk_widget_set_font (b, "cetus", 20);
        otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
        x += 50;
    }

    {
        b = otk_text_button_new (oxine->otk, x, y, 40, 40,
                                 "{", eject_cb, oxine);
        otk_widget_set_font (b, "cetus", 20);
        otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
        x += 50;
    }

    /* Show the progress bar. */
    int w = odk_osd_get_width (oxine->odk) - 20 - x;
    if (with_volume) {
        w -= 50;
    }

    {
        b = otk_slider_new (oxine->otk, x, y, w, 40,
                            0, 20, false,
                            OTK_SLIDER_HORIZONTAL, NULL, 0,
                            false, false, 0, 1, 0, 100,
                            get_stream_progress, oxine,
                            set_stream_progress, oxine);
        otk_widget_set_updated (b, true);
    }

    /* Show the volume controls. */
    if (!with_volume) {
        return;
    }

    otk_cb_t volume_button_cb = NULL;
    if (playback_menu_show_volume) {
        volume_button_cb = hide_volume_slider_cb;
    }
    else {
        volume_button_cb = show_volume_slider_cb;
    }

    {
        x = odk_osd_get_width (oxine->odk) - 60;
        y = odk_osd_get_height (oxine->odk) - 65;
        b = otk_vector_button_new (oxine->otk, x, y, 40, 40,
                                   OSD_VECTOR_VOLUME, 30, 30,
                                   volume_button_cb, oxine);
        set_volume_button (oxine, b);
        otk_button_set_update_cb (b, set_volume_button, oxine);
        otk_widget_set_updated (b, true);
        if (playback_menu_focus_buttons == B_FOCUS_VOL) {
            otk_widget_set_focused (b, true);
        }
    }

    /* Show the volume slider. */
    if (playback_menu_show_volume) {
        b = otk_slider_new (oxine->otk, x, y - 280, 40, 270, 36, 8, true,
                            OTK_SLIDER_VERTICAL, NULL, 0,
                            true, false, 0, 5, 0, 100,
                            volume_get, oxine, volume_set, oxine);
        otk_widget_set_updated (b, true);
    }
}


void
show_menu_playback (void *p)
{
    /* Just to make sure that we never enter the playback menu when not in
     * playback mode just because the developer fucked up. */
    if (odk_current_is_logo_mode (oxine->odk)) {
        debug ("HIT THE PROGRAMMER ON THE HEAD!");
        show_menu_main (NULL);
        return;
    }
    else if (!odk_current_is_playback_mode (oxine->odk)) {
        return;
    }

    bool is_normal_title = (!odk_current_is_dvd (oxine->odk)
                            && !odk_current_is_vcd (oxine->odk)
                            && !odk_current_is_v4l (oxine->odk)
                            && !odk_current_is_dvb (oxine->odk));

    create_new_window (false, true);
    show_controls (true);

    int x = 20;
    int y = 100;

    otk_widget_t *b = NULL;
    if (odk_current_is_dvd (oxine->odk)) {
        b = otk_text_button_new (oxine->otk, x, y, 180, 35,
                                 DVD_MENU_TITLE_2, dvd_menu2_cb, oxine);
        y += 40;

        b = otk_text_button_new (oxine->otk, x, y, 180, 35,
                                 DVD_MENU_TITLE_5, dvd_menu5_cb, oxine);
        y += 40;

        b = otk_text_button_new (oxine->otk, x, y, 180, 35,
                                 DVD_MENU_TITLE_7, dvd_menu7_cb, oxine);
        y += 40;
    }

    if (is_normal_title) {
        b = otk_text_button_new (oxine->otk, x, y, 180, 35, _("Loop Mode"),
                                 change_playlist_mode_cb, oxine);
        set_mode_button (oxine, b);
        otk_button_set_update_cb (b, set_mode_button, oxine);
        otk_widget_set_updated (b, true);
        if (playback_menu_focus_buttons == B_FOCUS_MODE) {
            otk_widget_set_focused (b, true);
        }
        y += 40;
    }
    {
        b = otk_text_button_new (oxine->otk, x, y, 180, 35, _("Settings"),
                                 show_playback_settings, oxine);
        if (playback_menu_focus_buttons == B_FOCUS_PARAM) {
            otk_widget_set_focused (b, true);
        }
        y += 40;
    }

#ifdef HAVE_IMAGE_ROTATION
    if (odk_current_is_image (oxine->odk)) {
        y += 10;
        otk_text_button_new (oxine->otk, x, y, 180, 35, _("Rotate left"),
                             rotate_right_cb, oxine);
        y += 40;
        otk_text_button_new (oxine->otk, x, y, 180, 35, _("Rotate right"),
                             rotate_left_cb, oxine);
        y += 40;
    }
#endif /* HAVE_IMAGE_ROTATION */

    y += 10;
    if (is_normal_title && (oxine->current_playlist == oxine->rw_playlist)) {
        otk_text_button_new (oxine->otk, x, y, 180, 35, _("Playlist"),
                             show_menu_playlist, oxine);
        y += 40;
    }
    if (is_normal_title && (oxine->current_playlist != oxine->rw_playlist)) {
        otk_text_button_new (oxine->otk, x, y, 180, 35, _("Media Browser"),
                             show_menu_filelist, oxine);
        y += 40;
    }
    {
        otk_text_button_new (oxine->otk, x, y, 180, 35, _("Mainmenu"),
                             show_menu_main, oxine);
        y += 40;
    }

    if (odk_current_is_dvb (oxine->odk)) {
        show_playback_menu_channel_list (oxine);
    }
    else {
        show_playback_menu_file_info (oxine);
    }

#ifdef HAVE_OSD_IMAGE
    y += 10;
    if (is_normal_title 
        && odk_current_get_mrl (oxine->odk)
        && !odk_current_is_cdda (oxine->odk)
        && !odk_current_is_image (oxine->odk)) {
        char *mrl = NULL;
        playitem_t *item = playlist_get_current (oxine->current_playlist);
        const char *thumbnail_mrl = playitem_get_thumbnail (item);

        if (thumbnail_mrl) {
            if (is_downloadable (thumbnail_mrl)) {
                mrl = download_to_cache (thumbnail_mrl, NULL, false);
            }
            else {
                mrl = ho_strdup (thumbnail_mrl);
            }

            if (!file_exists (mrl)) {
                ho_free (mrl);
            }
        }

        if (mrl) {
            int hi = odk_osd_get_height (oxine->odk) - 85 - y;
            int yi = y + (hi / 2);

            otk_image_new (oxine->otk, 110, yi, 180, hi, mrl,
                           ODK_ALIGN_CENTER | ODK_ALIGN_VCENTER, true);
        }

        ho_free (mrl);
    }
#endif

    set_backto_menu (show_menu_playback, NULL);
    set_current_menu (show_menu_playback, NULL);

    show_user_interface (NULL);
    show_menu_background (NULL);
}


void
show_playback_controls (void *p)
{
    /* Just to make sure that we never enter the playback menu when not in
     * playback mode just because the developer fucked up. */
    if (odk_current_is_logo_mode (oxine->odk)) {
        debug ("HIT THE PROGRAMMER ON THE HEAD!");
        show_menu_main (NULL);
        return;
    }
    else if (!odk_current_is_playback_mode (oxine->odk)) {
        return;
    }

    create_new_window (false, false);

    playback_menu_focus_buttons = B_FOCUS_PLAY;
    show_controls (false);

    oxine->playback_controls_are_visible = true;

    set_current_menu (show_playback_controls, NULL);

    show_user_interface (NULL);
    show_menu_background (NULL);

    schedule_hide_gui_job (6 * 1000);
}


/*
 * **************************************************************************
 * GUI-Layout of the info menu
 * **************************************************************************
 */
void
show_playback_info (void *p)
{
    const int al = OTK_ALIGN_LEFT | OTK_ALIGN_VCENTER;
    const int ar = OTK_ALIGN_RIGHT | OTK_ALIGN_VCENTER;
    const int ac = OTK_ALIGN_CENTER | OTK_ALIGN_VCENTER;

    /* Just to make sure that we never enter the playback menu when not in
     * playback mode just because the developer fucked up. */
    if (odk_current_is_logo_mode (oxine->odk)) {
        debug ("HIT THE PROGRAMMER ON THE HEAD!");
        show_menu_main (NULL);
        return;
    }
    else if (!odk_current_is_playback_mode (oxine->odk)) {
        return;
    }

    create_new_window (false, false);

    static int current_year = 0;
    if (!current_year) {
        time_t current = time (NULL);
        struct tm *brokentime = localtime (&current);
        current_year = brokentime->tm_year + 1900;
    }

    char *title = odk_get_meta_info (oxine->odk, META_INFO_TITLE);
    if (!title) {
        if (odk_current_get_title (oxine->odk)) {
            title = ho_strdup (odk_current_get_title (oxine->odk));
        }
        else {
            title = ho_strdup (_("Unknown Title"));
        }
    }
    char *artist = odk_get_meta_info (oxine->odk, META_INFO_ARTIST);
    char *album = odk_get_meta_info (oxine->odk, META_INFO_ALBUM);
    char *year = odk_get_meta_info (oxine->odk, META_INFO_YEAR);
    char *genre = odk_get_meta_info (oxine->odk, META_INFO_GENRE);
    char *track = odk_get_meta_info (oxine->odk, XINE_META_INFO_TRACK_NUMBER);

    int y = odk_osd_get_height (oxine->odk) / 2;
    int x2 = odk_osd_get_width (oxine->odk) / 2;
    int x1 = x2 - 10;
    int w0 = odk_osd_get_width (oxine->odk) - 40;
    int w1 = x1 - 20;
    int w2 = odk_osd_get_width (oxine->odk) - 20 - x2;

    if (album && (strlen (album) > 0)) {
        y -= 20;
    }
    if (artist && (strlen (artist) > 0)) {
        y -= 20;
    }
    if (year && (atoi (year) > 0) && (atoi (year) <= current_year)) {
        y -= 20;
    }
    if (track && (strlen (track) > 0)) {
        y -= 20;
    }
    if (genre && (strlen (genre) > 0)) {
        y -= 20;
    }

    otk_widget_t *l;
    l = otk_label_new (oxine->otk, x1, y, w0, ac, title);
    otk_widget_set_font (l, "sans", 40);
    otk_label_set_scrolling (l, true);
    y += 50;

    if (album && strlen (album) > 0) {
        otk_label_new (oxine->otk, x1, y, w1, ar, _("Album:"));
        otk_label_new (oxine->otk, x2, y, w2, al, album);
        y += 40;
    }

    if (artist && strlen (artist) > 0) {
        otk_label_new (oxine->otk, x1, y, w1, ar, _("Artist:"));
        otk_label_new (oxine->otk, x2, y, w2, al, artist);
        y += 40;
    }

    if (year && atoi (year) > 0 && atoi (year) <= current_year) {
        otk_label_new (oxine->otk, x1, y, w1, ar, _("Year:"));
        otk_label_new (oxine->otk, x2, y, w2, al, year);
        y += 40;
    }

    if (track && strlen (track) > 0) {
        otk_label_new (oxine->otk, x1, y, w1, ar, _("Track:"));
        otk_label_new (oxine->otk, x2, y, w2, al, track);
        y += 40;
    }

    if (genre && strlen (genre) > 0) {
        genre[0] = toupper (genre[0]);
        otk_label_new (oxine->otk, x1, y, w1, ar, _("Genre:"));
        otk_label_new (oxine->otk, x2, y, w2, al, genre);
        y += 40;
    }

    ho_free (title);
    ho_free (artist);
    ho_free (album);
    ho_free (year);
    ho_free (genre);
    ho_free (track);

    set_current_menu (show_playback_info, NULL);
    otk_draw (oxine->otk);

    schedule_hide_gui_job (10 * 1000);
}
