/***************************************************************************
 $RCSfile$
                             -------------------
    cvs         : $Id: transaction.cpp 396 2006-05-29 17:00:16Z martin $
    begin       : Mon Mar 01 2004
    copyright   : (C) 2004 by Martin Preuss
    email       : martin@libchipcard.de

 ***************************************************************************
 *          Please see toplevel file COPYING for license details           *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif


#include "transaction.h"
#include "app.h"

#include <stdlib.h>
#include <gwenhywfar/md.h>
#include <gwenhywfar/buffer.h>
#include <gwenhywfar/text.h>
#include <gwenhywfar/debug.h>
#include <qstring.h>

#if !defined(__GNUC__) && defined(WIN32)
// This is MSVC compiler which doesnt have snprintf()
# define snprintf _snprintf
#endif

Transaction::Transaction()
:_valutaDate(0)
,_date(0)
,_value(0)
,_transactionCode(0)
,_textKey(51)
,_transactionId(0)
,_modified(false) {

}



Transaction::Transaction(GWEN_DB_NODE *db)
:_valutaDate(0)
,_date(0)
,_value(0)
,_transactionCode(0)
,_textKey(51)
,_transactionId(0)
,_modified(false) {
  if (!fromDb(db)) {
    DBG_ERROR(0, "Error loading transaction from DB");
    abort();
  }
}



Transaction::Transaction(const AB_TRANSACTION *t)
:_valutaDate(0)
,_date(0)
,_value(0)
,_transactionCode(0)
,_textKey(51)
,_transactionId(0)
,_modified(false){
  const char *p;
  const GWEN_STRINGLIST *sl;
  const GWEN_TIME *ti;
  const AB_VALUE *v;

  assert(t);

  p=AB_Transaction_GetLocalCountry(t);
  if (p)
    _localCountry=p;
  p=AB_Transaction_GetLocalBankCode(t);
  if (p)
    _localBankCode=p;
  p=AB_Transaction_GetLocalAccountNumber(t);
  if (p)
    _localAccountNumber=p;
  p=AB_Transaction_GetLocalSuffix(t);
  if (p)
    _localSuffix=p;
  p=AB_Transaction_GetLocalName(t);
  if (p)
    _localOwnerName=p;

  p=AB_Transaction_GetRemoteCountry(t);
  if (p)
    _remoteCountry=p;
  p=AB_Transaction_GetRemoteBankCode(t);
  if (p)
    _remoteBankCode=p;
  p=AB_Transaction_GetRemoteBankName(t);
  if (p)
    _remoteBankName=p;
  p=AB_Transaction_GetRemoteBankLocation(t);
  if (p)
    _remoteBankLocation=p;
  p=AB_Transaction_GetRemoteAccountNumber(t);
  if (p)
    _remoteAccountNumber=p;
  p=AB_Transaction_GetRemoteSuffix(t);
  if (p)
    _remoteSuffix=p;
  sl=AB_Transaction_GetRemoteName(t);
  if (sl) {
    GWEN_STRINGLISTENTRY *se;

    se=GWEN_StringList_FirstEntry(sl);
    while(se) {
      p=GWEN_StringListEntry_Data(se);
      assert(p);
      _remoteName.push_back(p);
      se=GWEN_StringListEntry_Next(se);
    } /* while */
  }
  sl=AB_Transaction_GetPurpose(t);
  if (sl) {
    GWEN_STRINGLISTENTRY *se;

    se=GWEN_StringList_FirstEntry(sl);
    while(se) {
      p=GWEN_StringListEntry_Data(se);
      assert(p);
      _purpose.push_back(p);
      se=GWEN_StringListEntry_Next(se);
    } /* while */
  }
  v=AB_Transaction_GetValue(t);
  if (v)
    _value=AB_Value_dup(v);

  sl=AB_Transaction_GetCategory(t);
  if (sl) {
    GWEN_STRINGLISTENTRY *se;

    se=GWEN_StringList_FirstEntry(sl);
    while(se) {
      p=GWEN_StringListEntry_Data(se);
      assert(p);
      //_categories.push_back(p);  @TODO: Store category somewhere...
      se=GWEN_StringListEntry_Next(se);
    } /* while */
  }

  ti=AB_Transaction_GetValutaDate(t);
  if (ti)
    setValutaDate(ti);

  ti=AB_Transaction_GetDate(t);
  if (ti)
    setDate(ti);

  _textKey=AB_Transaction_GetTextKey(t);
  p=AB_Transaction_GetFiId(t);
  if (p)
    _fiid=p;

  p=AB_Transaction_GetTransactionKey(t);
  if (p)
    _transactionKey=p;
  _transactionCode=AB_Transaction_GetTransactionCode(t);
  p=AB_Transaction_GetCustomerReference(t);
  if (p)
    _customerReference=p;
  p=AB_Transaction_GetBankReference(t);
  if (p)
    _bankReference=p;
  p=AB_Transaction_GetTransactionText(t);
  if (p)
    _transactionText=p;
  p=AB_Transaction_GetPrimanota(t);
  if (p)
    _primanota=p;
  _transactionId=AB_Transaction_GetUniqueId(t);
}



AB_TRANSACTION *Transaction::toBankingTransaction() {
  AB_TRANSACTION *t;

  t=AB_Transaction_new();
  if (!_localCountry.empty())
    AB_Transaction_SetLocalCountry(t, _localCountry.c_str());
  if (!_localBankCode.empty())
    AB_Transaction_SetLocalBankCode(t, _localBankCode.c_str());
  if (!_localAccountNumber.empty())
    AB_Transaction_SetLocalAccountNumber(t, _localAccountNumber.c_str());
  if (!_localSuffix.empty())
    AB_Transaction_SetLocalSuffix(t, _localSuffix.c_str());
  if (!_localOwnerName.empty())
    AB_Transaction_SetLocalName(t, _localOwnerName.c_str());

  if (!_remoteCountry.empty())
    AB_Transaction_SetRemoteCountry(t, _remoteCountry.c_str());
  if (!_remoteBankCode.empty())
    AB_Transaction_SetRemoteBankCode(t, _remoteBankCode.c_str());
  if (!_remoteBankName.empty())
    AB_Transaction_SetRemoteBankName(t, _remoteBankName.c_str());
  if (!_remoteBankLocation.empty())
    AB_Transaction_SetRemoteBankLocation(t, _remoteBankLocation.c_str());
  if (!_remoteAccountNumber.empty())
    AB_Transaction_SetRemoteAccountNumber(t, _remoteAccountNumber.c_str());
  if (!_remoteName.empty()) {
    std::list<std::string>::iterator sit;

    for (sit=_remoteName.begin();
         sit!=_remoteName.end();
         sit++) {
      if (!(*sit).empty())
        AB_Transaction_AddRemoteName(t, (*sit).c_str(), 0);
    }
  }

#if 0
  if (!_categories.empty()) {
    std::list<std::string>::iterator sit;

    for (sit=_categories.begin();
         sit!=_categories.end();
         sit++) {
      if (!(*sit).empty())
        AB_Transaction_AddCategory(t, (*sit).c_str(), 0);
    }
  }
#endif

  if (_valutaDate)
    AB_Transaction_SetValutaDate(t, _valutaDate);
  if (_date)
    AB_Transaction_SetDate(t, _date);
  if (_value)
    AB_Transaction_SetValue(t, _value);
  if (!_transactionKey.empty())
    AB_Transaction_SetTransactionKey(t, _transactionKey.c_str());
  if (!_customerReference.empty())
    AB_Transaction_SetCustomerReference(t, _customerReference.c_str());
  if (!_bankReference.empty())
    AB_Transaction_SetBankReference(t, _bankReference.c_str());
  AB_Transaction_SetTransactionCode(t, _transactionCode);
  if (!_transactionText.empty())
    AB_Transaction_SetTransactionText(t, _transactionText.c_str());
  if (!_primanota.empty())
    AB_Transaction_SetPrimanota(t, _primanota.c_str());
  if (!_fiid.empty())
    AB_Transaction_SetFiId(t, _fiid.c_str());
  if (!_purpose.empty()) {
    std::list<std::string>::iterator sit;

    for (sit=_purpose.begin();
         sit!=_purpose.end();
         sit++) {
      if (!(*sit).empty())
        AB_Transaction_AddPurpose(t, (*sit).c_str(), 0);
    }
  }
  AB_Transaction_SetTextKey(t, _textKey);
  AB_Transaction_SetUniqueId(t, _transactionId);

  return t;
}



Transaction::Transaction(const Transaction &t)
:_valutaDate(0)
,_date(0)
,_value(0)
,_transactionCode(0)
,_textKey(51)
,_transactionId(0)
,_modified(false){
  _localCountry=t._localCountry;
  _localBankCode=t._localBankCode;
  _localAccountNumber=t._localAccountNumber;
  _localSuffix=t._localSuffix;
  _localOwnerName=t._localOwnerName;

  _remoteCountry=t._remoteCountry;
  _remoteBankCode=t._remoteBankCode;
  _remoteBankName=t._remoteBankName;
  _remoteBankLocation=t._remoteBankLocation;
  _remoteAccountNumber=t._remoteAccountNumber;
  _remoteSuffix=t._remoteSuffix;
  _remoteName=t._remoteName;
  _category=t._category;
  _payee=t._payee;

  if (t._valutaDate)
    _valutaDate=GWEN_Time_dup(t._valutaDate);
  if (t._date)
    _date=GWEN_Time_dup(t._date);

  if (t._value)
    _value=AB_Value_dup(t._value);

  _transactionKey=t._transactionKey;
  _customerReference=t._customerReference;
  _bankReference=t._bankReference;
  _transactionCode=t._transactionCode;
  _transactionText=t._transactionText;
  _primanota=t._primanota;
  _purpose=t._purpose;
  _textKey=t._textKey;
  _transactionId=t._transactionId;
  _fiid=t._fiid;

  _modified=false;
  _hash.erase();
}



Transaction::~Transaction(){
  GWEN_Time_free(_valutaDate);
  GWEN_Time_free(_date);
  AB_Value_free(_value);
}



bool Transaction::operator<(const Transaction &t){
  const GWEN_TIME *ti1;
  const GWEN_TIME *ti2;

  ti1=getDate();
  ti2=t.getDate();
  if (ti1 && ti2) {
    return (GWEN_Time_Diff(ti1, ti2)<0.0);
  }

  ti1=getValutaDate();
  ti2=t.getValutaDate();
  if (ti1 && ti2) {
    return (GWEN_Time_Diff(ti1, ti2)<0.0);
  }
  else {
    if (ti2)
      return true;
    return false;
  }

}



const std::string &Transaction::getLocalCountry() const{
  return _localCountry;
}



void Transaction::setLocalCountry(const std::string &s){
  _localCountry=s;
  _hash.erase();
  _modified=true;
}



const std::string &Transaction::getLocalBankCode() const{
  return _localBankCode;
}



void Transaction::setLocalBankCode(const std::string &s){
  _localBankCode=s;
  _hash.erase();
  _modified=true;
}



const std::string &Transaction::getLocalAccountNumber() const{
  return _localAccountNumber;
}



void Transaction::setLocalAccountNumber(const std::string &s){
  _localAccountNumber=s;
  _hash.erase();
  _modified=true;
}



const std::string &Transaction::getLocalSuffix() const{
  return _localSuffix;
}



void Transaction::setLocalSuffix(const std::string &s){
  _localSuffix=s;
  _hash.erase();
  _modified=true;
}



const std::string &Transaction::getLocalName() const{
  return _localOwnerName;
}



void Transaction::setLocalName(const std::string &s){
  _localOwnerName=s;
  _hash.erase();
  _modified=true;
}



const std::string &Transaction::getRemoteCountry() const{
  return _remoteCountry;
}



void Transaction::setRemoteCountry(const std::string &s){
  _remoteCountry=s;
  _hash.erase();
  _modified=true;
}



const std::string &Transaction::getRemoteBankCode() const{
  return _remoteBankCode;
}



void Transaction::setRemoteBankCode(const std::string &s){
  _remoteBankCode=s;
  _hash.erase();
  _modified=true;
}



const std::string &Transaction::getRemoteBankName() const{
  return _remoteBankName;
}



void Transaction::setRemoteBankName(const std::string &s){
  _remoteBankName=s;
  _hash.erase();
  _modified=true;
}



const std::string &Transaction::getRemoteBankLocation() const{
  return _remoteBankLocation;
}



void Transaction::setRemoteBankLocation(const std::string &s){
  _remoteBankLocation=s;
  _hash.erase();
  _modified=true;
}



const std::string &Transaction::getRemoteIban() const{
  return _remoteIban;
}



void Transaction::setRemoteIban(const std::string &s){
  _remoteIban=s;
  _hash.erase();
  _modified=true;
}



const std::string &Transaction::getRemoteAccountNumber() const{
  return _remoteAccountNumber;
}



void Transaction::setRemoteAccountNumber(const std::string &s){
  _remoteAccountNumber=s;
  _hash.erase();
  _modified=true;
}



const std::string &Transaction::getRemoteSuffix() const{
  return _remoteSuffix;
}



void Transaction::setRemoteSuffix(const std::string &s){
  _remoteSuffix=s;
  _hash.erase();
  _modified=true;
}



const std::list<std::string> &Transaction::getRemoteName() const{
  return _remoteName;
}



void Transaction::clearRemoteName(){
  _remoteName.clear();
  _hash.erase();
  _modified=true;
}



void Transaction::addRemoteName(const std::string &s){
  _remoteName.push_back(s);
  _hash.erase();
  _modified=true;
}



const GWEN_TIME *Transaction::getValutaDate() const{
  return _valutaDate;
}



void Transaction::setValutaDate(const GWEN_TIME *d){
  GWEN_Time_free(_valutaDate);
  _valutaDate=_transformedDate(d);
  _hash.erase();
  _modified=true;
}



const GWEN_TIME *Transaction::getDate() const{
  return _date;
}



void Transaction::setDate(const GWEN_TIME *d){
  GWEN_Time_free(_date);
  _date=_transformedDate(d);
  _hash.erase();
  _modified=true;
}



const AB_VALUE *Transaction::getValue() const{
  return _value;
}



void Transaction::setValue(const AB_VALUE *v){
  assert(v);
  AB_Value_free(_value);
  _value=AB_Value_dup(v);
  _hash.erase();
  _modified=true;
}



const std::string &Transaction::getFiId() const{
  return _fiid;
}



void Transaction::setFiId(const std::string &s){
  _fiid=s;
  _hash.erase();
  _modified=true;
}



const std::string &Transaction::getTransactionKey() const{
  return _transactionKey;
}



void Transaction::setTransactionKey(const std::string &s){
  _transactionKey=s;
  _hash.erase();
  _modified=true;
}



const std::string &Transaction::getCustomerReference() const{
  return _customerReference;
}



void Transaction::setCustomerReference(const std::string &s){
  _customerReference=s;
  _hash.erase();
  _modified=true;
}



const std::string &Transaction::getBankReference() const{
  return _bankReference;
}



void Transaction::setBankReference(const std::string &s){
  _bankReference=s;
  _hash.erase();
  _modified=true;
}



int Transaction::getTransactionCode() const{
  return _transactionCode;
}



void Transaction::setTransactionCode(int i){
  _transactionCode=i;
  _hash.erase();
  _modified=true;
}



const std::string &Transaction::getTransactionText() const{
  return _transactionText;
}



void Transaction::setTransactionText(const std::string &s){
  _transactionText=s;
  _hash.erase();
  _modified=true;
}



const std::string &Transaction::getPrimanota() const{
  return _primanota;
}



void Transaction::setPrimanota(const std::string &s){
  _primanota=s;
  _hash.erase();
  _modified=true;
}


int Transaction::getTextKey() const{
  return _textKey;
}



void Transaction::setTextKey(int i){
  _textKey=i;
  _hash.erase();
  _modified=true;
}



const std::string &Transaction::getHash(){
  std::string s;

  if (_hash.empty()) {
    _hash=_makeHash();
  }
  return _hash;
}



std::string Transaction::_makeHash() {
  std::string s;
  std::list<std::string>::iterator it;
  char buffer[32];
  GWEN_BUFFER *dbuf;
  char hbuf[32];
  unsigned int hsize;

  s=_localCountry;
  s+="#";
  s+=_localBankCode;
  s+="#";
  s+=_localAccountNumber;
  s+="#";
  s+=_localSuffix;
  s+="#";
  s+=_localOwnerName;
  s+="#";
  s+=_remoteCountry;
  s+="#";
  s+=_remoteBankCode;
  s+="#";
  s+=_remoteAccountNumber;
  s+="#";
  s+=_remoteSuffix;
  s+="#";
  for (it=_remoteName.begin(); it!=_remoteName.end(); it++){
    s+=*it;
    s+=" ";
  }
  s+="#";

  snprintf(buffer, sizeof(buffer), "%i", _textKey);
  s+=buffer;
  s+="#";

  if (_valutaDate) {
    snprintf(buffer, sizeof(buffer), GWEN_TYPE_TMPL_UINT32,
	     (unsigned int)GWEN_Time_Seconds(_valutaDate));
    s+=buffer;
  }
  s+="#";
  if (_date) {
    snprintf(buffer, sizeof(buffer), GWEN_TYPE_TMPL_UINT32,
	     (unsigned int)GWEN_Time_Seconds(_date));
    s+=buffer;
  }
  s+="#";
  if (_value) {
    if (AB_Value_IsValid(_value)) {
      const char *p;

      snprintf(buffer, sizeof(buffer), "%lf",
	       AB_Value_GetValue(_value));
      s+=buffer;
      s+=" ";
      p=AB_Value_GetCurrency(_value);
      if (p)
        s+=p;
    }
  }
  s+="#";
  s+=_transactionKey;
  s+="#";
  s+=_customerReference;
  s+="#";
  s+=_bankReference;
  s+="#";
  snprintf(buffer, sizeof(buffer), "%d", _transactionCode);
  s+=buffer;
  s+="#";
  s+=_transactionText;
  s+="#";
  s+=_primanota;
  s+="#";
  s+=_fiid;
  s+="#";
  for (it=_purpose.begin(); it!=_purpose.end(); it++){
    s+=*it;
    s+=" ";
  }

  /* s now contains all important data */
  dbuf=GWEN_Buffer_new(0, s.length(), 0, 1);
  GWEN_Buffer_AppendString(dbuf, s.c_str());
  /* remove duplicate blanks */
  GWEN_Text_CondenseBuffer(dbuf);

  /* now hash the data */
  hsize=sizeof(hbuf);
  if (GWEN_MD_Hash("rmd160",
                   GWEN_Buffer_GetStart(dbuf),
                   GWEN_Buffer_GetUsedBytes(dbuf),
                   hbuf,
                   &hsize)) {
    DBG_ERROR(0, "Could not hash transaction data");
    GWEN_Buffer_free(dbuf);
    return "";
  }
  GWEN_Buffer_Reset(dbuf);
  if (GWEN_Text_ToHexBuffer(hbuf, hsize, dbuf, 0, 0, 0)) {
    DBG_ERROR(0, "Error converting to hex");
    GWEN_Buffer_free(dbuf);
    return "";
  }
  s=std::string(GWEN_Buffer_GetStart(dbuf));
  GWEN_Buffer_free(dbuf);

  return s;
}


GWEN_TIME *Transaction::_transformedDate(const GWEN_TIME *ti) {
  if (ti) {
    GWEN_BUFFER *tbuf;
    GWEN_TIME *tnew;
    std::string s;

    tbuf=GWEN_Buffer_new(0, 32, 0, 1);
    if (GWEN_Time_toUtcString(ti, "YYYYMMDD", tbuf)) {
      DBG_ERROR(0, "Could not store date");
      GWEN_Buffer_free(tbuf);
      return 0;
    }

    GWEN_Buffer_AppendString(tbuf, "-12:00");
    tnew=GWEN_Time_fromUtcString(GWEN_Buffer_GetStart(tbuf),
                                 "YYYYMMDD-hh:mm");
    assert(tnew);
    GWEN_Buffer_free(tbuf);
    return tnew;
  }

  return 0;
}



bool Transaction::toDb(GWEN_DB_NODE *db) const {
  std::list<std::string>::const_iterator it;

  GWEN_DB_SetIntValue(db, GWEN_DB_FLAGS_OVERWRITE_VARS,
                      "transactionId", _transactionId);


  if (_date) {
    GWEN_BUFFER *tbuf;

    tbuf=GWEN_Buffer_new(0, 32, 0, 1);
    if (GWEN_Time_toUtcString(_date, "YYYYMMDD", tbuf)) {
      DBG_ERROR(0, "Could not store date");
      GWEN_Buffer_free(tbuf);
      return false;
    }
    GWEN_DB_SetCharValue(db, GWEN_DB_FLAGS_OVERWRITE_VARS,
                         "date", GWEN_Buffer_GetStart(tbuf));
    GWEN_Buffer_free(tbuf);
  }

  if (_valutaDate) {
    GWEN_BUFFER *tbuf;

    tbuf=GWEN_Buffer_new(0, 32, 0, 1);
    if (GWEN_Time_toUtcString(_valutaDate, "YYYYMMDD", tbuf)) {
      DBG_ERROR(0, "Could not store valuta date");
      GWEN_Buffer_free(tbuf);
      return false;
    }
    GWEN_DB_SetCharValue(db, GWEN_DB_FLAGS_OVERWRITE_VARS,
                         "valutaDate", GWEN_Buffer_GetStart(tbuf));
    GWEN_Buffer_free(tbuf);
  }

  if (_value) {
    GWEN_DB_NODE *dbV;

    dbV=GWEN_DB_GetGroup(db, GWEN_DB_FLAGS_OVERWRITE_GROUPS,
                         "value");
    assert(dbV);
    if (AB_Value_toDb(_value, dbV)) {
      return false;
    }
  }

  GWEN_DB_SetIntValue(db, GWEN_DB_FLAGS_OVERWRITE_VARS,
                      "textkey", _textKey);
  GWEN_DB_SetCharValue(db, GWEN_DB_FLAGS_OVERWRITE_VARS,
                       "key", _transactionKey.c_str());
  GWEN_DB_SetCharValue(db, GWEN_DB_FLAGS_OVERWRITE_VARS,
                       "text", _transactionText.c_str());
  GWEN_DB_SetCharValue(db, GWEN_DB_FLAGS_OVERWRITE_VARS,
                       "custref", _customerReference.c_str());
  GWEN_DB_SetCharValue(db, GWEN_DB_FLAGS_OVERWRITE_VARS,
                       "bankref", _bankReference.c_str());
  GWEN_DB_SetCharValue(db, GWEN_DB_FLAGS_OVERWRITE_VARS,
                       "primanota", _primanota.c_str());
  GWEN_DB_SetCharValue(db, GWEN_DB_FLAGS_OVERWRITE_VARS,
                       "fiid", _fiid.c_str());
  GWEN_DB_SetCharValue(db, GWEN_DB_FLAGS_OVERWRITE_VARS,
                       "payee", _payee.c_str());

  if (!_localCountry.empty())
    GWEN_DB_SetCharValue(db, GWEN_DB_FLAGS_OVERWRITE_VARS,
                         "localCountry", _localCountry.c_str());
  if (!_localBankCode.empty())
    GWEN_DB_SetCharValue(db, GWEN_DB_FLAGS_OVERWRITE_VARS,
                         "localBankCode", _localBankCode.c_str());
  if (!_localAccountNumber.empty())
    GWEN_DB_SetCharValue(db, GWEN_DB_FLAGS_OVERWRITE_VARS,
                         "localAccountNumber",
                         _localAccountNumber.c_str());
  if (!_localSuffix.empty())
    GWEN_DB_SetCharValue(db, GWEN_DB_FLAGS_OVERWRITE_VARS,
                         "localSuffix", _localSuffix.c_str());

  if (!_remoteCountry.empty())
    GWEN_DB_SetCharValue(db, GWEN_DB_FLAGS_OVERWRITE_VARS,
                         "remoteCountry", _remoteCountry.c_str());
  if (!_remoteBankCode.empty())
    GWEN_DB_SetCharValue(db, GWEN_DB_FLAGS_OVERWRITE_VARS,
                         "remoteBankCode", _remoteBankCode.c_str());
  if (!_remoteAccountNumber.empty())
    GWEN_DB_SetCharValue(db, GWEN_DB_FLAGS_OVERWRITE_VARS,
                         "remoteAccountNumber", _remoteAccountNumber.c_str());
  if (!_remoteIban.empty())
    GWEN_DB_SetCharValue(db, GWEN_DB_FLAGS_OVERWRITE_VARS,
                         "remoteIban", _remoteIban.c_str());
  if (!_remoteBankName.empty())
    GWEN_DB_SetCharValue(db, GWEN_DB_FLAGS_OVERWRITE_VARS,
                         "remoteBankName", _remoteBankName.c_str());
  if (!_remoteBankLocation.empty())
    GWEN_DB_SetCharValue(db, GWEN_DB_FLAGS_OVERWRITE_VARS,
                         "remoteBankLocation", _remoteBankLocation.c_str());
  if (!_remoteSuffix.empty())
    GWEN_DB_SetCharValue(db, GWEN_DB_FLAGS_OVERWRITE_VARS,
                         "remoteSuffix", _remoteSuffix.c_str());
  GWEN_DB_DeleteVar(db, "remoteName");
  for (it=_remoteName.begin(); it!=_remoteName.end(); it++)
    GWEN_DB_SetCharValue(db, GWEN_DB_FLAGS_DEFAULT,
                         "remoteName", (*it).c_str());

  GWEN_DB_DeleteVar(db, "purpose");
  for (it=_purpose.begin(); it!=_purpose.end(); it++)
    GWEN_DB_SetCharValue(db, GWEN_DB_FLAGS_DEFAULT,
                         "purpose", (*it).c_str());

  if (!_category.empty())
    GWEN_DB_SetCharValue(db, GWEN_DB_FLAGS_OVERWRITE_VARS,
                         "category", _category.c_str());

  return true;
}



bool Transaction::fromDb(GWEN_DB_NODE *db) {
  const char *p;
  int i;
  GWEN_DB_NODE *dbT;

  _transactionId=GWEN_DB_GetIntValue(db, "transactionId", 0, 0);

  p=GWEN_DB_GetCharValue(db, "date", 0, 0);
  if (p) {
    std::string s;
    GWEN_TIME *ti;

    s=p;
    s+="-12:00";

    ti=GWEN_Time_fromUtcString(s.c_str(), "YYYYMMDD-hh:mm");
    assert(ti);
    _date=ti;
  }
  else {
    dbT=GWEN_DB_GetGroup(db, GWEN_PATH_FLAGS_NAMEMUSTEXIST, "date");
    if (dbT) {
      GWEN_TIME *ti;

      ti=GWEN_Time_fromDb(dbT);
      if (!ti) {
        DBG_ERROR(0, "Bad format of date");
        return false;
      }
      _date=ti;
    }
  }

  p=GWEN_DB_GetCharValue(db, "valutaDate", 0, 0);
  if (p) {
    std::string s;
    GWEN_TIME *ti;

    s=p;
    s+="-12:00";

    ti=GWEN_Time_fromUtcString(s.c_str(), "YYYYMMDD-hh:mm");
    assert(ti);
    _valutaDate=ti;
  }
  else {
    dbT=GWEN_DB_GetGroup(db, GWEN_PATH_FLAGS_NAMEMUSTEXIST, "valutaDate");
    if (dbT) {
      GWEN_TIME *ti;

      ti=GWEN_Time_fromDb(dbT);
      if (!ti) {
        DBG_ERROR(0, "Bad format of date");
        return false;
      }
      _valutaDate=ti;
    }
  }

  dbT=GWEN_DB_GetGroup(db, GWEN_PATH_FLAGS_NAMEMUSTEXIST, "value");
  if (dbT)
    _value=AB_Value_fromDb(dbT);

  _textKey=GWEN_DB_GetIntValue(db, "textkey", 0, 0);
  _transactionKey=GWEN_DB_GetCharValue(db, "key", 0, "");
  _transactionText=GWEN_DB_GetCharValue(db, "text", 0, "");
  _customerReference=GWEN_DB_GetCharValue(db, "custref", 0, "");
  _bankReference=GWEN_DB_GetCharValue(db, "bankref", 0, "");
  _primanota=GWEN_DB_GetCharValue(db, "primanota", 0, "");
  _fiid=GWEN_DB_GetCharValue(db, "fiid", 0, "");
  _payee=GWEN_DB_GetCharValue(db, "payee", 0, "");

  _localCountry=GWEN_DB_GetCharValue(db, "localCountry", 0, "");
  _localBankCode=GWEN_DB_GetCharValue(db, "localBankCode", 0, "");
  _localAccountNumber=GWEN_DB_GetCharValue(db, "localAccountNumber", 0, "");

  _remoteCountry=GWEN_DB_GetCharValue(db, "remoteCountry", 0, "");
  _remoteBankCode=GWEN_DB_GetCharValue(db, "remoteBankCode", 0, "");
  _remoteBankName=GWEN_DB_GetCharValue(db, "remoteBankName", 0, "");
  _remoteBankLocation=GWEN_DB_GetCharValue(db, "remoteBankLocation", 0, "");
  _remoteIban=GWEN_DB_GetCharValue(db, "remoteIban", 0, "");
  _remoteAccountNumber=GWEN_DB_GetCharValue(db, "remoteAccountNumber",
                                            0, "");
  _category=GWEN_DB_GetCharValue(db, "category", 0, "");

  for (i=0; ; i++) {
    p=GWEN_DB_GetCharValue(db, "remoteName", i, 0);
    if (!p)
      break;
    if (*p)
      _remoteName.push_back(p);
  }

  for (i=0; ; i++) {
    p=GWEN_DB_GetCharValue(db, "purpose", i, 0);
    if (!p)
      break;
    if (*p)
      _purpose.push_back(p);
  }

  _modified=false;
  return true;
}



bool Transaction::modified(){
  return _modified;
}



void Transaction::setModified(bool m) {
  _modified=m;
}


const std::list<std::string> &Transaction::getPurpose() const {
  return _purpose;
}



void Transaction::clearPurpose(){
  _purpose.clear();
  _modified=true;
}



void Transaction::addPurpose(const std::string &s){
  _purpose.push_back(s);
  _modified=true;
}

std::string Transaction::toHtmlInfo() const {
  std::string tmp;
  const std::list<std::string> sl;
  std::list<std::string>::const_iterator it;
  const GWEN_TIME *ti;
  const AB_VALUE *v;

  tmp="<table>\n";
  tmp+="<tr>";

  tmp+="<td>";
  // date
  ti=getDate();
  if (!ti)
    ti=getValutaDate();
  if (ti) {
    int year, month, day;

    if (!GWEN_Time_GetBrokenDownDate(ti, &day, &month, &year)) {
      char dbuf[32];

      snprintf(dbuf, sizeof(dbuf),
               "%04d/%02d/%02d",
               year, month+1, day);
      tmp+=dbuf;
    }
  }
  tmp+="</td>\n";

  // institute
  tmp+="<td>";
  tmp+=getRemoteBankCode().c_str();
  tmp+="</td>\n";

  // account
  tmp+="<td>";
  tmp+=getRemoteAccountNumber().c_str();
  tmp+="</td>\n";

  // other name
  tmp+="<td>";
  for (it=getRemoteName().begin();
       it!=getRemoteName().end();
       it++) {
    tmp+=(*it).c_str();
    tmp+="<br>";
  } // for

  // description
  for (it=getPurpose().begin();
       it!=getPurpose().end();
       it++) {
    tmp+="<br>";
    tmp+=(*it).c_str();
  } // for

  tmp+="</td>\n";

  // type
  tmp+="<td>";
  tmp+=getTransactionText().c_str();
  tmp+="</td>\n";

  // value
  tmp+="<td>";
  v=getValue();
  if (v) {
    char numbuf[32];
    const char *s;

    if (AB_Value_GetValue(v)>0)
      tmp+="<font color=\"dark green\">";
    else
      tmp+="<font color=red>";

#if !defined(__GNUC__) && defined(WIN32)
    // This is MSVC compiler which doesnt have snprintf()
    sprintf(numbuf, "%.2lf", AB_Value_GetValue(v));
#else
    snprintf(numbuf, sizeof(numbuf), "%.2lf", AB_Value_GetValue(v));
#endif
    tmp+=numbuf;
    s=AB_Value_GetCurrency(v);
    if (s) {
      tmp+=" ";
      tmp+=s;
    }
    tmp+="</font>";
  }
  tmp+="</td>\n";

  tmp+="</tr>";
  tmp+="</table>";

  return tmp;
}



GWEN_TYPE_UINT32 Transaction::getTransactionId() const{
  return _transactionId;
}



void Transaction::setTransactionId(GWEN_TYPE_UINT32 id){
  _transactionId=id;
  _modified=true;
}



const std::string &Transaction::getPayee() const {
  return _payee;
}



void Transaction::setPayee(const std::string &s){
  _payee=s;
  _modified=true;
}



const std::string &Transaction::getCategory() const {
  return _category;
}



void Transaction::setCategory(const std::string &s) {
  _category=s;
}



bool Transaction::matchFuzzy(const Transaction &t){
  const AB_VALUE *v1, *v2;
  std::string b1, b2;
  std::string a1, a2;
  bool match;

  v1=getValue();
  v2=t.getValue();
  b1=getRemoteBankCode();
  b2=t.getRemoteBankCode();
  a1=getRemoteAccountNumber();
  a2=t.getRemoteAccountNumber();

  match=true;
  if (v1 && v2) {
    if (AB_Value_GetValue(v1)!=AB_Value_GetValue(v2))
      match=false;
  }

  if (match && !b1.empty() && !b2.empty()) {
    if (strcasecmp(b1.c_str(), b2.c_str())!=0)
      match=false;
  }

  if (match && !a1.empty() && !a2.empty()) {
    if (strcasecmp(a1.c_str(), a2.c_str())!=0)
      match=false;
  }

  return match;
}



bool Transaction::matchFuzzy(const Transaction &t,
                             GWEN_TYPE_UINT32 threshold){
  const AB_VALUE *v1, *v2;
  std::string b1, b2;
  std::string a1, a2;
  bool match;

  v1=getValue();
  v2=t.getValue();
  b1=getRemoteBankCode();
  b2=t.getRemoteBankCode();
  a1=getRemoteAccountNumber();
  a2=t.getRemoteAccountNumber();

  match=true;
  if (v1 && v2) {
    if (AB_Value_GetValue(v1)!=AB_Value_GetValue(v2))
      match=false;
  }

  if (match && !b1.empty() && !b2.empty()) {
    if (strcasecmp(b1.c_str(), b2.c_str())!=0)
      match=false;
  }

  if (match && !a1.empty() && !a2.empty()) {
    if (strcasecmp(a1.c_str(), a2.c_str())!=0)
      match=false;
  }

  if (match) {
    std::string p1, p2;
    std::list<std::string>::const_iterator sit;
    GWEN_TYPE_UINT32 fm;

    for (sit=_purpose.begin();
         sit!=_purpose.end();
         sit++)
      p1+=*sit;
    for (sit=t.getPurpose().begin();
         sit!=t.getPurpose().end();
         sit++)
      p2+=*sit;
    fm=App_FuzzyCompare(p1.c_str(), p2.c_str());
    DBG_ERROR(0, "Fuzzy match: "GWEN_TYPE_TMPL_UINT32, fm);
    if (fm<threshold) {
      match=false;
    }
  }

  return match;
}



void Transaction::gatherInfoFromPurpose() {
  if (_remoteBankCode.empty() || _remoteAccountNumber.empty()) {
    std::list<std::string>::const_iterator sit;

    for (sit=_purpose.begin();
         sit!=_purpose.end();
         sit++) {
      if (-1!=GWEN_Text_ComparePattern((*sit).c_str(),
                                       "KTO* BLZ*", 0)) {
        const char *p;
        std::string accountId;
        std::string bankCode;

        p=(*sit).c_str();
        /* skip "KTO " */
        while(*p && !isdigit(*p))
          p++;

        /* copy account id */
        while(*p && isdigit(*p))
          accountId+=*(p++);

        /* skip "BLZ" */
        while(*p && !isdigit(*p))
          p++;

        /* copy bank code */
        while(*p && isdigit(*p))
          bankCode+=*(p++);

        if (!bankCode.empty() && !accountId.empty()) {
          DBG_ERROR(0, "Setting remote bank code and/or account number");
          if (_remoteBankCode.empty())
            _remoteBankCode=bankCode;
          if (_remoteAccountNumber.empty())
            _remoteAccountNumber=accountId;
        }
        else {
          DBG_WARN(0,
                   "Could not extract remote "
                   "bank code/account number from \"%s\"",
                   (*sit).c_str());
        }
        break;
      }
    }
  }
}


std::string Transaction::getHashData() {
  std::string s;
  std::list<std::string>::iterator it;
  char buffer[32];
  GWEN_BUFFER *dbuf;

  s=_localCountry;
  s+="#";
  s+=_localBankCode;
  s+="#";
  s+=_localAccountNumber;
  s+="#";
  s+=_localSuffix;
  s+="#";
  s+=_localOwnerName;
  s+="#";
  s+=_remoteCountry;
  s+="#";
  s+=_remoteBankCode;
  s+="#";
  s+=_remoteAccountNumber;
  s+="#";
  s+=_remoteSuffix;
  s+="#";
  for (it=_remoteName.begin(); it!=_remoteName.end(); it++){
    s+=*it;
    s+=" ";
  }
  s+="#";

  snprintf(buffer, sizeof(buffer), "%i", _textKey);
  s+=buffer;
  s+="#";

  if (_valutaDate) {
    snprintf(buffer, sizeof(buffer), GWEN_TYPE_TMPL_UINT32,
	     (unsigned int)GWEN_Time_Seconds(_valutaDate));
    s+=buffer;
  }
  s+="#";
  if (_date) {
    snprintf(buffer, sizeof(buffer), GWEN_TYPE_TMPL_UINT32,
	     (unsigned int)GWEN_Time_Seconds(_date));
    s+=buffer;
  }
  s+="#";
  if (_value) {
    if (AB_Value_IsValid(_value)) {
      const char *p;

      snprintf(buffer, sizeof(buffer), "%lf",
	       AB_Value_GetValue(_value));
      s+=buffer;
      s+=" ";
      p=AB_Value_GetCurrency(_value);
      if (p)
        s+=p;
    }
  }
  s+="#";
  s+=_transactionKey;
  s+="#";
  s+=_customerReference;
  s+="#";
  s+=_bankReference;
  s+="#";
  snprintf(buffer, sizeof(buffer), "%d", _transactionCode);
  s+=buffer;
  s+="#";
  s+=_transactionText;
  s+="#";
  s+=_primanota;
  s+="#";
  s+=_fiid;
  s+="#";
  for (it=_purpose.begin(); it!=_purpose.end(); it++){
    s+=*it;
    s+=" ";
  }

  /* s now contains all important data */
  dbuf=GWEN_Buffer_new(0, s.length(), 0, 1);
  GWEN_Buffer_AppendString(dbuf, s.c_str());
  /* remove duplicate blanks */
  GWEN_Text_CondenseBuffer(dbuf);

  s=std::string(GWEN_Buffer_GetStart(dbuf));
  GWEN_Buffer_free(dbuf);

  return s;
}







