/***************************************************************************
                          ractmeasure.cpp  -  description
                             -------------------
    begin                : Thu Jan 27 2000
    copyright            : (C) 2000 by Gerald Pichler
    email                : geripi@sbox.tu-graz.ac.at
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <qapplication.h>
#include <qbitmap.h>
#include <qcursor.h>
#include <qstring.h>

#include "rappwin.h"
#include "ractmeasure.h"
#include "rbehaviordef.h"
#include "rfontdialog.h"
#include "rgraphic.h"
#include "rhatchdialog.h"
#include "rinputpanel.h"
#include "rlog.h"
#include "rmath.h"
#include "rsnap.h"
#include "rstatuspanel.h"
#include "rdlgline.h"
#include "rdlgcircle.h"
#include "rdlgarc.h"

RActMeasure::RActMeasure(){

}
RActMeasure::~RActMeasure(){

}


/*! Serve an event to the corresponding function
    \param _action The Action (e.g. ACT_MEASURE_PTPT)
    \param _myREvent The event (e.g. REVENT_BEGIN)
*/
void
RActMeasure::serveEvent(int _action, int _myREvent )
{
  if(_action>=ACT_MEASURE_FIRST &&
     _action<=ACT_MEASURE_LAST     ) {

    switch(_action) {

      case ACT_MEASURE_PTPT:
        measurePtPt(_myREvent);
        break;

      case ACT_MEASURE_LNPT:
        measureLnPt(_myREvent);
        break;

      case ACT_MEASURE_PARALN:
        measureParallelLines(_myREvent);
        break;

      case ACT_MEASURE_ANGLELNLN:
        measureAngleLnLn(_myREvent);
        break;

      case ACT_MEASURE_EDITELEMENT:
        measureEditElement(_myREvent);
        break;

			case ACT_MEASURE_AREA:
        measureArea(_myREvent);
        break;

      default:
        break;
    }
  }
}

/*! Lets the user measure the distance between two points
    <p>States:
    <ul>
      <li>1: Set 1st point
      <li>2: Set next point
    </ul>
*/
void
RActMeasure::measurePtPt(int _myREvent)
{
  static double cx1=DEF_AREAMAX;  // Coordinates of first (last) point
  static double cy1=DEF_AREAMAX;  //

  switch(_myREvent) {

    case REVENT_BEGIN:
      RAppWin::getRAppWin()->changeMenu(TOOLBAR_SNAP);
      currentDoc()->setCurrentState(1);
      currentDoc()->setCursor(*RAppWin::getRAppWin()->curCross);
      statusPanel()->setActionStatus(tr("L: First point"),
                                     tr("R: Back"),
                                     tr("Distance between two points"));
      break;

    case REVENT_MOUSEMOVE:

      // Move before 1st point is set:
      //
      if(currentDoc()->getCurrentState()==1) {
        snapper()->snapPoint();
      }

      // Move for setting next point:
      //
      else if(currentDoc()->getCurrentState()==2) {
        double px, py;
        if(snapper()->snapPoint(&px, &py)) {
          if(currentDoc()->setPreviewElement(T_LINE, cx1, cy1, px, py)) {
            currentDoc()->drawPreviewElement();
          }
        }
      }
      break;

    case REVENT_LBUTTONUP:

      // Set 1st point:
      //
      if(currentDoc()->getCurrentState()==1) {
        if(snapper()->snapPoint(&cx1, &cy1, true)) {
          currentDoc()->moveRelZero(cx1, cy1);
          statusPanel()->setLeftButtonStatus(tr("L: Second point"));
          currentDoc()->setCurrentState(2);
        }
      }

      // Set next point:
      //
      else if(currentDoc()->getCurrentState()==2) {
        double px, py;
        if(snapper()->snapPoint(&px, &py, true)) {
          currentDoc()->moveRelZero(px, py);
          snapper()->stop();
					
					/*Build the String for the Panels most right part*/
          QString outInfo, tempStr;
          outInfo="dx = ";
					tempStr.setNum(px-cx1);
          outInfo += tempStr += "    dist = ";
          tempStr.setNum(mtGetDistance(cx1, cy1, px, py));
          outInfo += tempStr += "\ndy = ";
          tempStr.setNum(py-cy1);
					outInfo += tempStr;
          statusPanel()->setFunctionStatus(outInfo);
					cx1=px;
          cy1=py;
        }
        snapper()->snapPoint();
      }

      break;

    case REVENT_RBUTTONUP:

      // Break before 1st point is set:
      //
      if(currentDoc()->getCurrentState()==1) {
        snapper()->stop();
        currentDoc()->setBehavior(BVR_NO);
        currentDoc()->resetCurrentState();
        currentDoc()->resetCurrentAction();
        currentDoc()->setCursor(arrowCursor);
        statusPanel()->clear();
      }

      // Go back to setting of first point:
      //
      else if(currentDoc()->getCurrentState()==2) {
        snapper()->stop();
        currentDoc()->setCurrentState(1);
        snapper()->snapPoint();
        statusPanel()->setLeftButtonStatus(tr("L: First point"));
      }
      break;

    default:
      break;
  }
}


/*! Lets the user measure the distance between a line and a point
    <p>States:
    <ul>
      <li>1: Set line
      <li>2: Set point
    </ul>
*/
void
RActMeasure::measureLnPt(int _myREvent)
{
  static RElement baseEl;
  static double cx1=DEF_AREAMAX;  // Coordinates of first (last) point
  static double cy1=DEF_AREAMAX;  //

  switch(_myREvent) {

    case REVENT_BEGIN:
      RAppWin::getRAppWin()->changeMenu(TOOLBAR_SNAP);
      currentDoc()->setCurrentState(1);
      currentDoc()->setCursor(*RAppWin::getRAppWin()->curCross);
      //currentDoc()->setBehavior(BVR_SNAP_NEXT_ELEMENT);
      statusPanel()->setActionStatus(tr("L: Point"),
                                     tr("R: Back"),
                                     tr("Distance between a line and a point"));
    break;

    case REVENT_MOUSEMOVE:
      // Move before point is set:
      //
      if(currentDoc()->getCurrentState()==1) {
        snapper()->snapPoint();
      }
      // Move before element is selected:
      //
      else if(currentDoc()->getCurrentState()==2) {
        snapper()->snapElement();
      }

		break;

    case REVENT_LBUTTONUP:
      // Set point:
      //
      if(currentDoc()->getCurrentState()==1) {
        if(snapper()->snapPoint(&cx1, &cy1, true)) {
          currentDoc()->moveRelZero(cx1, cy1);
          statusPanel()->setLeftButtonStatus(tr("L: Element"));
          currentDoc()->setBehavior(BVR_SNAP_NEXT_ELEMENT);
          currentDoc()->setCurrentState(2);
        }
      }
      //set Element
      else if(currentDoc()->getCurrentState()==2) {
        if(snapper()) {
          if( snapper()->snapElement(true) ) {
						double length=0.0;
            baseEl.getMeasuresFrom(snapper()->getSnappedElement());
						if (baseEl.getElementTyp()==T_LINE)
							length = baseEl.getDistanceFromLineToPoint(cx1,cy1,TRUE);
						else if (baseEl.getElementTyp()==T_CIRCLE)
							length = baseEl.getDistanceFromCircleToPoint(cx1,cy1);
						else if (baseEl.getElementTyp()==T_ARC)
							length = baseEl.getDistanceFromArcToPoint(cx1,cy1,TRUE);
						/*Build the String for the Panels most right part*/
          	QString outInfo, tempStr;
          	outInfo="dist = ";
						tempStr.setNum(length);
          	outInfo += tempStr;
          	statusPanel()->setFunctionStatus(outInfo);							
      		}
        }
      }
      break;

    case REVENT_RBUTTONUP:
      // Break before 1st point is set:
      //
      if(currentDoc()->getCurrentState()==1) {
        snapper()->stop();
        currentDoc()->setBehavior(BVR_NO);
        currentDoc()->resetCurrentState();
        currentDoc()->resetCurrentAction();
        currentDoc()->setCursor(arrowCursor);
        statusPanel()->clear();
      }
      // Go back to setting of first point:
      //
      else if(currentDoc()->getCurrentState()==2) {
        snapper()->stop();
        currentDoc()->setBehavior(BVR_NO);
        currentDoc()->setCurrentState(1);
        snapper()->snapPoint();
        statusPanel()->setLeftButtonStatus(tr("L: Point"));
      }
      break;

    default:
      break;
  }
}


/*! Lets the user measure the distance between two parallel lines
    <p>States:
    <ul>
      <li>1: Set 1st Line
      <li>2: Set 2nd Line
    </ul>
*/
void
RActMeasure::measureParallelLines(int _rEvent)
{
  static RElement* line1=0;            // 1st Line
//  static double mx1=0.0, my1=0.0;         // 1st mouse pos

  switch(_rEvent) {

    case REVENT_BEGIN:
      currentDoc()->setCurrentState(1);
      currentDoc()->setCursor(*RAppWin::getRAppWin()->curCross);
      currentDoc()->setBehavior(BVR_SNAP_NEXT_ELEMENT);
      currentDoc()->setHighlightFilter(T_LINE);
      statusPanel()->setActionStatus(tr("L: Fist line"),
                                     tr("R: Back"),
                                     tr("Distance of parallel lines"));
      break;

    case REVENT_MOUSEMOVE:
      if(currentDoc()->getCurrentState()==1) {
        currentDoc()->setBehavior(BVR_SNAP_NEXT_ELEMENT);
      	currentDoc()->setHighlightFilter(T_LINE);
      }
      else if(currentDoc()->getCurrentState()==2) {
        currentDoc()->setBehavior(BVR_SNAP_NEXT_ELEMENT);
        currentDoc()->setHighlightFilter(T_LINE, line1);
      }
      break;

    case REVENT_LBUTTONUP:
      if(currentDoc()->getCurrentState()==1) {

        // Set 1st Line:
        //
        if(snapper() &&
           snapper()->getSnappedElement()) {

          snapper()->stop();

          line1 = snapper()->getSnappedElement();

          currentDoc()->setHighlightFilter(T_LINE, line1);
          currentDoc()->doBehaviorSnapNextElement(REVENT_MOUSEMOVE);

          currentDoc()->setCurrentState(2);
          statusPanel()->setLeftButtonStatus(tr("L: Second line"));

        }
      }

      else if(currentDoc()->getCurrentState()==2) {

        // Set 2nd Line / compute Distance:
        //
        if(snapper() &&
           snapper()->getSnappedElement()) {

          currentDoc()->doBehaviorSnapNextElement(REVENT_MOUSEMOVE);

          // The 2nd Line:
          //
          RElement* line2 = snapper()->getSnappedElement();

					double normDx1=(line1->getX2() - line1->getX1()) / line1->getLength();
					double normDy1=(line1->getY2() - line1->getY1()) / line1->getLength();
					double normDx2=(line2->getX2() - line2->getX1()) / line2->getLength();
					double normDy2=(line2->getY2() - line2->getY1()) / line2->getLength();
					
					if (((normDx1>0) && (normDx2<0)) || ((normDx1<0) && (normDx2>0))) {
            normDx2*=-1.0;
            normDy2*=-1.0;
          }
          if (((normDy1>0) && (normDy2<0)) || ((normDy1<0) && (normDy2>0))) {
            normDx2*=-1.0;
            normDy2*=-1.0;
          }

          if (mtCompFloat(normDx1, normDx2, 0.000000000001) && // Lines are
          		mtCompFloat(normDy1, normDy2, 0.000000000001)){  // very parallel (maybe too parallel?)
						/*Build the String for the Panels most right part*/
          	QString outInfo, tempStr;
          	outInfo="dist = ";
						tempStr.setNum(line1->getDistanceFromLineToPoint(
          											line2->getX1(), line2->getY1(), true));
          	outInfo += tempStr;
          	statusPanel()->setFunctionStatus(outInfo);							
    				
          	line1 = 0;
    	      currentDoc()->setCurrentState(1);
         	  currentDoc()->setHighlightFilter(T_LINE);
            currentDoc()->doBehaviorSnapNextElement(REVENT_MOUSEMOVE);
      	    statusPanel()->setLeftButtonStatus(tr("L: Fist line"));
					}
					else {  // Lines are not parallel
						statusPanel()->setFunctionStatus(tr("Lines are not parallel"));
						
					}
        }
      }
      break;

    case REVENT_RBUTTONUP:
      if(currentDoc()->getCurrentState()==1) {
        snapper()->stop();
        currentDoc()->setBehavior(BVR_NO);
        currentDoc()->resetCurrentState();
        currentDoc()->resetCurrentAction();
        currentDoc()->setCursor(arrowCursor);
        statusPanel()->clear();
      }

      else if(currentDoc()->getCurrentState()==2) {
        snapper()->stop();
        currentDoc()->setCurrentState(1);
				line1 = 0;
        currentDoc()->doBehaviorSnapNextElement(REVENT_MOUSEMOVE);
        currentDoc()->setHighlightFilter(T_LINE);
        statusPanel()->setLeftButtonStatus(tr("L: Fist line"));
      }
      break;

    default:
      break;
  }

}

/*! Lets the user measure the angle between two lines
    <p>States:
    <ul>
      <li>1: Set 1st Line
      <li>2: Set 2nd Line
    </ul>
*/
void
RActMeasure::measureAngleLnLn(int _rEvent)
{
  static RElement* leg1=0;  // First leg
  static RElement* leg2=0;  // Second leg
  static RElement  prevEl;  // Preview
  static double cx=0.0;      // Intersection of the Lines
  static double cy=0.0;
  static double a1=0.0;      // Angles
  static double a2=0.0;
  static double angle=0.0;
  static double tpx1=0.0;    // Touch points of legs
  static double tpy1=0.0;
  static double tpx2=0.0;
  static double tpy2=0.0;

  switch(_rEvent) {

    case REVENT_BEGIN:
      RAppWin::getRAppWin()->changeMenu(TOOLBAR_SNAP);
      currentDoc()->setCurrentState(1);
      currentDoc()->setCursor(*RAppWin::getRAppWin()->curCross);
      currentDoc()->setBehavior(BVR_SNAP_NEXT_ELEMENT);
      currentDoc()->setHighlightFilter(T_LINE);
      statusPanel()->setActionStatus(tr("L: Fist line"),
                                     tr("R: Back"),
                                     tr("Angle of two lines"));
      break;

    case REVENT_MOUSEMOVE:

      // Set 1st leg:
      if(currentDoc()->getCurrentState()==1) {
        currentDoc()->setBehavior(BVR_SNAP_NEXT_ELEMENT);
        currentDoc()->setHighlightFilter(T_LINE);
      }

      else if(currentDoc()->getCurrentState()==2) {
        currentDoc()->setBehavior(BVR_SNAP_NEXT_ELEMENT);
        currentDoc()->setHighlightFilter(T_LINE, leg1);
      }

      break;

    case REVENT_LBUTTONUP:
      // Set first leg:
      if(currentDoc()->getCurrentState()==1) {
        if(snapper() &&
           snapper()->getSnappedElement()) {
          leg1 = snapper()->getSnappedElement();
          if(leg1) {
            leg1->getTouchPoint(&tpx1, &tpy1,
                                currentDoc()->getRealMousePosX(),
                                currentDoc()->getRealMousePosY());
            currentDoc()->setCurrentState(2);
            statusPanel()->setLeftButtonStatus(tr("L: Second line"));
            currentDoc()->setHighlightFilter(T_LINE, leg1);
          }
        }
      }

      // Set 2nd leg:
      else if(currentDoc()->getCurrentState()==2) {
        // Set 2nd leg / preview dimension:
        if(snapper() &&
           snapper()->getSnappedElement() &&
           leg1) {

          leg2 = snapper()->getSnappedElement();

          if(leg2 && leg2!=leg1) {
            bool inter=false;
            leg1->getIntersection(leg2, &inter, &cx, &cy, 0,0,0,0,false);

            if(inter) {  // intersection exists, angle>0
              a1 = mtGetAngle(cx, cy, tpx1, tpy1);
              leg2->getTouchPoint(&tpx2, &tpy2,
                                  currentDoc()->getRealMousePosX(),
                                  currentDoc()->getRealMousePosY());
              a2 = mtGetAngle(cx, cy, tpx2, tpy2);
							
              angle = mtGetAngleDiff(a1, a2);
              //if (angle<0.0) angle *= -1.0;

  						/*Build the String for the Panels most right part*/
            	QString outInfo, tempStr;
            	outInfo= tr("Angle");
            	outInfo += " = ";
	  					tempStr.setNum(angle);
            	outInfo += tempStr += " ";
            	statusPanel()->setFunctionStatus(outInfo);							
              currentDoc()->setCurrentState(1);
              currentDoc()->doBehaviorSnapNextElement(REVENT_MOUSEMOVE);
              currentDoc()->setHighlightFilter(T_LINE);
              statusPanel()->setLeftButtonStatus(tr("L: Fist line"));
						}
						else { // lines are Parallel
            	statusPanel()->setFunctionStatus(tr("Angle = 0"));							
              currentDoc()->setCurrentState(1);
              currentDoc()->doBehaviorSnapNextElement(REVENT_MOUSEMOVE);
              currentDoc()->setHighlightFilter(T_LINE);
              statusPanel()->setLeftButtonStatus(tr("L: Fist line"));
						}
          }
        }
      }
      break;

    case REVENT_RBUTTONUP:
      if(currentDoc()->getCurrentState()==1) {
        snapper()->stop();
        currentDoc()->setBehavior(BVR_NO);
        currentDoc()->resetCurrentState();
        currentDoc()->resetCurrentAction();
        currentDoc()->setCursor(arrowCursor);
        statusPanel()->clear();
      }
      else if(currentDoc()->getCurrentState()==2) {
        snapper()->stop();
        currentDoc()->setCurrentState(1);
        currentDoc()->doBehaviorSnapNextElement(REVENT_MOUSEMOVE);
        currentDoc()->setHighlightFilter(T_LINE);
        statusPanel()->setLeftButtonStatus(tr("L: Fist line"));
      }
      break;

    default:
      break;
  }
}

/*! Lets the user edit an element's geometry data
    <p>States:
    <ul>
      <li>1: Set element
    </ul>
*/
void
RActMeasure::measureEditElement(int _rEvent)
{
  static RElement* element=0;

  switch(_rEvent) {

    case REVENT_BEGIN:
      currentDoc()->setCurrentState(1);
      currentDoc()->setCursor(*RAppWin::getRAppWin()->curCross);
      currentDoc()->setBehavior(BVR_SNAP_NEXT_ELEMENT);
      statusPanel()->setActionStatus(tr("L: Element"),
                                     tr("R: Back"),
                                     tr("Edit geometry of an element"));
      break;

    case REVENT_MOUSEMOVE:
      if(currentDoc()->getCurrentState()==1) {
        currentDoc()->setBehavior(BVR_SNAP_NEXT_ELEMENT);
      }
      break;

    case REVENT_LBUTTONUP:
      if(currentDoc()->getCurrentState()==1) {
        // Set element:
        //
        if(snapper() &&
           snapper()->getSnappedElement()) {
          snapper()->stop();
          element = snapper()->getSnappedElement();
					
					if(element->getElementTyp()==T_LINE){
        		// Show Line Dialog:
        		//
        		RDlgLine* dlg = new RDlgLine(qApp->mainWidget(), "xxx", element);
        		if(dlg->exec()) {
        			bool okX1 = false; bool okY1=false; bool okX2=false; bool okY2=false;
        			// get the new points:
        			double newX1=dlg->getX1(&okX1);
        			double newY1=dlg->getY1(&okY1);
        			double newX2=dlg->getX2(&okX2);
        			double newY2=dlg->getY2(&okY2);
        			// if all values are ok, and any changes are made,
        			// make the changes
        			if((okX1 && okY1 && okX2 && okY2)
        					&& !((element->getX1()==newX1)
        						&& (element->getY1()==newY1)
        						&& (element->getX2()==newX2)
        						&& (element->getY2()==newY2))){
          			
          			currentDoc()->tagNothing(false);
          			element->setFlag(E_TAGGED);
      					currentDoc()->editDelete(true);
          			
          			int newEl=currentDoc()->addLine(newX1, newY1, newX2, newY2);
          			currentDoc()->elementAt(newEl)->setColor(element->getColor());
          			currentDoc()->elementAt(newEl)->setWidth(element->getWidth());
          			currentDoc()->elementAt(newEl)->setStyle(element->getStyle());
          			currentDoc()->elementAt(newEl)->setLayer(element->getLayer());
          			
          			currentDoc()->drawElement(newEl);
          			currentDoc()->terminateAction();
	       			}
        		}
        		delete dlg;
					}	
					else if(element->getElementTyp()==T_CIRCLE){
        		// Show Circle Dialog:
        		//
        		RDlgCircle* dlg = new RDlgCircle(qApp->mainWidget(), "xxx", element);
        		if(dlg->exec()) {
        			bool okCx = false; bool okCy=false; bool okCr=false;
        			// get the new points:
        			double newCx=dlg->getCx(&okCx);
        			double newCy=dlg->getCy(&okCy);
        			double newCr=dlg->getCr(&okCr);
        			// if all values are ok, and any changes are made,
        			// make the changes
        			if((okCx && okCy && okCr)
        					&& !((element->getCx()==newCx)
        						&& (element->getCy()==newCy)
        						&& (element->getCr()==newCr))){
          			
          			currentDoc()->tagNothing(false);
          			element->setFlag(E_TAGGED);
      					currentDoc()->editDelete(true);
          			
          			int newEl=currentDoc()->addCircle(newCx, newCy, newCr);
          			currentDoc()->elementAt(newEl)->setColor(element->getColor());
          			currentDoc()->elementAt(newEl)->setWidth(element->getWidth());
          			currentDoc()->elementAt(newEl)->setStyle(element->getStyle());
          			currentDoc()->elementAt(newEl)->setLayer(element->getLayer());
          			
          			currentDoc()->drawElement(newEl);
          			currentDoc()->terminateAction();
	       			}
        		}
        		delete dlg;
					}
					else if(element->getElementTyp()==T_ARC){
        		// Show Arc Dialog:
        		//
        		RDlgArc* dlg = new RDlgArc(qApp->mainWidget(), "xxx", element);
        		if(dlg->exec()) {
        			bool okCx = false; bool okCy=false; bool okCr=false; bool okA1=false; bool okA2=false;
        			// get the new points:
        			double newCx=dlg->getCx(&okCx);
        			double newCy=dlg->getCy(&okCy);
        			double newCr=dlg->getCr(&okCr);
        			double newA1=dlg->getA1(&okA1);
        			double newA2=dlg->getA2(&okA2);
        			bool newReversed=dlg->getReversed();
        			// if all values are ok, and any changes are made,
        			// make the changes
        			if((okCx && okCy && okCr && okA1 && okA2)
        					&& !((element->getCx()==newCx)
        						&& (element->getCy()==newCy)
        						&& (element->getCr()==newCr)
        						&& (element->getA1()==newA1)
        						&& (element->getA2()==newA2)
        						&& (element->getFlag(E_REVERSED)==newReversed))){
          			
          			currentDoc()->tagNothing(false);
          			element->setFlag(E_TAGGED);
      					currentDoc()->editDelete(true);
          			
          			int newEl=currentDoc()->addArc(newCx, newCy, newCr,
          																		 newA1, newA2, newReversed);
          			currentDoc()->elementAt(newEl)->setColor(element->getColor());
          			currentDoc()->elementAt(newEl)->setWidth(element->getWidth());
          			currentDoc()->elementAt(newEl)->setStyle(element->getStyle());
          			currentDoc()->elementAt(newEl)->setLayer(element->getLayer());
          			
          			currentDoc()->drawElement(newEl);
          			currentDoc()->terminateAction();
	       			}
        		}
        		delete dlg;
					}
					
	//			currentDoc()->doBehaviorSnapNextElement(REVENT_MOUSEMOVE);
      	currentDoc()->setBehavior(BVR_SNAP_NEXT_ELEMENT);
					
        }
      }
      break;

    case REVENT_RBUTTONUP:
      if(currentDoc()->getCurrentState()==1) {
        snapper()->stop();
        currentDoc()->setBehavior(BVR_NO);
        currentDoc()->resetCurrentState();
        currentDoc()->resetCurrentAction();
        currentDoc()->setCursor(arrowCursor);
        statusPanel()->clear();
      }
      break;

    default:
      break;
  }
}


/*! Lets the user measure the area defined through 3 or more points
    <p>States:
    <ul>
      <li>1: Set 1st point
      <li>2: Set next point
			<li>...
    </ul>
*/
void
RActMeasure::measureArea(int _myREvent)
{
  double cx;
  double cy;

  switch(_myREvent) {

    case REVENT_BEGIN:
      RAppWin::getRAppWin()->changeMenu(TOOLBAR_SNAP);
      //currentDoc()->setCurrentState(1);
      currentDoc()->setCursor(*RAppWin::getRAppWin()->curCross);
      statusPanel()->setActionStatus(tr("L: 1st point of area"), tr("R: Back"), tr("calculate area/circumference"));
      break;

    case REVENT_MOUSEMOVE:

      snapper()->snapPoint();
      break;

    case REVENT_LBUTTONUP:

      // Set point:
      //
      if(snapper()->snapPoint(&cx, &cy, true)) {
			  theArea.addPoint(cx, cy);
        currentDoc()->moveRelZero(cx, cy);
        //currentDoc()->setCurrentState(2);
				if (theArea.count() == 3)
				{
        	statusPanel()->setLeftButtonStatus(tr("L: next point of area"));
					statusPanel()->setRightButtonStatus(tr("R: Area/Circumference"));
				}
				else if (theArea.count() == 2)
        	statusPanel()->setLeftButtonStatus(tr("L: 3rd point of area"));
				else if (theArea.count() == 1)
					statusPanel()->setLeftButtonStatus(tr("L: 2nd point of area"));
      }

      break;

    case REVENT_RBUTTONUP:

			{
        // Calculate Area
        //
        snapper()->stop();
				if (theArea.count() < 3 )
				{
					if (theArea.count() == 0)
					{
            currentDoc()->setBehavior(BVR_NO);
            currentDoc()->resetCurrentState();
            currentDoc()->resetCurrentAction();
            currentDoc()->setCursor(arrowCursor);
					  statusPanel()->clear();
					}
					else
					{
  					theArea.reset();
            statusPanel()->setActionStatus(tr("L: 1st point of area"), tr("R: Back"), tr("calculate area/circumference"));
					}
				}
				else
				{
			    /*Build the String for the Panels most right part*/
          QString outInfo, tempStr;

					theArea.calculate();

          outInfo=tr("Area") + " = ";
			    tempStr.setNum(theArea.getArea());
          outInfo += tempStr;

					outInfo += "\n" + tr("Circumference") + " = ";
			    tempStr.setNum(theArea.getCircumference());
          outInfo += tempStr;

          statusPanel()->setFunctionStatus(outInfo);
          snapper()->snapPoint();
					theArea.reset();
          statusPanel()->setLeftButtonStatus(tr("L: 1st point of area"));
          statusPanel()->setRightButtonStatus(tr("R: Back"));
				}
			}

      break;

    default:
      break;
  }
}
