/***************************************************************************
                          rconfig.cpp  -  description
                             -------------------
    begin                : Sun Sep 12 1999
    copyright            : (C) 1999 by Andreas Mustun
    email                : andrew@ribbonsoft.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "rconfig.h"

#include <qapplication.h>
#include <qfile.h>
#include <qfont.h>
#include <qregexp.h>
#include <qtextstream.h>
#include <qtranslator.h>

#include "rfile.h"
#include "rfileparser.h"
#include "rinitialdialog.h"
#include "rlog.h"

/**
 * \class RConfig
 * Class RConfig provides access to global settings.
 */

RConfig* RConfig::theOneAndOnlyConfig=0;


/**
 * Gets the one and only Config-Object.
 *   Creates a new one on first call.
 */
RConfig*
RConfig::getConfig()
{
  if(theOneAndOnlyConfig==0) {
    theOneAndOnlyConfig = new RConfig;
    theOneAndOnlyConfig->ini();
  }
  return theOneAndOnlyConfig;
}



/**
 * Constructor.
 */
RConfig::RConfig()
 : RSettings('=', '"', '"')
{
  cadDimensionTextAutoScale=false;
  cadDimensionTextDistance=2.0;
  cadDimensionOverLength=2.0;

  languageList.setAutoDelete(true);
}



/**
 * Destructor.
 */
RConfig::~RConfig()
{
}



/**
 * Initializes the config object (only called once).
 */
void
RConfig::ini()
{
  addSetting( "Language:Language",  DEF_LANGUAGE );

  addSetting( "Application:Version", DEF_VERSION );

  addSetting( "Application:FontName", (RUNNING_WINDOWS ? "Arial" : "Helvetica") );
  addSetting( "Application:FontSize0", (RUNNING_WINDOWS ? "10" : "12") );
  addSetting( "Application:FontSize1", (RUNNING_WINDOWS ? "10" : "12") );
  addSetting( "Application:FontSize2", (RUNNING_WINDOWS ? "10" : "12") );
  addSetting( "Application:FontSize3", (RUNNING_WINDOWS ? "8" : "10") );

  addSetting( "Application:ApplicationWindowX", "10");
  addSetting( "Application:ApplicationWindowY", "10");
  addSetting( "Application:ApplicationWindowWidth", "690");
  addSetting( "Application:ApplicationWindowHeight", "480");
#ifdef DEF_CAM_EXPERT
  addSetting( "Application:CamDialogWindowX", "10");
  addSetting( "Application:CamDialogWindowY", "10");
  addSetting( "Application:CamDialogWindowWidth", "655");
  addSetting( "Application:CamDialogWindowHeight", "430");
#endif
  addSetting( "Application:FontDialogWindowX", "10");
  addSetting( "Application:FontDialogWindowY", "10");
  addSetting( "Application:FontDialogWindowWidth", "500");
  addSetting( "Application:FontDialogWindowHeight", "420");
  
  addSetting( "Application:OpenPath", "" );
  addSetting( "Application:OpenFilter", "Drawing Exchange (*.dxf)" );
  addSetting( "Application:SavePath", "" );

  addSetting( "CAD:Unit", (DEF_INCH_MODE ? mtUnitToString(Inch) : mtUnitToString(Millimeter)) );
  addSetting( "CAD:PreviewElements", "100" );
  addSetting( "CAD:SnapRange", "32" );
  addSetting( "CAD:MarkSize", "4" );
  addSetting( "CAD:AutoRedraw", "0" );
  addSetting( "CAD:GridFormat", (DEF_INCH_MODE ? mtValueFormatToString(Fractional) : mtValueFormatToString(Decimal)));
  addSetting( "CAD:MinGridX", (DEF_INCH_MODE ? "1/128" : "0.01"));
  addSetting( "CAD:MinGridY", (DEF_INCH_MODE ? "1/128" : "0.01"));
  addSetting( "CAD:PageFormat", (DEF_INCH_MODE ? mtPageFormatToString(Letter) : mtPageFormatToString(A4)));
  addSetting( "CAD:PageOrientation", "Portrait");
  addSetting( "CAD:PageSizeX", (DEF_INCH_MODE ? "8.5" : "210.0"));
  addSetting( "CAD:PageSizeY", (DEF_INCH_MODE ? "11.0" : "297.0"));
  addSetting( "CAD:PageOriginX", "0.0");
  addSetting( "CAD:PageOriginY", "0.0");
  addSetting( "CAD:PaperSpace", "1.0");
  addSetting( "CAD:ShowPageBorder", "1");
  addSetting( "CAD:MetaGrid", "1");
  addSetting( "CAD:AutoScaleGrid", "1");
  addSetting( "CAD:MinGridSpace", "8");
  addSetting( "CAD:DimensionUnit", (DEF_INCH_MODE ? mtUnitToString(Inch) : mtUnitToString(Millimeter)) );
  addSetting( "CAD:DimensionFormat", (DEF_INCH_MODE ? mtValueFormatToString(Fractional) : mtValueFormatToString(Decimal)) );
  addSetting( "CAD:DimensionExactness", (DEF_INCH_MODE ? "1/128" : "0.01"));
  addSetting( "CAD:DimensionShowUnits", (DEF_INCH_MODE ? "1" : "0"));
  addSetting( "CAD:AngleDimensionUnit", mtAngleUnitToString(Deg) );
  addSetting( "CAD:AngleDimensionExactness", "0.01");
  addSetting( "CAD:AngleDimensionFormat", mtAngleFormatToString(DecimalDegrees) );
  addSetting( "CAD:DimensionTextHeight", (DEF_INCH_MODE ? "2/8" : "5"));
  addSetting( "CAD:DimensionTextAutoScale", "0");
  addSetting( "CAD:DimensionTextDistance", (DEF_INCH_MODE ? "1/8" : "2"));
  addSetting( "CAD:DimensionOverLength", (DEF_INCH_MODE ? "1/8" : "2") );
  addSetting( "CAD:DimensionArrowLength", (DEF_INCH_MODE ? "3/8" : "3") );
  addSetting( "CAD:DimensionUnderLength", (DEF_INCH_MODE ? "1/8" : "2") );
  addSetting( "CAD:LayerListVisible", "1" );
  //addSetting( "CAD:InchMode", (DEF_INCH_MODE ? "1" : "0") );

#ifdef DEF_CAM_EXPERT
  addSetting( "Simulation:EndpointSize", "2");
  addSetting( "Simulation:ArrowSize", "3");
  addSetting( "Simulation:BrokenContourTolerance", "0.002");
  addSetting( "Simulation:DelayFactor", "1200");
#endif

  addSetting( "DXF:DxfSplineSegments", "16");
  addSetting( "DXF:DxfEllipseSegmentAngle", "5");

  addSetting( "EPS:EPSLineWithFactor", "4");

#ifdef DEF_CAM_EXPERT
  addSetting( "CAM:MachineGenerator", "gcode");
  addSetting( "CAM:SupportSubroutines", "0");
  addSetting( "CAM:SimulationPanelVisible", "1");
#endif

  addSetting( "Print:PrinterName", "lp");
  addSetting( "Print:PrintProgram", "");
  addSetting( "Print:PrintBorderLeft", "25");
  addSetting( "Print:PrintBorderTop", "15");
  addSetting( "Print:PrintBorderRight", "15");
  addSetting( "Print:PrintBorderBottom", "15");
  addSetting( "Print:PrintZeroSize", "5");
  addSetting( "Print:PrintEnhancement", "5");
  addSetting( "Print:PrintAdjustmentFactor", "1.0");

  addSetting( "Paths:BrowserPath", (RUNNING_WINDOWS ? "C:\\Progra~1\\Intern~1\\Iexplore.exe" : "netscape"));
  addSetting( "Paths:LibraryPath1", "/usr/share/qcad/libraries" );

  verbose=false;
}




/**
 * Creates not existing config files.
 *
 * \param  overwrite Overwrite contents.
 * \return Success.
 */
bool
RConfig::createConfigFile( bool overwrite )
{
  bool ret=true;

  // Config path under win is program path + /.manstyle.
  // Under Linux it's the home path + /.manstyle.
  //
#ifdef DEF_WINDOWS
  setConfigDir(QDir(prgDir.absPath()));
#else
  setConfigDir(QDir(QDir::home().absPath() + "/." + DEF_APPNAME_L));
#endif

  // Create config directory:
  //
  if(!configDir.exists()) {
    if(!configDir.mkdir(configDir.absPath(), true)) {
      // Can't create directory for config file!
      return false;
    }
  }

  // Create ".<app>.conf":
  //
  QFile configFile(configDir.absPath() + "/" + DEF_APPNAME_L + ".conf");

  // Show initial setup dialog if no config file exists:
  if(!configFile.exists()) {
    RInitialDialog dlg(0, "", true);
    dlg.show();
  }

  if(!configFile.exists() || overwrite) {
    if(configFile.open(IO_WriteOnly)) {    // file opened successfully
      QTextStream ts(&configFile);
      RSetting* setting;

      ts <<
      "# This file is automatically generated by " << DEF_APPNAME << ".\n"
      "# Please edit only if " << DEF_APPNAME << " is not running.\n";

      int sep;
      QString section;
      QString variable;
      QStringList sectionList;      // List of all sections

      // Collect section list:
      for(setting=settingList.first(); setting!=0; setting=settingList.next()) {
        sep = setting->getName().find( ':' );
        section = setting->getName().left( sep );

        if( sectionList.find( section ) == sectionList.end() ) {
          sectionList += section;
        }
      }

      sectionList.sort();

      // Write the sections:
      //
      for( QStringList::Iterator it = sectionList.begin(); it!=sectionList.end(); ++it ) {
        ts << "\n[" << (*it) << "]\n";
        for( setting=settingList.first(); setting!=0; setting=settingList.next() ) {
          sep = setting->getName().find( ':' );
          section = setting->getName().left( sep );

          if( section==(*it) ) {
            variable = setting->getName().right( setting->getName().length()-sep-1 );
            ts << variable << "=\"" << setting->getValue() << "\"\n";
          }
        }
      }

      configFile.close();
    }

    else {
      // Can't create file
      ret=false;
    }
  }

  return ret;
}



/**
 * Reads the config file.
 */
bool
RConfig::readConfigFile()
{
  QString configPath;
  configPath = configDir.absPath() + "/" + DEF_APPNAME_L + ".conf";
  int i=0, j, l=0;                    // Index, length of matching string
  QRegExp regSection("\\[[^]]*\\]");  // Reg exp for a section including brackets
  QRegExp regName("[^=[]*");          // Reg exp for a setting name (lvalue)
  QRegExp regValue("\"[^\"]*\"");     // Reg exp for a setting value (rvalue) including quote marks
  QString lSectionName;                // Section name excluding brackets
  QString setName;                    // Setting name
  QString setValue;                   // Setting value

  // Get file contents without comments:
  //
  QString cont = RFileParser::getContents(configPath, false);

  do {
    // Read next section (name/contents):
    //
    i=regSection.match(cont, i, &l);
    if(i==-1) break;
    lSectionName = cont.mid(i+1, l-2);
    i+=l;

    // Read next setting:
    //
    do {
      j=regName.match(cont, i, &l);
      if(j==-1) break;
      setName = cont.mid(j, l);
      if(setName.stripWhiteSpace().isEmpty()) break;
      i=j+l;

      j=regValue.match(cont, i, &l);
      if(j==-1) break;
      setValue = cont.mid(j+1, l-2);
      i=j+l;

      addSetting(lSectionName + ":" + setName.stripWhiteSpace(), setValue);

    } while(true);

  } while(true);


  // Add some values which were not saved in config file:
  //

  // Calculate factor for Speed-Slider (Simulation delay):
  //
  if( getSettingInt( "Simulation:DelayFactor" )==0 ) {
    QTime startTime = QTime::currentTime();
    for(int t=0; t<30000; ++t) {
      qApp->processEvents(1);
    }
    QTime stopTime = QTime::currentTime();

    uint iStartTime = startTime.hour()*3600000 +
                      startTime.minute()*60000 +
                      startTime.second()* 1000 +
                      startTime.msec();
    uint iStopTime =  stopTime.hour()*3600000 +
                      stopTime.minute()*60000 +
                      stopTime.second()* 1000 +
                      stopTime.msec();

    // Day change:
    //
    if( iStopTime<iStartTime ) iStopTime+=86400000;

    uint mil = iStopTime-iStartTime;
    int delayFactor;
    if(mil>0) delayFactor = (int)((1.0/(double)mil)*810000.0);
    else      delayFactor = 1000;

    addSetting( "Simulation:DelayFactor", delayFactor );

    RLOG("\nTime to count from 0 to 30000: ");
    RLOG((int)mil);
    RLOG("\ndelayFactor: ");
    RLOG(delayFactor);
  }

  // Ini some public variables provided for faster access:
  //
  cadDimensionTextAutoScale = (bool)getSettingInt( "CAD:DimensionTextAutoScale" );
  cadDimensionTextDistance  = getSettingDouble( "CAD:DimensionTextDistance" );
  cadDimensionOverLength    = getSettingDouble( "CAD:DimensionOverLength" );
  cadDimensionUnderLength   = getSettingDouble( "CAD:DimensionUnderLength" );
  
  // Correct some values:
  //
  if( getSettingDouble("CAD:DimensionExactness")<0.000001 ) {
    addSetting( "CAD:DimensionExactness", (DEF_INCH_MODE ? "1/128" : "0.01"));
  }

  return false;
}



// Creates a list of all available languages.
//   The list is stored in "languageList"
//
void
RConfig::createLanguageList()
{
    QString messagePath = fileSearchSubdir("messages");
    QDir messageDir(messagePath, "qcad_*.qm");
    QStringList lst = messageDir.entryList();
    QFile langFile(messagePath + "/qcad.lang");
    QTextStream txt(&langFile);
    QString str, sign, name, charset;
    languageStruct *newLanguage;
    int pos1, pos2;

    languageList.clear();
    
    // Default language
    newLanguage = new languageStruct;
    newLanguage->name = "English";
    newLanguage->sign = "en";
    newLanguage->charset = QFont::Latin1;
    languageList.append(newLanguage);
    
    if (langFile.open(IO_ReadOnly)) {
        while (!txt.atEnd()) {
            str = txt.readLine().simplifyWhiteSpace();
            pos1 = str.find(' ', 0);
            sign = str.left(pos1);
            if (!lst.contains("qcad_" + sign + ".qm"))
                continue;
            pos2 = str.find(' ', pos1 + 1);
            charset = str.mid(pos1 + 1, pos2 - pos1 - 1);
            name = str.mid(pos2 + 1);
            
            newLanguage = new languageStruct;
            newLanguage->name = name;
            newLanguage->sign = sign;
            newLanguage->charset = mtStringToCharset(charset);

            languageList.append(newLanguage);
        }
        langFile.close();
    } else {
        qWarning("Cannot open " + messagePath + "/qcad.lang");
    }
}


// Read messages from .qm file:
//
bool
RConfig::readQmFile()
{
    static QTranslator *translator = 0;
    
    if (translator) {
        qApp->removeTranslator(translator);
        delete translator;
    }
    
    // Search for language file
    QString qmPath;
    qmPath = searchFile(QString("messages/qcad_") +
                        getSetting("Language:Language") + ".qm");
    if (qmPath.isEmpty()) {
        addSetting("Language:Language", "en");
    } else {
        translator = new QTranslator(0);
        translator->load(qmPath);
        qApp->installTranslator(translator);
    }

    // Add charset for this language to the settings
    if (getSetting("Application:Charset").isEmpty()) {
        addSetting("Application:Charset",
                   mtCharsetToString(getLanguageCharset(getLanguageName(getSetting("Language:Language")))));
    }

    // Ini fonts:
    //
    font0 = new QFont( getSetting("Application:FontName"),
                       getSettingInt("Application:FontSize0"),
                       QFont::Normal,
                       false,
                       mtStringToCharset(getSetting("Application:Charset")) );
    font1 = new QFont( getSetting("Application:FontName"),
                       getSettingInt("Application:FontSize1"),
                       QFont::Normal,
                       false,
                       mtStringToCharset(getSetting("Application:Charset")) );
    font2 = new QFont( getSetting("Application:FontName"),
                       getSettingInt("Application:FontSize2"),
                       QFont::Normal,
                       false,
                       mtStringToCharset(getSetting("Application:Charset")) );
    font3 = new QFont( getSetting("Application:FontName"),
                       getSettingInt("Application:FontSize3"),
                       QFont::Normal,
                       false,
                       mtStringToCharset(getSetting("Application:Charset")) );

    return true;
}


/**
 * Searchs for a file with relative constant path given
 *   and returns the absolute path of the file.
 * Possible paths are: <BR>
 * /usr/local/qcad/... <BR>
 * ~/.qcad/...         <BR>
 * /usr/share/qcad/... <BR>
 */
QString
RConfig::searchFile(const QString& _relPath)
const
{
  QString path;

  // Search in program dir (/usr/local/qcad/...):
  path = getPrgDir().path()+"/"+_relPath;

#ifndef DEF_WINDOWS
  // Search in ~/.qcad/...:
  if(!QFileInfo(path).exists()) {
    path = getConfigDir().path()+"/"+_relPath;
  }

  // Search in /usr/share/qcad/... dir:
  if(!QFileInfo(path).exists()) {
    path = QString("/usr/share/")+DEF_APPNAME_L+"/"+_relPath;
  }
#endif

  if(!QFileInfo(path).exists()) {
    path="";
  }

  return path;
}


/*! Gets the short two letter sign for a given language.
*/
QString
RConfig::getLanguageSign( QString _languageName )
{
  languageStruct* ls;
  for( ls=languageList.first(); ls!=0; ls=languageList.next() ) {
    if( ls->name.lower()==_languageName.lower() ) {
      return ls->sign;
    }
  }

  return QString("en");
}



/*! Gets the full name of a language for a given two letter sign.
*/
QString
RConfig::getLanguageName( QString _languageSign )
{
  languageStruct* ls;
  for( ls=languageList.first(); ls!=0; ls=languageList.next() ) {
    if( ls->sign.lower()==_languageSign.lower() ) {
      return ls->name;
    }
  }

  return QString("English");
}



QFont::CharSet
RConfig::getLanguageCharset( QString _languageName )
{
  languageStruct* ls;
  for( ls=languageList.first(); ls!=0; ls=languageList.next() ) {
    if( ls->name.lower()==_languageName.lower() ) {
      return ls->charset;
    }
  }

  return QFont::Latin1;
}




// Gets the number of languages available in languageList.
//
int
RConfig::getLanguageNum()
{
  return languageList.count();
}


QString
RConfig::getLanguage(int _index)
{
  if(_index<(int)languageList.count()) {
    return languageList.at(_index)->name;
  }
  return QString("English");
}



QFont
RConfig::getFont(int _num)
{
  switch(_num) {
    default:
    case 0: return *font0;
    case 1: return *font1;
    case 2: return *font2;
    case 3: return *font3;
  }
  return *font0;
}


// EOF

























