/***************************************************************************
                          rlibrarybrowser.cpp  -  description
                             -------------------
    begin                : Sun Jan 9 2000
    copyright            : (C) 2000 by Andrew Mustun
    email                : andrew@qcad.org
 ***************************************************************************/



#include "rlibrarybrowser.h"

#include <qapplication.h>
#include <qlabel.h>
#include <qlist.h>
#include <qheader.h>
#include <qscrollview.h>
#include <qsplitter.h>
#include <qpushbutton.h>

#include "rappwin.h"
#include "rcombobox.h"
#include "rconfig.h"
#include "rgraphic.h"
#include "rlibraryitem.h"
#include "rlineedit.h"
#include "rlog.h"
#include "rlayoutdef.h"
#include "rlistview.h"
#include "rscrollview.h"

RLibraryBrowser::RLibraryBrowser(QWidget *_parent, const char *_name )
  : RWidget(_parent,_name,WStyle_Tool|WStyle_StaysOnTop)
{
  //setCaption( tr("Library browser") );

  // create label:
  lLabel = new QLabel( tr("Library browser"), this );
  lLabel->setAlignment( QLabel::AlignCenter|QLabel::AlignVCenter );
  lLabel->setFrameStyle( QLabel::WinPanel|QLabel::Raised );
  lLabel->setLineWidth( 2 );
  lLabel->setBackgroundColor( qApp->palette().normal().light() );

  itemList.setAutoDelete( false );
  selectedItem=0;

  splitter = new QSplitter( Qt::Vertical, this );

  lvLibrary = new RListView( splitter );
  lvLibrary->setMultiSelection( false );
  lvLibrary->addColumn( tr("Libraries") );
  lvLibrary->setTreeStepSize( 20 );
  lvLibrary->setSorting( 0 );
  lvLibrary->header()->setClickEnabled( false );
  lvLibrary->header()->setMovingEnabled( false );
  lvLibrary->header()->setResizeEnabled( false );
  lvLibrary->setHScrollBarMode( QScrollView::AlwaysOff );
  lvLibrary->setVScrollBarMode( QScrollView::AlwaysOn );
  lvLibrary->setMinimumSize( 0, 0 );

  scrollView = new RScrollView( splitter, "scrollView" );
  scrollView->enableClipper( true );
  scrollView->setVScrollBarMode( QScrollView::AlwaysOn );
  scrollView->setHScrollBarMode( QScrollView::AlwaysOff );
  scrollView->resizeContents( scrollView->width(), scrollView->height() );
  scrollView->setMinimumSize( 0, 0 );

  splitter->setResizeMode( lvLibrary, QSplitter::KeepSize );
  splitter->setResizeMode( scrollView, QSplitter::KeepSize );

  lHeight = new QLabel( tr("Height:"), this );
  lHeight->setAlignment( AlignRight|AlignVCenter );
  eHeight = new RLineEdit( this );

  lLength = new QLabel( tr("Length:"), this );
  lLength->setAlignment( AlignRight|AlignVCenter );
  eLength = new RLineEdit( this );

  lFactor = new QLabel( tr("Factor:"), this );
  lFactor->setAlignment( AlignRight|AlignVCenter );
  eFactor = new RLineEdit( this );

  lAngle = new QLabel( tr("Angle:"), this );
  lAngle->setAlignment( AlignRight|AlignVCenter );
  eAngle = new RLineEdit( this );

  bInsert = new QPushButton( tr("Insert"), this );
  connect( bInsert, SIGNAL(clicked()), this, SLOT(insertPart()) );

  // Fill directory list with possible directories for library trees:
  //
  fillDirectoryList();

  updateLibraryTree();

  // Fill scrollview with files from library directories:
  //
  updatePreview(0);

  connect( lvLibrary, SIGNAL(selectionChanged(QListViewItem*)),
           this, SLOT(updatePreview(QListViewItem*)) );
}


RLibraryBrowser::~RLibraryBrowser()
{

}


void
RLibraryBrowser::resizeEvent( QResizeEvent* _ev )
{
  RWidget::resizeEvent(_ev);
  resizeChildren();
}


void
RLibraryBrowser::resizeChildren()
{
  int ly = height()-(DEF_BSIZE0+2)*5;  // Begin of labels / edit fields
  int lx = width()/2+1;                // x of second column
  int w = width()/2-2;                // width of column

  if(lLabel) {
    lLabel->setGeometry( 1, 1,
                         width()-2, DEF_BSIZE0);
  }

  //int headerHeight = lvLibrary->header()->sizeHint().height();

  splitter->setGeometry( 1, DEF_BSIZE0+2,
                         width()-2, ly-DEF_BSIZE0-4 );

  QValueList<int> vl;
  vl << 2 << 3;
  splitter->setSizes( vl );

  //lvLibrary->setGeometry( 2, 2,
  //                        width()-4, DEF_BSIZE0*5 );
  lvLibrary->setColumnWidth( 0, width()-lvLibrary->verticalScrollBar()->width()-10 );

  //scrollView->setGeometry( 2, 4+DEF_BSIZE0*5,
  //                         width()-4, ly-DEF_BSIZE0*5-6 );

  lHeight->setGeometry( 1, ly,
                        w, DEF_BSIZE0 );
  eHeight->setGeometry( lx, ly,
                        w, DEF_BSIZE0 );
  ly+=DEF_BSIZE0+2;

  lLength->setGeometry( 1, ly,
                        w, DEF_BSIZE0 );
  eLength->setGeometry( lx, ly,
                        w, DEF_BSIZE0 );
  ly+=DEF_BSIZE0+2;

  lFactor->setGeometry( 1, ly,
                        w, DEF_BSIZE0 );
  eFactor->setGeometry( lx, ly,
                        w, DEF_BSIZE0 );
  ly+=DEF_BSIZE0+2;

  lAngle->setGeometry( 1, ly,
                       w, DEF_BSIZE0 );
  eAngle->setGeometry( lx, ly,
                        w, DEF_BSIZE0 );

  ly+=DEF_BSIZE0+2;

  bInsert->setGeometry( 1, ly,
                        width()-4, DEF_BSIZE0 );
}



/** Fills directories which can contain libraries into directoryList
*/
void
RLibraryBrowser::fillDirectoryList()
{
  directoryList.clear();
  directoryList.append( RCONFIG->getPrgDir().path()+"/libraries" );
  if(!RUNNING_WINDOWS) {
    directoryList.append( RCONFIG->getConfigDir().path()+"/libraries" );
  }

  for(int i=1; i<=10; ++i) {
    QString num;
    num.setNum( i );
    QString path = RCONFIG->getSetting( "Paths:LibraryPath" + num );
    if( !path.isEmpty() ) {
      directoryList.append( path );

      RLOG( "\nLibraryPath: " );
      RLOG( path );
    }
  }
}


/** Fills the tree with all directories / subdirectories of the libray paths
*/
void
RLibraryBrowser::updateLibraryTree()
{
  QStringList::Iterator it;
  QDir dir;

  // Collect all sub directories of library paths:
  //
  for( it=directoryList.begin(); it!=directoryList.end(); ++it ) {
    dir.setPath( (*it) );
    if( dir.exists() ) {
      appendTree( 0, dir.path() );
    }
  }
}


/** Appends the directory tree in the given directory to the given item.
*/
void
RLibraryBrowser::appendTree( RListViewItem* _item, QString _directory )
{
  QStringList::Iterator it;
  QDir dir( _directory );

  RLOG( "\nAppend tree: " );
  RLOG( _directory );

  // Read directories in this directory:
  //
  if( dir.exists() ) {
    QStringList lDirectoryList = dir.entryList( QDir::Dirs, QDir::Name );

    RListViewItem* newItem;
    RListViewItem* searchItem;
    for( it=lDirectoryList.begin(); it!=lDirectoryList.end(); ++it ) {
      if( (*it)!="." && (*it)!="..") {

        newItem=0;

        // Look for an item already existing and take this
        //   instead of making a new one:
        //
        if(_item) searchItem = (RListViewItem*)_item->firstChild();
        else      searchItem = (RListViewItem*)lvLibrary->firstChild();
        while( searchItem ) {
          if( searchItem->text(0)==(*it) ) {
            newItem=searchItem;
            break;
          }
          searchItem = (RListViewItem*)searchItem->nextSibling();
        }

        // Create new item if no existing was found:
        //
        if(!newItem) {
          if(_item) newItem = new RListViewItem( _item, (*it) );
          else      newItem = new RListViewItem( lvLibrary, (*it) );
        }

        appendTree( newItem, _directory+"/"+(*it) );
      }
    }
  }
}



/** Update the preview (scroll view with library items)
*/
void
RLibraryBrowser::updatePreview( QListViewItem* _item )
{
  if(_item) {
    QStringList::Iterator it;
    QStringList::Iterator it2;
    QDir itemDir;
    QStringList itemPathList;

    QString relPath = "/"+((RListViewItem*)_item)->getFullPath();

    for( RLibraryItem* i=itemList.first(); i!=0; i=itemList.next() ) {
      delete i;
    }

    itemList.clear();

    // Collect all file names of items:
    //
    for( it=directoryList.begin(); it!=directoryList.end(); ++it ) {
      itemDir.setPath( (*it)+relPath );
      if( itemDir.exists() ) {
        QStringList itemNameList = itemDir.entryList( QDir::Files, QDir::Name );
        for( it2=itemNameList.begin(); it2!=itemNameList.end(); ++it2 ) {
          itemPathList += itemDir.path()+"/"+(*it2);
        }
      }
    }

    // Sort entries:
    //
    itemPathList.sort();

    // Fill items into scroll view:
    //
    RLibraryItem* newItem;
    int lx=0, ly=0, bottom=0, s=64;
    for( it=itemPathList.begin(); it!=itemPathList.end(); ++it ) {
      newItem = new RLibraryItem( scrollView->viewport() );
      connect( newItem, SIGNAL(activated(RLibraryItem*)),
               this, SLOT(slotItemActivated(RLibraryItem*)) );
      itemList.append( newItem );
      newItem->setFixedSize( s, s );
      scrollView->addChild( newItem, lx, ly );
      newItem->show();
      newItem->setFile( QFileInfo(*it) );
      bottom = ly+s;
      lx+=s;
      if( lx+s>scrollView->width() ) {
        lx=0;
        ly+=s;
      }
    }

    scrollView->resizeContents( scrollView->width(), bottom );

    selectedItem = 0;
  }

}



/** Called if the user clicked an item.
*/
void
RLibraryBrowser::slotItemActivated( RLibraryItem* _item )
{
  // Mark selected item:
  //
  for( RLibraryItem* i=itemList.first(); i!=0; i=itemList.next() ) {
    if( i!=_item && i->isActive() ) {
      i->setActive(false);
    }
  }

  if( !_item->isActive() ) _item->setActive(true);

  // Get reference height from graphic:
  //
  eHeight->setText( _item->getReferenceValue('h') );
  eLength->setText( _item->getReferenceValue('l') );
  eFactor->setText( "1.0" );
  eAngle->setText( "0.0" );

  // Deactivate unused layout objects:
  //
  eHeight->setEnabled( !eHeight->text().isEmpty() );
  eLength->setEnabled( !eLength->text().isEmpty() );

  selectedItem = _item;
}


/** Returns the selected graphic (pointer) or 0
*/
RGraphic*
RLibraryBrowser::getSelectedGraphic()
{
  if(selectedItem) return selectedItem->getGraphic();
  else return 0;
}


double
RLibraryBrowser::getHeight()
{
  return eHeight->getFloat();
}


double
RLibraryBrowser::getLength()
{
  return eLength->getFloat();
}


double
RLibraryBrowser::getAngle()
{
  return eAngle->getFloat();
}


double
RLibraryBrowser::getFactor()
{
  return eFactor->getFloat();
}


void
RLibraryBrowser::insertPart()
{
  // Add layers of the part to graphic:
  if( selectedItem ) {
    //RAppWin::getRAppWin()->raise();
    //RAppWin::getRAppWin()->setActiveWindow();
    RAppWin::getRAppWin()->editInsertPart();
  }
}


















