/**********************************************************************************************
    Copyright (C) 2006, 2007 Oliver Eichler oliver.eichler@gmx.de

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA

  Garmin and MapSource are registered trademarks or trademarks of Garmin Ltd.
  or one of its subsidiaries.

**********************************************************************************************/

#include "CGarminDBRoute.h"
#include "CGarminRoute.h"
#include "CToolViewDist.h"
#include "CToolViewRoute.h"
#include "CCentralResources.h"
#include "icons.h"
#include "CGpx.h"
#include "GarminIcons.h"

#include <IDevice.h>

#include <QtGui>

CGarminDBRoute::CGarminDBRoute(QTabWidget * parent)
: QObject(parent)
, tab(parent)
{
    toolviewRoute = new CToolViewRoute(parent,this);
    toolviewRoute->show();
    parent->addTab(toolviewRoute,QPixmap(iconRoute16x16),"");
    parent->setTabToolTip(parent->indexOf(toolviewRoute), tr("Route View"));

    toolviewDist = new CToolViewDist(parent,this);
    toolviewDist->show();
    parent->addTab(toolviewDist,QPixmap(iconRuler16x16),"");
    parent->setTabToolTip(parent->indexOf(toolviewDist), tr("Distance View"));

}


CGarminDBRoute::~CGarminDBRoute()
{
}


void CGarminDBRoute::clear()
{
    removeDistances(distances.keys());
}


void CGarminDBRoute::gainFocus(const QString& toolViewName)
{
    if(tab) {
        if((tab->currentWidget() != toolviewDist) && (toolViewName == toolviewDist->objectName())) {
            tab->setCurrentWidget(toolviewDist);
        }
        else if((tab->currentWidget() != toolviewRoute) && (toolViewName == toolviewRoute->objectName())) {
            tab->setCurrentWidget(toolviewRoute);
        }

    }
}


void CGarminDBRoute::highlightDistance(const QString& key)
{
    if(distances.contains(key)) {
        pRoute = distances[key];
    }

    emit sigDistListChanged();
}


void CGarminDBRoute::highlightRoute(const QString& key)
{
    if(routes.contains(key)) {
        pRoute = routes[key];
    }

    emit sigRouteListChanged();
}


CGarminRoute * CGarminDBRoute::getCurrentRoute()
{
    return pRoute;
}


void CGarminDBRoute::newDistance()
{
    pRoute = new CGarminRoute(tr("Distance %1").arg(distances.size()),this);
    distances[pRoute->getName()] = pRoute;
    connect(pRoute,SIGNAL(sigRouteChanged()),SIGNAL(sigDistListChanged()));

    emit sigDistListChanged();
}


void CGarminDBRoute::newRoute()
{
    pRoute = new CGarminRoute(tr("Route %1").arg(routes.size()+1),this);
    routes["user|" + pRoute->getName()] = pRoute;
    connect(pRoute,SIGNAL(sigRouteChanged()),SIGNAL(sigRouteListChanged()));

    emit sigRouteListChanged();
}


void CGarminDBRoute::removeDistances(const QStringList& keys)
{
    QString key;
    foreach(key,keys) {
        if(!distances.contains(key)) continue;
        delete distances.take(key);
    }

    emit sigDistListChanged();
}


void CGarminDBRoute::removeRoutes(const QStringList& keys)
{
    QString key;
    foreach(key,keys) {
        if(!routes.contains(key)) continue;
        delete routes.take(key);
    }

    emit sigRouteListChanged();

}


void CGarminDBRoute::saveGPX(CGpx& gpx)
{
    QDomElement root = gpx.documentElement();
    QMap<QString,CGarminRoute*>::iterator route = routes.begin();
    while(route != routes.end()) {
        QDomElement gpxRoute = gpx.createElement("rte");
        root.appendChild(gpxRoute);

        QDomElement name = gpx.createElement("name");
        gpxRoute.appendChild(name);
        QDomText _name_ = gpx.createTextNode((*route)->getName());
        name.appendChild(_name_);

        QString symname; QPixmap dummy;
        getWptIconById((*route)->getSmbl(), dummy, symname);

        unsigned cnt = 0;
        QList<XY>::const_iterator rtept = (*route)->begin();
        while(rtept != (*route)->end()) {
            QDomElement gpxRtept = gpx.createElement("rtept");
            gpxRoute.appendChild(gpxRtept);

            gpxRtept.setAttribute("lat",QString::number(rtept->v,'f',6));
            gpxRtept.setAttribute("lon",QString::number(rtept->u,'f',6));

            QString str = QString("%1").arg(++cnt,3,10,QChar('0'));

            QDomElement name = gpx.createElement("name");
            gpxRtept.appendChild(name);
            QDomText _name_ = gpx.createTextNode(str);
            name.appendChild(_name_);

            QDomElement cmt = gpx.createElement("cmt");
            gpxRtept.appendChild(cmt);
            QDomText _cmt_ = gpx.createTextNode(str);
            cmt.appendChild(_cmt_);

            QDomElement desc = gpx.createElement("desc");
            gpxRtept.appendChild(desc);
            QDomText _desc_ = gpx.createTextNode(str);
            desc.appendChild(_desc_);

            QDomElement sym = gpx.createElement("sym");
            gpxRtept.appendChild(sym);
            QDomText _sym_ = gpx.createTextNode(symname);
            sym.appendChild(_sym_);

            ++rtept;
        }

        ++route;
    }
}


void CGarminDBRoute::loadGPX(CGpx& gpx)
{
    const QDomNodeList& rtes = gpx.elementsByTagName("rte");
    uint N = rtes.count();
    for(uint n = 0; n < N; ++n) {
        const QDomNode& rte = rtes.item(n);

        CGarminRoute * r = 0;
        /* name is not a required element. */
        if(rte.namedItem("name").isElement()) {
            r = new CGarminRoute(rte.namedItem("name").toElement().text(),this);
        }
        else {
            /* Use desc if name is unavailable, else give it no name. */
            if (rte.namedItem("desc").isElement())
                r = new CGarminRoute(rte.namedItem("desc").toElement().text(),this);
            else
                r = new CGarminRoute("Unnamed",this);
        }

        QDomElement rtept = rte.firstChildElement("rtept");

        while (!rtept.isNull()) {
            XY pt;
            QDomNamedNodeMap attr = rtept.attributes();

            pt.u = attr.namedItem("lon").nodeValue().toDouble();
            pt.v = attr.namedItem("lat").nodeValue().toDouble();

            r->addPosition(pt.u,pt.v);

            if(rtept.namedItem("sym").isElement()) {
                QString symname = rtept.namedItem("sym").toElement().text();
                uint16_t smbl = 0;
                getWptIconByName(symname, smbl);
                r->setSmbl(smbl);
            }

            rtept = rtept.nextSiblingElement("rtept");
        }

        if(routes.contains("gpx|" + r->getName())) {
            delete routes.take("gpx|" + r->getName());
        }

        routes["gpx|" + r->getName()] = r;

        connect(r,SIGNAL(sigRouteChanged()),SIGNAL(sigRouteListChanged()));

    }

    emit sigRouteListChanged();
}


void CGarminDBRoute::uploadRoutes()
{
    int cnt = 0;
    std::list<Garmin::Route_t> rtes;
    QMap<QString,CGarminRoute*>::iterator route = routes.begin();
    while(route != routes.end()) {
        rtes.push_back((*route)->toGarminRoute(++cnt));
        ++route;
    }
    // nothing to do?
    if(rtes.size() == 0) return;

    Garmin::IDevice * dev = 0;
    try
    {
        dev = gpResources->device();
        if(dev) {
            dev->uploadRoutes(rtes);
        }
    }
    catch(int e) {
        if(dev == 0) return;
        QMessageBox::warning(0,tr("Device Link Error"),dev->getLastError().c_str(),QMessageBox::Ok,QMessageBox::NoButton);
        if(e == Garmin::errSync) {
            gpResources->resetDevice();
        }
    }
}
