/**********************************************************************************************
    Copyright (C) 2007 Oliver Eichler oliver.eichler@gmx.de

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA
**********************************************************************************************/

#include "CToolViewLog.h"
#include "CGarminLiveLog.h"
#include "GeoMath.h"
#include "CCentralResources.h"
#include "CGarminMap.h"
#include <IDevice.h>

#include <limits>
#include <QtGui>

#undef DBG_SHOW_POSITION_DATA

CToolViewLog::CToolViewLog(QWidget * parent, CGarminLiveLog * log)
: QWidget(parent)
{
    setupUi(this);
    setObjectName("Log");

    connect(log,SIGNAL(sigPosition(const CGarminLiveLog::Pvt_t&)),this,SLOT(slotPosition(const CGarminLiveLog::Pvt_t&)));

}


CToolViewLog::~CToolViewLog()
{

}


void CToolViewLog::slotPosition(const CGarminLiveLog::Pvt_t& pvt)
{

#ifdef DBG_SHOW_POSITION_DATA
    qDebug() << "fix        " << pvt.fix;
    qDebug() << "lon        " << pvt.lon;
    qDebug() << "lat        " << pvt.lat;

    qDebug() << "epe        " << pvt.epe;
    qDebug() << "eph        " << pvt.eph;
    qDebug() << "epv        " << pvt.epv;
#endif

    float speed_km_h = pvt.velocity() * 3.6;
    float heading = pvt.heading();
    if( speed_km_h < 0.2 ) {

        // some pretty arbitrary threshold ...
        // with a horizontal error of +/-5m it never goes above
        // 0.06 km/h while standing still, but you don't always
        // have +/-5m...
        speed_km_h = 0.0;
        heading = std::numeric_limits<float>::quiet_NaN();
    }
    QDateTime t = pvt.localTime();

    if(pvt.fix > 1) {
        QString str;
        GPS_Math_Deg_To_Str(pvt.lon, pvt.lat, str);
        if(pvt.lockToCenter) str += " (centered)";

        QPointF infoPoint;
        QMultiMap<QString,QString> dict;

        XY pt;

        pt.u = DEG_TO_RAD * pvt.lon;
        pt.v = DEG_TO_RAD * pvt.lat;

        pt = pj_fwd(pt,*gpProj);

        infoPoint.setX(pt.u);
        infoPoint.setY(pt.v);

        gpResources->mapdb().getInfo(infoPoint,dict,gpResources->canvas().getMap());

        QString key;
        QStringList keys = dict.uniqueKeys();
        qSort(keys);
        QString info;

        info.clear();
        foreach(key,keys) {
            info += "<b>" + key + ":</b><br/>";
            const QStringList& values = dict.values(key).toSet().toList();
            info += values.join("<br/>");
            info += "<br/>";
        }

        lblStreet->setText(info);

        lblPosition->setText(str);
        lblAltitude->setText(QString::number(pvt.alt + pvt.msl_hght,'f',0) + "m");
        lblErrorHoriz->setText(QString("\261%1m").arg(pvt.eph/2,0,'f',0));
        lblErrorVert->setText(QString("\261%1m").arg(pvt.epv/2,0,'f',0));
        lblSpeed->setText(QString("%1km/h").arg(speed_km_h, 0, 'f', 1));
        lblHeading->setText(QString("%1\260T").arg(nearbyintf(heading),3,'f',0,'0'));
        lblTime->setText(t.toString());
    }
    else {
        lblPosition->setText(tr("GPS signal low"));
        lblAltitude->setText("-");
        lblErrorHoriz->setText("-");
        lblErrorVert->setText("-");
        lblSpeed->setText("-");
        lblHeading->setText("-");
        lblTime->setText("-");
    }
}
