/**********************************************************************************************
    Copyright (C) 2007 Oliver Eichler oliver.eichler@gmx.de

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA
**********************************************************************************************/

#include "CToolViewRoute.h"
#include "CCentralResources.h"
#include "CGarminDBRoute.h"
#include "CGarminRoute.h"
#include "CDlgEditRoute.h"
#include "icons.h"

#include <QtGui>

CToolViewRoute::CToolViewRoute(QWidget * parent, CGarminDBRoute* db)
: QWidget(parent)
, originator(false)
{
    setupUi(this);
    setObjectName("Route");

    connect(db,SIGNAL(sigRouteListChanged()),this,SLOT(slotUpdateRouteList()));
    connect(db,SIGNAL(sigRouteChanged()),this,SLOT(slotUpdateRouteList()));

    connect(listRoutes,SIGNAL(itemClicked(QTreeWidgetItem*,int) ),this,SLOT(slotItemClicked(QTreeWidgetItem*)));
    connect(listRoutes,SIGNAL(itemDoubleClicked(QTreeWidgetItem*,int) ),this,SLOT(slotItemDoubleClicked(QTreeWidgetItem*)));

    contextMenu = new QMenu(this);
    contextMenu->addAction(QPixmap(iconEdit16x16),tr("Edit..."),this,SLOT(slotEdit()));
    contextMenu->addAction(QPixmap(iconTrash16x16),tr("Delete"),this,SLOT(slotDelete()));

    connect(listRoutes,SIGNAL(customContextMenuRequested(const QPoint&)),this,SLOT(slotContextMenu(const QPoint&)));

}


CToolViewRoute::~CToolViewRoute()
{

}


void CToolViewRoute::keyPressEvent(QKeyEvent * e)
{
    if(e->key() == Qt::Key_Delete) {
        QStringList keys;
        QTreeWidgetItem * item;
        const QList<QTreeWidgetItem*>& items = listRoutes->selectedItems();
        foreach(item,items) {
            keys << item->data(eName,Qt::UserRole).toString();
            delete item;
        }
        originator = true;
        gpResources->routedb().removeRoutes(keys);
        originator = false;
    }
    else {
        QWidget::keyPressEvent(e);
    }
}


void CToolViewRoute::slotUpdateRouteList()
{
    if(originator) return;
    CGarminRoute * actRoute = gpResources->routedb().getCurrentRoute();
    listRoutes->clear();

    const QMap<QString,CGarminRoute*>& routes = gpResources->routedb().getRouteDict();
    QMap<QString,CGarminRoute*>::const_iterator route = routes.begin();
    while(route != routes.end()) {
        QTreeWidgetItem * entry = new QTreeWidgetItem(listRoutes);
        entry->setIcon(eIcon,(*route)->getIcon());
        entry->setText(eName,(*route)->getName());
        entry->setData(eName,Qt::UserRole,route.key());
        entry->setText(eDistance,route.key().section('|',0,0));
        if(*route == actRoute) entry->setSelected(true);
        ++route;
    }

    listRoutes->header()->setResizeMode(eIcon,QHeaderView::Interactive);
    listRoutes->resizeColumnToContents(eIcon);
    listRoutes->header()->setResizeMode(eName,QHeaderView::Interactive);
    listRoutes->resizeColumnToContents(eName);

}


void CToolViewRoute::slotItemClicked(QTreeWidgetItem * item)
{
    originator = true;
    gpResources->routedb().highlightRoute(item->data(eName,Qt::UserRole).toString());
    originator = false;
}


void CToolViewRoute::slotItemDoubleClicked(QTreeWidgetItem * item)
{
    QString key = item->data(eName,Qt::UserRole).toString();
    CGarminRoute* route = gpResources->routedb().getRouteDict()[key];

    QRectF area;

    QList<XY>::const_iterator pt1 = route->routeDegree.begin();
    QList<XY>::const_iterator pt2 = route->routeDegree.begin();

    if(pt1 == route->routeDegree.end()) return;

    area.setTopLeft(QPointF(pt1->v,pt1->u));
    area.setBottomRight(QPointF(pt1->v,pt1->u));
    ++pt1;
    while(pt1 != route->routeDegree.end()) {
        QRectF r;
        r.setTopLeft(QPointF(pt1->u,pt1->v));
        r.setBottomRight(QPointF(pt2->u,pt2->v));

        area |= r;
        ++pt1; ++pt2;
    }

    qDebug() << area;
    gpResources->canvas().zoom(area);
}


void CToolViewRoute::slotContextMenu(const QPoint& pos)
{
    if(!listRoutes->selectedItems().isEmpty()) {
        QPoint p = listRoutes->mapToGlobal(pos);
        p.setY(p.y() + listRoutes->header()->height());
        contextMenu->exec(p);
    }
}


void CToolViewRoute::slotDelete()
{
    QStringList keys;
    QTreeWidgetItem * item;
    const QList<QTreeWidgetItem*>& items = listRoutes->selectedItems();
    foreach(item,items) {
        keys << item->data(eName,Qt::UserRole).toString();
        delete item;
    }
    originator = true;
    gpResources->routedb().removeRoutes(keys);
    originator = false;
}


void CToolViewRoute::slotEdit()
{
    QTreeWidgetItem * item = listRoutes->currentItem();
    if(item == 0) return;

    QString key         = item->data(eName,Qt::UserRole).toString();
    CGarminRoute* route = gpResources->routedb().getRouteDict()[key];
    if(route == 0) return;

    CDlgEditRoute dlg(*route, this);
    dlg.exec();
}
