/*
 * QNix      (C) 2001 Derek Greene     <del@chek.com>
 * Based on code from Default, ModSystem, NextStep & Redmond kwin clients
 *	         Matthias Ettrich <ettrich@kde.org>
 *					 Karol Szwed <gallium@kde.org>
 *  			   Daniel M. Duley  <mosfet@kde.org>
 * 					 Karol Szwed <gallium@kde.org>
 *
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

/*
 * CHANGES:
 *
 * 0.2.1
 *     - Animated min/max now works
 * 0.2
 *     - Right/middle clicking on the max button causes a MaximizeHorizontal/MaximizeVertical (like all other KWin clients)
 *     - Top-resizing problem fixed by change to QNiXClient::mousePosition()
 *     - Added optional mouseover highlighting of titlebar buttons
 *     - When window is maximized, max button appearance changes to "Restore"
 *		 - Added option border decoration in bottom right hand corner
 *     - Generally cleaned up code... hopefully a bit more maintainable
 *
*/

// QT headers
#include <qlayout.h>
#include <qpainter.h>
#include <qdrawutil.h>
#include <qbitmap.h>
// KDE2 headers
#include <kconfig.h>
#include <kstddirs.h>
#include <kdebug.h>
#include <kconfig.h>
#include <kglobal.h>
#include <klocale.h>
#include <kwin/workspace.h>
#include <kwin/options.h>
#include <kpixmapeffect.h>
#include <kdrawutil.h>
// project headers
#include "qnix_client.h"

//////////////////////////////////////////////////////////////////////////////////////////////////////////////

#define BUTTON_SIZE 				13
#define MENU_BUTTON_SIZE 		16
#define MINI_BUTTON_SIZE		12
#define TITLE_PIX_WIDTH			32
#define TITLE_PIX_HEIGHT		20

// [0.2] - Shortcut macros for setting pen colour (useful for highlighting)
#define SET_HCOLOR( r, g, b ) p->setPen( QColor(r + m_highlight, g+ m_highlight, b+ m_highlight ) )
#define SET_COLOR( r, g, b )  p->setPen( QColor(r, g, b ) )

using namespace KWinInternal;

//////////////////////////////////////////////////////////////////////////////////////////////////////////////
// GLOBALS
//////////////////////////////////////////////////////////////////////////////////////////////////////////////

// PIXMAP CREATION
// -----------------------------
static void     drawButtonFrame( KPixmap *pix, const QColorGroup &g, bool sunken );
static void     drawTitleDetails( KPixmap *pix, bool active );

// SETTINGS
// -----------------------------
// Indicates whether the application icon should be shown
static bool     config_showMenuButtonIcon;
// Indicates whether the left detail should be shown or not
static bool     config_showLeftDetail;
// [0.2] - Indicates whether titlebar buttons should be highlighted on mouseover
static bool     config_highlightButtons;
// [0.2] - Indicates whether the detail in the bottom right corner should be shown
static bool     config_showCornerDetail;

// TITLEBAR RELATED
// -----------------------------
// Indicates whether the theme pixmaps have been initialised yet or not
static bool     pixmaps_created = false;
// Title Bar images
static KPixmap* pixTitleActive = 0;
static KPixmap* pixTitleInactive = 0;
static KPixmap* pixTitleLeftActive = 0;
static KPixmap* pixTitleLeftInactive = 0;
static KPixmap* pixTitleRight = 0;
// Default app icon pixmap
static QPixmap* pixDefaultAppIcon;
// Left detail pixmap
static QPixmap* pixLeftDetailActive;
static QPixmap* pixLeftDetailInactive;

// BUTTON RELATED
// -----------------------------
// Standard Button Pixmaps
static KPixmap* pixButtonNormal;
static KPixmap* pixButtonDown;
// [0.2] - Highlighted Button Pixmaps
static KPixmap* pixButtonHighlighted;
static KPixmap* pixMiniButtonHighlighted;
// Mini Button pixmap
static KPixmap* pixMiniButtonNormal;
static KPixmap* pixMiniButtonDown;
// menu Button pixmap
static KPixmap* pixMenuButton;

//////////////////////////////////////////////////////////////////////////////////////////////////////////////
// STATIC FUNCTIONS
//////////////////////////////////////////////////////////////////////////////////////////////////////////////

// ----------------------------------------------------------------
// Create all necessary pixmaps
// ----------------------------------------------------------------
static void create_pixmaps()
{
	// Only create pixmaps if they haven't already been created
	if(pixmaps_created)
			return;

	pixmaps_created = true;
	QColor c(options->color(Options::ButtonBg, false));

	// active title gradient
	// -------------------------
	pixTitleActive = new KPixmap();
	pixTitleActive->resize(TITLE_PIX_WIDTH, TITLE_PIX_HEIGHT);
	KPixmapEffect::gradient(*pixTitleActive,options->color(Options::TitleBar, true), options->color(Options::TitleBlend, true), KPixmapEffect::VerticalGradient);
 	drawTitleDetails( pixTitleActive, true );

	// inactive title gradient
	// -------------------------
	pixTitleInactive = new KPixmap();
	pixTitleInactive->resize(TITLE_PIX_WIDTH, TITLE_PIX_HEIGHT);
	// reverse gradient looks better for inactive
	KPixmapEffect::gradient(*pixTitleInactive, options->color(Options::TitleBlend, false), options->color(Options::TitleBar, false), KPixmapEffect::VerticalGradient);
 	drawTitleDetails( pixTitleInactive, false );

	// right title pix
	// ------------------------
	pixTitleRight = new KPixmap();
	pixTitleRight->resize(TITLE_PIX_WIDTH, TITLE_PIX_HEIGHT);
	// NOTHC : KPixmapEffect::gradient(*pixTitleRight, c.dark(120), c.light(110), KPixmapEffect::VerticalGradient);
	KPixmapEffect::gradient(*pixTitleRight, QColor(192, 196, 192), QColor(224, 228, 224), KPixmapEffect::VerticalGradient);

	// left title pix
	// ------------------------
	pixTitleLeftActive = new KPixmap();
	pixTitleLeftActive->resize(TITLE_PIX_WIDTH, TITLE_PIX_HEIGHT);
	KPixmapEffect::gradient(*pixTitleLeftActive,options->color(Options::TitleBar, true).dark(115), options->color(Options::TitleBlend, true).dark(115), KPixmapEffect::VerticalGradient);
	pixTitleLeftInactive = new KPixmap();
	pixTitleLeftInactive->resize(TITLE_PIX_WIDTH, TITLE_PIX_HEIGHT);
	KPixmapEffect::gradient(*pixTitleLeftInactive,options->color(Options::TitleBar, false).dark(115), options->color(Options::TitleBlend, false).dark(115), KPixmapEffect::VerticalGradient);


	// Default icon
	pixDefaultAppIcon = new QPixmap(kdelogo);
	// left detail icon
	pixLeftDetailActive = new QPixmap( qnixlogo );
	pixLeftDetailInactive = new QPixmap( iqnixlogo );

	// Buttons
	pixButtonNormal = new KPixmap;
	pixButtonDown = new KPixmap;
	pixMiniButtonNormal = new KPixmap;
	pixMiniButtonDown = new KPixmap;
	pixButtonHighlighted = new KPixmap;
	pixMiniButtonHighlighted = new KPixmap;

	// buttons (active/inactive, sunken/unsunken)
	QColorGroup g = options->colorGroup(Options::ButtonBg, true);
	c = g.background();
	pixButtonNormal->resize(BUTTON_SIZE, BUTTON_SIZE);
	pixButtonDown->resize(BUTTON_SIZE, BUTTON_SIZE);

	pixButtonHighlighted->resize(BUTTON_SIZE, BUTTON_SIZE);
	pixMiniButtonHighlighted->resize(MINI_BUTTON_SIZE, MINI_BUTTON_SIZE);

	pixMiniButtonNormal->resize(MINI_BUTTON_SIZE, MINI_BUTTON_SIZE);
	pixMiniButtonDown->resize(MINI_BUTTON_SIZE, MINI_BUTTON_SIZE);

	KPixmapEffect::gradient(*pixButtonNormal, QColor(214, 218, 214), QColor(184, 184, 184), KPixmapEffect::VerticalGradient);
	KPixmapEffect::gradient(*pixButtonDown, QColor(174, 178, 174), QColor(184, 184, 184), KPixmapEffect::VerticalGradient);
	KPixmapEffect::gradient(*pixMiniButtonNormal, QColor(216, 220, 216), QColor(184, 184, 184), KPixmapEffect::VerticalGradient);
	KPixmapEffect::gradient(*pixMiniButtonDown, QColor(174, 178, 174), QColor(184, 184, 184), KPixmapEffect::VerticalGradient);

	g = options->colorGroup(Options::ButtonBg, false);
	c = g.background();
	KPixmapEffect::gradient(*pixButtonHighlighted, QColor(239, 243, 239), QColor(218, 218, 218), KPixmapEffect::VerticalGradient);
	KPixmapEffect::gradient(*pixMiniButtonHighlighted, QColor(239, 243, 239), QColor(218, 218, 218), KPixmapEffect::VerticalGradient);
	// draw active buttons frames
	g = options->colorGroup(Options::ButtonBg, true);
	drawButtonFrame(pixButtonNormal, g, false);
	drawButtonFrame(pixButtonDown, g, true);
	drawButtonFrame(pixMiniButtonNormal, g, false);
	drawButtonFrame(pixMiniButtonDown, g, true);
	// [0.2] - highlight
	drawButtonFrame(pixButtonHighlighted, g, false);
	drawButtonFrame(pixMiniButtonHighlighted, g, false);

	// menu button
	pixMenuButton = new KPixmap;
	pixMenuButton->resize(BUTTON_SIZE, BUTTON_SIZE-2);
	KPixmapEffect::gradient(*pixMenuButton, QColor(214, 218, 214), QColor(184, 184, 184), KPixmapEffect::VerticalGradient);
	drawButtonFrame( pixMenuButton, g, false );

}

// ----------------------------------------------------------------
// Clean up afterwards
// ----------------------------------------------------------------
static void delete_pixmaps()
{
  delete pixTitleActive;
  delete pixTitleInactive;
  delete pixTitleRight;
	delete pixTitleLeftActive;
	delete pixTitleLeftInactive;

	delete pixButtonNormal;
  delete pixButtonHighlighted;
  delete pixButtonDown;

  delete pixMiniButtonNormal;
  delete pixMiniButtonHighlighted;
  delete pixMiniButtonDown;

	delete pixDefaultAppIcon;
	delete pixLeftDetailActive;
	delete pixLeftDetailInactive;
	delete pixMenuButton;
}

// ----------------------------------------------------------------
// Create the pixmap which will be used for the titlebar
// ----------------------------------------------------------------
static void drawTitleDetails( KPixmap *pix, bool active )
{
  QPainter p;
  int x2 = pix->width();
  int y2 = pix->height();
  p.begin(pix);

	// draw details
	// top of titlebar - v dark
	p.setPen( options->color(Options::TitleBlend, active ).dark( 160 )  );
	p.drawLine(0, 0, x2, 0);
	// top of titlebar - light
	p.setPen( options->color(Options::TitleBar,active ).light( 111 ) );
	p.drawLine(0, 1, x2, 1);
	// top of titlebar - mid
	p.setPen( options->color(Options::TitleBar, active ));
	p.drawLine(0, 2, x2, 2);
	// top of titlebar - dark
	p.setPen( options->color(Options::TitleBlend, active ).dark( 130 )  );
	p.drawLine(0, 3, x2, 3);

	// bottom of titlebar - quite light
	p.setPen( options->color(Options::TitleBlend, active ).light( 110 )  );
	p.drawLine(0, y2-2, x2, y2-2);
	// bottom of titlebar - almost v dark
	// NOT HC : p.setPen( options->color(Options::TitleBlend, active ).dark( 145 )  );
	p.setPen( QColor( 151, 151, 151 ) );
	p.drawLine(0, y2-1, x2, y2-1);
	// NOT HC : p.setPen( options->color(Options::TitleBlend, active ).dark( 130 )  );
	p.setPen( QColor( 193, 197, 193 ) );
	p.drawLine(0, y2, x2 + 1, y2);
	// bottom of titlebar - v dark - only if inactive
	if( !active )
	{
		p.setPen( QColor( 68, 67, 63 ) );
		p.drawLine(1, y2, x2 - 6, y2 );
	}

}

// ----------------------------------------------------------------
// Create a blank button "frame" which will be used as a background
// for all window buttons
// ----------------------------------------------------------------
static void drawButtonFrame( KPixmap *pix, const QColorGroup &g, bool sunken )
{
	QPainter p;
	int y1 = 0;

	int x2 = pix->width() - 1;
	int y2 = pix->height() - 1;
	p.begin(pix);

	// Outer Frame
	p.setPen( sunken ? QColor( 14, 14, 14 ) : QColor( 56, 60, 56 ) );
	// across top
	p.drawLine(0, y1, x2, y1);
	// down right
	p.drawLine(x2, y1, x2, y2);
	if( sunken)
		p.setPen( QColor( 56, 60, 56 ) );
	// across bottom
	p.drawLine(0, y2, x2, y2);
	// down left
	p.drawLine(	0, y1, 0, y2);

	// Inner Frame
	p.setPen( sunken ? QColor( 101, 101, 101) : QColor(232, 232, 232 ) );
	// top
	p.drawLine(1, y1+1, x2-2, y1+1);
	// left
	p.drawLine(	1, y1+1, 1, y2-2);
	// top right corner
	p.setPen( sunken ? QColor(91, 89, 91) : QColor(208, 204, 208 ) );
	p.drawPoint( x2-1, y1+1 );
	// bottom right corner
	p.drawPoint( 1, y2-1 );
	// bottom
	p.setPen( QColor(144, 144, 144 ) );
	p.drawLine(2, y2-1, x2-2, y2-1);
	// right
	p.setPen( sunken ? QColor( 152, 152, 152) : QColor(160, 160, 160 ) );
	p.drawLine(x2-1, 2, x2-1, y2-1);
}

// ----------------------------------------------------------------
// Read client configuration data
// ----------------------------------------------------------------
static bool read_config()
{
	KConfig config("kwinqnixrc");
	config.setGroup("General");

	config_showMenuButtonIcon = config.readBoolEntry( "ShowMenuButtonIcon", false );
	config_showLeftDetail     = config.readBoolEntry( "ShowLeftDetail", true );
	config_showCornerDetail		= config.readBoolEntry( "ShowCornerDetail", true );
	config_highlightButtons   = config.readBoolEntry( "HighlightButtons", false );

	return true;
}


//////////////////////////////////////////////////////////////////////////////////////////////////////////////
// CLASS : QNiXClient
//////////////////////////////////////////////////////////////////////////////////////////////////////////////

// ----------------------------------------------------------------
// Constructor - initialise the client
// ----------------------------------------------------------------
QNiXClient::QNiXClient(Workspace * ws, WId w, QWidget * parent, const char * name)
  : Client(ws, w, parent, name, WResizeNoErase), titlebar(0)
{

  setBackgroundMode(NoBackground);
  //QFontMetrics fm(options->font(isActive(), tool_));

	// handle toolwindows
  if ( isTool() )
	{
  	titleHeight = 14;
    smallButtons = true;
	}
	else
	{
  	titleHeight = 22;
    smallButtons = false;
 	}

  setupLayout();

	// Make sure that the menu button uses the correct mini-icon
  iconChange();

  connect(options, SIGNAL(resetClients()), this, SLOT(slotReset()));
}

// ----------------------------------------------------------------
// Setup the layout of this KWin Theme
// i.e. where title bar, buttons & window content go
// ----------------------------------------------------------------
void QNiXClient::setupLayout()
{
  const uint sideMargin    = 6;
  const uint bottomMargin  = 6;

	// create titlebar
	titlebar = new QSpacerItem( 0, titleHeight,  QSizePolicy::Expanding, QSizePolicy::Minimum);

	connect(options, SIGNAL(resetClients()), this, SLOT(slotReset()));

	// setup layout
	mainLayout = new QVBoxLayout(this);
	titleLayout = new QHBoxLayout();
	windowLayout = new QHBoxLayout();

	mainLayout->addLayout(titleLayout);
	mainLayout->addLayout(windowLayout, 1);
	// add bottom margin
	mainLayout->addSpacing( bottomMargin );

	// add left margin
	windowLayout->addSpacing( sideMargin );
	// add window
	windowLayout->addWidget( windowWrapper() );
	// add right margin
	windowLayout->addSpacing( sideMargin );

	// create buttons
	button[BtnMenu]    = new QNixButton(this, "menu", BtnMenu, NULL, config_showMenuButtonIcon, smallButtons,  i18n("Menu"));
	button[BtnClose]   = new QNixButton(this, "close", BtnClose, NULL, false, smallButtons,  i18n("Close"));
	button[BtnMin] 		 = new QNixButton(this, "iconify", BtnMin, NULL, false, smallButtons, i18n("Minimize"));
	button[BtnMax]     = new QNixButton(this, "maximize", BtnMax, NULL, false, smallButtons,  i18n("Maximize"));
	// connect buttons to their appropriate actions
	connect( button[BtnMenu],    SIGNAL(pressed()), this, SLOT( slotMenu() ));
	connect( button[BtnClose], SIGNAL(clicked()), this, SLOT( closeWindow() ) );
	connect( button[BtnMin], SIGNAL(clicked()), this, SLOT( iconify() ) );
	//[0.2] - changed to fix min/max problem
	connect( button[BtnMax], SIGNAL( clicked() ), this, ( SLOT( slotMaximize() ) ) );

	// Add everything to the main layout
	titleLayout->addSpacing(5);
	titleLayout->addWidget( button[BtnMenu] );
	titleLayout->addItem(titlebar);
	titleLayout->addSpacing(5);
	titleLayout->addWidget( button[BtnMin] );
	titleLayout->addSpacing(4);
	titleLayout->addWidget( button[BtnMax] );
	titleLayout->addSpacing(4);
	titleLayout->addWidget( button[BtnClose] );
	titleLayout->addSpacing(6);
}

// ================================================================================
// EVENTS
// ================================================================================

// ----------------------------------------------------------------
// Paint the title bar details
// Most of the interesting (messy) code is here
// ----------------------------------------------------------------
void QNiXClient::paintEvent(QPaintEvent* pe)
{
	QColorGroup g;
  QPainter p(this);
  int w = width();
  int h = height();

	QRect t = titlebar->geometry();
	t.setTop(1);

	// draw titlebar right gradient (silver)
	p.drawTiledPixmap(t.width() + t.x() - 4, t.y(), width() - (t.width() + t.x() -  2) , t.height()-2, *pixTitleRight );

  // outer frame ( margin )
	// -------------------------------------
	// left margin
	p.setPen(Qt::white);
	p.drawLine( 2, 19, 2, h-4);
	p.setPen( QColor( 219, 219, 219 ) );
	p.drawLine( 3, 19, 3, h-4);
	p.setPen( QColor(158, 158, 158 ) );
	p.drawLine( 4, 19, 4, h-4);
	p.setPen( QColor( 71, 71, 71 ) );
	p.drawLine( 5, 19, 5, h-4);   // innermost

	// bottom margin
	p.setPen( QColor( 153, 153, 153 ) );
	p.drawLine( 2, h-3, w-2, h-3);
	p.setPen( QColor( 219, 219, 219 ) );
	p.drawLine( 2, h-4, w-2, h-4);
	p.setPen(Qt::white);
	p.drawLine( 2, h-5, w-2, h-5);
	p.setPen( QColor( 71, 71, 71 ) );
	p.drawLine( 5, h-6, w-5, h-6);		// innermost

	// right margin
	p.setPen( QColor(158, 158, 158 ) );
	p.drawLine( w-3, 3, w-3, h-4);
	p.setPen( QColor( 219, 219, 219 ) );
	p.drawLine( w-4, 3, w-4, h-4);
	p.setPen(Qt::white);
	p.drawLine( w-5, 21, w-5, h-4);
	p.setPen( QColor( 71, 71, 71 ) );
	p.drawLine( w-6, 22, w-6, h-6);   // innermost

	// draw bottom horiz lines on right - bottom
	p.setPen( QColor( 68, 67, 63 ) );
	p.drawLine( t.width() + t.x() - 4, t.bottom(), width()-6, t.bottom() );
	p.setPen( options->color(Options::TitleBlend, false ).dark( 145 )  );
	p.drawLine( t.width() + t.x() - 4, t.bottom()-1, width()-6, t.bottom()-1 );

	// draw bottom horiz lines on right - top
	p.setPen( Qt::white );
	p.drawLine( t.width() + t.x() - 4, 2, width()-4, 2);

	// draw titlebar gradient
	// note : it must go all the way from x = 1 to cover the menu button background
	p.drawTiledPixmap( 1, t.y(), t.width()+t.x()-2, t.height()+1, isActive() ? *pixTitleActive : *pixTitleInactive);

	// Is the client configured to show the extra detail on the left?
	if( config_showLeftDetail )
	{
		int lw = 20; // left detail width
		p.setPen( options->color(Options::TitleBlend, isActive() ).light( 110 )  );
		p.drawLine(2, t.y(), 2, t.height()+1);
		p.drawTiledPixmap( 2, t.y(), lw-1, t.height()+1, isActive() ? *pixTitleLeftActive : *pixTitleLeftInactive);
		// draw split detail - 1st in titlebar colour
		p.setPen( options->color(Options::TitleBlend, isActive() ).dark( 145 )  );
		p.drawLine(lw+1, 2, lw+1, 4);
		p.drawLine(lw+2, 4, lw+2, t.bottom()-3);
		p.drawLine(lw+1, t.bottom()-3, lw+1, t.bottom()-1);
		// 2nd in dark gray
		p.setPen( QColor( 72, 72, 72 ) );
		p.drawLine(lw, 2, lw, 4);
		p.drawLine(lw+1, 4, lw+1, t.bottom()-3);
		p.drawLine(lw, t.bottom()-3, lw, t.bottom()-1);
	}

	// just an extra detail on left top corner
	p.setPen( options->color(Options::TitleBar,isActive() ).light( 110 ) );
	p.drawLine (1, 4, 1, t.bottom() - 2);
	p.setPen( options->color(Options::TitleBar,isActive() ).dark( 110 ) );
	p.drawLine (1, t.bottom() - 2, 1, t.bottom());

	// draw bottom horiz lines
	p.setPen( options->color(Options::TitleBlend, isActive() ).dark( 145 )  );
	p.drawLine(1, t.bottom()-1, t.right(), t.bottom()-1);

	// draw split detail - 1st in titlebar colour
	p.drawLine(t.width()+t.x()-1, 2, t.width()+t.x()-1, 4);
	p.drawLine(t.width()+t.x()-2, 4, t.width()+t.x()-2, t.bottom()-3);
	p.drawLine(t.width()+t.x()-1, t.bottom()-3, t.width()+t.x()-1, t.bottom()-1);
	// 2nd in dark gray
	p.setPen( QColor( 72, 72, 72 ) );
	p.drawLine(t.width()+t.x(), 2, t.width()+t.x(), 4);
	p.drawLine(t.width()+t.x()-1, 4, t.width()+t.x()-1, t.bottom()-3);
	p.drawLine(t.width()+t.x(), t.bottom()-3, t.width()+t.x(), t.bottom()-1);
	// 3rd in light gray
	// NOT HC : p.setPen( options->color(Options::TitleBlend, isActive() ).dark( 130 )  );
	p.setPen( QColor( 193, 197, 193 ) );
	p.drawLine(t.width()+t.x()+1, 2, t.width()+t.x()+1, 4);
	p.drawLine(t.width()+t.x(), 4, t.width()+t.x(), t.bottom()-3);
	p.drawLine(t.width()+t.x()+1, t.bottom()-3, t.width()+t.x()+1, t.bottom()-2);

	// draw bottom line
	p.setPen( QColor( 72, 72, 72 ) );
	p.drawLine( 1, t.bottom(), width()-6, t.bottom() );

	// draw title text
	// -----------------------
	// Align text center
  //p.drawText(t, AlignCenter, caption());
 	QFont fnt = options->font(true);
	// Set smaller font size if tool window
  if ( smallButtons )
  {
       fnt.setPointSize( fnt.pointSize() - 3 );  // Shrink font by 3pt
       fnt.setWeight( QFont::Normal );           // and disable bold
  }
	p.setFont( fnt );
  p.setPen( options->color(Options::Font, isActive() ));
  p.drawText( t.x() + 14, 1, t.width()-13, t.height()-1, AlignLeft | AlignVCenter, caption() );


	// [0.2] - If required, draw bottom right corner details
	// -----------------------
	if( config_showCornerDetail )
	{
		// light
		p.setPen( options->color(Options::TitleBar,isActive() ).light( 120 ) );
		p.drawLine(w-9, h-5, w-5, h-5); 	// horizontal
		p.drawLine(w-5, h-9, w-5, h-6);		// vertical
		//  mid
		p.setPen( options->color(Options::TitleBar, isActive() ).dark(110) );
		p.drawLine(w-9, h-4, w-4, h-4);   // horizontal
		p.drawLine(w-4, h-9, w-4, h-5);		// vertical
		// v dark
		p.setPen( options->color(Options::TitleBlend, isActive() ).dark( 160 )  );
		p.drawLine(w-9, h-3, w-3, h-3);   // horizontal
		p.drawLine(w-3, h-9, w-3, h-4);		// vertical
		// gray lines
		p.setPen( QColor( 72, 72, 72 ) );
		p.drawLine(w-5, h-10, w-3, h-10);		// horizontal on side
		p.drawLine(w-10, h-5, w-10, h-3); 	// vertical on bottom
	}

	// Outer Frames
	// -----------------------------
	// outer border around whole window
	p.setPen(Qt::black);
 	p.drawRect(0, 0, w, h);
	// inner border around whole window
	p.setPen( QColor( 72, 72, 72 ) );
 	p.drawRect(1, 1, w-2, h-2);
}

// ----------------------------------------------------------------
// Resize
// ----------------------------------------------------------------
void QNiXClient::resizeEvent(QResizeEvent * e)
{
	Client::resizeEvent( e );

  if ( isVisibleToTLW() && !testWFlags( WNorthWestGravity ))
	{
  	QPainter p( this );
		QRect t = titlebar->geometry();
		t.setTop( 0 );
		QRegion r = rect();
		r = r.subtract( t );
		p.setClipRegion( r );
		p.eraseRect( rect() );
  }
}

// ----------------------------------------------------------------
// This event occurs if the application icon changes
// Depending on the theme config, the new icon may need to be drawn
// ----------------------------------------------------------------
void QNiXClient::iconChange()
{
	// should we show the menu icon?
	if( config_showMenuButtonIcon )
	{
		if(!miniIcon().isNull())
			button[BtnMenu]->setPixmap(miniIcon());
		else
			button[BtnMenu]->setPixmap(*pixDefaultAppIcon);

		if (button[BtnMenu]->isVisible())
			button[BtnMenu]->repaint(false);
	}
	// special default menu button when left detail is on
	else if ( config_showLeftDetail )
	{
		if( isActive() )
			button[BtnMenu]->setPixmap( *pixLeftDetailActive );
		else
			button[BtnMenu]->setPixmap( *pixLeftDetailInactive );

		if (button[BtnMenu]->isVisible())
			button[BtnMenu]->repaint(false);
	}

}

// ----------------------------------------------------------------
// Redraw everything when the caption changes
// ----------------------------------------------------------------
void QNiXClient::captionChange(const QString &)
{
  repaint();
}

void QNiXClient::mouseDoubleClickEvent(QMouseEvent * e)
{
  if (titlebar->geometry().contains(e->pos()))
    workspace()->performWindowOperation(this, options->operationTitlebarDblClick());
}

// ----------------------------------------------------------------
// Return mouse position
// Note: this also determines how window can be resized
// ----------------------------------------------------------------
Client::MousePosition QNiXClient::mousePosition(const QPoint & p) const
{
	// [0.2] - fix window-top resizing problem
	// now using code from nextclient.cpp
	MousePosition m = Nowhere;

  if (p.y() < (height() - 6))
	{
    m = Client::mousePosition(p);
	}
  else
	{
    if (p.x() >= (width() - 25))
      m = BottomRight;
    else if (p.x() <= 25)
      m = BottomLeft;
    else
      m = Bottom;
  }

  return m;
}

// ----------------------------------------------------------------
// Reset the client
// Re-read config info, reset state of buttons, and then redraw
// ----------------------------------------------------------------
void QNiXClient::slotReset()
{
	read_config();
  button[BtnMax]->reset();
	button[BtnClose]->reset();
	button[BtnMin]->reset();
  repaint();
}

// ----------------------------------------------------------------
// Called when window changes from active to inactive, or vice versa
// ----------------------------------------------------------------
void QNiXClient::activeChange( bool )
{
	// Update active/inactive default icon
	if( config_showLeftDetail )
		iconChange();

  button[BtnMax]->repaint(false);
	button[BtnClose]->repaint(false);
	button[BtnMin]->repaint(false);
	button[BtnMenu]->repaint(false);
  repaint(false);
}

// ----------------------------------------------------------------
// Called when "sticky" state is changed
// ----------------------------------------------------------------
void QNiXClient::stickyChange(bool b)
{
  emit(stkyChange(b));
}

// ----------------------------------------------------------------
// Called when max state is changed
// ----------------------------------------------------------------
void QNiXClient::maximizeChange(bool b)
{
  emit(maxChange(b));
}

// ================================================================================
// BUTTON EVENTS
// ================================================================================

// ----------------------------------------------------------------
// Maximize button was clicked...
// ----------------------------------------------------------------
void QNiXClient::slotMaximize()
{
	//[0.2] - changed to fix min/max problem
	switch ( button[BtnMax]->last_button  )
	{
		case MidButton:
			maximize( MaximizeVertical );
			break;
    case RightButton:
			maximize( MaximizeHorizontal );
			break;
    default: // Left mouse button:
			maximize();
	}
}

// ----------------------------------------------------------------
// "Menu"(icon) button was clicked...
// ----------------------------------------------------------------
void QNiXClient::slotMenu()
{
	// from redmond.cpp
	static QTime* t = 0;
  static QNiXClient* tc = 0;
  if ( !t )
  	t = new QTime;

	if ( tc != this || t->elapsed() > QApplication::doubleClickInterval() )
  {
  	// KS - move the menu left by 3 pixels, and down 2 pixels.
    QPoint menupoint ( button[BtnMenu]->rect().bottomLeft().x()-3, button[BtnMenu]->rect().bottomLeft().y()+2 );
    workspace()->clientPopup(this)->popup( button[BtnMenu]->mapToGlobal( menupoint ));
 	}
  else
	{
  	closeWindow();
	}
  t->start();
  tc = this;
}


//////////////////////////////////////////////////////////////////////////////////////////////////////////////
// CLASS : QNixButton
//////////////////////////////////////////////////////////////////////////////////////////////////////////////
QNixButton::QNixButton(Client *parent, const char *name, Buttons type, const unsigned char *bitmap, bool menuButton, bool isMini, const QString& tip)
    : KWinButton(parent, name, tip)
{
	// Eliminate background flicker
	setBackgroundMode( QWidget::NoBackground );

	m_type = type;
	menuBtn = menuButton;
	miniBtn = isMini;
	client  = parent;
	b_highlight = false;

	// Use larger button for the menu, or mini-buttons for toolwindows.
	if ( isMini )
	{
		setFixedSize(MINI_BUTTON_SIZE, MINI_BUTTON_SIZE);
		resize(MINI_BUTTON_SIZE, MINI_BUTTON_SIZE);
	}
	else if ( menuButton )
	{
		setFixedSize(MENU_BUTTON_SIZE, MENU_BUTTON_SIZE);
		resize(MENU_BUTTON_SIZE, MENU_BUTTON_SIZE);
	}
	else
	{
		setFixedSize(BUTTON_SIZE, BUTTON_SIZE);
		resize(BUTTON_SIZE, BUTTON_SIZE);
	}

	if(bitmap)
		setBitmap(bitmap);
}


QSize QNixButton::sizeHint() const
{
	if ( miniBtn )
			return(QSize(MINI_BUTTON_SIZE, MINI_BUTTON_SIZE));
	else if ( menuBtn )
			return(QSize(MENU_BUTTON_SIZE, MENU_BUTTON_SIZE));
	else
			return(QSize(BUTTON_SIZE, BUTTON_SIZE));
}

void QNixButton::setBitmap(const unsigned char *bitmap)
{
  pix.resize(0, 0);
  deco = QBitmap(10, 10, bitmap, true);
  deco.setMask(deco);
	repaint( false );
}


void QNixButton::setPixmap( const QPixmap &p )
{
	deco.resize(0, 0);
	pix = p;

	if (miniBtn)
		setMask(QRect(0, 0, MINI_BUTTON_SIZE, MINI_BUTTON_SIZE));
	else if (menuBtn)
		setMask(QRect(0, 0, MENU_BUTTON_SIZE, MENU_BUTTON_SIZE+1));
	else
		setMask(QRect(0, 0, BUTTON_SIZE, BUTTON_SIZE));

	repaint( false );
}

// ----------------------------------------------------------------
// Button's status is reset => redraw the button
// ----------------------------------------------------------------
void QNixButton::reset()
{
   repaint(false);
}

// ----------------------------------------------------------------
// Event : Button clicked
// ----------------------------------------------------------------
void QNixButton::mousePressEvent( QMouseEvent* e )
{
	last_button = e->button();
	QMouseEvent me ( e->type(), e->pos(), e->globalPos(), LeftButton, e->state() );
	KWinButton::mousePressEvent( &me );
}

// ----------------------------------------------------------------
// Event : Button released
// ----------------------------------------------------------------
void QNixButton::mouseReleaseEvent( QMouseEvent* e )
{
	last_button = e->button();
	QMouseEvent me ( e->type(), e->pos(), e->globalPos(), LeftButton, e->state() );
	KWinButton::mouseReleaseEvent( &me );
}

// ----------------------------------------------------------------
// Responds to "MouseIN" events
// Sets highlight if highlighting is enabled
// ----------------------------------------------------------------
void QNixButton::enterEvent( QEvent *e)
{
	// Only highlight if config option is selected
	if( config_highlightButtons )
	{
		b_highlight = true;
		repaint();		// update button's appearance
	}
}

// ----------------------------------------------------------------
// Responds to "MouseOut" events
// Removes highlight if highlighting is enabled
// ----------------------------------------------------------------
void QNixButton::leaveEvent( QEvent *e)
{
	// Only remove highlight if config option is selected
	if( config_highlightButtons )
	{
		b_highlight = false;
		repaint();		// update button's appearance
	}
}

// ----------------------------------------------------------------
// Actually draw the button
// ----------------------------------------------------------------
void QNixButton::drawButton(QPainter *p)
{
	int w = width();
	int h = height();
	// [0.2] - Highlight when mouse is over button, but not if the button is down!
	int m_highlight = ( ( b_highlight && !isDown() ) ? 20 : 0);

	// menu button
	if( !pix.isNull() )
	{
		// draw part of titlebar that was covered up
		QPixmap px;
		px.resize( w, h+2 );
		bitBlt( &px, 0, 0, (client->isActive() ? pixTitleLeftActive : pixTitleLeftInactive), 1, 2, w, h+2 );
 		p->drawPixmap( 0, 0, px );
		// draw image
		if ( menuBtn && miniBtn )
    {
    	QPixmap tmpPix;
      // Smooth scale the menu button pixmap
      tmpPix.convertFromImage( pix.convertToImage().smoothScale(12, 12));
      p->drawPixmap( 0, 0, tmpPix );
	  }
		else
    	p->drawPixmap( 0, 0, pix );
		// finished
		return;
	}

	// Never paint if the pixmaps have not been created
	if( !pixmaps_created )
		return;

	// special menu button - detail without pixmap
	if( m_type == BtnMenu )
	{
		p->drawPixmap(0, 2, *pixMenuButton );
		// draw part of titlebar that was covered up
		p->setPen( options->color(Options::TitleBlend, client->isActive() ).dark( 130 )  );
		p->drawLine(0, 0, BUTTON_SIZE, 0);
		p->setPen( options->color(Options::TitleBar, client->isActive()) );
		p->drawLine(0, 1, BUTTON_SIZE, 1);
		// pattern
		SET_HCOLOR( 144, 144, 144 );
		p->drawLine( 1, 5, w - 2, 5 );  // 1.1
		p->drawLine( 1, 8, w - 2, 8 );  // 2.1
		//SET_HCOLOR( 56, 60, 56);
		SET_HCOLOR( 176, 172, 176 );
		p->drawLine( 1, 6, w - 2, 6);  // 1.2
		p->drawLine( 1, 9, w - 2, 9 );  // 2.1

		return;
	}

	// Draw background pixmap
	if(isDown())
		p->drawPixmap(0, 0, miniBtn ? *pixMiniButtonDown : *pixButtonDown);
	else if( b_highlight )
		p->drawPixmap(0, 0, miniBtn ? *pixMiniButtonHighlighted : *pixButtonHighlighted);
	else
		p->drawPixmap(0, 0, miniBtn ? *pixMiniButtonNormal : *pixButtonNormal);

	// Draw specific detail for a given button type
	switch( m_type )
	{
		// MAX/RESTORE BUTTON
		case( BtnMax ):
			// Window isn't maximized => draw MAX button
			if( !client->isMaximized() )
			{
				if( isDown() )
				{
					// extra line
					SET_COLOR( 162, 162, 162 );
					p->drawLine( 2, 2, 2, h - 2 );
					// the rest
					SET_COLOR( 176, 172, 176 );
					p->drawLine( 2, 4, w - 3, 4 );
					SET_COLOR( 200, 200, 200 );
					p->drawLine( 2, 5, w - 3, 5 );
					SET_COLOR( 198, 198, 198 );
					p->drawPoint( w-2, 5 );
				}
				else
				{
					SET_HCOLOR( 176, 172, 176 );
					p->drawLine( 1, 4, w - 3, 4 );
					SET_HCOLOR( 229, 229, 229 );
					p->drawLine( 1, 5, w - 3, 5 );
					SET_HCOLOR( 216, 212, 216 );
					p->drawPoint( w-2, 5 );
				}
			}
			// [0.2] - Window is maximized => draw RESTORE button
			else
			{
				if( isDown() )
				{
					// extra line
					SET_COLOR( 152, 152, 152 );
					p->drawLine( 2, 2, 2, h - 2 );
					// the rest
					SET_COLOR( 160, 156, 160 );
					p->drawLine( 2, 7, w - 3, 7 );
					SET_COLOR( 184, 184, 184 );
					p->drawLine( 2, 8, w - 3, 8 );
					SET_COLOR( 182, 182, 182 );
					p->drawPoint( w-2, 8 );
				}
				else
				{
					SET_HCOLOR( 150, 146, 150 );
					p->drawLine( 1, 7, w - 3, 7 );
					SET_HCOLOR( 203, 203, 203 );
					p->drawLine( 1, 8, w - 3, 8 );
					SET_HCOLOR( 190, 186, 190 );
					p->drawPoint( w-2, 8 );
				}
			}
			break;
		// CLOSE BUTTON
		case( BtnClose ):
			// NOTE : same for up/down
			// outer rectangle
			SET_HCOLOR( 144, 144, 144);
			p->drawLine( 3, 3, 3, 8 );  // down left
			p->drawLine( 3, 3, 9, 3 );  // across top
			SET_HCOLOR( 232, 232, 232);
			p->drawLine( 9, 3, 9, 8 ); // down right
			p->drawLine( 3, 9, 9, 9 );  // across bottom
			SET_HCOLOR( 192, 188, 192);
			p->drawPoint( 9, 3 ); // right top corner
			p->drawPoint( 3, 9 ); // left bottom corner
			// inner rectangle
			SET_HCOLOR( 56, 60, 56);
			p->drawRect( 4, 4, 5, 5 );
			SET_HCOLOR( 192, 188, 192);
			p->drawRect( 5, 5, 2, 2 );
			SET_HCOLOR( 128, 128, 128);
			p->drawRect( 6, 6, 2, 2 );
			SET_HCOLOR( 160, 160, 160);
			p->drawLine( 5, 7, 7, 5 );
			break;
		// MIN BUTTON
		case( BtnMin ):
			if( isDown() )
			{
				// extra line
				SET_HCOLOR( 162, 162, 162 );
				p->drawLine( 2, 2, 2, h - 2 );
				// the rest
				SET_HCOLOR( 176, 172, 176 );
				p->drawLine( 2, 4, w - 3, 4 );
				SET_HCOLOR( 200, 200, 200 );
				p->drawLine( 2, 5, w - 3, 5 );
				SET_HCOLOR( 198, 198, 198 );
				p->drawPoint( w-2, 5 );
			}
			else
			{
				SET_HCOLOR( 208, 204, 208);
				p->drawPoint( 1, h-6 );
			}
			// the rest is common for up/down
			SET_HCOLOR( 144, 144, 144);
			p->drawLine( 2, h - 6, w - 2, h - 6 );  // horiz line 1
			SET_HCOLOR( 56, 60, 56);
			p->drawLine( 1, h - 5, w - 2, h - 5 );  // horiz line 2
			// arrow:
			SET_HCOLOR( 80, 80, 80 );
			p->drawLine( 4, 3, 8, 3 );
			p->drawLine( 5, 4, 7, 4 );
			p->drawPoint( 6, 5 );
			break;
		// shouldn't get here...
		default:
			break;
	}
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////
// INIT
//////////////////////////////////////////////////////////////////////////////////////////////////////////////
extern "C"
{
  Client * allocate(Workspace * ws, WId w, int tool)
  {
    return new QNiXClient( ws, w );
  }
	void init()
	{
		create_pixmaps();
    read_config();
	}
	void reset()
	{
		delete_pixmaps();
		create_pixmaps();
    read_config();
		// Ensure change in tooltip state gets applied
		Workspace::self()->slotResetAllClientsDelayed();
	}
	void deinit()
	{
		delete_pixmaps();
	}
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////

#include "qnix_client.moc"
