/* Copyright (C) 2004 Nikos Chantziaras.
 *
 * This file is part of the QTads program.  This program is free
 * software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place - Suite 330, Boston,
 * MA 02111-1307, USA.
 */

/* Main entrypoint.
 */

#include "config.h"

#include <iostream>
#include <cstdlib>

#include <qapplication.h>
#include <qeventloop.h>
#include <qstatusbar.h>
#include <qfile.h>
#include <qfileinfo.h>
#include <qtranslator.h>
#include <qtextcodec.h>
#include <qmessagebox.h>
#include <qaction.h>

#include "os.h"
#include "trd.h"
#include "t3std.h"
#include "vmvsn.h"
#include "vmmain.h"
#include "vmconsol.h"

#include "qtadsvmhostifc.h"
#include "qtadssettings.h"
#include "qtadsmainwindow.h"
#include "qtadsstatusline.h"
#include "qtadsgamewindow.h"
#include "qtadsio.h"
#include "qtadscvmmainclientconsole.h"
#include "install_dirs.h"
#include "qtadswarning.h"
#include "qtadsappctx.h"


// This contains the filename of the game to run.
QString gameFileName;

// This contains the filename of the most recently run game.
QString lastGameFileName;


int main( int argc, char* argv[] )
{
	const QString QTDIR(std::getenv("QTDIR"));

	// The application object.
	QApplication app(argc, argv);

	// Check the QTDIR env. variable.
	// TODO: This warning dialog is annoying.  Find a way to locate
	// the translation files automaticly without using QTDIR.
	/*
	if (not QFile::exists(QTDIR)) {
		QMessageBox::warning(0, "Qt directory not found",
				     QObject::tr("<p>The QTDIR environment variable is not"
						 " set or points to a directory that doesn't"
						 " exist. Although QTads will run fine, some"
						 " features (internationalization for"
						 " example) may not work. QTDIR points to the"
						 " directory where the Qt library is"
						 " installed (usually /usr/lib/qt, "
						 " /usr/local/lib/qt or similar).</p>"),
				     QMessageBox::Ok, QMessageBox::NoButton,
				     QMessageBox::NoButton);
	}
	*/

	// Translation file for the Qt libary.
	QTranslator qtTrans(0);
	qtTrans.load(QString("qt_") + QTextCodec::locale(), QTDIR + "/translations");
	//qtTrans.load("qt_de", QTDIR + "/translations");
	app.installTranslator(&qtTrans);

	// Translation file for QTads.
	QTranslator qtadsTrans(0);
	qtadsTrans.load(QString("qtads_") + QTextCodec::locale(), I18N_INST_DIR);
	//qtadsTrans.load("qtads_de", I18N_INST_DIR);
	app.installTranslator(&qtadsTrans);

	if (argc > 2) {
		std::cerr << "Usage: qtads [gamefile]" << std::endl;
		return 1;
	}

	if (argc == 2) {
		if (QFile::exists(argv[1])) {
			::gameFileName = argv[1];
		} else if (QFile::exists(QString(argv[1]) + ".gam")) {
			::gameFileName = QString(argv[1]) + ".gam";
		} else if (QFile::exists(QString(argv[1]) + ".t3")) {
			::gameFileName = QString(argv[1]) + ".t3";
		} else {
			std::cerr << "File `" << argv[1] << "' not found." << std::endl;
			return 2;
		}
	} else {
		::gameFileName = QString::null;
	}

	// Construct the "Welcome" message.
	QString welcomeMsg(QString("<blockquote><p><br>") + QObject::tr("Welcome to")
			   + " " QTADSVERSION ".<br>"
			   + QObject::tr("Tads 2 Virtual Machine version:")
			   + " " TADS_RUNTIME_VERSION "<br>"
			   + QObject::tr("Tads 3 Virtual Machine version:")
			   + " " T3VM_VSN_STRING
			   "</p></blockquote>");

	// If false, don't show the "Welcome" message.
	bool showWelcomeMsg = true;

	// Create and set-up the Tads 2 application container context.
	// We make it static for no other reason than automatic null
	// initialization of the fields; unused fields *must* be 0, or
	// else the VM would crash.
	static appctxdef appctx;
	appctx.get_io_safety_level = QTadsAppctx::getIoSafetyLevel;

	// The Tads 3 host and client services interfaces.
	CVmHostIfc* hostifc = new QTadsCVmHostIfc(argv[0]);
	QTadsCVmMainClientConsole clientifc;

	// Initialize the I/O system, the main window's title (caption)
	// and show the main window.
	QTadsIO::init();
	QTadsIO::mainWindow().setCaption("QTads");
	// Show the main window's status bar before the window itself.
	// This ensures that status bar and main window will show up
	// at the same time, thus avoiding an ugly update later.
	QTadsIO::mainWindow().statusBar()->show();
	QTadsIO::mainWindow().show();

	// Process the above events.
	app.processEvents();

	// TODO: I need to change this, since we can't start QTads with
	// more than one argument.
	int appArgc = 2;
	char* appArgv[2];
	appArgv[0] = "qtads";
	appArgv[1] = new char[512];

	bool firstRun = true;
	while (not QTadsIO::quitApp) {
		if (::gameFileName.isEmpty()) {
			if (QTadsIO::gameWindow().text().isEmpty() and showWelcomeMsg) {
				// Show the "Welcome" message.
				QTadsIO::gameWindow().setTextFormat(Qt::RichText);
				QTadsIO::gameWindow().setText(welcomeMsg);
				QTadsIO::gameWindow().setTextFormat(Qt::PlainText);

			}
			// Prompt for a game file.
			if (not (firstRun and not QTadsIO::settings().promptAtStartup())) {
				::gameFileName = QTadsIO::openFile(QString::null,
								   QObject::tr("Tads Games")
								   + " (*.gam *.Gam *.GAM *.t3 *.T3)"
								   ";;"
								   + QObject::tr("All Files")
								   + " (*)",
								   QObject::tr("Select a game to load"));

			}
		}
		firstRun = false;

		if (not ::gameFileName.isEmpty()) {

			// Convert the filename to an absolute path...
			::gameFileName = QFileInfo(::gameFileName).absFilePath();
			// ...and change to the game's directory.  This
			// makes it simpler to access files that the
			// game might request (for example registration
			// key-files, like regkey.txt with "Losing Your
			// Grip").
			if (not QDir::setCurrent(QFileInfo(::gameFileName).dirPath(true))) {
				qtadsWarning(QObject::tr("Could not change current directory")
					     + " (" + QFileInfo(::gameFileName).dirPath(true)
					     + ")");
			}

			os_init(&appArgc, appArgv, 0, 0, 0);
			QTadsIO::gameRunning = true;
			QTadsIO::currentGame = ::gameFileName;
			QTadsIO::enableCommandActions(true);
			if (QTadsIO::gameWindow().isReadOnly()) {
				QTadsIO::gameWindow().setReadOnly(false);
			}
			QTadsIO::reset();

			// Add the filename to the "Recent Games" menu...
			QTadsIO::settings().addRecentGame(::gameFileName);
			// ...and update it.
			QTadsIO::mainWindow().updateRecentGamesList();

			// Enable the "Game->Game Information" action.
			QTadsIO::mainWindow().gameInfoAction->setEnabled(true);

			QTadsIO::mainWindow().startTimer();
			int ret;
			if (vm_get_game_type(::gameFileName.ascii(), 0, 0, 0, 0)
			    == VM_GGT_TADS2)
			{
				QTadsIO::t3Mode(false);
				std::strncpy(appArgv[1], ::gameFileName.ascii(), 512);
				ret = trdmain(appArgc, appArgv, &appctx, "sav");
				// Reset the text codec back to the default
				// (in case the game was using a character
				// mapping file).
				QTextCodec::setCodecForCStrings(0);
			} else {
				QTadsIO::t3Mode(true);
				ret = vm_run_image(&clientifc, ::gameFileName.ascii(),
						   hostifc, appArgv, appArgc, 0, false, 0, 0, false,
						   false, 0, 0, 0, 0);
			}

			QTadsIO::mainWindow().setCaption("QTads");
			QTadsIO::mainWindow().stopTimer();
			::lastGameFileName = ::gameFileName;
			showWelcomeMsg = false;
			QTadsIO::mainWindow().gameInfoAction->setEnabled(false);
			QTadsIO::enableCommandActions(false);
			QTadsIO::gameRunning = false;
			QTadsIO::currentGame = QString::null;
			QTadsIO::enableCommandActions(true);
			QTadsIO::flush();
			QTadsIO::gameWindow().setReadOnly(true);
			if (ret == 0 and not QTadsIO::quitApp and not QTadsIO::restartGame) {
				QTadsIO::gameWindow().setTextFormat(Qt::RichText);
				QTadsIO::gameWindow().append("<br><br><br>"
							     "<blockquote align=\"center\">"
							     "<hr><i>"
							     + QObject::tr("The game has ended.")
							     + "</i><hr></blockquote>");
				QTadsIO::gameWindow().setTextFormat(Qt::PlainText);
			}

		}

		Q_ASSERT(not (QTadsIO::openNewGame and QTadsIO::restartGame));

		while (true) {
			qApp->eventLoop()->processEvents(QEventLoop::WaitForMore
							 | QEventLoop::AllEvents);
			if (QTadsIO::quitApp) {
				break;
			}
			if (QTadsIO::restartGame) {
				if (::gameFileName.isEmpty()) {
					::gameFileName = ::lastGameFileName;
				}
				QTadsIO::restartGame = false;
				break;
			}
			if (QTadsIO::openNewGame) {
				if (QTadsIO::newGameToOpen.isEmpty()) {
					::gameFileName = QString::null;
				} else {
					::gameFileName = QTadsIO::newGameToOpen;
					QTadsIO::newGameToOpen = QString::null;
				}
				QTadsIO::openNewGame = false;
				break;
			}
		}
	}

	QTadsIO::done();
	os_uninit();
	os_term(0);

	return 0;
}
