/***************************************************************************
 *   Copyright (C) 2006-2009 by Tomasz Moń                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; under version 2 of the License.         *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.         *
 ***************************************************************************/

#include <QtGui>
#include <QTime>
#include "qtscrob.h"
#include "settings.h"
#include "about.h"
#include "help.h"
#include "browsedirtree.h"
#include "progress.h"
#include "threadscrobble.h"
#include "console.h"

QTScrob::QTScrob(QWidget *parent) : QMainWindow( parent ) {
	scrob = new ThreadScrobble(this);
	readSettings();

	if (!scrob->init(conf.timeout, conf.connect_timeout))
	{
		QMessageBox::warning(this, QCoreApplication::applicationName(), tr("Failed to init the libscrobble library: %1").arg(scrob->get_error_str().c_str()));
		QCoreApplication::quit();
	}

	logTable = new QTableWidget();
	tableLabels << tr("Artist") << tr("Album") << tr("Title") << tr("TrackNumber") << tr("Length") << tr("Played") << tr("Date (UTC)");
	logTable->setColumnCount(7);
	logTable->setRowCount(0);
	logTable->setHorizontalHeaderLabels(tableLabels);
	logTable->setAlternatingRowColors(true);
	logTable->setSelectionBehavior(QAbstractItemView::SelectRows);

	baseDT = QDateTime::currentDateTime();
}

QTScrob::~QTScrob() {
#ifdef HAVE_MTP
	scrob->mtp_disconnect();
#endif

	saveWindow();

	if (winSettings!=NULL)
		delete winSettings;

	if (winConsole != NULL)
		delete winConsole;

	delete scrob;
}

void QTScrob::readSettings() {
	QSettings settings("qtscrob");
	settings.beginGroup("application");
	conf.username = settings.value("username","").toString();
	conf.password = settings.value("password","").toString();
	conf.lastDirectory = settings.value("lastDirectory","").toString();
	conf.utc_offset = settings.value("utc_offset","").toInt();
	conf.tz_override = settings.value("tz_override", "false").toBool();
	conf.checkAutoOpen = settings.value("checkAutoOpen","true").toBool();
	conf.checkDelPlayList = settings.value("checkDelPlayList","false").toBool();
	conf.timeout = settings.value("timeout",60).toInt();
	conf.connect_timeout = settings.value("connect_timeout",30).toInt();
	settings.endGroup();
	settings.beginGroup("proxy");
	conf.proxy_host = settings.value("proxy_host", "").toString();
	conf.proxy_port = settings.value("proxy_port", "").toInt();
	conf.proxy_user = settings.value("proxy_user", "").toString();
	conf.proxy_pass = settings.value("proxy_pass", "").toString();
	conf.proxy_winauth = settings.value("proxy_winauth", "").toBool();
	settings.endGroup();
	settings.deleteLater();

	scrob->set_proxy(conf.proxy_host.toStdString(),
					conf.proxy_port,
					conf.proxy_user.toStdString(),
					conf.proxy_pass.toStdString(),
					conf.proxy_winauth);
}

void QTScrob::saveSettings() {
	QSettings settings("qtscrob");

	if (winSettings!=NULL) {
		settings.beginGroup("application");
		settings.setValue("username",winSettings->username());
		settings.setValue("utc_offset",winSettings->utc_offset());
		settings.setValue("tz_override",winSettings->tz_override());
		settings.setValue("checkAutoOpen",winSettings->checkAutoOpen());
		settings.setValue("checkDelPlayList",winSettings->checkDelPlayList());

		if ( winSettings->password()!="" && passchanged) {
			settings.setValue("password",winSettings->password());
		} else {
			settings.setValue("password",conf.password);
		}
		settings.endGroup();

		settings.beginGroup("proxy");
		settings.setValue("proxy_host",winSettings->proxy_host());
		settings.setValue("proxy_port",winSettings->proxy_port());
		settings.setValue("proxy_user",winSettings->proxy_user());
		settings.setValue("proxy_pass",winSettings->proxy_pass());
		settings.setValue("proxy_winauth",winSettings->proxy_winauth());
		settings.endGroup();

		winSettings->close();
	}

	settings.deleteLater();
	readSettings(); // let's update conf.password and conf.userName
}

void QTScrob::setupWindow() {
	readWindow();
	setupWidgets();
	connectSlots();
}

void QTScrob::readWindow() {
	QSettings settings("qtscrob");
	settings.beginGroup("window");
	resize(settings.value("size", QSize(705, 458)).toSize());
	move(settings.value("pos", QPoint(100, 100)).toPoint());
	if (settings.value("maximized", "false").toBool())
		setWindowState(windowState() | Qt::WindowMaximized);
	settings.endGroup();
	settings.deleteLater();
}

void QTScrob::saveWindow() {
	QSettings settings("qtscrob");
	settings.beginGroup("window");
	settings.setValue("maximized",  isMaximized());
	if (!isMaximized())
	{
		settings.setValue("size", size());
		settings.setValue("pos", pos());
	}
	settings.endGroup();
	settings.deleteLater();
}

void QTScrob::passEdited(const QString& newstring) {
	(void)newstring;
	passchanged = true;
}

void QTScrob::setupWidgets() {
	actOpen = new QAction(tr("Open"),this);
	actOpeniTunes = new QAction(tr("Open iTunesDB"),this);
	actExit = new QAction(tr("Exit"),this);
	actHelp = new QAction(tr("Help"),this);
	actAbout = new QAction(tr("About"),this);
	actAboutQt = new QAction(tr("AboutQt"),this);
	actSettings = new QAction(tr("Settings"),this);
	actConsole = new QAction(tr("Console"), this);

	centralWidget = new QWidget(this);

	btnOpen = new QPushButton(tr("Open .scrobbler.log"));
	btnOpeniTunes = new QPushButton(tr("Open iTunesDB"));
	btnDelete = new QPushButton(tr("Delete selected item"));
	btnSubmit = new QPushButton(tr("Submit"));

	btnOpen->setIcon(QIcon(":/resources/icons/document-open.png"));
	btnOpeniTunes->setIcon(QIcon(":/resources/icons/multimedia-player-ipod-standard-color.png"));
	btnDelete->setIcon(QIcon(":/resources/icons/edit-clear.png"));
	btnSubmit->setIcon(QIcon(":/resources/icons/mail-send-receive.png"));

#ifdef HAVE_MTP
	actOpenMTP = new QAction(tr("Open MTP Device"), this);
	actOpenMTP->setIcon(QIcon(":/resources/icons/multimedia-player-iriver-s10_16.png"));
	btnOpenMTP = new QPushButton(tr("Open MTP Device"));
	btnOpenMTP->setIcon(QIcon(":/resources/icons/multimedia-player-iriver-s10_16.png"));
#endif

	QGridLayout *layout = new QGridLayout(centralWidget);
	layout->setMargin(0);
	layout->setColumnStretch(0, 1);
	layout->addWidget(btnOpen, 0, 1);
	layout->addWidget(btnOpeniTunes, 0, 2);
#ifdef HAVE_MTP
	layout->addWidget(btnOpenMTP, 0, 3);
#endif
	layout->addWidget(btnDelete, 0, 4);
	layout->addWidget(btnSubmit, 0, 5);
	layout->setColumnStretch(6, 1);
	layout->addWidget(logTable, 1, 0, 1, 7);

	this->setCentralWidget(centralWidget);

	menuBar = new QMenuBar(this);
	menuBar->setGeometry(QRect(0, 0, 705, 26));
	menuHelp = new QMenu(tr("Help"),menuBar);
	menuGlobal = new QMenu(tr("Global"),menuBar);
	this->setMenuBar(menuBar);
	statusbar = new QStatusBar(this);
	this->setStatusBar(statusbar);

	lblStatus = new QLabel(tr("Ready"));
	DTedit = new QDateTimeEdit();
	btnRecalcDT = new QPushButton(tr("Recalc Date/Time"));
	btnRecalcDT->setIcon(QIcon(":/resources/icons/appointment-new.png"));
	statusbar->addWidget(lblStatus, 1);
	statusbar->addWidget(DTedit);
	statusbar->addWidget(btnRecalcDT);

	DTedit->setDateTime(baseDT);

	actSettings->setIcon(QIcon(":/resources/icons/preferences-system.png"));
	actHelp->setIcon(QIcon(":/resources/icons/help-browser.png"));
	actExit->setIcon(QIcon(":/resources/icons/system-log-out.png"));
	actOpen->setIcon(QIcon(":/resources/icons/document-open.png"));
	actOpeniTunes->setIcon(QIcon(":/resources/icons/multimedia-player-ipod-standard-color.png"));

	actHelp->setShortcut(QKeySequence::HelpContents);

	menuBar->addAction(menuGlobal->menuAction());
	menuBar->addAction(menuHelp->menuAction());
	menuHelp->addAction(actHelp);
	menuHelp->addAction(actConsole);
	menuHelp->addAction(actAbout);
	menuHelp->addAction(actAboutQt);
	menuGlobal->addAction(actOpen);
	menuGlobal->addAction(actOpeniTunes);
#ifdef HAVE_MTP
	menuGlobal->addAction(actOpenMTP);
#endif
	menuGlobal->addAction(actSettings);
	menuGlobal->addSeparator();
	menuGlobal->addAction(actExit);

	this->setWindowTitle(tr("QTScrobbler"));

	winSettings = NULL;
	winAbout = NULL;
	winHelp = NULL;
	winProgress = NULL;
	winConsole = NULL;
	passchanged = false;
}

void QTScrob::resizeEvent(QResizeEvent *e)
{
	int cellWidth;
	if (e->size().width() - logTable->viewport()->width() > 4)
		cellWidth = (e->size().width() - 4) / logTable->columnCount();
	else
		cellWidth = logTable->viewport()->width() / logTable->columnCount();

	for(int colIndex = 0; colIndex < logTable->columnCount(); colIndex++)
		logTable->horizontalHeader()->resizeSection(colIndex, cellWidth-2);
}

void QTScrob::connectSlots() {
	connect(actOpen, SIGNAL(triggered()), btnOpen, SLOT(click()));
	connect(actExit, SIGNAL(triggered()), this, SLOT(close()));
	connect(actHelp, SIGNAL(triggered()), this, SLOT(help()));
	connect(actAbout, SIGNAL(triggered()), this, SLOT(about()));
	connect(actAboutQt, SIGNAL(triggered()), this, SLOT(aboutQt()));
	connect(actOpeniTunes, SIGNAL(triggered()), this, SLOT(openiTunesDB()));
	connect(actConsole, SIGNAL(triggered()), this, SLOT(console()));
#ifdef HAVE_MTP
	connect(actOpenMTP, SIGNAL(triggered()), this, SLOT(openMTP()));
	connect(btnOpenMTP, SIGNAL(clicked()), this, SLOT(openMTP()));
#endif
	connect(btnOpeniTunes, SIGNAL(clicked()), this, SLOT(openiTunesDB()));
	connect(btnOpen, SIGNAL(clicked()), this, SLOT(open()));
	connect(btnDelete, SIGNAL(clicked()), this, SLOT(deleteRow()));
	connect(btnSubmit, SIGNAL(clicked()), this, SLOT(scrobble()));
	connect(actSettings, SIGNAL(triggered()), this, SLOT(settings()));
	connect(btnRecalcDT, SIGNAL(clicked()), this, SLOT(recalcDT()));
	connect(DTedit, SIGNAL(dateTimeChanged(QDateTime)), this, SLOT(updatebaseDT(QDateTime)));
	connect(logTable, SIGNAL(cellChanged(int, int)), this, SLOT(changeRow(int, int)));
	connect(scrob, SIGNAL(finished()), this, SLOT(scrobbled()));
}

void QTScrob::help() {
	if (winHelp==NULL)
		winHelp = new Help(this);
	winHelp->show();
}

void QTScrob::about() {
	if (winAbout==NULL)
		winAbout = new About(this);
	winAbout->show();
}

void QTScrob::settings() {
	if (winSettings==NULL)
		winSettings = new Settings(this);
	winSettings->show();
}

void QTScrob::console() {
	if (winConsole==NULL)
		winConsole = new Console(this);
	winConsole->show();
}

void QTScrob::aboutQt() {
	QApplication::aboutQt();
}

void QTScrob::open() {
	if (conf.checkAutoOpen && (QFile(conf.lastDirectory+"/.scrobbler.log").exists() || QFile(conf.lastDirectory+"/.scrobbler-timeless.log").exists()))
		parse(conf.lastDirectory);
	else
	{
		BrowseDirtree browser(this, tr("Select jukebox mountpoint"));
		browser.setFilter(QDir::AllDirs | QDir::NoDotAndDotDot);

		#if defined(Q_OS_MACX)
		browser.setRoot("/Volumes");
		#endif
		if(QFile(conf.lastDirectory).exists())
		{
			browser.setDir(conf.lastDirectory);
		}

		if(browser.exec() == QDialog::Accepted)
		{
			qDebug() << "Open Folder: " << browser.getSelected();
			parse(browser.getSelected());
		}
	}
}

void QTScrob::parse(QString fileDir) {
	QString entry;
	if ( !fileDir.isEmpty()) {

		file.setFileName(fileDir + "/.scrobbler-timeless.log");

		if (file.exists()) {
			timeless = true;
			qWarning("Timeless file detected");
		} else {

			file.setFileName(fileDir + "/.scrobbler.log");

			if (file.exists()) {
				timeless = false;
			} else {
				QMessageBox::warning(this, QCoreApplication::applicationName(), ("Cannot find .scrobbler.log in selected directory"));
				return;
			}
		}
	}
	file.close();
	if (!scrob->parse_file(file.fileName().toStdString(), conf.tz_override?conf.utc_offset:scrob->get_offset())) {
		QMessageBox::warning(this, QCoreApplication::applicationName(), QString("Error parsing file: %1").arg(scrob->get_error_str().c_str()));
		return;
	}

	loadtable();

	lblStatus->setText(tr("Loaded %1 entries").arg(logTable->rowCount()));

	if (timeless)
		scrob->recalc_now();

	if (QString::compare(fileDir, conf.lastDirectory) != 0)
	{
		QSettings settings("qtscrob");
		settings.setValue("application/lastDirectory",fileDir);
		settings.deleteLater();
	}

	set_open_buttons();
}

void QTScrob::openiTunesDB() {
	if (QFile(conf.lastDirectory+"/iPod_Control/iTunes/iTunesDB").exists() && QFile(conf.lastDirectory+"/iPod_Control/iTunes/Play Counts").exists())
		parseiTunesDB(conf.lastDirectory);
	else
	{
		BrowseDirtree browser(this, tr("Select iPod mountpoint"));
		browser.setFilter(QDir::AllDirs | QDir::NoDotAndDotDot);

		#if defined(Q_OS_MACX)
		browser.setRoot("/Volumes");
		#endif
		if(QFile(conf.lastDirectory).exists())
		{
			browser.setDir(conf.lastDirectory);
		}

		if(browser.exec() == QDialog::Accepted)
		{
			qDebug() << "Open Folder: " << browser.getSelected();
			parseiTunesDB(browser.getSelected());
		}
	}
}

void QTScrob::parseiTunesDB(QString fileDir) {
	if (!scrob->parse_db(fileDir.toStdString(), conf.tz_override?conf.utc_offset:scrob->get_offset())) {
		QMessageBox::warning(this, QCoreApplication::applicationName(), QString("Error parsing file: %1").arg(scrob->get_error_str().c_str()));
		return;
	}
	else
		file.setFileName(fileDir + "/iPod_Control/iTunes/Play Counts");

	loadtable();

	lblStatus->setText(tr("Loaded %1 entries").arg(logTable->rowCount()));

	if (QString::compare(fileDir, conf.lastDirectory) != 0)
	{
		QSettings settings("qtscrob");
		settings.setValue("application/lastDirectory",fileDir);
		settings.deleteLater();
	}

	set_open_buttons();
}

#ifdef HAVE_MTP
void QTScrob::openMTP()
{
	this->setCursor(Qt::WaitCursor);
	if (!scrob->mtp_connect())
	{
		QMessageBox::warning(this, QCoreApplication::applicationName(), QString::fromStdString(scrob->get_error_str()));
		this->unsetCursor();
		return;
	}

	parseMTP();
}

void QTScrob::parseMTP()
{
	scrob->mtp_get_tracks();

	loadtable();

	lblStatus->setText(tr("Loaded %1 entries").arg(logTable->rowCount()));

	set_open_buttons();
	this->unsetCursor();
}
#endif // HAVE_MTP

void QTScrob::loadtable(void) {
	size_t i;
	scrob_entry scrob_tmp;

	logTable->disconnect(SIGNAL(cellChanged(int, int)));

	logTable->clearContents();

	logTable->setRowCount(scrob->get_num_tracks());

	QApplication::setOverrideCursor(Qt::WaitCursor);

	// using fromUtf8 rather than fromStdString since it doesn't work with UTF8 strings
	for (i = 0; i < scrob->get_num_tracks(); i++) {
		scrob_tmp = scrob->get_track(i);

		QTableWidgetItem *temp_artist = new QTableWidgetItem;
		temp_artist->setText(QString::fromUtf8(scrob_tmp.artist.c_str()));
		logTable->setItem(i, 0, temp_artist );

		QTableWidgetItem *temp_album = new QTableWidgetItem;
		temp_album->setText(QString::fromUtf8(scrob_tmp.album.c_str()));
		logTable->setItem(i, 1, temp_album );

		QTableWidgetItem *temp_title = new QTableWidgetItem;
		temp_title->setText(QString::fromUtf8(scrob_tmp.title.c_str()));
		logTable->setItem(i, 2, temp_title );

		QTableWidgetItem *temp_tracknum = new QTableWidgetItem;
		temp_tracknum->setText(QString::fromUtf8(scrob_tmp.tracknum.c_str()));
		logTable->setItem(i, 3, temp_tracknum );

		QTableWidgetItem *temp_length = new QTableWidgetItem;
		temp_length->setText(QString::number(scrob_tmp.length, 10));
		logTable->setItem(i, 4, temp_length );

		QTableWidgetItem *temp_played = new QTableWidgetItem;
		temp_played->setText(QChar(scrob_tmp.played));
		logTable->setItem(i, 5, temp_played );

		QTableWidgetItem *temp_when = new QTableWidgetItem;
		temp_when->setText((QDateTime::fromTime_t(scrob_tmp.when).toUTC()).toString("yyyy-MM-dd hh:mm:ss"));
		logTable->setItem(i, 6, temp_when );

	}
	QApplication::restoreOverrideCursor();

	connect(logTable, SIGNAL(cellChanged(int, int)), this, SLOT(changeRow(int, int)));
}

void QTScrob::updatebaseDT(QDateTime newbaseDT) {
	baseDT = newbaseDT;
}

void QTScrob::recalcDT(void) {
	// Since QT will always convert to a GMT unixtime stamp, use this rather
	// horrible method to create one ourselves, and adjust it according to
	// OUR settings (automatic or manual override)
/*
	time_t offset_timestamp = scrob.calc_when(
		baseDT.toString("yyyy M d h m s").toAscii().data() );

	offset_timestamp -= conf.tz_override?conf.utc_offset:scrob.offset;
*/
	time_t offset_timestamp = baseDT.toTime_t();

	offset_timestamp -= conf.tz_override?conf.utc_offset:scrob->get_offset();

	scrob->recalc_dt(offset_timestamp);

	loadtable();
}

void QTScrob::deleteRow() {
	if (logTable->rowCount()) { // Items are displayed. 
	if (logTable->rowCount() == 1) { // Handle it as one item. 
		QTableWidgetItem *artist = new QTableWidgetItem();
		QTableWidgetItem *title = new QTableWidgetItem();
		artist = logTable->item(logTable->currentRow(),0);
		title = logTable->item(logTable->currentRow(),2);
		QMessageBox mb(QCoreApplication::applicationName(), \
				tr("Are you sure you want to delete: ") + artist->text() + " - " + title->text() + " ?", \
				QMessageBox::Information, \
				QMessageBox::Yes | QMessageBox::Default, \
				QMessageBox::No, 0);
			mb.setButtonText(QMessageBox::Yes, "Yes");
			mb.setButtonText(QMessageBox::No, "No");
			switch(mb.exec()) {
				case QMessageBox::Yes:
					scrob->remove_track(logTable->currentRow());
					break;
				case QMessageBox::No:
					break;
			}
	} else { // More than one item. 
		QMessageBox mb(QCoreApplication::applicationName(), \
			tr("Are you sure you want to delete the selected items") + "?", \
			QMessageBox::Information, \
			QMessageBox::Yes | QMessageBox::Default, \
			QMessageBox::No, 0);
		mb.setButtonText(QMessageBox::Yes, "Yes");
		mb.setButtonText(QMessageBox::No, "No");
		switch(mb.exec()) {
			case QMessageBox::Yes:
				for (int i = logTable->rowCount() - 1; i >= 0; i--) { 
					if (logTable->item(i, 0)->isSelected())
						scrob->remove_track(i); 
				}

				break;
			case QMessageBox::No:
				break;
			}
		}
	}
	loadtable();
	if (scrob->get_num_tracks() == 0)
	{
		scrob->clear_method();
		set_open_buttons();
	}
}

void QTScrob::changeRow(int r, int c) {
	scrob_entry tmp = scrob->get_track(r);

	// album name and tracknum are allowed to be blank
	switch (c) {
		case 0:
			if (!logTable->item(r, c)->text().isEmpty())
				tmp.artist = logTable->item(r, c)->text().toUtf8().data();
			break;
		case 1:
			tmp.album = logTable->item(r, c)->text().toUtf8().data();
			break;
		case 2:
			if (!logTable->item(r, c)->text().isEmpty())
				tmp.title = logTable->item(r, c)->text().toUtf8().data();
			break;
		case 3:  // only allow numbers - convert to a int and then back again... 
			tmp.tracknum = QString::number(logTable->item(r, c)->text().toInt(), 10).toAscii().data();
			break;
		case 4:
			if (!logTable->item(r, c)->text().isEmpty())
				tmp.length = logTable->item(r, c)->text().toInt();
			break;
		case 5:
			if (!logTable->item(r, c)->text().isEmpty())
			{
				char rating = logTable->item(r, c)->text().toAscii().data()[0];
				if ('L' == rating || 'S' == rating)
					tmp.played = rating;
			}
			break;
		case 6:
			if (!logTable->item(r, c)->text().isEmpty())
			{
				QDateTime dt = QDateTime::fromString(logTable->item(r, c)->text(), "yyyy-MM-dd hh:mm:ss");
				dt.setTimeSpec(Qt::UTC);
				tmp.when = dt.toTime_t();
			}
			break;
	}
	scrob->update_track(tmp, r);
	loadtable();
}

void QTScrob::scrobble() {
	if ( logTable->rowCount()==0 ) {
		QMessageBox::warning(this, QCoreApplication::applicationName(), "Nothing to submit");
		return;
	}

	if ( conf.username=="" || conf.password=="" ) {
		QMessageBox::warning(this, QCoreApplication::applicationName(), "Missing username and/or password");
		return;
	}


	if (winProgress==NULL)
		winProgress = new Progress(this);
	winProgress->show();
	// make sure the widget appears before we submit()
	QCoreApplication::flush();

	// get an accurate number of tracks
	scrob->cleanup_tracks();
	winProgress->totaltracks = scrob->get_num_tracks();

	qDebug() << "Submitting data";
	lblStatus->setText(tr("Submitting data to server..."));
	btnSubmit->setEnabled(false);

	scrob->start();
}

void QTScrob::scrobbled() {
	qDebug() << "Post submit";
	if (winProgress != NULL)
	{
		winProgress->close();
		delete winProgress;
		winProgress = NULL;
	}

	if(scrob->val) {
		switch (scrob->get_last_method())
		{
			case SCROBBLE_LOG:
				file.remove();
				break;
			case SCROBBLE_IPOD:
				if (conf.checkDelPlayList && file.fileName().endsWith("Play Counts"))
					file.remove();
				break;
#ifdef HAVE_MTP
			case SCROBBLE_MTP:
				scrob->mtp_clear_tracks();
				scrob->mtp_disconnect();
				break;
#endif
			case SCROBBLE_NONE:
			default:
				qDebug() << "Unknown scrobble_method encountered!";
		}
		lblStatus->setText(tr("Data submitted Ok"));
		qDebug() << "Submission complete";
	} else {
		qDebug() << "Submission failed: " << QString::fromStdString( scrob->get_error_str() );

		QMessageBox::warning(this, QCoreApplication::applicationName(), tr("There was a problem submitting data to the server\n(reason: %1)\n\nPlease check your internet connection, or try again later").arg( QString::fromStdString( scrob->get_error_str() )));
		lblStatus->setText(tr("Submission Failed - server error"));
	}

	loadtable();
	scrob->clear_method();
	set_open_buttons();
	btnSubmit->setEnabled(true);
}

void QTScrob::set_open_buttons()
{
// don't allow more than one method to be opened at a time
// otherwise, what to clear/delete afterwards??

	if (scrob->get_last_method() == SCROBBLE_NONE)
	{
		actOpen->setEnabled(true);
		actOpeniTunes->setEnabled(true);
		btnOpen->setEnabled(true);
		btnOpeniTunes->setEnabled(true);
#ifdef HAVE_MTP
		actOpenMTP->setEnabled(true);
		btnOpenMTP->setEnabled(true);
#endif
	}
	else
	{
		actOpen->setEnabled(false);
		actOpeniTunes->setEnabled(false);
		btnOpen->setEnabled(false);
		btnOpeniTunes->setEnabled(false);
#ifdef HAVE_MTP
		actOpenMTP->setEnabled(false);
		btnOpenMTP->setEnabled(false);
#endif
	}
}
