/***************************************************************************
 *            main.c
 *
 *  Sun Sep 24 12:11:40 2006
 *  Copyright  2006-2007  Neil Williams
 *  linux@codehelp.co.uk
 ****************************************************************************/
/** @file main.c
	@brief Utility and conversion functions.
	@author Copyright 2006-2007  Neil Williams <linux@codehelp.co.uk>
	@author Copyright 1999  Robert Lissner
*/
/*
    This package is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "config.h"
#define _GNU_SOURCE
#include <errno.h>
#include <math.h>
#include <time.h>
#include <stdlib.h>
#include <string.h>
#include <locale.h>
#include <ctype.h>
#include <glib.h>
#include <gtk/gtk.h>
#include <glib/gi18n.h>
#include <gtkextra/gtksheet.h>
#include <libgnomevfs/gnome-vfs.h>
#include "types.h"
#include "dialog_initial.h"
#include "dim_list_menu.h"
#include "edit.h"
#include "filein.h"
#include "fileout.h"
#include "main.h"

/** \brief convert old quicklist date formats to new

\since 0.9.0

Previous versions had limited date support, v0.9.0 starts
a wider progression to a more comprehensive date, time and
number formatting options.
*/
const gchar *
convert_old_df (QlDateFormat df)
{
	switch (df)
	{
		case QL_DF_USSHORT     : { return QL_DF_ISO99;   break; }
		case QL_DF_USLONG      : { return "%m/%d/%Y";    break; }
		case QL_DF_USSHORT_2   : { return "%m.%d.%y";    break; }
		case QL_DF_USLONG_2    : { return QL_DF_US;      break; }
		case QL_DF_LOCAL_MONTH : { return "%b %m %y";    break; }
		case QL_DF_LONGYEAR    : { return QL_DF_ISO8601; break; }
		case QL_DF_LOCAL_DAY   : { return "%m %b %Y";    break; }
		case QL_DF_EURSHORT    : { return QL_DF_UK;      break; }
		case QL_DF_EURLONG     : { return "%d/%m/%Y";    break; }
		case QL_DF_LAST : { return NULL; }
	}
	return NULL;
}

const gchar *
convert_old_tf (QlTimeFormat tf)
{
	switch (tf)
	{
		case QL_TF_12P :   { return "%l:%M %p";    break; }
		case QL_TF_24  :   { return "%R";          break; }
		case QL_TF_DOT24 : { return "%H.%M";       break; }
		case QL_TF_24P :   { return "%H:%M%P";     break; }
		case QL_TF_T :     { return "%T";          break; }
		case QL_TF_12TP :  { return "%l:%M:%S %p"; break; }
		case QL_TF_LAST :  { return NULL; }
	}
	return NULL;
}

/** old values are mapped to the new */
const gchar *
convert_old_cf (QlCurrFormat cf)
{
	switch (cf)
	{
		/* "1,234.56" %! formatting style for formatting == 0 */
		case QL_NF_GRP :   { return "%!i"; break;  }
		/* "1.234,56" used for files only */
		case QL_NF_OLD_A : { return "%!i"; break;  }
		/* "1234.56"  %^! */
		case QL_NF :       { return "%^!i"; break; }
		/* "1234,56" used for files only */
		case QL_NF_OLD_B : { return "%^!i"; break; }
		/* "$1,234.56"  default (%n) */
		case QL_NF_CURR :  { return "%n"; break;  }
		/* "1,234.56%" used for files only */
		case QL_NF_OLD_C : { return "%!i%%"; break;  }
		/* "1.234,56%" used for files only */
		case QL_NF_PERCENT : { return "%!i%%"; break;  }
		/** unused check value */
		case QL_NF_LAST :  { return NULL; }
	}
	return NULL;
}

/** \bug replace with QofDate */
gchar *
display_date (GDate * gd, const gchar * fmt)
{
	gchar buf[MAX_DATE_BUFFER];
	struct tm tm;
	gsize len;

	g_return_val_if_fail (g_date_valid (gd), NULL);
	g_return_val_if_fail (fmt, NULL);
	buf[0] = '\1';
	g_date_to_struct_tm (gd, &tm);
	len  = strftime (buf, MAX_DATE_BUFFER, fmt, &tm);
	if (len == 0 && buf[0] != '\0')
		return NULL;
	return g_strdup (buf);
}

/** \bug replace with QofTime */
gchar *
display_time (GTimeVal * gt, const gchar * fmt)
{
	gchar buf[MAX_DATE_BUFFER];
	struct tm tm;
	gsize len;

	g_return_val_if_fail (gt, NULL);
	g_return_val_if_fail (fmt, NULL);
	buf[0] = '\1';
	tm = *localtime_r (&gt->tv_sec, &tm);
	len  = strftime (buf, MAX_DATE_BUFFER, fmt, &tm);
	if (len == 0 && buf[0] != '\0')
		return NULL;
	return g_strdup (buf);
}

/** \brief general purpose date parser

 \since 0.9.0

Due to be extended further with QofDate from the
Query Object Framework but currently using GDate.
*/
GDate *
parse_date (const gchar * string, const gchar * format)
{
	GDate * gd;
	g_return_val_if_fail (string, NULL);
	if (strlen (string) >= MAX_DATE_BUFFER)
		return NULL;
	gd = g_date_new ();
	g_date_set_parse (gd, string);
	if (g_date_valid (gd))
		return gd;
	else
	{
		/** \bug improve this handling with QofDate */
		const gchar *cp;
		struct tm tm;
		time_t now;
		now = time (NULL);
		gmtime_r (&now, &tm);
		cp = strptime (string, format, &tm);
		g_date_set_dmy (gd, tm.tm_mday, tm.tm_mon + 1, tm.tm_year);
		return gd;
	}
}

void
close_dlg (GtkWidget G_GNUC_UNUSED * w, gpointer data)
{
	g_return_if_fail (data);
	gtk_widget_destroy (GTK_WIDGET(data));
}

int
main (int argc, char **argv)
{
	QlContext * qlc;
	gchar * filename, *cwd;

	/* I18N */
	setlocale (LC_ALL, "");
	bindtextdomain (PACKAGE, LOCALE_DIR);
	textdomain (PACKAGE);

	filename = NULL;
	gnome_vfs_init();
	gtk_init (&argc, &argv);
	qlc = ql_context_init ();
	main_window(qlc);
	if (argc >= 2 && argv[1])
	{
		GnomeVFSFileInfo *vfsinfo;
		GnomeVFSResult vfsresult;
		gchar * arg = g_strdup (argv[1]);

		vfsinfo = gnome_vfs_file_info_new ();
		vfsresult = gnome_vfs_get_file_info (arg, vfsinfo, GNOME_VFS_FILE_INFO_DEFAULT);
		if (vfsinfo->type == GNOME_VFS_FILE_TYPE_UNKNOWN)
		{
			/* relative locations seem to cause trouble with VFS */
			cwd = g_strconcat (g_get_current_dir(), "/", NULL);
			g_free (arg);
			arg = gnome_vfs_uri_make_full_from_relative (cwd, argv[1]);
			g_free (cwd);
			/* refresh info for the new location */
			vfsresult = gnome_vfs_get_file_info (arg, vfsinfo, GNOME_VFS_FILE_INFO_DEFAULT);
		}
		if ((vfsinfo->type == GNOME_VFS_FILE_TYPE_REGULAR) &&
			(vfsinfo->size > 0))
		{
			filename = g_strdup(arg);
			actually_open (filename, MODE_OPEN, QL_OLD_QLF, qlc);
		}
		g_free (filename);
	}
	gtk_main ();
	ql_free_context (qlc);
	return 0;
}								/* end of main */

void
add1row (QlTabData * tab)
{
	gtk_sheet_add_row (tab->view->sheet, 1);
	gtk_sheet_row_button_add_label (tab->view->sheet,
		++tab->file->last_row, " ");
	front_is_changed (tab);
}

void
big_draw_start (QlTabData * tab)
{
	if (tab->view->display_mode == DISPLAY_LIST)
	{
		g_signal_handler_block
			(tab->view->sheet, tab->view->activate_signal);
		g_signal_handler_block
			(tab->view->sheet, tab->view->new_column_width_signal);
		g_signal_handler_block
			(tab->view->sheet, tab->view->select_range_signal);
		gtk_sheet_freeze (tab->view->sheet);
	}
	else
		gtk_sheet_freeze (tab->view->report_sheet);
}

void
big_draw_end (QlTabData * tab)
{
	if (tab->view->display_mode == DISPLAY_LIST)
	{
		gtk_sheet_thaw (tab->view->sheet);
		g_signal_handler_unblock
			(tab->view->sheet, tab->view->activate_signal);
		g_signal_handler_unblock
			(tab->view->sheet, tab->view->new_column_width_signal);
		g_signal_handler_unblock
			(tab->view->sheet, tab->view->select_range_signal);
	}
	else
		gtk_sheet_thaw (tab->view->report_sheet);
}

gint
check_entry (QlTabData * tab, gchar * entry)
{
	gint length;
	gchar *error_message = _("Text must contain valid characters "
		"and not contain \\ * or %");

	if (!entry)
		return 0;
	length = strlen (entry);
	if (!length)
		return 0;
	while (length && entry[length - 1] == ' ')
		entry[--length] = '\0';
	if (length && !strpbrk (entry, "\\*%"))
		return 0;
	level2_error (tab, error_message);
	return -1;
}

gboolean
check_if_changed (QlTabData * tab)
{
	gint fieldx;
	gdouble temp_double;
	gchar linebuf[40]; /**< \bug remove the static buffer */
	gchar *newtext;
	QlFieldInfo * field;

	g_return_val_if_fail (tab, FALSE);
	/* initial vals */
	if (tab->view->activate_row < 0 || tab->view->activate_col < 0)
		return FALSE;
	if (!tab->view->sheet)
		return FALSE;
	if (tab->file->changed)
		return TRUE;
	newtext = gtk_sheet_cell_get_text (tab->view->sheet,
		tab->view->activate_row, tab->view->activate_col);

	/** \todo replace with safe_strcmp from QOF */
	/* both zero */
	if (!tab->view->activate_text && !newtext)
		return FALSE;

	if (tab->view->activate_text && newtext)
		if (!(strcmp (newtext, tab->view->activate_text)))
		return FALSE;

	/* entries clearly don't match */
	front_is_changed (tab);
	if (!newtext)
		return FALSE;

	/* check that input is valid in non-text fields */
	fieldx = tab->file->col_to_field[tab->view->activate_col];
	field = ql_get_fieldinfo (tab, fieldx);
	if (field->type != FIELD_TYPE_TEXT)
	{
		temp_double = qls2d (newtext, field->type,
			field->formatting);

		if (temp_double >= HUGE_VAL)
		{
			level2_error (tab, _("The cell contents are not valid."));
			return (TRUE);
		}
		d2qls (linebuf, temp_double, field->type,
			field->formatting,
			field->decimal_places);
		gtk_sheet_set_cell_text (tab->view->sheet,
			tab->view->activate_row, tab->view->activate_col, linebuf);
	}
	return TRUE;
}								/* end of check_if_changed () */

/** \brief old compatibility function

Converts an internal double to an old quicklist string.

 \bug formatting is overridden for use by QlCurrFormat as well.

*/
gint
d2qls (gchar * texto, gdouble input, QlFieldType type,
	   QlDateFormat formatting, gint dec_places)
{
	/* these are for numeric conversions */
	gboolean is_negative;
	gboolean is_currency;
	gboolean is_percent;
	gchar * texti;
	gdouble temp_val;

	/* these are for date conversion */
	GDateYear year;
	GDateMonth month;
	GDateDay day;

	/* these are for time conversions */
	gint hour;
	gint min;

	temp_val = input;
	if (temp_val >= HUGE_VAL)
	{
		texto = g_strdup ("######");
		return 6;
	}
	switch (type)
	{
		case FIELD_TYPE_NUMERIC:
		{
			is_negative = FALSE;
			if (temp_val < 0)
				is_negative = TRUE;
			is_percent = FALSE;
			is_currency = FALSE;
			switch (formatting)
			{
			case 0:
				{
//					use_for_comma = ',';
					break;
				}
			case 1:
				{
//					use_for_comma = '.';
//					use_for_decimal = ',';
					break;
				}
				/** \todo check why this is empty. */
			case 2:
				break;
			case 3:
				{
//					use_for_decimal = ',';
					break;
				}
			case 4:
				{
//					use_for_currency = '$';
//					use_for_comma = ',';
					break;
				}
			case 5:
				{
//					use_for_comma = ',';
//					is_percent = TRUE;
					break;
				}
				/* this would be case 6 */
			default:
				{
//					use_for_comma = '.';
//					use_for_decimal = ',';
					is_percent = TRUE;
				}
			}
		}
		if (is_percent)
			temp_val = temp_val * 100;
		switch (dec_places)
		{
			case 0:
			{
				texti = g_strdup_printf ("%.0f", temp_val);
				break;
			}
			case 1:
			{
				texti = g_strdup_printf ("%.1f", temp_val);
				break;
			}
			case 2:
			{
				texti = g_strdup_printf ("%.2f", temp_val);
				break;
			}
			case 3:
			{
				texti = g_strdup_printf ("%.3f", temp_val);
				break;
			}
			case 4:
			{
				texti = g_strdup_printf ("%.4f", temp_val);
				break;
			}
			case 5:
			{
				texti = g_strdup_printf ("%.5f", temp_val);
				break;
			}
			case 6:
			{
				texti = g_strdup_printf ("%.6f", temp_val);
				break;
			}
			case 7:
			{
				texti = g_strdup_printf ("%.7f", temp_val);
				break;
			}
			case 8:
			{
				texti = g_strdup_printf ("%.8f", temp_val);
				break;
			}
			/* this would be case 9 */
			default:
			{
				texti = g_strdup_printf ("%.9f", temp_val);
				break;
			}
		}

		/* now count the digits before the decimal place or the end */
/*		if (use_for_comma)
			for (inx = 0; inx < 40; inx++)
			{
				if (texti[inx] >= '0' && texti[inx] <= '9')
					digits_so_far++;
				else if (!texti[inx] || texti[inx] == '.')
					break;
			}
*/
		/* have counted places in front of decimal.  Now set
		   up counter of how many digits to skip before inserting
		   a comma */
/*		digits_so_far %= 3;
		if (!digits_so_far)
			digits_so_far = 3;
*/
		/* start formatting output */
/*		outx = 0;
		if (use_for_currency)
		{
			if (is_negative)
				texto[outx++] = '(';
			texto[outx++] = use_for_currency;
		}
		else if (is_negative)
			texto[outx++] = '-';
		for (inx = 0; inx < 40; inx++)
		{
			if (texti[inx] >= '0' && texti[inx] <= '9')
			{
				if (use_for_comma && !digits_so_far)
				{
					texto[outx++] = use_for_comma;
					digits_so_far = 3;
				}
				texto[outx++] = texti[inx];
				digits_so_far--;
			}
			else if (texti[inx] == '.')
			{
*/				/* turn off commas past decimal point */
/*				use_for_comma = '\0';
				texto[outx++] = use_for_decimal;
			}
			else if (!texti[inx])
				break;
		}
*/
		/* finished moving everything.  Put percent or minus on end */
/*		if (use_for_currency && is_negative)
			texto[outx++] = ')';
		if (is_percent)
			texto[outx++] = '%';
		texto[outx] = '\0';*/
		texto = texti;
		return 0;
		break;					/* end of FIELD_TYPE_NUMERIC */

		case FIELD_TYPE_DATE:
		{
			GDate * gd;
			const gchar * fmt;
			fmt = convert_old_df (formatting);
			year = (gint) temp_val / 10000;
			month = (gint) temp_val / 100 % 100;
			day = (gint) temp_val % 100;
			if (g_date_valid_dmy (day, month, year))
			{
				gd = g_date_new_dmy (day, month, year);
				strcpy (texto, display_date (gd, fmt));
				return strlen (texto);
			}
			return 0;
			break;
		}
		/* FIELD_TYPE_TIME */
		default:
		{
			hour = temp_val / 60;
			min = (gint) temp_val % 60;
			if (!formatting)
			{					/* 0 is 7:15 PM */
				if (!hour)
					hour = 12;
				if (hour > 12)
					hour -= 12;
				if (temp_val > 719)	/* half a day */
					return (sprintf (texto, "%u:%u%u PM", hour, min / 10,
							min % 10));
				else
					return (sprintf (texto, "%u:%u%u AM", hour, min / 10,
							min % 10));
			}

			if (formatting == 1)
				return (sprintf (texto, "%u%u:%u%u", hour / 10, hour % 10,
						min / 10, min % 10));

			/* all that remains is 18.00 */
			return (sprintf (texto, "%u.%u%u", hour, min / 10, min % 10));
			break;
		}
	}
	return 0;
}								/* end of d2qls */

void
connect_signals (QlTabData * tab)
{
	tab->view->activate_signal = g_signal_connect
		(G_OBJECT (tab->view->sheet),
		"activate", G_CALLBACK (activate_callback), tab);
	tab->view->new_column_width_signal = g_signal_connect
		(G_OBJECT (tab->view->sheet),
		"new_column_width", G_CALLBACK (new_column_width), tab);
	tab->view->select_range_signal = g_signal_connect
		(G_OBJECT (tab->view->sheet),
		"select_range", G_CALLBACK (select_range_callback), tab);
}								/* end of connect_signals */

/** \brief notebook-aware dirty flag

\todo Rename function to indicate the move
away from a static 'front' to a notebook tab,
as used internally.
*/
void
front_is_changed (QlTabData * tab)
{
	if (tab->file->changed)
		return;
	tab->file->changed = TRUE;
	dim_list_file_menu (tab->qlc);
}

/** Save the x and y and height and width of window
 \todo retain for next session using GConf */
void
get_window_size_loc (GtkWidget * win)
{
	gint x, y;
	QlContext * qlc;
	QlTabData * tab;

	qlc = ql_get_context (GTK_WIDGET(win));
	tab = ql_get_tabdata (qlc);

	tab->view->width = win->allocation.width;
	tab->view->height = win->allocation.height;
	gdk_window_get_deskrelative_origin (win->window, &x, &y);
	tab->view->x = x;
	tab->view->y = y;
}

void
level1_error (QlTabData * tab, gchar * message)
{
	GtkWidget * dlg;

	dlg = gtk_message_dialog_new (GTK_WINDOW(tab->qlc->parent),
		GTK_DIALOG_DESTROY_WITH_PARENT,
		GTK_MESSAGE_INFO, GTK_BUTTONS_CLOSE,
		"%s", message);
	g_signal_connect (GTK_OBJECT (dlg), "response",
		G_CALLBACK (gtk_widget_destroy), dlg);
	gtk_dialog_run (GTK_DIALOG (dlg));
	gtk_widget_destroy (dlg);
}								/* End of level1_error */

/** Error dialog box when there's already a dialogue on
   the screen, and it has to stay

\bug pointless function.
*/
void
level2_error (QlTabData * tab, gchar * message)
{
	GtkWidget * dlg;

	dlg = gtk_message_dialog_new (GTK_WINDOW(tab->qlc->parent),
		GTK_DIALOG_DESTROY_WITH_PARENT,
		GTK_MESSAGE_INFO, GTK_BUTTONS_CLOSE,
		"%s", message);
	gtk_dialog_run (GTK_DIALOG (dlg));
	gtk_widget_destroy (dlg);
}								/* level2_error */

/** \todo replace with custom dialog functions */
void
make_basic_dialog1 (void)
{
	GtkWidget * dialog1_win;
	dialog1_win = gtk_dialog_new ();
	gtk_window_set_modal (GTK_WINDOW (dialog1_win), TRUE);
	gtk_window_set_position (GTK_WINDOW (dialog1_win), GTK_WIN_POS_CENTER);
	gtk_window_set_resizable (GTK_WINDOW (dialog1_win), TRUE);
	gtk_container_set_border_width (GTK_CONTAINER (dialog1_win), 5);
}

/**  converts a quicklist string (1,234.56) to double.
	\bug Have to check for NULL and newlines at end of
	line because of reading input files.

	\todo replace with QOF/GLib functions.
*/
gdouble
qls2d (const gchar * input, gint type, gint formatting)
{
	gdouble temp_double;
	gint inx;

	/* these are for time conversion */
	gchar before_char[15];
	gint beforex;
	gchar after_char[15];
	gint afterx;
	gboolean got_sep;
	gboolean got_pm;
	gboolean got_am;

	if (!input || !input[0])
		return HUGE_VAL;
	for (inx = 0; input[inx] == ' '; inx++);
	if (!input[inx])
		return HUGE_VAL;

	if (type == FIELD_TYPE_NUMERIC)
	{
		gchar *tail;

		temp_double = strtod (input, &tail);
		if (errno)
			return HUGE_VAL;
		return temp_double;
	}
	else if (type == FIELD_TYPE_DATE)
	{
		GDate * gd;
		GDateDay day = 20;
		GDateMonth month = 3;
		GDateYear year = 2004;
		gdouble retval;

		/** \bug default format is day/month/year

		parse_date is locale sensitive, default was not. */
		gd = parse_date (input, convert_old_df (formatting));
		if (!gd)
		{
			fprintf (stderr, "null gdate\n");
			fprintf (stderr, "format=%s\n",
				convert_old_df (formatting));
			retval = year * 10000 + month * 100 + day;
			fprintf (stderr, "val=%f\n", retval);
			return retval;
		}
		if (g_date_valid (gd))
		{
			day = g_date_get_day (gd);
			month = g_date_get_month (gd);
			year = g_date_get_year (gd);
		}
		retval = year * 10000 + month * 100 + day;
	return retval;
	}

	else
	{
		/* this would be FIELD_TYPE_TIME */
		beforex = afterx = 0;
		got_sep = got_pm = got_am = FALSE;
		for (; inx < 15; inx++)
		{
			if (input[inx] >= '0' && input[inx] <= '9')
			{
				if (got_sep)
					after_char[afterx++] = input[inx];
				else
					before_char[beforex++] = input[inx];
			}
			/* for efficient typists */
			else if (input[inx] == ':' || input[inx] == ';'
				|| input[inx] == '.')
				got_sep = TRUE;
			else if (input[inx] == ' ')
			{
				if (beforex && !afterx)
					got_sep = TRUE;
			}
			else if ((input[inx] == 'P' || input[inx] == 'p')
				&& !got_am && !got_pm)
				got_pm = TRUE;
			else if ((input[inx] == 'A' || input[inx] == 'a')
				&& !got_am && !got_pm)
				got_am = TRUE;
			else if (input[inx] == 'm' || input[inx] == 'M');
			else if (input[inx] < ' ')	/* end of string */
				break;
			else
				return HUGE_VAL;	/* invalid characters */
		}
		/* wrap up the time conversion */
		if (!beforex)
			return HUGE_VAL;	/* ie, no data at all */
		before_char[beforex] = '\0';
		temp_double = atof (before_char) * 60;
		/** \bug another failing comparison */
		if ((glong)temp_double == 720	/* 12:00 */
			&& got_am)
			temp_double = 0;
		if (afterx)
		{
			after_char[afterx] = '\0';
			temp_double += atof (after_char);
		}
		/* for 4:00 pm type formatting, change an entry of 4 to 4:00 pm */
		if (got_pm && temp_double < 720)
			temp_double += 720;	/* 720 minutes is half a day */
		if (!got_am && temp_double < 419 && !formatting)
			temp_double += 720;	/* if 1:00 to 6:59 make it pm by default */
		if (temp_double < 1440)
			return temp_double;
	}
	return HUGE_VAL;
}								/* end of qls2d */

/**  Resets the table point at fields from columns */
void
reset_col_to_field (QlTabData * tab)
{
	gint fieldx;
	gint colx;
	QlFieldInfo * field;
	for (fieldx = 0; fieldx <= tab->file->last_field; fieldx++)
	{
		field = ql_get_fieldinfo (tab, fieldx);
		if (!field)
			continue;
		colx = field->sheet_column;
		tab->file->col_to_field[colx] = fieldx;
	}
}

/**  Force the size of the window we just opened */
void
set_window_size_loc (GtkWidget * win)
{
	QlContext * qlc;
	QlTabData * tab;

	qlc = ql_get_context (GTK_WIDGET(win));
	tab = ql_get_tabdata (qlc);
	gtk_widget_set_size_request (win, tab->view->width, tab->view->height);
}

void
unselect (QlTabData * tab)
{
	if (tab && tab->view->display_mode == DISPLAY_LIST && tab->view->sel_type)
	{
		tab->view->sel_type = SELECT_NONE;
		gtk_sheet_unselect_range (tab->view->sheet);
	}
}
