/*
 * Mike Glover
 * mpg4@duluoz.net
 *
 * Johan Ekenberg
 * johan@ekenberg.nu
 *
 * parse.c
 * command line parsing routines
 */

#include <config.h>

#include <stdio.h>
#include <stdlib.h>

#include "output.h"
#include "parse.h"
#include "quota.h"
#include "quotatool.h"
#include "system.h"



int main (int argc, char **argv) 
{
  int  block_sav, inode_sav;
  int ok;
  int id;
  int old;
  argdata_t *argdata;
  quota_t *quota;
  

  
  /* parse commandline and fill argdata */
  argdata = parse_commandline (argc, argv);
  if ( ! argdata ) {
    exit (ERR_PARSE);
  }


  /* initialize the id to use */
  if ( ! argdata->id ) {
    id = 0;
  }
  else if ( argdata->id_type == QUOTA_USER ) {
    id = (int) system_getuid (argdata->id);
  }
  else {
    id = (int) system_getgid (argdata->id);
  }
  if ( id < 0 ) {
    exit (ERR_ARG);
  }
  
  
  /* get the quota info */
  quota = quota_new (argdata->id_type, id, argdata->qfile);
  if ( ! quota ) {
    exit (ERR_SYS);
  }
  
  if ( ! quota_get(quota) ) {
    exit (ERR_SYS);
  }
  


  /* print a header for verbose info */
  output_info ("");
  output_info ("%-14s %-16s %-16s", "Limit", "Old", "New");
  output_info ("%-14s %-16s %-16s", "-----", "---", "---");



  /*
   *  BEGIN  setting global grace periods
   */

  if ( argdata->block_grace ) {
    old = quota->block_grace;
    quota->block_grace = parse_timespan (old, argdata->block_grace);
    output_info ("%-14s %-16d %-16d", "block grace:", old, quota->block_grace);
  }

  if ( argdata->inode_grace ) {
    old = quota->inode_grace;
    quota->inode_grace = parse_timespan (old, argdata->inode_grace);
    output_info ("%-14s %-16d %-16d", "inode grace:", old, quota->block_grace);
  }



  /* 
   *  FINISH setting global grace periods
   *  BEGIN  setting quotas
   */  


  /* update quota info from the command line */
  if ( argdata->block_hard ) {
    old = quota->block_hard;
    quota->block_hard = parse_size (old, argdata->block_hard);
    if ( quota->block_hard < 0 ) {
      exit (ERR_ARG);
    }
    output_info ("%-14s %-16d %-16d", "block hard:", old, quota->block_hard);
  }

  if ( argdata->block_soft ) {
    old = quota->block_soft;
    quota->block_soft= parse_size (old, argdata->block_soft);
    if ( quota->block_soft < 0 ) {
      exit (ERR_ARG);
    }
    output_info ("%-14s %-16d %-16d", "block soft:", old, quota->block_soft);
  }

  if ( argdata->inode_hard ) {
    old = quota->inode_hard;
    quota->inode_hard = parse_size (old, argdata->inode_hard);
    if ( quota->inode_hard < 0 ) {
      exit (ERR_ARG);
    }
    output_info ("%-14s %-16d %-16d", "inode hard:", old, quota->inode_hard);
  }

  if ( argdata->inode_soft ) {
    old = quota->inode_soft;
    quota->inode_soft = parse_size (old, argdata->inode_soft);
    if ( quota->inode_soft < 0 ) {
      exit (ERR_ARG);
    }
    output_info ("%-14s %-16d %-16d", "inode_soft:", old, quota->inode_soft);

  }




  /* 
   * FINISH setting quotas
   *  BEGIN  resetting grace periods
   *   
   * to "reset" the grace period, we really
   * set the current used {blocks,inodes}
   * to the soft limit - 1, call quota_set,
   * then reinstate the original usage.
   */


  block_sav = quota->block_used;
  inode_sav = quota->inode_used;

  if ( argdata->block_reset ) {
    quota->block_used = quota->block_soft - 1;
  }
  if ( argdata->inode_reset ) {
    quota->inode_used = quota->inode_soft - 1;
  }

  if ( ! argdata->noaction ) {
    ok = quota_set (quota);
    if ( ! ok ) {
      exit (ERR_SYS);
    }
  }
  
  quota->block_used = block_sav;
  quota->inode_used = inode_sav;



  /*
   * FINISH resetting grace periods
   */

  
  if ( ! argdata->noaction ) {
    ok = quota_set (quota);
    if ( ! ok ) {
      exit (ERR_SYS);
    }
  }


  quota_delete (quota);
  exit (0);
}
