/*
** TAPIIR, A software multitap delay with realtime I/O
**
** This software is copyright (c) 2000-2002 by Maarten de Boer
**
** You are free to distribute this software under the terms of
** the GNU General Public License.
** On Debian systems, the complete text of the GNU General Public
** License can be found in /usr/share/common-licenses/GPL file.
**
*/
#ifndef __MTD__
#define __MTD__

typedef unsigned int delaylength_t;

/* template class for multi tap delay
**
** class T = internal processing data type, typically float or int
** class T_IO = I/O data type, typically float or short
*/
template <class T, class T_IO> class MTD
{
public:
	const int nchannels;
	const int ndelays;
	delaylength_t *delaylen;
private:
	const int size;
	const int mask;

	T* buffer;
	T* gain;

	delaylength_t *writeIndex;

	/**** the following might be specialized for specific data types ****/
	float FromGain(const T& g) const { return (float) g; }
	T ToGain(const float& v) const { return (T) v; }
	void MulAdj( T& i ) const { }	
	void Clip( T&) const {}
	
public:
	MTD(const int& _size,const int& _nchannels,const int& _ndelays)
	:nchannels(_nchannels),ndelays(_ndelays),
	 size(_size),mask(_size-1)
	{
		int i;

		buffer = new T[size];
		
		gain = new T[ (ndelays+nchannels) * (ndelays+nchannels+1) ];

		writeIndex = new delaylength_t[ndelays];
		delaylen = new delaylength_t[ndelays];

		for (i=0;i<ndelays;i++) {
			writeIndex[i]=i*(size/ndelays);
			delaylen[i]=1;
		}

		T *aptr = gain;

		for (i = 0; i<(ndelays+nchannels) * (ndelays+nchannels+1); i++)
		{
			(*aptr++) = 0;
		}
	}

	~MTD()
	{
		delete [] buffer;
		delete [] gain;
		delete [] writeIndex;
		delete [] delaylen;
	}

	int ChannelID(const int& chn) const
	{
		return chn + ndelays;
	}

	int OverallID(void) const
	{
		return ndelays + nchannels;
	}

	/* set gain, normalized to 1 */	
	void SetGain(int mixerid,int gainid,const float& val)
	{
		gain[ mixerid * (ndelays+nchannels+1) + gainid ] = ToGain(val);
	}

	/* get gain, normalized to 1 */	
	float GetGain(int mixerid,int gainid) const
	{
		return FromGain(gain[ mixerid * (ndelays+nchannels+1) + gainid ]);
	}

	void Apply(T_IO* inframes,int insize)
	{
		T val[256];
		T *a = gain;

		T_IO* isamples = inframes;
		T_IO* osamples = inframes;
		int i,j;
		while (insize)
		{
			register T* aptr;
			register T* valptr;
			register T acc;
			
			valptr=val;

			for (i=0;i<ndelays;i++) {
				*valptr++ = buffer[(writeIndex[i]+size-delaylen[i])&mask];
			}
			for (i=0;i<nchannels;i++) {
				*valptr++ = (T) *isamples++;
			}

			aptr=a;
			for (i=0;i<ndelays;i++) {
				valptr=val;

				acc = (*aptr++) * (*valptr++);
				for (j=1;j<ndelays + nchannels;j++) {
					acc += (*aptr++) * (*valptr++);
				}
				MulAdj(acc);				

				acc*=*aptr++;
				MulAdj(acc);				

				Clip(acc);
				
				buffer[writeIndex[i]]=acc;
				writeIndex[i]=(writeIndex[i]+1)&mask;
			}
			for (i=0;i<nchannels;i++) {
				valptr=val;

				acc = (*aptr++) * (*valptr++);
				for (j=1;j<ndelays + nchannels;j++) {
					acc += (*aptr++) * (*valptr++);
				}
				MulAdj(acc);				

				acc*=*aptr++;
				MulAdj(acc);				

				Clip(acc);
				
				*osamples++ = (T_IO) acc;
			}
			insize--;
		}
	}
};

/****************************** specializations ******************************/

float MTD<int,short>::FromGain(const int& g) const { return float(g)/256.; }
float MTD<int,float>::FromGain(const int& g) const { return float(g)/256.; }

int   MTD<int,short>::ToGain(const float& f) const { return (int)(f*256.); }
int   MTD<int,float>::ToGain(const float& f) const { return (int)(f*256.); }

void  MTD<float,short>::Clip( float& f) const
{ if (f>32767.) f = 32767.; else if (f<-32767.) f = -32767.; }
void  MTD<int  ,short>::Clip( int&   i) const
{ if (i>32767.) i = 32767; else if (i<-32767.) i = -32767; }

void  MTD<int,float>::MulAdj( int& i) const { i>>=8; }
void  MTD<int,short>::MulAdj( int& i) const { i>>=8; }

/*****************************************************************************/

#endif
