/***************************************************************************
                             rmmsgpckt.c
                             -------------------
    begin                : May 2001
    copyright            : (C) 2001 by Jorge Allyson Azevedo
                                       Milena Scanferla
                                       Magnos Martinello
                                       Daniel Sadoc
    email                : {allyson,milena,magnos,sadoc}@land.ufrj.br
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
/******************************************************************************************************
 *																									  *
 * Reliable Multicast Protocol (rmmsgpckt.c)														  *
 *																									  *
 * Functions to manipulate (mount and unmount) message packets.										  *
 *																									  *
 * use tab spacing = 4																				  *
 *																									  *
 ******************************************************************************************************/

#ifndef RMMSGPCKT_C

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include <ctype.h>

#include "rmstruct.h"
#include "rmmsgpckt.h"

#define RMMSGPCKT_C

#define MIN(a,b)  ((a<b)?a:b) 

#define get(dest, org, bytes) { bzero(dest,bytes); memcpy(dest,org, bytes); org+=bytes; }

#define getPCKInfo(dest, org,bytes) { bzero(dest,bytes); memcpy(dest,org, bytes); dest+=bytes; packet_size+=bytes; }

#define get_member_id(member_id, message)			\
{													\
	fprintf(stderr,"\tMember id:\n");				\
	get(&member_id,message,sizeof(MEMBER_ID));		\
	fprintf(stderr,"\t\tip: "); 					\
	for (k=0; k<MAX_IP_STRING_SIZE; k++)			\
	{												\
	if ((int)(member_id.ip[k])==0) break;			\
	fprintf(stderr,"%c",member_id.ip[k]);			\
	}												\
	fprintf(stderr,"\n");							\
	fprintf(stderr,"\t\tpid: %d\n",member_id.pid);	\
}													\

extern int rmcast_listening_tcp_port;
	
char *MESSAGE_TYPE_NAME[NUMBER_OF_MESSAGE_TYPES] = {  
                                "UN", /* "UNKNOWN PACKET TYPE" */
                                "DT", /* "DATA PACKET        ", */
                                "RT", /* "RETRANSMISSION PACKET ", */
                                "NK", /* "NAK PACKET         ", */
                                "SM", /* "SM PACKET          ", */
                                "UN", /* "UNKNOWN PACKET TYPE" */
                                "RF", /*"REFRESH PACKET     " */ 
                                "JR", /*"JOIN REQUEST PACKET     " */ 
                                "JA" /*"JOIN ACCEPT PACKET     " */ 
                                };


#define EXCHANGE_BYTE_ORDER(in_var)                                             \
    {                                                                           \
        int i = 0, j=sizeof(in_var)-1;                                          \
        unsigned char * var = (unsigned char*)&in_var;                          \
        char temp = 0;                                                          \
        for (i=0;i<j;i++, j--)                                                  \
        {                                                                       \
            temp       = var[ i ];                                              \
            var[ i ]   = var[ j ];                                              \
            var[ j ] = temp;                                                    \
        }                                                                       \
    }



/*******************************************************************************************************
 *
 * void msgPcktShowMessage(BYTE *message)
 *
 * Shows the message on the screen.
 *
 ******************************************************************************************************/

void msgPcktShowMessage(BYTE *message)
{

#ifdef DEBUG_SHOW

	char type;	
	
	MEMBER_ID member_id;  char string[1001];
	
	int sn, pckt_size, data_size, k, port;
   
    int i;
	
	get(&type,message,sizeof(char));

	fprintf(stderr,"\ntype: %d\n",type);
	fprintf(stderr,"member info: \n");

	get_member_id(member_id, message);
    
#ifdef SOLARIS
    EXCHANGE_BYTE_ORDER(member_id.pid);
    
    fprintf(stderr,"SOLARIS byte order pid #: %d\n",member_id.pid);
#endif
    
    
	get(&pckt_size,message,sizeof(int));  
    
    
#ifdef SOLARIS
    EXCHANGE_BYTE_ORDER(pckt_size);
#endif
    

    fprintf(stderr,"packet size: %d\n",pckt_size);

	
	switch (type)
	{
		
	case DATA_PACKET_TYPE:
    
		fprintf(stderr,"data packet: \n");

		get(&sn,message,sizeof(int)); 

#ifdef SOLARIS
        EXCHANGE_BYTE_ORDER(sn);
#endif

        fprintf(stderr,"\tsn: %d\n",sn);

		get(&data_size,message,sizeof(int)); 

#ifdef SOLARIS
        EXCHANGE_BYTE_ORDER(data_size);
#endif

        fprintf(stderr,"\tdata size: %d\n",data_size);

		memcpy(string,message, MIN((sizeof(BYTE)*data_size),1000)); string[MIN(data_size,1000)]='\0'; 

		fprintf(stderr,"\tdata: ");


		for(i=0; i<data_size; i++)
		{
			if (isalnum((int)string[i]))
			{
				fprintf(stderr,"%c ",string[i]);
			}
			else
			{
				fprintf(stderr,"\\%d ",string[i]);
			}
		}
        
		fprintf(stderr,"\n\n");
        
		break;
		
	case RETRANSM_PACKET_TYPE: 
		
		get_member_id(member_id, message);
        
#ifdef SOLARIS
        EXCHANGE_BYTE_ORDER(member_id.pid);
    
        fprintf(stderr,"SOLARIS byte order pid #: %d\n",member_id.pid);
#endif
        
        get(&sn,message,sizeof(int)); 
		get(&data_size,message,sizeof(int)); 
		memcpy(string,message, sizeof(BYTE)*data_size); 
        string[data_size]='\0'; 

        fprintf(stderr,"retransmition packet: \n"); 
		fprintf(stderr,"\tdata packet: \n");
        
#ifdef SOLARIS
        EXCHANGE_BYTE_ORDER(sn);
        EXCHANGE_BYTE_ORDER(data_size);        
#endif
        
        fprintf(stderr,"\t\tsn: %d\n",sn);
        fprintf(stderr,"\t\tdata size: %d\n",data_size);
 		fprintf(stderr,"\tdata: %s\n",string);

			
		for(i=0; i<data_size; i++)
		{
			if (isalnum((int)string[i]))
			{
				fprintf(stderr,"%c ",string[i]);
			}
			else
			{
				fprintf(stderr,"\\%d ",string[i]);
			}
		}
       
		fprintf(stderr,"\n\n");


	break;
		
	case NAK_PACKET_TYPE:
		
		get_member_id(member_id, message);	

#ifdef SOLARIS
        EXCHANGE_BYTE_ORDER(member_id.pid);
    
        fprintf(stderr,"SOLARIS byte order pid #: %d\n",member_id.pid);
#endif
        
		
		get(&sn,message,sizeof(int)); 
        
        fprintf(stderr,"nak packet: \n");

#ifdef SOLARIS
        EXCHANGE_BYTE_ORDER(sn);
#endif  
    
        
        fprintf(stderr,"\tsn: %d\n",sn);
/*        
        get(&sn,message,sizeof(int)); 
        
        fprintf(stderr,"\tlast sn: %d\n",sn);
*/		
    break;
		
		
	case LEAVE_GROUP_PACKET_TYPE:

		fprintf(stderr,"leave group packet: \n");

		break;
        
    case REFRESH_PACKET_TYPE:
    
		get(&sn,message,sizeof(int)); 

#ifdef SOLARIS
        EXCHANGE_BYTE_ORDER(sn);
#endif  
        
    	
        fprintf(stderr, "\tsequence number of last message sent: %d\n", sn);
    
		break;
        
	case JOIN_ACCEPT_PACKET_TYPE: 
    
  		get(&port,message,sizeof(int)); 

#ifdef SOLARIS
        EXCHANGE_BYTE_ORDER(port);
#endif  
        
		
  		fprintf(stderr,"join accept packet (port: %d). \n",port);
        
        break;
        
	case JOIN_REQUEST_PACKET_TYPE:         

        break;

    default:
    
        fprintf(stderr,"Unknown packet type: %d\n",type);
        
        
	}

#endif

	return;
}


/*******************************************************************************************************
 *
 * void msgPcktExchangeByteOrder(PACKET_INFO *pckt_info)
 *
 * Exchange byte order in message (Linux versus Solaris).
 *
 * Arguments:	pckt_info,	a pointer to the strucutre holding the message.
 *
 ******************************************************************************************************/

#ifdef SOLARIS

void msgPcktExchangeByteOrder(PACKET_INFO *pckt_info)
{
    EXCHANGE_BYTE_ORDER(pckt_info->sender_id.pid);
    EXCHANGE_BYTE_ORDER(pckt_info->packet_size);    
    switch (pckt_info->type)
    {
    case DATA_PACKET_TYPE:

        EXCHANGE_BYTE_ORDER(pckt_info->packet_data.data_packet.sn);
        EXCHANGE_BYTE_ORDER(pckt_info->packet_data.data_packet.data_size);
        break;

    
	case RETRANSM_PACKET_TYPE: 
        

        EXCHANGE_BYTE_ORDER(pckt_info->packet_data.retransm_packet.original_sender_id.pid);
        EXCHANGE_BYTE_ORDER(pckt_info->packet_data.retransm_packet.data_packet.sn);        
        EXCHANGE_BYTE_ORDER(pckt_info->packet_data.retransm_packet.data_packet.data_size);
        break;

    
	case NAK_PACKET_TYPE:    
    
        EXCHANGE_BYTE_ORDER(pckt_info->packet_data.nak_packet.requested_member_id.pid);
        EXCHANGE_BYTE_ORDER(pckt_info->packet_data.nak_packet.sn);
        break;                
    
    case REFRESH_PACKET_TYPE:    

        EXCHANGE_BYTE_ORDER(pckt_info->packet_data.refresh_packet.sn_of_last_msg_sent);
        break;
    
	case LEAVE_GROUP_PACKET_TYPE:     
        break;
    
	case JOIN_ACCEPT_PACKET_TYPE:     
        EXCHANGE_BYTE_ORDER(pckt_info->packet_data.join_accept_packet.port);
        break;
    
	case JOIN_REQUEST_PACKET_TYPE:             
        break;
    }
}

#endif

/*******************************************************************************************************
 *
 * void msgPcktUnmountMessage(PACKET_INFO *pckt_info, BYTE *message)
 *
 * Unmount the message come from the network, filling PACKET_INFO structure with the apropriate values.
 *
 * Arguments:	message,	a pointer to the message to be unmounted (source of information);
 *				pckt_info,	a pointer to the target strucutre, which will hold the message info.
 *							(this structure must have been previously allocated).
 *
 ******************************************************************************************************/


void msgPcktUnmountMessage(PACKET_INFO *pckt_info, BYTE *message)
{
#ifdef DEBUG_PCKT
    fprintf(stderr,"msgPcktUnmountMessage() [char: %d MEMBER_ID: %d int: %d]\n",
        sizeof(char),
        sizeof(MEMBER_ID),
        sizeof(int));
#endif    


	
	get(&(pckt_info->type),message,sizeof(char));
	get(&(pckt_info->sender_id),message,sizeof(MEMBER_ID));
	get(&(pckt_info->packet_size),message,sizeof(int));
	
	switch (pckt_info->type)
	{
	case DATA_PACKET_TYPE:
		
		get(&(pckt_info->packet_data.data_packet.sn),message,sizeof(int)); 
		get(&(pckt_info->packet_data.data_packet.data_size),message,sizeof(int));
		(pckt_info->packet_data.data_packet.data)=(BYTE*)message;
		break;
		
	case RETRANSM_PACKET_TYPE: 
		
		get(&(pckt_info->packet_data.retransm_packet.original_sender_id),message,sizeof(MEMBER_ID));
		get(&(pckt_info->packet_data.retransm_packet.data_packet.sn),message,sizeof(int));
		get(&(pckt_info->packet_data.retransm_packet.data_packet.data_size),message,sizeof(int));
		(pckt_info->packet_data.retransm_packet.data_packet.data) = (BYTE*)message;
		break;
		
	case NAK_PACKET_TYPE:
		
		get(&(pckt_info->packet_data.nak_packet.requested_member_id),message,sizeof(MEMBER_ID));	
        get(&(pckt_info->packet_data.nak_packet.sn),message,sizeof(int));	
/*FIXME get(&(pckt_info->packet_data.nak_packet.last_sn),message,sizeof(int));	*/
		break;
		
#ifdef REFRESH
    
    case REFRESH_PACKET_TYPE:
    
    	get(&(pckt_info->packet_data.refresh_packet.sn_of_last_msg_sent), message, sizeof(int));
        break;
#endif
		
	case LEAVE_GROUP_PACKET_TYPE: 
		
		/* do nothing. We already have the sender id*/
		
		break;
        
	case JOIN_ACCEPT_PACKET_TYPE: 
    
       	get(&(pckt_info->packet_data.join_accept_packet.port), message, sizeof(int));
		break;

	case JOIN_REQUEST_PACKET_TYPE:         
        
        break;
        
    default:
    
        fprintf(stderr,"Unknown packet type: %d\n",pckt_info->type);
        
   
        break;
        
	}
#ifdef SOLARIS    

    fprintf(stderr,"That's SOLARIS so, excanching by order:\n[sender pid] ");
    fprintf(stderr,"%d %x\n",pckt_info->sender_id.pid,pckt_info->sender_id.pid);

    msgPcktExchangeByteOrder(pckt_info);

    fprintf(stderr,"[sender pid]: %d %x\n",pckt_info->sender_id.pid,pckt_info->sender_id.pid);
    
#endif    

#ifdef DEBUG_PCKT
    fprintf(stderr,"returning from msgPcktUnmountMessage()\n");
#endif    

	return; 
    
}

/*******************************************************************************************************
 *
 * void msgPcktMountMessage(PACKET_INFO *pckt_info, BYTE **message, int *msgsize)
 *
 * Mount the message to be sent over the network, getting data from the PACKET_INFO structure.
 *
 * Arguments:	pckt_info,	a pointer to the source structure, from which data will be retrieved;
 *				message,	a pointer to the message to be mounted - 
 *							(the space needed to store the message will be allocated as necessary);
 *				msgsize,	a pointer to an integer which will return the effective size of the message.
 *
 ******************************************************************************************************/

void msgPcktMountMessage(PACKET_INFO *pckt_info, BYTE **message, int *msgsize)
{
	BYTE *aux_message_pt;
	BYTE *common_header;
	
	int *sn_pointer;
	int common_header_size = sizeof(char)+sizeof(MEMBER_ID)+sizeof(int); 
	int msg_specific = 0, data_size = 0;
	int packet_size = 0;

#ifdef DEBUG_PCKT
	fprintf(stderr,"mounting packet\n");
#endif

#ifdef SOLARIS
    fprintf(stderr,"That's SOLARIS so, excanching by order:\n[sender pid] ");
    fprintf(stderr,"%d %x\n",pckt_info->sender_id.pid,pckt_info->sender_id.pid);

    msgPcktExchangeByteOrder(pckt_info);
    
#endif

#ifdef DEBUG_PCKT
    fprintf(stderr,"[sender pid] ");
    fprintf(stderr,"%d %x\n",pckt_info->sender_id.pid,pckt_info->sender_id.pid);
#endif    

	
	/* info about type of the message, sender_id and packet_size */
	
	common_header = (BYTE*)malloc(common_header_size);
	
	aux_message_pt = common_header;

#ifdef DEBUG_PCKT
	fprintf(stderr,"common_header allocated at: %p\n", common_header);
#endif	  
	
	getPCKInfo(aux_message_pt, &(pckt_info->type), sizeof(char));
	getPCKInfo(aux_message_pt, &(pckt_info->sender_id), sizeof(MEMBER_ID));
	getPCKInfo(aux_message_pt, &(pckt_info->packet_size), sizeof(int));
    
	
	packet_size = 0;
	
	/* specific info plus data */
	
	switch(pckt_info->type)
	{
		
	case DATA_PACKET_TYPE: 
		
		msg_specific = sizeof(DATA_PACKET)-sizeof(BYTE*);
		data_size = (pckt_info)->packet_data.data_packet.data_size;
		
		(*message) = (BYTE*)malloc(common_header_size+msg_specific+data_size);
		aux_message_pt = *message;
		getPCKInfo(aux_message_pt,common_header,common_header_size);
		
		sn_pointer = (int*)aux_message_pt;

		getPCKInfo(aux_message_pt,&(pckt_info->packet_data.data_packet.sn), sizeof(int));
		
		getPCKInfo(aux_message_pt,&(pckt_info->packet_data.data_packet.data_size), sizeof(int));
		getPCKInfo(aux_message_pt, (pckt_info->packet_data.data_packet.data), data_size);
		
		
		break;
		
	case RETRANSM_PACKET_TYPE: 
		
		msg_specific = sizeof(RETRANSM_PACKET)-sizeof(BYTE*);
		
		
		data_size = sizeof(BYTE)*(pckt_info)->packet_data.retransm_packet.data_packet.data_size;
		
		
		(*message) = (BYTE*)malloc(common_header_size+msg_specific+data_size);
		aux_message_pt = *message;
		getPCKInfo(aux_message_pt,common_header,common_header_size);
		
		
		getPCKInfo(aux_message_pt,(void*)(&(pckt_info->packet_data.retransm_packet)), msg_specific);
		getPCKInfo(aux_message_pt,(pckt_info->packet_data.retransm_packet.data_packet.data), data_size);		   
		
		break;
		
	case NAK_PACKET_TYPE:
		
		(*message) = (BYTE*)malloc(common_header_size+sizeof(NAK_PACKET));
		aux_message_pt = *message;
		getPCKInfo(aux_message_pt,common_header,common_header_size);
		getPCKInfo(aux_message_pt, &(pckt_info->packet_data.nak_packet.requested_member_id), sizeof(MEMBER_ID));        
		getPCKInfo(aux_message_pt, &(pckt_info->packet_data.nak_packet.sn), sizeof(int));
/*		getPCKInfo(aux_message_pt, &(pckt_info->packet_data.nak_packet.last_sn), sizeof(int));	
FIXME relativo a possibilidade de envio de nak pra um grupo de pacotes*/	
		break;
		

#ifdef REFRESH

    case REFRESH_PACKET_TYPE:

		(*message) = (BYTE*)malloc(common_header_size+sizeof(int));
		aux_message_pt = *message;
		getPCKInfo(aux_message_pt,common_header,common_header_size);
            	
        getPCKInfo(aux_message_pt, &(pckt_info->packet_data.refresh_packet.sn_of_last_msg_sent), sizeof(int));
    	
		break;    

#endif

	case LEAVE_GROUP_PACKET_TYPE:
		
		/* do nothing. The packet is already built */
		
		break;	
        
	case JOIN_ACCEPT_PACKET_TYPE:    
    
		(*message) = (BYTE*)malloc(common_header_size+sizeof(int));
		aux_message_pt = *message;
		getPCKInfo(aux_message_pt,common_header,common_header_size);
        
        getPCKInfo(aux_message_pt, &(pckt_info->packet_data.join_accept_packet.port), sizeof(int));

        break;
    
    case JOIN_REQUEST_PACKET_TYPE:  
        (*message) = (BYTE*)malloc(common_header_size);
        aux_message_pt = *message;

		getPCKInfo(aux_message_pt,common_header,common_header_size);
    
        break;
        
    default:
    
        fprintf(stderr,"Unknown packet type: %d\n",pckt_info->type);
	}
	
	packet_size++;
	
	
	aux_message_pt =  *message + sizeof(char) + sizeof(MEMBER_ID);
	memcpy(aux_message_pt, &packet_size,sizeof(int));
	
	*msgsize = packet_size;
    
#ifdef DEBUG_PCKT
    {
        int cont;
        fprintf(stderr,"Mounted message: \n");
        for (cont = 0; cont < packet_size; cont++)
        {
            fprintf(stderr,"%d [%c] ", ((signed char*)(*message))[cont], (*message)[cont]);
        }	
    }
#endif 
    

	free (common_header);
	
}
#endif
