/* -*- mode: c; c-file-style: "gnu" -*-
 * dirindex.c -- directory indexer
 * Copyright (C) 2002, 2003, 2004 Gergely Nagy <algernon@bonehunter.rulez.org>
 *
 * This file is part of Thy.
 *
 * Thy is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 dated June, 1991.
 *
 * Thy is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

/** @file dirindex.c
 * Directory indexer.
 *
 * This file implements the directory indexer, a very simple one. No
 * icons or anything, just two plain lists: first, the directories,
 * then the files.
 *
 * Adding icons would be quite trivial, but that would result in lots
 * of mime-type lookups, which would be rather slow. If one wants
 * icons, he can pre-generate a directory listing himself.
 */

#include <ctype.h>
#include <dirent.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <time.h>
#include <unistd.h>

#include "compat/compat.h"

#include "dirindex.h"
#include "fabs.h"
#include "misc.h"
#include "thy.h"

/** @internal Converts a time_t to a string.
 * Converts a time_t to a string, and returns it in a format suitable
 * for displaying in a directory index (DAY-SHORT_MONTH_NAME-YEAR
 * HH:MM).
 *
 * @return A static buffer with the converted time. It must not be
 * freed!
 */
static const char *
_dirindex_file_date(time_t when)
{
  struct tm *stm;
  static char buffer[128];

  stm = gmtime(&when);
  strftime (buffer, sizeof (buffer), "%d-%b-%Y %H:%M", stm);
  return buffer;
}

/** @internal URL-encode a given string.
 * Encodes FROM in a way to be valid as an URL.
 *
 * @return The URL-encoded string on a newly allocated area, which
 * must be freed later.
 */
static char *
_dirindex_urlencode (const char *from)
{
  size_t tolen, tosize = strlen (from) * 3 + 2;
  char *to = (char *)bhc_malloc (tosize);
  int i = 0;

  for (tolen = 0; from[i] != '\0' && tolen + 4 < tosize; ++i)
    {
      if (isalnum (from[i]) || strchr ("/_.-~", from[i]))
	  to[tolen++] = from[i];
      else
	{
	  sprintf (&to[tolen], "%%%02x", (int) from[i] & 0xff);
	  tolen += 3;
	}
    }
  to[tolen] = '\0';
  return to;
}

/** @internal Encode a string to be HTML-safe.
 * HTML-encode SRC. That is, replace <, > and & with their HTML-entity
 * counterparts.
 *
 * @returns A newly allocated, HTML-safe string.
 */
static char *
_dirindex_htmlencode (const char *src)
{
    char *buf, *dest;
    unsigned char c;

    dest = (char *)bhc_malloc (strlen (src) * 5 + 1);

    if (!dest)
      return NULL;

    buf = dest;
    while ((c = *src++))
      {
	switch (c)
	  {
	  case '>':
	    *dest++ = '&';
	    *dest++ = 'g';
	    *dest++ = 't';
	    *dest++ = ';';
	    break;
	  case '<':
	    *dest++ = '&';
	    *dest++ = 'l';
	    *dest++ = 't';
	    *dest++ = ';';
	    break;
	  case '&':
	    *dest++ = '&';
	    *dest++ = 'a';
	    *dest++ = 'm';
	    *dest++ = 'p';
	    *dest++ = ';';
	    break;
	  default:
	    *dest++ = c;
	  }
      }
    *dest = '\0';
    return buf;
}

/** Generate a directory index.
 * Generates a directory index of a given path, for a given URL. No
 * icons, no fancy stuff, but hey! :)
 *
 * @param path is the path to index.
 * @param url is the URL we do the indexing for.
 *
 * @returns The directory index, or NULL on error.
 */
char *
directory_index (const char *path, const char *url)
{
  struct dirent **namelist;
  struct stat st;
  char *html, *h_files = NULL, *h_dirs = NULL, *cwd, *hurl;
  const char *t = rfc822_date (time (NULL));
  const char *servername = thy_servername (NULL);
  int n, m = 0;
  int d_dirs = 1, d_files = 0;

  cwd = bhc_getcwd ();
  if (chdir (path))
    {
      free (cwd);
      return NULL;
    }

  h_files = bhc_strdup ("  <tr><td colspan=3><h3>Files</h3></td></tr>\n");
  if (!strcmp ("/", url))
    {
      h_dirs = bhc_strdup ("  <tr><td colspan=3><h3>Directories</h3></td>"
			   "</tr>\n");
      d_dirs = 0;
    }
  else
    h_dirs =
      bhc_strdup ("  <tr><td colspan=3><h3>Directories</h3></td></tr>\n"
		  "  <tr><td colspan=3><tt><a href=\"../\">Parent "
		  "directory</a></tt></td></tr>\n"
		  "  <tr><td colspan=3>&nbsp;</td></tr>\n");

  n = scandir (".", &namelist, 0, alphasort);
  while (m < n)
    {
      char *canon = fabs_realpath (namelist[m]->d_name);

      if (canon == NULL)
	asprintf (&canon, "%s/%s", path, namelist[m]->d_name);

      if (fabs_stat (canon, &st) ||
	  namelist[m]->d_name[0] == '.')
	{
	  free (canon);
	  free (namelist[m]);
	  m++;
	  continue;
	}
      free (canon);

      if (S_ISDIR (st.st_mode))
	{
	  char *tmp, *ue, *he;
	  size_t len, tmplen;

	  d_dirs = 1;
	  ue = _dirindex_urlencode (namelist[m]->d_name);
	  he = _dirindex_htmlencode (namelist[m]->d_name);
	  asprintf (&tmp,
		    "  <tr><td colspan=3><tt><a href=\"%s/\">%s</a>"
		    "</tt></td></tr>\n", ue, he);
	  free (ue);
	  free (he);

	  len = strlen (h_dirs);
	  tmplen = strlen (tmp);
	  h_dirs = (char *)bhc_realloc (h_dirs, len + tmplen + 2);
	  memcpy (h_dirs + len, tmp, tmplen);
	  h_dirs [len + tmplen] = 0;
	  free (tmp);
	}

      if (S_ISREG (st.st_mode))
	{
	  char *tmp, *ue, *he;
	  size_t len, tmplen;

	  d_files = 1;
	  ue = _dirindex_urlencode (namelist[m]->d_name);
	  he = _dirindex_htmlencode (namelist[m]->d_name);
	  asprintf (&tmp,
		    "  <tr><td width=\"40%%\"><tt><a href=\"%s\">%s</a>"
		    "</tt></td>\n"
		    "      <td align=\"right\"><tt>%s</tt></td>\n"
		    "      <td align=\"right\"><tt>" SIZET_FORMAT "kB"
		    "</tt></td></tr>\n",
		    ue, he, _dirindex_file_date(st.st_mtime),
		    (size_t)(st.st_size/1024));
	  free (ue);
	  free (he);

	  len = strlen (h_files);
	  tmplen = strlen (tmp);
	  h_files = (char *)bhc_realloc (h_files, len + tmplen + 2);
	  memcpy (h_files + len, tmp, tmplen);
	  h_files [len + tmplen] = 0;
	  free (tmp);
	}
      free (namelist[m]);
      m++;
    }
  chdir (cwd);

  hurl = _dirindex_htmlencode (url);
  asprintf (&html,
	    "<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01 "
	    "Transitional//EN\">"
	    "\n"
	    "<html>\n<head>\n <title>Directory listing of %s</title>\n"
	    " <meta name=\"generator\" content=\"%s\">\n"
	    "</head>\n<body bgcolor=\"#ffffff\" text=\"#000000\">\n"
	    "<h2>Index of %s</h2>\n<table width=\"100%%\">\n"
	    "%s"
	    "  <tr><td colspan=3>&nbsp;</td></tr>\n"
	    "%s"
	    "</table>\n<hr noshade>\nIndex generated %s by %s\n"
	    "</body>\n</html>\n",
	    hurl, thy_servername (NULL), hurl, (d_dirs) ? h_dirs : "",
	    (d_files) ? h_files : "",  t, servername);

  free (hurl);
  free (h_files);
  free (h_dirs);
  free (namelist);
  free (cwd);

  return html;
}
