/* -*- mode: c; c-file-style: "gnu" -*-
 * types.h -- Type declarations
 * Copyright (C) 2002, 2003, 2004 Gergely Nagy <algernon@bonehunter.rulez.org>
 *
 * This file is part of Thy.
 *
 * Thy is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 dated June, 1991.
 *
 * Thy is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

/** @file types.h
 * Type declarations.
 */

#ifndef _THY_TYPES_H
#define _THY_TYPES_H 1 /**< types.h multi-inclusion guard. */

#include "options.h"
#include "system.h"

#include "bh-libs/list.h"

#include <stddef.h>
#ifdef HAVE_STDINT_H
#include <stdint.h>
#endif
#include <sys/types.h>
#include <time.h>

#if THY_OPTION_TLS
#include <gnutls/gnutls.h>
#endif

/** Special boolean type.
 * We have three states - unset, true and false. Unset is used by
 * config_get_mapped(), so it can weed options together nicely.
 */
typedef enum
{
  THY_BOOL_UNSET, /**< Unset. */
  THY_BOOL_TRUE, /**< True. */
  THY_BOOL_FALSE /**< False. */
} thy_bool_t;

/** Result of parsing.
 * HTTP request parsing can end only three ways: success, and two kind
 * of failures: premature end of headers, and unknown error. We don't
 * do much to go after unknown errors, there's no point in that.
 *
 * @see http_request_parse().
 */
typedef enum
{
  HTTP_PARSE_OK, /**< Successful parsing. */
  HTTP_PARSE_PREMATURE, /**< Premature end of headers. */
  HTTP_PARSE_UNKNOWN /**< Unknown error. */
} parse_result_t;

/** HTTP request method.
 * This enumeration contains the HTTP request methods Thy supports
 * (plus the unknown method, which might be handled by an external
 * program).
 *
 * @see session_handle() and http_request_parse().
 */
typedef enum
{
  HTTP_METHOD_GET, /**< HTTP GET method used. */
  HTTP_METHOD_HEAD, /**< HTTP HEAD method used. */
  HTTP_METHOD_POST, /**< HTTP POST method used. */
  HTTP_METHOD_OPTIONS, /**< HTTP OPTIONS method used. */
  HTTP_METHOD_TRACE, /**< HTTP TRACE method used. */
  HTTP_METHOD_UNKNOWN /**< Unknown HTTP method. */
} http_method_t;

/** HTTP status codes.
 * @see #session_code_map and the HTTP/1.1 RFC (rfc2616).
 */
typedef enum
{
  HTTP_STATUS_100 = 0,
  HTTP_STATUS_101,
  HTTP_STATUS_200,
  HTTP_STATUS_206,
  HTTP_STATUS_301,
  HTTP_STATUS_302,
  HTTP_STATUS_304,
  HTTP_STATUS_400,
  HTTP_STATUS_401,
  HTTP_STATUS_403,
  HTTP_STATUS_404,
  HTTP_STATUS_405,
  HTTP_STATUS_408,
  HTTP_STATUS_411,
  HTTP_STATUS_412,
  HTTP_STATUS_413,
  HTTP_STATUS_416,
  HTTP_STATUS_417,
  HTTP_STATUS_500,
  HTTP_STATUS_501,
  HTTP_STATUS_503,
  HTTP_STATUS_505,
  HTTP_STATUS_UNKNOWN
} http_status_t;

/** Content-encoding type.
 * @see _http_parse_line(), _session_out(), _session_putfile and
 * _session_canonicalize_gzip()
 */
typedef enum
{
  CONTENT_ENCODING_NONE, /**< No encoding at all. */
  CONTENT_ENCODING_GZIP, /**< gzip encoding. */
  CONTENT_ENCODING_DEFLATE /**< deflate encoding. */
} encoding_t;

/** Content-encoding method.
 * @see thy_send(), _session_out(), _session_putfile() and
 * _session_canonicalize_gzip().
 */
typedef enum
{
  ENCODING_TYPE_UNSET, /**< Encoding type not specified.
			This is used by config_get_mapped(). */
  ENCODING_TYPE_NONE, /**< No encoding at all. */
  ENCODING_TYPE_STATIC, /**< Use precompressed files if possible. */
  ENCODING_TYPE_DYNAMIC /**< Dynamic, on the fly encoding. Not
			   supported yet */
} enctype_t;

/** Type of an Upgrade request.
 * This type enumerates the supported Upgrade requests.
 */
typedef enum
{
  THY_UPGRADE_NONE, /**< No upgrade requested. */
  THY_UPGRADE_HTTP11, /**< Upgrade to HTTP/1.1 is requested. */
  THY_UPGRADE_TLS10, /**< Upgrade to TLS/1.0 is requested. */
  THY_UPGRADE_TLS10_POST /**< Session already upgraded to TLS/1.0. */
} thy_upgrade_t;

/** Type of a HTTP request.
 * This largish structure holds information about the HTTP request, in
 * a parsed, easy to access form.
 *
 * @see http_request_parse().
 */
typedef struct
{
  http_method_t method; /**< The request method. */
  char *method_str; /**< The request method, as string. */
  char *url; /**< The requested URL. */
  char *host; /**< The requested host (optional in HTTP/1.\ 0). */
  char *ua; /**< User-Agent (optional). */
  char *referer; /**< Referer (optional). */
  struct tm modified_since; /**< Parsed value of the If-Modified-Since
			       header (optional). */
  struct tm unmodified_since; /**< Parsed value of the
				 If-Unmodified-Since header (optional). */
  int http_major; /**< Major HTTP version. Only "1" is supported. */
  int http_minor; /**< Minor HTTP version. Only "0" and "1" are
		     supported. */
  off_t range_start; /**< Start of the range specified in the Range:
			header. */
  off_t range_end; /**< End of the range specified in the Range:
		      header. */
  int keepalive; /**< Boolean variable controlling keep-alive. */
  char *content_type; /**< Content-Type of a POST request. */
  size_t content_length; /**< Length of a POST request */
  char *query_string; /**< Query string passed to a CGI script. */
  char *path_info; /**< Extra path information passed to a CGI script. */
  char **cgienv; /**< Array of environment variables for a CGI
		    script. */
  size_t cgienvalloc; /**< Entries allocated in the CGI environment
			 array. */
  size_t cgienvlen; /**< Entries used in the CGI environment array. */
  char *raw; /**< The raw HTTP request. Used for HTTP TRACE. */
  char *auth_token; /**< Authentication token. */
  char *auth_realm; /**< Authentication realm. */
  char *auth_file; /**< File to get authentication info from. */
  char *if_range; /**< Value of the If-Range header, if any.*/
  encoding_t encoding; /**< Requested encoding type. */
  bhl_list_t *if_match; /**< List of If-Match header values. */
  bhl_list_t *if_none_match; /**< List of If-None-Match header
				values. */
  char *file; /**< File to be served. */
  char *resolved; /**< The resolved URL. */
  char *expect; /**< Value of the Expect: field, if any. */
  thy_upgrade_t upgrade; /**< Parsed value of the Upgrade: field, if
			    any. */
} request_t;

/** Session state type.
 * @note This is meant to be changed via session_state_change()
 * <b>ONLY</b>!
 */
typedef enum
{
  SESSION_STATE_HANDSHAKE, /**< Session is in TLS handshake state. */
  SESSION_STATE_INPUT_REQUEST, /**< Session is waiting for input. */
  SESSION_STATE_OUTPUT_HEAD, /**< Session is sending headers. */
  SESSION_STATE_OUTPUT_BODY, /**< Session is sending the body. */
  SESSION_STATE_PROCESS, /**< Session is being processed. */
  SESSION_STATE_CGI_HEADER_INPUT, /**< Session is waiting for CGI
				     headers. */
  SESSION_STATE_CGI_HEADER_OUTPUT, /**< Session is sending CGI headers
				      to the client. */
  SESSION_STATE_CGI_BODY_INPUT, /**< Session is waiting for CGI
				   body. */
  SESSION_STATE_CGI_BODY_OUTPUT, /**< Session is sending CGI body to
				    the client. */
  SESSION_STATE_POST_INPUT, /**< Session is reading POST body. */
  SESSION_STATE_POST_OUTPUT, /**< Session is sending POST body. */
  SESSION_STATE_OUTPUT_100, /**< Session is sending a 100 Continue
			       response. */
  SESSION_STATE_OUTPUT_101, /** Session is sending a 101 Switching
				Protocols response. */
  SESSION_STATE_INPUT_DISCARD /**< Session is reading & discarding
				 input data. */
} session_state_t;

/** Output method used for the session.
 * @note These are only meant to be set by session_finalise() and the
 * throttling methods only.
 */
typedef enum
{
  SESSION_OUTPUT_METHOD_BUFFER, /**< Session is sent from a buffer. */
  SESSION_OUTPUT_METHOD_SENDFILE, /**< Session is pushed from a file,
				     with sendfile(). */
  SESSION_OUTPUT_METHOD_SIMPLEFILE, /**< Session is pushed from a file
				       with read()/write(). */
  SESSION_OUTPUT_METHOD_NONE /**< Session should not reach output
				(HEAD). */
} session_output_method_t;

/** TLS Certificate type.
 * @see _thy_tls_session_create().
 */
typedef enum
{
  THY_TLS_CERT_TYPE_X509, /**< Use an X.509 certificate. */
  THY_TLS_CERT_TYPE_OPENPGP /**< Use an OpenPGP certificate. */
} thy_tls_cert_type_t;

/** Prototype for the session throttling functions.
 */
typedef int (*session_throttle_f) (void *session);

/** Opaque declaration of thy_zlib_session_t.
 * @see _thy_zlib_session_t.
 */
typedef struct _thy_zlib_session_t thy_zlib_session_t;

/** Session descriptor type.
 * Each and every session in Thy has an associated structure of this
 * type. It holds all the information needed to handle the request, to
 * keep track of the state, progress and other things.
 */
typedef struct
{
  request_t *request; /**< The HTTP request, in parsed form. */
  parse_result_t result; /**< Result of the request parsing. */
  session_state_t state; /**< State of the session. */
  int request_count; /**< Number of kept-alive requests. */

  char *origin; /**< Originating IP. */
  char *root; /**< Root directory of the request. Used for virtual
		 hosting. */
  int filefd; /**< File descriptor of the input file, if any. */
  int port; /**< The port this connection was accepted on. */
  int ssl; /**< Mode of the session. If in SSL mode, 1, otherwise
	      zero. */
  time_t conntime; /**< Time of the connect. */
  time_t acttime; /**< Time of the last action. */
  char *mime; /**< Mime type of the request. */
  char *absuri; /**< The absolute URI. */
  char *etag; /**< ETag of the response, if any. */

  /** Network I/O filedescriptors.
   */
  struct
  {
    int in;
    int out;
  } io;

  /** Response header structure.
   */
  struct
  {
    http_status_t code;
    char *fields;
    size_t len;
    off_t offset;
  } header;

  /** Response body structure.
   */
  struct
  {
    char *buffer;
    size_t len;
    size_t size;
    size_t content_length;
    size_t content_size;
    off_t offset;
    char *fn;
    session_output_method_t method;

    /** Body header structure.
     */
    struct
    {
      char *buff;
      size_t size;
      off_t offset;
    } header;

    /** Body footer structure.
     */
    struct
    {
      char *buff;
      size_t size;
      off_t offset;
    } footer;
  } body;

  /** CGI-specific structure.
   */
  struct
  {
    pid_t child;
    char **argv;
    char *handler;
    int status;
    int running;
    int eof;
    size_t content_length;

    /** POST buffer.
     */
    struct
    {
      size_t size, length, len;
      off_t offset;
      char *buffer;
    } post;

    /** CGI I/O pipes.
     */
    struct
    {
      int in[2];
      int out[2];
    } pipes;
  } cgi;

  /** Transfer-Encoding: chunked support struct.
   */
  struct
  {
    int enabled;

    /** Pre-chunk data tructure.
     */
    struct
    {
      char *buff;
      size_t size;
      off_t offset;
    } prechunk;

    /** Post-chunk data structure.
     */
    struct
    {
      char *buff;
      size_t size;
      off_t offset;
    } postchunk;
  } chunked;

  /** Vary header support struct.
   */
  struct
  {
    int accept_encoding;
  } vary;

  session_throttle_f throttle; /**< Throttling function. */
#if THY_OPTION_TLS
  gnutls_session tls_session; /**< GnuTLS session object. */
  char *cert_verified; /**< Result of the verification of the client's
			  certificate. */
#endif

#if THY_OPTION_ZLIB
  /** Variables for on-the-fly compression.
   */
  struct
  {
    enctype_t encoding;
    thy_zlib_session_t *gz_session;
  } compress;
#endif
} session_t;

/** HTTP code to message map entry.
 */
typedef struct
{
  int code; /**< HTTP status code. */
  char *message; /**< HTTP status message. */
} session_code_map_entry_t;

/** HTTP code to message mapping table.
 */
extern session_code_map_entry_t session_code_map[];

/** Enumerated list of supported waiters.
 */
typedef enum
{
#if HAVE_EPOLL_CREATE
  THY_NQUEUE_WAITER_EPOLL, /**< epoll() */
#endif
#if HAVE_KQUEUE
  THY_NQUEUE_WAITER_KQUEUE, /**< kqueue() */
#endif
#if defined(HAVE_SYS_DEVPOLL_H) || defined(HAVE_LINUX_DEVPOLL_H)
  THY_NQUEUE_WAITER_DEVPOLL, /**< /dev/poll */
#endif
#if HAVE_SYS_POLL_H
  THY_NQUEUE_WAITER_POLL, /**< poll() */
#endif
#if HAVE_SYS_SELECT_H
  THY_NQUEUE_WAITER_SELECT, /**< select() */
#endif
  THY_NQUEUE_WAITER_NONE /**< Nothing, aka select automatically. */
} thy_nqueue_waiter_t;

/** Generic field - value pair.
 */
typedef struct
{
  char *field; /**< The name of the field. */
  char *value; /**< Value of the field. */
} pair_t;

/** Base used for counting expiration times.
 */
typedef enum
{
  THY_EXPIRY_BASE_UNKNOWN, /* Unknown expiry base. */
  THY_EXPIRY_BASE_NOW, /* Expiration time based on access time. */
  THY_EXPIRY_BASE_MODIFICATION /* Expiration time based on last
				  modification time. */
} thy_expiry_base_t;

/** Mappable configuration structure.
 */
typedef struct
{
  char *webroot; /**< Our root directory, where web documents are
		    stored. */
  char *default_type; /**< Default content-type. */
  char *userdir; /**< User web directory. Used for /~username/
		    requests. */

  char *http_status[HTTP_STATUS_UNKNOWN]; /**< HTTP status code to
					     error document mapping
					     table. */

  /** Options settable via -o.
   */
  struct
  {
    thy_bool_t vhosting;
    thy_bool_t auth;
    thy_bool_t userdir;
    thy_bool_t cgi;
    thy_bool_t usercgi;
    thy_bool_t casemime;
    thy_bool_t followall;
    thy_bool_t vary;
    thy_bool_t dirindex;
    thy_bool_t lazycgi;
  } options;

  /** Options settable via -X.
   */
  struct
  {
    thy_bool_t no_cache;
    thy_bool_t no_store;
    thy_bool_t no_transform;
    thy_bool_t must_revalidate;
    thy_bool_t do_max_age;
    thy_expiry_base_t expiry_base;
    long int max_age;
  } cache;

  /** Options settable via -g.
   */
  struct
  {
    enctype_t type;
    int level;
  } encoding;

  /** ETag options
   */
  struct
  {
    thy_bool_t dirtag;
    thy_bool_t etag;
  } etag;

  bhl_list_t *indexes; /**< List of files to consider directory
			  indexes. */
  bhl_list_t *cgiexts; /**< List of extensions to consider as CGI
			  extensions. */
  bhl_list_t *handlers; /**< List of special handlers. */
  bhl_list_t *mime_types; /**< List of user-specified MIME-type
			     mappings. */
  bhl_list_t *methods; /**< Additional request methods, handled by
			  CGIs. */
  bhl_list_t *headers; /**< Additional HTTP response headers. */
  bhl_list_t *env; /**< Additional environment variables for CGIs. */
  bhl_list_t *cgidirs; /**< Special CGI directories. */
  bhl_list_t *aliases; /**< Aliases. */
} thy_mappable_config_t;

/** Thy configuration structure.
 */
typedef struct
{
  thy_mappable_config_t *defconf; /**< Default mapped configuration. */

  bhl_list_t *ips; /**< IP address to listen on. */
  int type; /**< Type of the listening socket (IPv4 or IPv6 or
	       both). */
  time_t timeout; /**< Maximum number of seconds for which a session
		     might remain inactive before being
		     disconnected. */
  time_t ktimeout; /**< Timeout for keep-alive requests. */
  int keep_max; /**< Maximum kept-alive requests in a session. */
  long stimeout; /**< Timeout for the select() system call. */
  char *pidfile; /**< Path of the file to log our pid into. */
  uid_t uid; /**< User ID to run as. */
  size_t buffer_size; /**< Size of the data buffer. */

  /** Options settable via -o.
   */
  struct
  {
    int buffers;
    thy_bool_t daemon;
    thy_bool_t chroot;
    thy_bool_t worker;
    thy_bool_t expect;
    thy_bool_t hardlimit;
    unsigned int stats;
    thy_nqueue_waiter_t waiter;
  } options;

  /** Various limits.
   */
  struct
  {
    size_t post_buffer;
    size_t header;
    size_t cgis;
    size_t mmap;
  } limits;

  /** Options settable via -s.
   */
  struct
  {
    thy_tls_cert_type_t cert_type;
    int verify_level;

    /** X.509 settings.
     */
    struct
    {
      bhl_list_t *certfiles;
      bhl_list_t *keyfiles;
      bhl_list_t *cafiles;
    } x509;

    /** OpenPGP settings.
     */
    struct
    {
      char *keyfile;
      char *certfile;
      char *keyring;
      char *trustdb;
    } openpgp;

    /** SRP settings.
     */
    struct
    {
      char *passwd;
      char *conf;
    } srp;
  } ssl;

  /** Authenticator options.
   */
  struct
  {
    /** Arguments passed to the Authoriser.
     */
    struct
    {
      char **argv;
      int argc;
    } args;
    char *path;
    uid_t uid;
    char *file;
  } auth;

  /** Worker options.
   */
  struct
  {
    /** Arguments passed to the Worker.
     */
    struct
    {
      char **argv;
      int argc;
    } args;
    char *path;
    uid_t uid;
  } worker;

  /** LRU options.
   */
  struct
  {
    size_t fcache;
    size_t ucache;
    time_t expire;
  } lru;

  bhl_list_t *maps; /**< List of configuration maps. */
} config_t;

/** The way SSL was set (or unset) on a listener.
 */
typedef enum
{
  LISTENER_SSL_AUTO, /**< SSL mode was not explicitly set, nor
			unset. */
  LISTENER_SSL_SET, /**< SSL mode was explicitly set. */
  LISTENER_SSL_UNSET /**< SSL mode was explicitly unset. */
} thy_listener_ssl_t;

/** Type of a listener socket.
 */
typedef struct
{
  thy_listener_ssl_t ssl; /**< Signals TLS mode. */
  int fd; /**< The FD of the socket. */
  struct sockaddr_storage *sock; /**< Socket info storage. Only used
				    by the configuration parser. */
} thy_listener_t;

/** Available events on a file descriptor in the network event queue.
 */
typedef enum
{
  THY_NQ_EVENT_NONE, /**< No event. */
  THY_NQ_EVENT_INPUT, /**< Input is available. */
  THY_NQ_EVENT_OUTPUT, /**< FD is ready for output. */
  THY_NQ_EVENT_TRIGGER, /**< FD is always read. */
  THY_NQ_EVENT_ERROR /**< An error occurred. */
} thy_nq_event_t;

/** Configuration map type.
 */
typedef struct
{
  char *url; /**< Regex where this config map applies. */
  void *preg; /**< Compiled regex. */
  thy_mappable_config_t *config; /**< Map configuration. */
} thy_config_map_t;

/** Flag to track if TLS is enabled or not.
 * This is needed to properly support RFC2817 - ie, not support it
 * when we have no certificate even though TLS is compiled in.
 */
thy_bool_t thy_tls_enabled;

#endif /* !_THY_TYPES_H */
