# $Id: chats.tcl,v 1.153 2006/05/28 14:54:24 aleksey Exp $

package require textutil

option add *Chat.theyforeground        red3   widgetDefault
option add *Chat.meforeground          blue   widgetDefault
option add *Chat.serverlabelforeground green  widgetDefault
option add *Chat.serverforeground      violet widgetDefault
option add *Chat.infoforeground        blue   widgetDefault
option add *Chat.errforeground         red    widgetDefault
option add *urlforeground              blue   widgetDefault
option add *urlactiveforeground        red    widgetDefault
option add *Chat.inputheight           3      widgetDefault


namespace eval chat {
    array set opened {}
    set urlid 0
    set enrichid 0
    custom::defgroup Chat [::msgcat::mc "Chat options."] -group Tkabber
    custom::defvar options(smart_scroll) 0 \
	[::msgcat::mc "Enable chat window autoscroll only when last message is shown."] \
	-type boolean -group Chat
    custom::defvar options(stop_scroll) 0 \
	[::msgcat::mc "Stop chat window autoscroll."] \
	-type boolean -group Chat
    custom::defvar options(emphasize) 1 \
	[::msgcat::mc "Enable messages emphasize."] \
	-type boolean -group Chat \
	-command chat::switch_emphasize
    custom::defvar options(display_status_description) 1 \
	[::msgcat::mc "Display description of user status in chat windows."] \
	-type boolean -group Chat
    custom::defvar options(default_message_type) chat \
	[::msgcat::mc "Default message type (if not specified explicitly)."] \
	-type radio -group Chat \
	-values [list normal [::msgcat::mc "Normal"] \
		      chat [string trim [::msgcat::mc "Chat "]]]

    custom::defvar open_chat_list {} [::msgcat::mc "List of users for chat."] \
	    -group Hidden

    set url_regexp {^
		([^\w\d]*)
		(

		 (?:
			(?: ftp|https?)://[-\w]+(\.\w[-\w]*)+
		   |
			(?: [a-z0-9][-a-z0-9]* \. )+
			(?: com
			  | edu
			  | biz
			  | gov
			  | in(?:t|fo)
			  | mil
			  | net
			  | org
			  | name
			  | aero
			  | arpa
			  | coop
			  | museum
			  | pro
			  | travel
			  | [a-z][a-z]
			)
		)
		(?: : \d+ )?

		(?:
			/
			[^.,?!:;"'<>()\[\]{}\s\x7F-\xFF]*
			(?:
				[.,?!:;]+ [^.,?!:;"'<>()\[\]{}\s\x7F-\xFF]+

			)*

		)?
	     )
	     ([^\w\d]*)$
    }

}

set chat_width 50
set chat_height 18

plugins::load [file join plugins chat]

proc chat::open_chat_dialog {} {
    variable open_chat_jid
    variable open_chat_list
    variable open_chat_connid

    if {[lempty [jlib::connections]]} return

    set gw .open_chat
    catch { destroy $gw }

    set connid [lindex [jlib::connections] 0]
    set open_chat_connid [jlib::connection_jid $connid]

    Dialog $gw -title [::msgcat::mc "Open chat"] -separator 1 -anchor e \
	    -default 0 -cancel 1

    set gf [$gw getframe]
    grid columnconfigure $gf 1 -weight 1

    set open_chat_jid ""

    label $gf.ljid -text [::msgcat::mc "JID:"]
    ecursor_entry [ComboBox $gf.jid -textvariable [namespace current]::open_chat_jid \
	    -values $open_chat_list -width 35].e
    
    grid $gf.ljid -row 0 -column 0 -sticky e
    grid $gf.jid  -row 0 -column 1 -sticky ew

    if {[llength [jlib::connections]] > 1} {
	set connections {}
	foreach c [jlib::connections] {
	    lappend connections [jlib::connection_jid $c]
	}
	label $gf.lconnection -text [::msgcat::mc "Connection:"]
	ComboBox $gf.connection -textvariable [namespace current]::open_chat_connid \
				-values $connections

	grid $gf.lconnection -row 1 -column 0 -sticky e
	grid $gf.connection  -row 1 -column 1 -sticky ew
    }

    $gw add -text [::msgcat::mc "Chat"] -command "[namespace current]::open_chat $gw"
    $gw add -text [::msgcat::mc "Cancel"] -command "destroy $gw"

    $gw draw $gf.jid
}

proc chat::open_chat {gw} {
    variable open_chat_jid
    variable open_chat_list
    variable open_chat_connid

    destroy $gw

    foreach c [jlib::connections] {
	if {[jlib::connection_jid $c] == $open_chat_connid} {
	    set connid $c
	}
    }
    if {![info exists connid]} {
	set connid [jlib::route $open_chat_jid]
    }

    set open_chat_list [update_combo_list $open_chat_list $open_chat_jid 20]
    set custom::saved([namespace current]::open_chat_list) $open_chat_list
    custom::store
    open_to_user $connid $open_chat_jid
}

proc chat::get_nick {connid jid type} {
    variable chats

    set nick $jid
    switch -- $type {
	chat {
	    set group [node_and_server_from_jid $jid]
	    set chatid1 [chatid $connid $group]
	    if {[is_groupchat $chatid1]} {
		set nick [resource_from_jid $jid]
	    } else {
		set nick [roster::itemconfig $connid \
			      [roster::find_jid $connid $jid] -name]
		if {$nick == ""} {
		    set nick [node_from_jid $jid]
		}
	    }
	}
	groupchat {
	    set nick [resource_from_jid $jid]
	}
    }
    return $nick
}

proc chat::winid {chatid} {
    set connid [get_connid $chatid]
    set jid [get_jid $chatid]
    set tag [jid_to_tag $jid]
    return .chat_${connid}_$tag
}

proc chat::chatid {connid jid} {
    return [list $connid $jid]
}

proc chat::get_connid {chatid} {
    return [lindex $chatid 0]
}

proc chat::get_jid {chatid} {
    return [lindex $chatid 1]
}

proc client:message {connid from id type is_subject subject body err thread priority x} {
    debugmsg chat "MESSAGE: $connid; $from; $id $type; $is_subject; $subject; $body; $err;\
$thread; $priority; $x"

    if {![cequal [info commands ::ssj::encrypted:input] ""]} {
	set xs {}

	foreach xe $x {
	    jlib::wrapper:splitxml $xe tag vars isempty chdata children

	    if {![cequal [jlib::wrapper:getattr $vars xmlns] \
			 jabber:x:encrypted]} {
		lappend xs $xe
		continue
	    } elseif {[cequal $chdata ""]} {
# in case the sender didn't check the exit code from gpg...
	    } elseif {[catch { ssj::encrypted:input $connid $from $chdata } body]} {
		set body [format [::msgcat::mc ">>> Unable to decipher data: %s <<<"] $body]
	    } else {
		lappend xs $xe
	    }
	}

	set x $xs
    }

    chat::process_message \
	$connid $from $id $type $is_subject $subject $body $err $thread $priority $x
}

proc chat::process_message {connid from id type is_subject subject body err thread priority x} {
    variable options

    if {$type == ""} {
	set type $options(default_message_type)
    }

    set from [tolower_node_and_domain $from]

    hook::run process_message_hook \
	$connid $from $id $type $is_subject $subject $body $err $thread $priority $x
}

proc chat::process_message_fallback {connid from id type is_subject subject body err thread priority x} {
    global font
    variable opened
    variable chats

    set chatid [chatid $connid $from]
    hook::run normalize_chat_id_hook chatid $type


    switch -- $type {
	chat {
	    if {$thread != ""} {
		set chats(thread,$chatid) $thread
	    }
	    set chats(id,$chatid) $id
	}
	groupchat {
	    set chatid [chatid $connid [node_and_server_from_jid $from]]

	    if {![is_groupchat $chatid]} return

	    if {$is_subject} {
		set_subject $chatid $subject
		if {[cequal $body ""]} {
		    set nick [resource_from_jid $from]
		    if {[cequal $nick ""]} {
			set body [format [::msgcat::mc "Subject is set to: %s"] $subject]
		    } else {
			set body [format [::msgcat::mc "/me has set the subject to: %s"] $subject]
		    }
		}
	    }
	}
	error {
	    if {$is_subject} {
		set body "subject: "
		restore_subject $chatid
	    } else {
		set body ""
	    }
	    append body [error_to_string $err]
	}
	default {
	    debugmsg chat "MESSAGE: UNSUPPORTED message type '$type'"
	}
    }

    #chat::open_window $chatid $type

    chat::add_message $chatid $from $type $body $x
    if {[llength $x] > 0} {
        message::show_dialog $connid $from $type $subject $body $thread $priority $x 0
    }
}

hook::add process_message_hook \
    [namespace current]::chat::process_message_fallback 99


proc chat::window_titles {chatid} {
    variable chats

    set connid [get_connid $chatid]
    set jid [get_jid $chatid]
    set chatname [roster::itemconfig $connid \
		      [roster::find_jid $connid $jid] -name]
    if {$chatname != ""} {
	set tabtitlename $chatname
	set titlename $chatname
    } else {
	set titlename $jid
	if {[is_groupchat $chatid]} {
	    set tabtitlename [node_from_jid $jid]
	} else {
	    set tabtitlename [get_nick $connid $jid chat]
	}
	if {$tabtitlename == ""} {
	    set tabtitlename $titlename
	}
    }
    return [list $tabtitlename [format [::msgcat::mc "Chat with %s"] $titlename]]
}

proc chat::reconnect_groupchats {connid} {
    variable chats
    global grouproster
	
    if {![info exists chats(opened)]} return

    foreach chatid $chats(opened) {
	if {[cequal [get_connid $chatid] $connid]} {
	    if {[is_groupchat $chatid]} {
		if {[info exists ::muc::muc_password($chatid)]} {
		    set password $::muc::muc_password($chatid)
		} else {
		    set password ""
		}
		join_group [get_jid $chatid] \
		    -nick [get_our_groupchat_nick $chatid] \
		    -password $password \
		    -connection $connid
	    } else {
		set chats(status,$chatid) connected
	    }
	}
    }
}

hook::add connected_hook [namespace current]::chat::reconnect_groupchats 99

proc chat::disconnect_groupchats {connid} {
    variable chats

    if {![info exists chats(opened)]} return

    foreach chatid $chats(opened) {
	if {[cequal $connid {}] || [cequal [get_connid $chatid] $connid]} {
	    set chatw [chat_win $chatid]
	    if {[winfo exists $chatw] && \
		    $chats(status,$chatid) != "disconnected"} {
		set chats(status,$chatid) disconnected
		add_message $chatid [get_jid $chatid] error \
		    [::msgcat::mc "Disconnected"] {}
	    }
	}
    }
}

hook::add disconnected_hook [namespace current]::chat::disconnect_groupchats 99

proc chat::open_window {chatid type args} {
    global font font_bold font_italic font_bold_italic
    global chat_width chat_height
    variable opened
    variable chats
    variable chat_id
    variable options
    global grouproster
    global statusdesc

    set connid [get_connid $chatid]
    set jid [get_jid $chatid]
    set jid [tolower_node_and_domain $jid]
    set chatid [chatid $connid $jid]

    set cw [winid $chatid]

    set cleanroster 1
    foreach {key val} $args {
	switch -- $key {
	    -cleanroster { set cleanroster $val }
	}
    }

    if {[winfo exists $cw]} {
	if {$type == "groupchat" && \
		$chats(status,$chatid) == "disconnected" && \
		$cleanroster} {
	    set grouproster(users,$chatid) {}
	}

	if {!$::usetabbar && \
		[info exists ::raise_on_activity] && $::raise_on_activity} {
	    if {$type == "chat"} {
		wm deiconify $cw
	    }
	    raise $cw
	}
	return
    }

    hook::run open_chat_pre_hook $chatid $type

    set chats(type,$chatid) $type
    set chats(subject,$chatid) ""
    if {$type == "groupchat"} {
	set chats(status,$chatid) disconnected
    } else {
	set chats(status,$chatid) connected
    }
    set chats(exit_status,$chatid) ""
    debugmsg chat "OURJID: [our_jid $chatid]"
    lappend chats(opened) $chatid
    
    set opened($chatid) $cw
    set chat_id($cw) $chatid
    #debugmsg chat $cw
    #debugmsg chat [array names opened_chats]
    #debugmsg chat $opened_chats($chatid)

    lassign [chat::window_titles $chatid] chats(tabtitlename,$chatid) \
	chats(titlename,$chatid)

    add_win $cw -title $chats(titlename,$chatid) \
	-tabtitle $chats(tabtitlename,$chatid) \
	-class Chat -type $type \
	-raisecmd "focus [list $cw.input]
		   tab_set_updated [list $cw]
                   hook::run raise_chat_tab_hook [list $cw] [list $chatid]"

    frame $cw.status
    pack $cw.status -side top -fill x
    if {[cequal $type chat]} {
	set status [get_user_status $connid $jid]
	Label $cw.status.icon \
	    -image [ifacetk::roster::get_jid_icon $connid $jid] \
	    -helptext [get_user_status_desc $connid $jid]
	pack $cw.status.icon -side left
	if {$options(display_status_description)} {
	    Label $cw.status.desc -text "($statusdesc($status))" -font $font \
		-helptext [get_user_status_desc $connid $jid]
	    pack $cw.status.desc -side left
	}
    }
    if {[cequal $type chat]} {
	menubutton $cw.status.mb -text $jid -font $font \
	    -menu $cw.status.mb.menu
	create_user_menu $cw.status.mb.menu $chatid
	pack $cw.status.mb -side left
    } else {
        menubutton $cw.status.mb -text [::msgcat::mc "Subject:"] \
	    -direction below \
	    -menu $cw.status.mb.menu -relief flat
	create_conference_menu $cw.status.mb.menu $chatid
	pack $cw.status.mb -side left

	entry $cw.status.subject -font $font \
	    -xscrollcommand [list [namespace current]::set_subject_tooltip $chatid]
	pack $cw.status.subject -side left -fill x -expand yes

	bind $cw.status.subject <Return> \
	    [list chat::change_subject [double% $chatid]]

	bind $cw.status.subject <Escape> \
	    [list chat::restore_subject [double% $chatid]]

	bind $cw.status.subject <Any-Enter> \
	    +[list eval balloon::set_text \
		   \[[namespace current]::set_subject_balloon [list [double% $chatid]]\] \
		   -width \[winfo width %W\]]

	bind $cw.status.subject <Any-Motion> \
	    [list eval balloon::on_mouse_move \
		  \[[namespace current]::set_subject_balloon [list [double% $chatid]]\] %X %Y]

	bind $cw.status.subject <Any-Leave> {+ balloon::destroy}
    }
    foreach tag [bind Menubutton] {
        if {[string first 1 $tag] >= 0} {
	    regsub -all 1 $tag 3 new
	    bind $cw.status.mb $new [bind Menubutton $tag]
	}
    }

    PanedWin $cw.pw0 -side right -pad 0 -width 8
    pack $cw.pw0 -fill both -expand yes


    set upw [$cw.pw0 add -weight 1 -minsize 0]
    set dow [$cw.pw0 add -weight 0 -minsize 0]

    set isw [ScrolledWindow $cw.isw -scrollbar vertical]
    pack $cw.isw -fill both -expand yes -side bottom -in $dow
    textUndoable $cw.input -width $chat_width \
	-height [option get $cw inputheight Chat] -font $font -wrap word
    $isw setwidget $cw.input
    [winfo parent $dow] configure -height [winfo reqheight $cw.input]
    set chats(inputwin,$chatid) $cw.input

    if {[cequal $type groupchat]} {
	PanedWin $cw.pw -side bottom -pad 2 -width 8
	pack $cw.pw -fill both -expand yes -in $upw

	set cf [$cw.pw add -weight 1 -minsize 0]

	set uw [$cw.pw add -weight 0 -minsize 0]
	set chats(userswin,$chatid) $uw.users

	set rosterwidth [option get . chatRosterWidth [winfo class .]]
	if {$rosterwidth == ""} {
	    set rosterwidth [winfo pixels . 3c]
	}

	ifacetk::roster::create $uw.users -width $rosterwidth \
	    -popup ifacetk::roster::groupchat_popup_menu \
	    -singleclick [list [namespace current]::user_singleclick $chatid] \
	    -doubleclick ::ifacetk::roster::jid_doubleclick
	pack $uw.users -fill both -side right -expand yes
	[winfo parent $uw] configure -width $rosterwidth

	set grouproster(users,$chatid) {}

	set pack_in $cf
    } else {
	set cf $cw
	set pack_in $upw
    }

    set chats(chatwin,$chatid) $cf.chat

    set csw [ScrolledWindow $cf.csw -scrollbar vertical -auto none]
    pack $csw -expand yes -fill both -side top -in $pack_in
    text $cf.chat -width $chat_width -height $chat_height -font $font -wrap word
    $csw setwidget $cf.chat

    $cf.chat tag configure they \
	-foreground [option get $cw theyforeground Chat]
    $cf.chat tag configure me \
	-foreground [option get $cw meforeground Chat]
    $cf.chat tag configure highlight \
	-foreground [option get $cw meforeground Chat]
    $cf.chat tag configure server_lab \
	-foreground [option get $cw serverlabelforeground Chat]
    $cf.chat tag configure server \
	-foreground [option get $cw serverforeground Chat]
    $cf.chat tag configure info \
	-foreground [option get $cw infoforeground Chat]
    $cf.chat tag configure err \
	-foreground [option get $cw errforeground Chat]

    $cf.chat tag configure bold -font $font_bold
    $cf.chat tag configure italic -font $font_italic
    $cf.chat tag configure bold_italic -font $font_bold_italic
    $cf.chat tag configure underlined -underline 1
    if {$options(emphasize)} {
	$cf.chat tag configure emphasized -elide 0
	$cf.chat tag configure nonemphasized -elide 1
    } else {
	$cf.chat tag configure emphasized -elide 1
	$cf.chat tag configure nonemphasized -elide 0
    }


    $cf.chat configure -state disabled
    focus $cw.input

    bind $cw.input <Shift-Key-Return> { }

    bind $cw.input <Key-Return> [double% "
	chat::send_message [list $cw] [list $chatid] [list $type]
	break"]

    bind $cw.input <Control-z> { %W undo }
    bind $cw.input <Control-Z> { %W redo }

    regsub -all %W [bind Text <Prior>] $cf.chat prior_binding
    regsub -all %W [bind Text <Next>] $cf.chat next_binding
    bind $cw.input <Meta-Prior> $prior_binding
    bind $cw.input <Meta-Next> $next_binding
    bind $cw.input <Alt-Prior> $prior_binding
    bind $cw.input <Alt-Next> $next_binding

    bind $cw <Destroy> [list chat::close_window [double% $chatid]]

    hook::run open_chat_post_hook $chatid $type
}

###############################################################################

proc chat::user_singleclick {chatid cjid} {
    lassign $cjid connid jid
    set nick [get_nick $connid $jid groupchat]
    hook::run groupchat_roster_user_singleclick_hook $chatid $nick
}

###############################################################################

proc chat::bind_window_click {chatid type} {
    set cw [chat::chat_win $chatid]
    bind $cw <ButtonPress-1><ButtonRelease-1> \
	[list hook::run chat_window_click_hook [double% $chatid] %W %x %y]
}

hook::add open_chat_post_hook [namespace current]::chat::bind_window_click

###############################################################################

proc chat::close_window {chatid} {
    variable opened
    variable chats
    variable chat_id

    if {![info exists opened($chatid)]} return

    unset opened($chatid)

    set connid [get_connid $chatid]
    set jid [get_jid $chatid]

    set cw [winid $chatid]
    unset chat_id($cw)

    set idx [lsearch -exact $chats(opened) $chatid]
    set chats(opened) [lreplace $chats(opened) $idx $idx]
    debugmsg chat $chats(opened)

    if {[is_groupchat $chatid]} {
	muc::leave_group $connid $jid [get_our_groupchat_nick $chatid] \
			 $chats(exit_status,$chatid)
	set chats(status,$chatid) disconnected
	client:presence $connid $jid unavailable "" {}
	foreach jid [get_jids_of_user $connid $jid] {
	    client:presence $connid $jid unavailable "" {}
	}
    }

    destroy $cw
    hook::run close_chat_post_hook $chatid
}

proc chat::opened {} {
    variable chats
    if {[info exists chats(opened)]} {
	return $chats(opened)
    } else {
	return {}
    }
}

proc chat::users_win {chatid} {
    variable chats
    return $chats(userswin,$chatid)
}

proc chat::chat_win {chatid} {
    variable chats
    return $chats(chatwin,$chatid)
}

proc chat::input_win {chatid} {
    variable chats
    return $chats(inputwin,$chatid)
}

proc chat::is_groupchat {chatid} {
    variable chats
    if {[info exists chats(type,$chatid)]} {
	return [cequal $chats(type,$chatid) groupchat]
    } else {
	return 0
    }
}

proc chat::create_user_menu {m chatid} {
    menu $m -tearoff 0

    set connid [get_connid $chatid]
    set jid [get_jid $chatid]

    hook::run chat_create_user_menu_hook $m $connid $jid

    return $m
}

proc chat::create_conference_menu {m chatid} {
    menu $m -tearoff 0

    set connid [get_connid $chatid]
    set jid [get_jid $chatid]

    hook::run chat_create_conference_menu_hook $m $connid $jid

    return $m
}

proc chat::add_separator {m connid jid} {
    $m add separator
}

hook::add chat_create_user_menu_hook chat::add_separator 40
hook::add chat_create_user_menu_hook chat::add_separator 42
hook::add chat_create_user_menu_hook chat::add_separator 50
hook::add chat_create_user_menu_hook chat::add_separator 70
hook::add chat_create_user_menu_hook chat::add_separator 85
hook::add chat_create_conference_menu_hook chat::add_separator 40
hook::add chat_create_conference_menu_hook chat::add_separator 42
hook::add chat_create_conference_menu_hook chat::add_separator 50

proc chat::add_chat_win_popup_menu {m chatwin X Y x y} {
    set tags [$chatwin tag names "@$x,$y"]
    set idx [lsearch -glob $tags url*]
    if {$idx >= 0} {
	$m add command -label [::msgcat::mc "Copy URL to clipboard"] \
	    -command [list [namespace current]::copy_url $chatwin \
			   [lindex $tags $idx]]
    }
}

hook::add chat_win_popup_menu_hook \
    [namespace current]::chat::add_chat_win_popup_menu 10

proc chat::copy_url {chatwin tag} {
    regsub -all %% [$chatwin tag bind $tag <ButtonPress-1><ButtonRelease-1>] % res
    # assuming "browseurl url"
    clipboard clear -displayof $chatwin
    clipboard append -displayof $chatwin [lrange $res 1 end]
}

proc chat::send_message {cw chatid type} {
    set iw $cw.input

    set connid [get_connid $chatid]

    if {[catch { set user [jlib::connection_user $connid] }]} {
	set user ""
    }

    set body [$iw get 1.0 "end -1 chars"]

    debugmsg chat "SEND_MESSAGE:\
 [list $chatid] [list $user] [list $body] [list $type]"

    set chatw [chat_win $chatid]
    $chatw mark set start_message "end -1 chars"
    $chatw mark gravity start_message left

    hook::run chat_send_message_hook $chatid $user $body $type

    $iw delete 1.0 end
}

proc chat::add_message {chatid from type body x} {
    variable chats
    variable opened
    variable options

    if {[info exists opened($chatid)]} {
	set chatw [chat_win $chatid]

	if {[lindex [$chatw yview] 1] == 1} {
	    set scroll 1
	} else {
	    set scroll 0
	}
    } else {
	set scroll 1
    }

    hook::run draw_message_hook $chatid $from $type $body $x

    if {[info exists opened($chatid)]} {
	set chatw [chat_win $chatid]

	if {![$chatw compare "end -1 chars linestart" == "end -1 chars"]} {
	    $chatw insert end "\n"
	}

	if {$body != "" && !$options(stop_scroll) && \
		(!$options(smart_scroll) || \
		     ($options(smart_scroll) && $scroll) || \
		     [chat::is_our_jid $chatid $from])} {
	    after idle [list catch [list $chatw yview moveto 1]]
	}

	$chatw configure -state disabled
    }
}

proc chat::highlighttext {chatw tag color cursor} {
	$chatw configure -cursor $cursor
	$chatw tag configure $tag -foreground $color
}

proc chat::add_emoteiconed_text {chatw body defaulttag {highlightlist {}}} {
    if {[lempty $highlightlist]} {
	add_emoteiconed_text1 $chatw $body $defaulttag
    } else {
	set ind 0
	foreach {i1 i2} $highlightlist {
	    add_emoteiconed_text1 $chatw [crange $body $ind [expr {$i1 - 1}]] $defaulttag
	    $chatw insert end [crange $body $i1 [expr {$i2 - 1}]] highlight
	    set ind $i2
	}
	chat::add_emoteiconed_text1 $chatw [crange $body $ind end] $defaulttag
    }
}

proc chat::encode_url {url} {
    set utf8_url [encoding convertto utf-8 $url]
    set len [string length $utf8_url]
    set encoded_url ""
    for {set i 0} {$i < $len} {incr i} {
	binary scan $utf8_url @${i}c sym
	set sym [expr {$sym & 0xFF}]
	if {$sym >= 128 || $sym <= 32} {
	    append encoded_url [format "%%%02X" $sym]
	} else {
	    append encoded_url [binary format c $sym]
	}
    }
    return $encoded_url
}

proc chat::add_url {tw msg url args} {
    variable urlid

    set command ""
    foreach {key val} $args {
	switch -- $key {
	    -command { set command $val }
	}
    }

    set tag url$urlid
    set urlfg    [option get $tw urlforeground       Text]
    set urlactfg [option get $tw urlactiveforeground Text]
    $tw tag configure $tag -foreground $urlfg -underline 1
    $tw tag bind $tag <ButtonPress-1><ButtonRelease-1> \
	[list browseurl [double% [encode_url $url]]]
    if {$command != ""} {
	$tw tag bind $tag <ButtonPress-1><ButtonRelease-1> \
	    +[list eval $command]
    }
    $tw tag bind $tag <Any-Enter> \
	[list chat::highlighttext $tw $tag $urlactfg hand2]
    $tw tag bind $tag <Any-Leave> \
	[list chat::highlighttext $tw $tag $urlfg xterm]
	    
    $tw insert end $msg $tag
    incr urlid
}

proc chat::on_browse_url {chatid w x y} {
    set tags [$w tag names "@$x,$y"]
    set idx [lsearch -glob $tags url*]
    if {$idx >= 0} {
	return stop
    }
    return
}

hook::add chat_window_click_hook chat::on_browse_url 10
	
proc chat::add_emoteiconed_text1 {chatw body defaulttag} {
    variable url_regexp
    variable options

    set num_of_emoticons 0
    set words [textutil::splitx $body {([\t \r\n]+)}]

    foreach word $words {
	if {$num_of_emoticons < 200} {
	    if {[emoteicons::get $word] != ""} {
		emoteicons::put $chatw $word
		incr num_of_emoticons
		continue
	    }
	}
	if {[regexp -expanded -nocase -- $url_regexp $word ignore head url ignore2 tail]} {
	    $chatw insert end $head $defaulttag
	    add_url $chatw $url $url
	    $chatw insert end $tail $defaulttag
	} else {
	    if {0 && !$options(emphasize)} {
		$chatw insert end $word $defaulttag
	    } else {
		set enrichedText(bold) 0
		set enrichedText(italic) 0
		set enrichedText(underline) 0

		set tmp $word
		while {[regexp -- "^(_|/|\\*)(.+)(_|/|\\*)$" $tmp wholeMatch e1 subWord e2]} {
		    if { $e1 != $e2 } {
			break
		    }
		    if { "*" == $e1 } {
			if {$enrichedText(bold)} break
			set enrichedText(bold) 1
		    } elseif { "/" == $e1 } {
			if {$enrichedText(italic)} break
			set enrichedText(italic) 1
		    } elseif { "_" == $e1 } {
			if {$enrichedText(underline)} break
			set enrichedText(underline) 1
		    }
		    set tmp $subWord
		}

		set tag $defaulttag
		switch -- $enrichedText(bold),$enrichedText(italic) {
		    1,1 {lappend tag bold_italic}
		    1,0 {lappend tag bold}
		    0,1 {lappend tag italic}
		}
		if {$enrichedText(underline) == 1} {
		    lappend tag underlined
		}
		if {$tmp != $word} {
		    lappend tag emphasized
		    $chatw insert end $tmp $tag
		    $chatw insert end $word [concat nonemphasized $defaulttag]
		} else {
		    $chatw insert end $word $defaulttag
		}
	    }
	}
    }
}

proc chat::switch_emphasize {args} {
    variable options
    variable opened

    foreach chatid [array names opened] {
	set cw [chat_win $chatid]
	if {$options(emphasize)} {
	    $cw tag configure emphasized -elide 0
	    $cw tag configure nonemphasized -elide 1
	} else {
	    $cw tag configure emphasized -elide 1
	    $cw tag configure nonemphasized -elide 0
	}
    }
}

proc chat::add_open_to_user_menu_item {m connid jid} {
    $m add command -label [::msgcat::mc "Start chat"] \
	-command [list chat::open_to_user $connid $jid]
}

hook::add roster_create_groupchat_user_menu_hook \
    [namespace current]::chat::add_open_to_user_menu_item 10
hook::add roster_jid_popup_menu_hook \
    [namespace current]::chat::add_open_to_user_menu_item 10
hook::add message_dialog_menu_hook \
    [namespace current]::chat::add_open_to_user_menu_item 10
hook::add search_popup_menu_hook \
    [namespace current]::chat::add_open_to_user_menu_item 10

proc chat::open_to_user {connid user args} {
    set msg ""
    foreach {opt val} $args {
	switch -- $opt {
	    -message { set msg $val }
	}
    }

    if {$connid == ""} {
	set connid [jlib::route $user]
    }

    set jid [get_jid_of_user $connid $user]

    if {[cequal $jid ""]} {
	set jid $user
    }

    set chatid [chatid $connid $jid]
    set cw [winid $chatid]
    if {[winfo exists $cw]} {
	if {$::usetabbar} {
	    .nb raise [ifacetk::nbpage $cw]
	}
	focus -force $cw.input
    } else {
	chat::open_window $chatid chat
    }

    if {![cequal $msg ""]} {
	debugmsg chat "SEND_MESSAGE ON OPEN:\
	    [list $chatid] [jlib::connection_user $connid] [list $msg] chat"

	set chatw [chat_win $chatid]
	$chatw mark set start_message "end -1 chars"
	$chatw mark gravity start_message left

	hook::run chat_send_message_hook $chatid [jlib::connection_user $connid] \
	    $msg chat
    }
}

###############################################################################

proc chat::change_presence {connid jid type x args} {
    global grouproster
    variable chats
    variable opened
    global statusdesc

    switch -- $type {
	error -
	unavailable { set status $type }
	available {
	    set status available
	    foreach {key val} $args {
		switch -- $key {
		    -show { set status [normalize_show $val] }
		}
	    }
	}
	default { return }
    }

    set group [node_and_server_from_jid $jid]
    set chatid [chatid $connid $group]
    set nick [get_nick $connid $jid groupchat]

    if {[info exists opened($chatid)]} {
	if {[is_groupchat $chatid]} {
	    debugmsg chat "ST: $connid $jid $status"
	    if {[resource_from_jid $jid] == ""} {
		return
	    }
	    if {[cequal $status unavailable] || [cequal $status error]} {
		if {$status == "error" && [is_our_jid $chatid $jid]} {
		    add_message $chatid $group error \
			[format [::msgcat::mc "Error %s"] \
			     [get_jid_presence_info status $connid $jid]] {}
			set chats(status,$chatid) disconnected
			client:presence $connid $group unavailable "" {}
		}
		if {$status == "unavailable" && [is_our_jid $chatid $jid]} {
		    if {[muc::is_compatible $group] || \
			    ![muc::is_changing_nick $chatid]} {
			add_message $chatid $group error \
			    [::msgcat::mc "Disconnected"] {}
			set chats(status,$chatid) disconnected
			client:presence $connid $group unavailable "" {}
		    }
		}
		debugmsg chat "$jid UNAVAILABLE"
		muc::process_unavailable $chatid $nick
		hook::run chat_user_exit $chatid $nick
		lvarpop grouproster(users,$chatid) \
		    [lsearch -exact $grouproster(users,$chatid) $jid]
		catch { unset grouproster(status,$chatid,$jid) }
		debugmsg chat "GR: $grouproster(users,$chatid)"
		chat::redraw_roster_after_idle $chatid
	    } else {
		if {$chats(status,$chatid) == "disconnected" && \
			[is_our_jid $chatid $jid]} {
		    set chats(status,$chatid) connected
		    client:presence $connid $group "" "" {}
		}
		set userswin [users_win $chatid]
		if {![lcontain $grouproster(users,$chatid) $jid]} {
		    #roster::addline $userswin jid $nick $jid
		    lappend grouproster(users,$chatid) $jid
		    set grouproster(status,$chatid,$jid) $status

		    muc::process_available $chatid $nick
		    hook::run chat_user_enter $chatid $nick
		}
		set grouproster(status,$chatid,$jid) $status
		ifacetk::roster::changeicon $userswin $jid roster/user/$status
		ifacetk::roster::changeforeground $userswin $jid $status
		chat::redraw_roster_after_idle $chatid
	    }
	}
    }

    set cw [winid [chatid $connid $jid]]

    if {[winfo exists $cw.status.icon]} {
	$cw.status.icon configure -image roster/user/$status \
	    -helptext [get_user_status_desc $connid $jid]
    }

    if {[winfo exists $cw.status.desc]} {
	$cw.status.desc configure -text "($statusdesc([get_user_status $connid $jid]))" \
	    -helptext [get_user_status_desc $connid $jid]
    }

    set user [node_and_server_from_jid $jid]
    set cw [winid [chatid $connid $user]]

    if {[winfo exists $cw.status.icon]} {
	$cw.status.icon configure \
	    -image roster/user/[get_user_status $connid $user]
    }
}

hook::add client_presence_hook chat::change_presence 70

###############################################################################

proc chat::redraw_roster {group} {
    global grouproster

    set connid [get_connid $group]
    set userswin [users_win $group]

    set grouproster(users,$group) [lsort $grouproster(users,$group)]
    ifacetk::roster::clear $userswin 0

    set levels {}
    foreach jid $grouproster(users,$group) {
	if {[info exists ::muc::users(role,$connid,$jid)]} {
	    if {$::muc::users(role,$connid,$jid) == "" && \
		    $::muc::users(affiliation,$connid,$jid) == ""} {
		set level "aGreen mans"
		lappend levels $level
		lappend levelusers($level) $jid
	    } else {
		if {$::muc::users(role,$connid,$jid) != "" && \
			$::muc::users(role,$connid,$jid) != "none"} {
		    set level $::muc::users(role,$connid,$jid)
		    switch -- $level {
			moderator   {set level 4[::msgcat::mc "Moderators"]}
			participant {set level 5[::msgcat::mc "Participants"]}
			visitor     {set level 6[::msgcat::mc "Visitors"]}
		    }
		    lappend levels $level
		    lappend levelusers($level) $jid
		}

		#if {$::muc::users(affiliation,$jid) != "" && \
		#	$::muc::users(affiliation,$jid) != "none"} {
		#    set level $::muc::users(affiliation,$jid)
		#    switch -- $level {
		#	owner   {set level 0Owners}
		#	admin   {set level 1Admins}
		#	member  {set level 2Members}
		#	outcast {set level 9Outcasts}
		#    }
		#    lappend levels $level
		#    lappend levelusers($level) $jid
		#}
	    }
	} else {
	    set level 2[::msgcat::mc "Users"]
	    lappend levels $level
	    lappend levelusers($level) $jid
	}	    
    }
    set levels [lrmdups $levels]

    foreach level $levels {
	ifacetk::roster::addline $userswin group \
	    "[crange $level 1 end] ([llength $levelusers($level)])" \
	    [crange $level 1 end] [crange $level 1 end] 0
	set jid_nicks {}
	foreach jid $levelusers($level) {
	    lappend jid_nicks [list $jid [get_nick $connid $jid groupchat]]
	}
	set jid_nicks [lsort -index 1 -dictionary $jid_nicks]

	foreach item $jid_nicks {
	    lassign $item jid nick
	    set status $grouproster(status,$group,$jid)

	    ifacetk::roster::addline $userswin jid $nick \
		[list $connid $jid] [crange $level 1 end] 0
	    ifacetk::roster::changeicon $userswin \
		[list $connid $jid] roster/user/$status
	    if {$::plugins::nickcolors::options(use_colored_roster_nicks)} {
		ifacetk::roster::changeforeground $userswin \
		    [list $connid $jid] $status \
		    [plugins::nickcolors::get_nick_color $nick]
	    } else {
		ifacetk::roster::changeforeground $userswin \
		    [list $connid $jid] $status
	    }
	}
    }

    ifacetk::roster::update_scrollregion $userswin
}

proc chat::redraw_roster_after_idle {group} {
    variable afterid

    if {[info exists afterid($group)]} \
	return

    set afterid($group) [after idle "
	chat::redraw_roster [list $group]
	unset [list chat::afterid($group)]
    "]
}

proc chat::restore_subject {chatid} {
    variable opened
    variable chats

    set cw [winid $chatid]

    if {[info exists opened($chatid)]} {
	$cw.status.subject delete 0 end
	$cw.status.subject insert 0 $chats(subject,$chatid)
    }
}

proc chat::set_subject {chatid subject} {
    variable opened
    variable chats

    set cw [winid $chatid]

    if {[info exists opened($chatid)]} {
	$cw.status.subject delete 0 end
	$cw.status.subject insert 0 $subject
	set chats(subject,$chatid) $subject
    }
}

proc chat::change_subject {chatid} {
    set cw [winid $chatid]
    set connid [get_connid $chatid]
    set jid [get_jid $chatid]

    set subject [$cw.status.subject get]

    message::send_msg $jid -connection $connid \
	-type groupchat -subject $subject
}

proc chat::set_subject_balloon {chatid} {
    variable chats
    if {[info exists chats(subject_tooltip,$chatid)]} {
	return $chats(subject_tooltip,$chatid)
    } else {
	return ""
    }
}

proc chat::set_subject_tooltip {chatid lo hi} {
    variable chats

    set sw [winid $chatid].status.subject

    if {![winfo exists $sw]} return

    if {($lo == 0) && ($hi == 1)} {
	set chats(subject_tooltip,$chatid) ""
    } else {
	set chats(subject_tooltip,$chatid) [$sw get]
    }
}

proc chat::is_our_jid {chatid jid} {
    return [cequal [our_jid $chatid] $jid]
}

proc chat::our_jid {chatid} {
    variable chats

    set connid [get_connid $chatid]
    set jid [get_jid $chatid]
    switch -- $chats(type,$chatid) {
	groupchat {
	    return $jid/[get_our_groupchat_nick $chatid]
	}
	chat {
	    set group [node_and_server_from_jid $jid]
	    set groupid [chatid $connid $group]
	    if {[is_groupchat $groupid]} {
		return $group/[get_our_groupchat_nick $groupid]
	    } else {
		return [jlib::connection_jid $connid]
	    }
	}
    }
    return ""
}

###############################################################################

proc chat::add_invite_menu_item {m connid jid} {
    $m add command -label [::msgcat::mc "Invite to conference..."] \
	   -command [list chat::invite_dialog $jid 0 -connection $connid]
}

hook::add chat_create_user_menu_hook \
    [namespace current]::chat::add_invite_menu_item 20
hook::add roster_create_groupchat_user_menu_hook \
    [namespace current]::chat::add_invite_menu_item 20
hook::add roster_jid_popup_menu_hook \
    [namespace current]::chat::add_invite_menu_item 20
hook::add message_dialog_menu_hook \
    [namespace current]::chat::add_invite_menu_item 20
hook::add search_popup_menu_hook \
    [namespace current]::chat::add_invite_menu_item 20

proc chat::add_invite2_menu_item {m connid jid} {
    $m add command -label [::msgcat::mc "Invite users..."] \
	   -command [list chat::invite_dialog2 $jid 0 -connection $connid]
}

hook::add chat_create_conference_menu_hook \
    [namespace current]::chat::add_invite2_menu_item 20

###############################################################################

proc chat::invite_dialog {user {ignore_muc 0} args} {
    variable chats
    variable opened
    global invite_gc

    foreach {opt val} $args {
	switch -- $opt {
	    -connection { set connid $val }
	}
    }
    if {![info exists connid]} {
	set connid [jlib::route $user]
    }

    set jid [get_jid_of_user $connid $user]

    if {[cequal $jid ""]} {
        set jid $user
    }

    set gw .invite
    catch { destroy $gw }

    if {[catch { set nick [roster::get_label $user] }]} {
	if {[catch {set nick [roster::get_label \
				  [node_and_server_from_jid $user]] }]} {
	    if {[catch { set nick [chat::get_nick $connid \
						  $user groupchat] }]} {
		set nick $user
	    }
	}
    }

    set titles {}
    set jids {}
    foreach chatid [lsort [array names opened]] {
        if {([get_connid $chatid] == $connid) && [is_groupchat $chatid]} {
	    lappend jids $chatid [get_jid $chatid]
            lappend titles $chatid [node_from_jid [get_jid $chatid]]
	}
    }
    if {[llength $titles] == 0} {
        MessageDlg ${gw}_err -aspect 50000 -icon info \
	    -message \
		[format [::msgcat::mc "No conferences for %s in progress..."] \
		    [jlib::connection_jid $connid]] \
	    -type user \
	    -buttons ok -default 0 -cancel 0
        return
    }

    CbDialog $gw [format [::msgcat::mc "Invite %s to conferences"] $nick] \
	[list [::msgcat::mc "Invite"] "chat::invitation [list $jid] 0 $ignore_muc
				       destroy $gw" \
	      [::msgcat::mc "Cancel"] "destroy $gw"] \
	invite_gc $titles $jids
}

proc chat::invite_dialog2 {jid ignore_muc args} {
    variable chats
    variable opened
    global invite_gc

    foreach {opt val} $args {
	switch -- $opt {
	    -connection { set connid $val }
	}
    }
    if {![info exists connid]} {
	set connid [jlib::route $jid]
    }

    set gw .invite
    catch { destroy $gw }

    set title [node_from_jid $jid]

    set choices {}
    set balloons {}
    foreach choice [roster::get_jids $connid] {
	if {![cequal [roster::itemconfig $connid $choice -category] conference]} {
	    lappend choices [list $connid $choice] [roster::get_label $connid $choice]
	    lappend balloons [list $connid $choice] $choice
	} 
    } 
    if {[llength $choices] == 0} {
        MessageDlg ${gw}_err -aspect 50000 -icon info \
	    -message \
		[format [::msgcat::mc "No users in %s roster..."] \
		     [jlib::connection_jid $connid]] \
	    -type user \
	    -buttons ok -default 0 -cancel 0
	return
    }

    CbDialog $gw [format [::msgcat::mc "Invite users to %s"] $title] \
	[list [::msgcat::mc "Invite"] "chat::invitation [list $jid] 1 $ignore_muc
				       destroy $gw" \
	      [::msgcat::mc "Cancel"] "destroy $gw"] \
	invite_gc $choices $balloons
}

proc chat::invitation {jid usersP ignore_muc {reason ""}} {
    global invite_gc

    foreach choice [array names invite_gc] {
        if {$invite_gc($choice)} {
	    lassign $choice con gc
	    if {$usersP} {
		set to $gc
		set group $jid
	    } else {            
		set to $jid
		set group $gc
	    }
	    if {[cequal $reason ""]} {
		set reas [format [::msgcat::mc "Please join %s"] $group]
	    } else {
		set reas $reason
	    }
	    if {!$ignore_muc && [muc::is_compatible $group]} {
		muc::invite_muc $con $group $to $reas
	    } else {
		muc::invite_xconference $con $group $to $reas
	    }
	}
    }
}

