/*
 *  ISEM - Instructional Sparc EMulator and tkisem
 *  Copyright (C) 1993, 1994, 1995, 1996
 *	Department of Computer Science,
 *      The University of New Mexico
 *
 *  Please send questions, comments, and bug reports to: isem@cs.unm.edu
 *
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

//
// Memory.c -- definition of the  Memory  class
//

#if __GNUC__
#define UNUSED __attribute__ ((unused)) 
#else
#define UNUSED
#endif

static char rcsid[] UNUSED = "$Id: Memory.cpp 1.1 Fri, 25 Oct 1996 18:04:04 -0600 maccabe $";

#include "Assert.h"
#include "sizedefs.h"
#include "Config.h"

#include "sys_bus.h"
#include "Memory.h"


const UInt32 WordMask[] = {
    0,			// 0000
    0x000000ff,		// 0001
    0x0000ff00,		// 0010
    0x0000ffff,		// 0011

    0x00ff0000,		// 0100
    0,			// 0101
    0,			// 0110
    0,			// 0111

    0xff000000,		// 1000
    0,			// 1001
    0,			// 1010
    0,			// 1011

    0xffff0000,		// 1100
    0,			// 1101
    0,			// 1110
    0xffffffff,		// 1111
};


//
// public Constructors / Destructors
//

//
// Memory(start in 1K units, size in 1K units)
//
Memory::Memory(SystemBus& sb)
    : StorageDevice(sb, Config::PHYS_MEM_START << 10,
	((Config::PHYS_MEM_START + Config::PHYS_MEM_SIZE) << 10)-1)
{
    data = new UInt32 [ Config::PHYS_MEM_SIZE << (10 - 2) ];
    Assert(data, "new failed");

    for (int i = 0; i < Config::PHYS_MEM_SIZE << (10 - 2); i++)
	data[i] = 0;
}


//
// public Member functions
//
void Memory::reset() {
    for (int i = 0; i < Config::PHYS_MEM_SIZE << (10 - 2); i++)
	data[i] = 0;
}

void Memory::run() {
    // do nothing for this device
}

UInt32 Memory::read(UInt32 addr) {
    // If this could fail for 2 reasons
    //    1 - caller makes mistake
    //    2 - memory isn't an even number of pages
    //
    Assert (addr >= lowAddress() && addr <= highAddress(),
	"address out of range");

    return data[(addr-lowAddress()) >> 2];
}

void Memory::write(UInt32 addr, int byteMask, UInt32 value) {
    UInt32 mask = WordMask[byteMask];

    // 
    // If any of these assertions fail, the caller made a mistake
    //
    Assert(byteMask > 0, "illegal byte mask");
    Assert(byteMask < 16, "illegal byte mask");
    Assert(mask != 0, "illegal byte mask");

    // If this could fail for 2 reasons
    //    1 - caller makes mistake
    //    2 - memory isn't an even number of pages
    //
    Assert (addr >= lowAddress() && addr <= highAddress(),
	"address out of range");
    
    // fetch old value from memory and mask out
    //   any bits that will could be changed by write
    //
    UInt32 oldValue = data[(addr-lowAddress()) >> 2] & ~mask;

    // write new value
    //
    data[(addr-lowAddress()) >> 2] = oldValue | (value & mask);
}

