#include "os.h"
#include "pool.h"

static void*	sbrkalloc(ulong);
static int		sbrkmerge(void*, void*);
static void		plock(Pool*);
static void		punlock(Pool*);
static void		pprint(Pool*, char*, ...);
static void		ppanic(Pool*, char*, ...);

typedef struct Private Private;
struct Private {
	Lock		lk;
	int		printfd;	/* gets debugging output if set */
};

Private sbrkmempriv;

static Pool sbrkmem = {
	"sbrkmem",
	2UL*1024*1024*1024,
	0,
	0,
	0,
	4*1024,
	32,
	0,
	0,
	0,
	sbrkalloc,
	sbrkmerge,
	0,
	0,
	0,
	0,
	plock,
	punlock,
	pprint,
	ppanic,
	0,
	&sbrkmempriv
};
Pool *mainmem = &sbrkmem;
Pool *imagmem = &sbrkmem;

/*
 * we do minimal bookkeeping so we can tell pool
 * whether two blocks are adjacent and thus mergeable.
 */
static void*
sbrkalloc(ulong n)
{
	intptr t;
	u32int *x;

	n += 8;		/* 64-bit alignment (just in case) */
	n += 8;		/* two 32-bit words for us */
	x = (u32int*)sbrk(n);
	if(~(intptr)x == 0)
		return nil;
	t = (intptr)x & 7;
	if(t){
		t = 8-t;
		n -= t;
		x = (u32int*)((intptr)x+t);
	}
	x[0] = n;	/* sbrk rounds size up to mult. of 8 */
	x[1] = 0xDeadBeef;
	return x+2;
}

static int
sbrkmerge(void *x, void *y)
{
	u32int *lx, *ly;

	lx = x;
	if(lx[-1] != 0xDeadBeef)
		abort();

	if((uchar*)lx+lx[-2] == (uchar*)y) {
		ly = y;
		lx[-2] += ly[-2];
		return 1;
	}
	return 0;
}

static void
plock(Pool *p)
{
	Private *pv;
	pv = p->private;
	lock(&pv->lk);
}

static void
punlock(Pool *p)
{
	Private *pv;
	pv = p->private;
	unlock(&pv->lk);
}

static void
pprint(Pool *p, char *fmt, ...)
{
	va_list v;
	int n;
	char buf[128];
	char *msg;
	Private *pv;

	pv = p->private;
	if(pv->printfd <= 0)
		pv->printfd = 2;

	msg = buf;
	va_start(v, fmt);
	n = vseprint(buf, buf+sizeof buf, fmt, v)-msg;
	write(pv->printfd, buf, n);
	va_end(v);
}

static char panicbuf[256];
static void
ppanic(Pool *p, char *fmt, ...) 
{
	va_list v;
	int n;
	char *msg;
	Private *pv;

	pv = p->private;
	assert(canlock(&pv->lk)==0);

	if(pv->printfd <= 0)
		pv->printfd = 2;

	msg = panicbuf;
	va_start(v, fmt);
	n = vseprint(msg, msg+sizeof panicbuf, fmt, v) - msg;
	write(2, "panic: ", 7);
	write(2, msg, n);
	write(2, "\n", 1);
	if(pv->printfd != 2){
		write(pv->printfd, "panic: ", 7);
		write(pv->printfd, msg, n);
		write(pv->printfd, "\n", 1);
	}
	va_end(v);
	unlock(&pv->lk);
	abort();
}

/* - everything from here down should be the same in libc, libdebugmalloc, and the kernel - */
/* - except the code for malloc(), which alternately doesn't clear or does. - */

/*
 * Npadlong is the number of 32-bit longs to leave at the beginning of 
 * each allocated buffer for our own bookkeeping.  We return to the callers
 * a pointer that points immediately after our bookkeeping area.  Incoming pointers
 * must be decremented by that much, and outgoing pointers incremented.
 * The malloc tag is stored at MallocOffset from the beginning of the block,
 * and the realloc tag at ReallocOffset.  The offsets are from the true beginning
 * of the block, not the beginning the caller sees.
 *
 * The extra if(Npadlong != 0) in various places is a hint for the compiler to
 * compile out function calls that would otherwise be no-ops.
 */

/*	non tracing
 *
enum {
	Npadlong	= 0,
	MallocOffset = 0,
	ReallocOffset = 0,
};
 *
 */

/* tracing */
enum {
	Npadlong	= 2,
	MallocOffset = 0,
	ReallocOffset = 1
};

void*
malloc(ulong size)
{
	void *v;

	v = poolalloc(mainmem, size+Npadlong*sizeof(u32int));
	if(Npadlong && v != nil) {
		v = (u32int*)v+Npadlong;
		setmalloctag(v, getcallerpc(&size));
		setrealloctag(v, 0);
	}
	return v;
}

void*
mallocz(ulong size, int clr)
{
	void *v;

	v = poolalloc(mainmem, size+Npadlong*sizeof(u32int));
	if(Npadlong && v != nil){
		v = (u32int*)v+Npadlong;
		setmalloctag(v, getcallerpc(&size));
		setrealloctag(v, 0);
	}
	if(clr && v != nil)
		memset(v, 0, size);
	return v;
}

void
free(void *v)
{
	if(v != nil)
		poolfree(mainmem, (u32int*)v-Npadlong);
}

void*
realloc(void *v, ulong size)
{
	void *nv;

	if(size == 0){
		free(v);
		return nil;
	}

	if(v)
		v = (u32int*)v-Npadlong;
	size += Npadlong*sizeof(u32int);

	if((nv = poolrealloc(mainmem, v, size)) != nil){
		nv = (u32int*)nv+Npadlong;
		setrealloctag(nv, getcallerpc(&v));
		if(v == nil)
			setmalloctag(nv, getcallerpc(&v));
	}		
	return nv;
}

ulong
msize(void *v)
{
	return poolmsize(mainmem, (u32int*)v-Npadlong)-Npadlong*sizeof(u32int);
}

void*
calloc(ulong n, ulong szelem)
{
	void *v;
	if((v = mallocz(n*szelem, 1)) != nil)
		setmalloctag(v, getcallerpc(&n));
	return v;
}

void
setmalloctag(void *v, ulong pc)
{
	u32int *u;
	USED(v);
	USED(pc);
	if(Npadlong <= MallocOffset || v == nil)
		return;
	u = v;
	u[-Npadlong+MallocOffset] = pc;
}

void
setrealloctag(void *v, ulong pc)
{
	u32int *u;
	USED(v);
	USED(pc);
	if(Npadlong <= ReallocOffset || v == nil)
		return;
	u = v;
	u[-Npadlong+ReallocOffset] = pc;
}

ulong
getmalloctag(void *v)
{
	USED(v);
	if(Npadlong <= MallocOffset)
		return ~0;
	return ((u32int*)v)[-Npadlong+MallocOffset];
}

ulong
getrealloctag(void *v)
{
	USED(v);
	if(Npadlong <= ReallocOffset)
		return ((u32int*)v)[-Npadlong+ReallocOffset];
	return ~0;
}

void*
malloctopoolblock(void *v)
{
	if(v == nil)
		return nil;

	return &((u32int*)v)[-Npadlong];
}
