/*
	**
	** main.c
	**
	** Main module for collector module for traffic-vis program
	**
	** Copyright 1998-1999 Damien Miller <dmiller@ilogic.com.au>
	**
	** This software is licensed under the terms of the GNU General 
	** Public License (GPL). Please see the file COPYING for details.
	** 
	** $Id: traffic-collector.c,v 1.4 1999/03/27 00:12:50 dmiller Exp $
	**
 */

#include <stdlib.h>
#include <stdio.h>
#include <getopt.h>
#include <unistd.h>
#include <signal.h>
#include <syslog.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <math.h>

#include <pcap.h>

#include "packet-parse.h"
#include "packet-summary.h"
#include "util.h"

#include "summary-output.h"

#define NET_READ_TIMEOUT	50

/* Global variables */
static char			rcsid[] = "$Id: traffic-collector.c,v 1.4 1999/03/27 00:12:50 dmiller Exp $";
static pcap_t		*pcap_handle;
static summary_t	*psum;

static int			exit_flag;
static int			delete_state;
static int			write_summary;

/* Prototypes */
int				main(int argc, char **argv);
void				cleanup(int signum);
const char 		*append_unique_timestamp(const char *rootname);
void				configure_capture_file(pcap_t **p, const char *capture_file, int *link_type);
void				configure_live_interface(pcap_t **p, char *dev, int promisc_flag, bpf_u_int32 *localnet, bpf_u_int32 *netmask, int *link_type);
void				exit_main_loop(int signum);
void 				summary_request(int signum);
void				restart(int signum);
void 				request_and_restart(int signum);
void				pcap_packet_handler(u_char *user_data, const struct pcap_pkthdr* phdr, const u_char *pkt);
void				set_packet_filter(pcap_t *p, char *filter, bpf_u_int32 netmask);
void				daemonise(void);
void				usage(void);
void				version(void);

#ifdef HAVE_GETOPT_LONG
/* Commandline options */
static struct option long_options[] =
{
	{"no-promisc", 0, NULL, 'p'},
	{"interface", 1, NULL, 'i'},
	{"filter", 1, NULL, 'f'},
	{"read-file", 1, NULL, 'r'},
	{"summary-file", 1, NULL, 's'},
	{"timestamp", 0, NULL, 't'},
	{"version", 0, NULL, 'V'},
	{"help", 0, NULL, 'h'},
	{NULL, 0, NULL, 0}
};
#endif /* HAVE_GETOPT_LONG */

/* Called by pcap_dispatch for each packet processed */
void pcap_packet_handler(u_char *user_data, const struct pcap_pkthdr* phdr, const u_char *pkt)
{
	static packet_info_t pinfo;


	if (parse_packet(*(unsigned int *)user_data, pkt, phdr->caplen, &pinfo))
	{
		pinfo.received_time = phdr->ts.tv_sec;
		store_packet_info(psum, &pinfo);
	} else
	{
#ifdef DEBUG
		syslog(LOG_DEBUG, "Packet parse error: %s", parse_error());
#endif
	}
}

/* Signal handler, shuts down pcap and exits cleanly */
void cleanup(int signum)
{
	if (pcap_handle != 0)
		pcap_close(pcap_handle);
		
	syslog(LOG_CRIT, "Caught signal %i - exiting.", signum);
	exit(1);
}

/* Signal handler for SIGINT, exits main loop */
void exit_main_loop(int signum)
{
	/* Exit from loop */
	exit_flag = 1;

	signal(signum, exit_main_loop);
}

/* Signal handler for SIGUSR1, sets "write summary" flag */
void summary_request(int signum)
{
	write_summary = 1;
	
	signal(signum, summary_request);
}

/* Signal handler for SIGHUP, sets "delete state" flag */
void restart(int signum)
{
	delete_state = 1;
	
	signal(signum, restart);
}

/* Signal handler for SIGUSR2, sets "write summary" and */ 
/* "delete state" flags */
void request_and_restart(int signum)
{
	write_summary = 1;
	delete_state = 1;
	
	signal(signum, request_and_restart);
}

/* Sets up interface and returns information about network address, */
/* netmask and datalink type. Can be called with a NULL interface, */
/* in which case pcap will pick one. */
void configure_live_interface(pcap_t **p, char *dev, 
                              int promisc_flag, bpf_u_int32 *localnet, 
										bpf_u_int32 *netmask, int *link_type)
{
	char error_buf[1024];
	
	/* If no device selected, use pcap to select one */
	if (dev == NULL)
	{
		dev = pcap_lookupdev(error_buf);
		if (dev == NULL)
		{
			syslog(LOG_CRIT, "pcap_lookupdev: %s", error_buf);
			exit(-1);
		}
		syslog(LOG_WARNING, "No interface specified, using \'%s\'.", dev);
	}
		
	/* Get netmask of interface */
	if (pcap_lookupnet(dev, localnet, netmask, error_buf) < 0)
	{
		syslog(LOG_CRIT, "pcap_lookupnet: %s", error_buf);
		exit(-1);
	}
	
	/* Open interface */
	*p = pcap_open_live(dev, 256, promisc_flag, NET_READ_TIMEOUT, error_buf);
	if (pcap_handle == NULL)
	{
		syslog(LOG_CRIT, "pcap_open_live: %s", error_buf);
		exit(-1);
	}

	*link_type = pcap_datalink(*p);
}

/* Sets up reading from a previously recorded pcap capture file */
void configure_capture_file(pcap_t **p, const char *capture_file, 
                            int *link_type)
{
	char error_buf[1024];
	
	/* Open capture file */
	*p = pcap_open_offline(capture_file, error_buf);
	if (pcap_handle == NULL)
	{
		syslog(LOG_CRIT, "pcap_open_offline: %s", error_buf);
		exit(-1);
	}

	*link_type = pcap_datalink(*p);
}

/* Sets a filter on packet capture */
void set_packet_filter(pcap_t *p, char *filter, bpf_u_int32 netmask)
{
	struct bpf_program bpf_prog;
	
	if (filter == NULL)
		return;
	
	if (pcap_compile(p, &bpf_prog, filter, 1, netmask) == -1)
	{
		syslog(LOG_CRIT, "pcap_compile: %s", pcap_geterr(p));
		exit(-1);
	}
	
	if (pcap_setfilter(p, &bpf_prog) == -1)
	{
		syslog(LOG_CRIT, "pcap_setfilter: %s", pcap_geterr(p));
		exit(-1);
	}
}

void daemonise(void)
{
	switch (fork())
	{
		case -1:
			syslog(LOG_CRIT, "Couldn't fork.");
			fprintf(stderr, "Couldn't fork.\n");
			exit(2);
		case 0:
			/* We are the child */
			break;
		default:
			/* We are the parent */
			_exit(0);
	}

	setsid();

	switch (fork())
	{
		case -1:
		syslog(LOG_CRIT, "Couldn't fork.");
		exit(2);
	case 0:
		/* We are the child */
		break;
	default:
		/* We are the parent */
		_exit(0);
	}

	chdir("/");

	umask(077);

	close(STDIN_FILENO);
	close(STDOUT_FILENO);
	close(STDERR_FILENO);
}

/* Appends a unique timestamp to the specified filename */
/* If it cannot find a unique filename, will return an 'overflow' name */
const char *append_unique_timestamp(const char *rootname)
{
	static char	s[1024];
	time_t		t;
	struct tm	*lt;
	int			c;

	t = time(NULL);
	lt = localtime(&t);
		
	lt->tm_year += 1900;
	lt->tm_mon++;
		
	/* Try max of 100 times to find an non-existing file */
	/* Filename of the form rootname.YYYYMMDD-HHMMSS-XX */
	c = 0;
	do 
	{
		g_snprintf(s, sizeof(s), "%s.%04d%02d%02d-%02d%02d%02d-%02d",
						rootname, lt->tm_year, lt->tm_mon, lt->tm_mday, 
						lt->tm_hour, lt->tm_min, lt->tm_sec, c);
		c++;
	} while ((access(s, W_OK) == 0) && c < 100);
	
	/* If we cannot find one log it and use the overflow file */
	if (c == 100)
	{
		g_snprintf(s, sizeof(s), "%s.%04d%02d%02d-%02d%02d%02d-XX",
						rootname, lt->tm_year, lt->tm_mon, lt->tm_mday, 
						lt->tm_hour, lt->tm_min, lt->tm_sec);
		syslog(LOG_ERR, "Could not find unique name, using %s", s);
	}
	
	return(s);
}

void usage(void)
{
	fprintf(stderr, "Usage: traffic-collector [OPTION]\n");

	fprintf(stderr, "\n");

	fprintf(stderr, "Collect statistics on network traffic, including total traffic\n");
	fprintf(stderr, "(bytes and packets), traffic per host and which hosts communicated.\n");

	fprintf(stderr, "\n");

	fprintf(stderr, "traffic-collector will run as a daemon until stopped with a SIGINT or a\n");
	fprintf(stderr, "SIGTERM. It will dump its state and restart upon a SIGHUP and write a\n");
	fprintf(stderr, "report to the file %s upon SIGUSR1\n", DEFAULT_OUTPUT_FILE);

	fprintf(stderr, "\n");

#ifdef HAVE_GETOPT_LONG
	fprintf(stderr, "  -p, --no-promisc       Don\'t put the network interface into promiscious mode.\n");
	fprintf(stderr, "  -i, --interface xxx    Specify network interface to collect packets from.\n");
	fprintf(stderr, "  -f, --filter    xxx    Specify BPF filter to be applied before packet \n");
	fprintf(stderr, "                         processing. See tcpdump(8) for details.\n");
	fprintf(stderr, "  -r, --read-file xxx    Read a pcap capture file, such as those created by\n");
	fprintf(stderr, "                         tcpdump(8), instead of listening to live traffic.\n");
	fprintf(stderr, "  -s, --summary-file xxx Write reports to specified summary file.\n");
	fprintf(stderr, "  -V, --version          Print program version.\n");
	fprintf(stderr, "  -h, --help             Display this help text.\n");
#else /* HAVE_GETOPT_LONG */
	fprintf(stderr, "  -p   	 Don\'t put the network interface into promiscious mode.\n");
	fprintf(stderr, "  -i xxx   Specify network interface to collect packets from.\n");
	fprintf(stderr, "  -f xxx   Specify BPF filter to be applied before packet \n");
	fprintf(stderr, "       	 processing. See tcpdump(8) for details.\n");
	fprintf(stderr, "  -r xxx   Read a pcap capture file, such as those created by\n");
	fprintf(stderr, "       	 tcpdump(8), instead of listening to live traffic.\n");
	fprintf(stderr, "  -s xxx   Write report s to specified summary file.\n");
	fprintf(stderr, "  -V   	 Print program version.\n");
	fprintf(stderr, "  -h   	 Display this help text.\n");
#endif /* HAVE_GETOPT_LONG */

	fprintf(stderr, "\n");

	fprintf(stderr, "Please report bugs to dmiller@ilogic.com.au\n");
}

void version(void)
{
	fprintf(stderr, "traffic-collector %s\n", VERSION);
}

int main(int argc, char **argv)
{
	bpf_u_int32 		localnet;
	bpf_u_int32			netmask;
	unsigned int		link_type;
	int 					c;
	int					dispatch_result;
	extern char 		*optarg;
	static const char	*summary_file;
	
	/* Commandline options */
	static char			*dev = NULL;
	static char 		*filter = NULL;
	static char 		*capture_file = NULL;
	static char 		*summary_root = NULL;
	static int 			promisc_flag = 1;
	static int 			timestamp = 0;

	/* Fetch commandline options */
	while (1)
	{
#ifdef HAVE_GETOPT_LONG
		c = getopt_long(argc, argv, "pi:f:r:s:tVh?", long_options, NULL);
#else /* HAVE_GETOPT_LONG */
		c = getopt(argc, argv, "pi:f:r:s:tVh?");
#endif /* HAVE_GETOPT_LONG */

		if (c == -1)
			break;
		
		switch (c)
		{
			case 'p':
				promisc_flag = 0;
				break;
			
			case 'i':
				dev = util_strdup(optarg);
				break;

			case 'f':
				filter = util_strdup(optarg);
				break;
				
			case 'r':
				capture_file = util_strdup(optarg);
				break;
			
			case 's':
				summary_root = util_strdup(optarg);
				break;
			
			case 't':
				timestamp = 1;
				break;
			
			case 'V':
				version();
				exit(0);
				
			case 'h':
			case '?':
				usage();
				exit(0);
				
			default:
				fprintf(stderr, "Invalid commandline options.\n\n");
				usage();
				exit(-1);
		}
	}

	if (summary_root == NULL)
		summary_root = util_strdup(DEFAULT_OUTPUT_FILE);

	/* Set up logging */
	openlog("traffic-collector", LOG_PID, LOG_DAEMON);
	
	/* Detatch from tty and run as a daemon */
	daemonise(); 
					
	/* Initialise packet summary data structure */
	packet_summary_init(&psum);

	/* Exits main loop on user signal */
	signal(SIGINT, exit_main_loop);
	signal(SIGTERM, exit_main_loop);

	/* Restart on SIGHUP */
	signal(SIGHUP, restart);

	/* Shut down cleanly is something has gone wrong */
#ifndef DEBUG
	signal(SIGABRT, cleanup);
	signal(SIGSEGV, cleanup);
	signal(SIGQUIT, cleanup);
	signal(SIGILL, cleanup);
#endif
	
	/* Write a report on SIGUSR1 */
	signal(SIGUSR1, summary_request);

	/* Write a report and restart on SIGUSR2 */
	signal(SIGUSR2, request_and_restart);

	/* Set up packet capture */
	if (capture_file != NULL)
	{
		configure_capture_file(&pcap_handle, capture_file, &link_type);
		localnet = 0;
		netmask = 0;
	} else
	{
		configure_live_interface(&pcap_handle, dev, promisc_flag, &localnet, 
	    								 &netmask, &link_type);
	}
	set_packet_filter(pcap_handle, filter, netmask);
	
	/* Main loop */
	exit_flag = 0;
	write_summary = 0;
	delete_state = 0;
	while(!exit_flag)
	{
		/* Read packets */
		dispatch_result = pcap_dispatch(pcap_handle, 0, pcap_packet_handler, 
			                             (u_char*)&link_type);
		if (dispatch_result == -1)
		{
			/* An error has occurred */
			syslog(LOG_CRIT, "pcap_dispatch: %s", pcap_geterr(pcap_handle));
			pcap_close(pcap_handle);
			exit(-1);
		} else if ((dispatch_result == 0) && (capture_file != NULL))
		{
			/* An EOF has occurred, set exit and summary flags */
			exit_flag = 1;
			write_summary = 1;
		}

		if (write_summary == 1)
		{
			write_summary = 0;
			
			if (timestamp == 1)
				summary_file = append_unique_timestamp(summary_root);
			else 
				summary_file = summary_root;
			
			syslog(LOG_INFO, "Writing summary to '%s'", summary_file);
			output_summary(summary_file, psum);
		}
		
		if (delete_state == 1)
		{
			delete_state = 0;
			syslog(LOG_INFO, "Clearing collected information");
			packet_summary_free(psum);
			packet_summary_init(&psum);
		}
	}
	
	/* Close packet capture interface */
	pcap_close(pcap_handle);

	/* Free options variables */	
	if (dev != NULL)
		free(dev);
		
	if (filter != NULL)
		free(filter);
		
	if (capture_file != NULL)
		free(capture_file);

	if (summary_file != NULL)
		free(capture_file);

	/* Reset signals */
	signal(SIGINT, SIG_DFL);
	signal(SIGTERM, SIG_DFL);
	signal(SIGSEGV, SIG_DFL);
	signal(SIGABRT, SIG_DFL);
	signal(SIGQUIT, SIG_DFL);
	signal(SIGILL, SIG_DFL);
	signal(SIGUSR1, SIG_DFL);

	/* Free packet summary data structure */
	packet_summary_free(psum);

	syslog(LOG_INFO, "Exiting normally.");

	return(0);
}
