/*
	**
	** util.h
	**
	** Miscellaneous utility routines
	**
	** Copyright 1998-1999 Damien Miller <dmiller@ilogic.com.au>
	**
	** This software is licensed under the terms of the GNU General 
	** Public License (GPL). Please see the file COPYING for details.
	** 
	** $Id: util.c,v 1.5 1999/02/10 12:29:20 dmiller Exp $
	**
 */

#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>
#include <string.h>
#include <signal.h>
#include <time.h>
#include <netdb.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/types.h>
#include <math.h>

#include <glib.h>

#include "util.h"

static char rcsid[] = "$Id: util.c,v 1.5 1999/02/10 12:29:20 dmiller Exp $";

/* Allocates s bytes, checking for allocation errors */
void *util_malloc(size_t s)
{
	void *m;
	
	m = g_malloc(s);

	if (m == NULL)
	{
		fprintf(stderr, "\nOut of memory.\n");
		raise(SIGSEGV);
	}
	
	return(m);
}

/* Allocates s zero-filled bytes, checking for allocation errors */
void *util_zmalloc(size_t s)
{
	void *m;
	
	m = g_malloc(s);

	if (m == NULL)
	{
		fprintf(stderr, "\nOut of memory.\n");
		raise(SIGSEGV);
	}
	
	memset(m, 0, s);
	
	return(m);
}

/* Duplicates string s, checking for allocation errors */
void *util_strdup(const char *s)
{
	char *m;
	
	m = g_strdup(s);

	if (m == NULL)
	{
		fprintf(stderr, "\nOut of memory.\n");
		raise(SIGSEGV);
	}
	
	return(m);
}

/* Resizes allocation m to s bytes, checking for allocation errors */
void *util_realloc(void *m, size_t s)
{
	m = g_realloc(m, s);

	if (m == NULL)
	{
		fprintf(stderr, "\nOut of memory.\n");
		raise(SIGSEGV);
	}
	
	return(m);
}

/* Seeds RNG  */
void util_seed_rng(void)
{
	srandom(time(NULL) ^ (getpid() + (getppid() << 15)));
}

/* Parses an address and mask */
int parse_address_and_mask(const char *addrspec, u_int32_t *address, u_int32_t *mask)
{
	char address_str[20];
	char mask_str[20];
	struct in_addr i_addr;
	struct hostent *h;
	int cidr_bits;
	int c;
	
	/* Check and split address specification */
	c = sscanf(addrspec, "%18[0-9.]/%18[0-9.]", address_str, mask_str);
	
	/* If no values were converted, try to look up name in DNS */
	if (c == 0)
	{
		h = gethostbyname(addrspec);
		
		/* If no host of suitable type found, return an error */
		if ((h == NULL) || (h->h_addrtype != AF_INET))
			return(0);
		
		memcpy(address, h->h_addr_list[0], sizeof(*address));
		*mask = 0xFFFFFFFF;
		
		return(1);
	}

	/* Convert address part, returning an error if conversion failed */
	if (inet_aton(address_str, &i_addr) == 0)
		return(0);
	*address = i_addr.s_addr;

	/* If only address was specified, return it with all-ones mask */
	if (c == 1)
	{
		*mask = 0xFFFFFFFF;
		return(1);
	}

	/* Check whether a CIDR network-length was specified */
	if (strchr(mask_str, '.') == NULL)
	{
		cidr_bits = atoi(mask_str);

		/* Check for out of bounds CIDR bits */
		if ((cidr_bits < 0) || (cidr_bits > 32))
			return(0);

		/* Check for special case of num. bits == width of address */
		/* As the nice mask calculation formula doesn't work in that case */
		if (cidr_bits == 32)
			*mask = 0xFFFFFFFF;
		else 
			*mask = htonl(0xFFFFFFFF ^ (0xFFFFFFFF >> cidr_bits));
			
		return(1);
	}

	/* Otherwise, assume a netmask was specified */
	/* Attempt to convert it, returning an error if conversion failed */
	if (inet_aton(mask_str, &i_addr) == 0)
		return(0);
	*mask = i_addr.s_addr;
	
	return(1);
}
