/*
 * @(#)MidiScheduler.cpp 3.00 2 August 2000
 *
 * Copyright (c) 2000 Pete Goodliffe (pete.goodliffe@pace.co.uk)
 *
 * This file is part of TSE3 - the Trax Sequencer Engine version 3.00.
 *
 * This library is modifiable/redistributable under the terms of the GNU
 * General Public License.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; see the file COPYING. If not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */

#include "tse3/util/MidiScheduler.h"

#include "tse3/util/NoteNumber.h"
#include <iomanip>
#include <iostream>

using namespace TSE3;
using namespace TSE3::Util;

/******************************************************************************
 * StreamMidiScheduler class
 *****************************************************************************/

StreamMidiScheduler::StreamMidiScheduler(std::ostream &stream)
: out(stream)
{
    out << "[StreamMidiScheduler::ctor] " << implementationName() << "\n";
}


StreamMidiScheduler::~StreamMidiScheduler()
{
    out << "[StreamMidiScheduler::dtor]\n";
}


const char *StreamMidiScheduler::implementationName()
{
    return "StreamMidiScheduler version 0.10 [dev].";
}


size_t StreamMidiScheduler::ports() const
{
    return 1;
}


const char *StreamMidiScheduler::portName(size_t /*port*/) const
{
    return "StreamMidiScheduler port";
}


const char *StreamMidiScheduler::portType(size_t /*port*/) const
{
    return "StreamMidiScheduler port";
}


bool StreamMidiScheduler::portReadable(size_t /*port*/) const
{
    return false;
}


bool StreamMidiScheduler::portWriteable(size_t /*port*/) const
{
    return true;
}


void StreamMidiScheduler::tx(MidiCommand mc)
{
    out << "[StreamMidiScheduler::tx] ";
    outMidiCommand(mc);
    out << "\n";
}


void StreamMidiScheduler::start(const Clock start)
{
    out << "[StreamMidiScheduler::start] ";
    if (!_running)
    {
        outClock(start);
        out << "\n";
        startClock = start;
        _running   = true;
        notify(&MidiSchedulerListener::MidiScheduler_Started);
    }
    else
    {
        out << "Invalid - already running\n";
    }
}


void StreamMidiScheduler::stop(Clock stopTime)
{
    out << "[StreamMidiScheduler::stop] ";
    if (_running)
    {
        if (stopTime == -1) stopTime = clock();
        outClock(stopTime);
        out << "\n";
        restingClock = stopTime;
        _running     = false;
        notify(&MidiSchedulerListener::MidiScheduler_Stopped);
    }
    else
    {
        out << "Invalid - not running\n";
    }
}


void StreamMidiScheduler::moveTo(const Clock /*moveTime*/, const Clock newTime)
{
    out << "[StreamMidiScheduler::moveTo] ";
    outClock(newTime);
    out << "\n";
    if (_running)
    {
        startClock = newTime;
    }
    else
    {
        restingClock = newTime;
    }
    notify(&MidiSchedulerListener::MidiScheduler_Moved);
}


Clock StreamMidiScheduler::clock()
{
    if (_running)
    {
        startClock += 1;
        return startClock;
    }
    else
    {
        return restingClock;
    }
}


int StreamMidiScheduler::msecs()
{
    return 0;
}


void StreamMidiScheduler::setTempo(const int newTempo,
                                   const Clock changeTime)
{
    out << "[StreamMidiScheduler::setTempo] ";
    outClock(changeTime);
    out << " - " << newTempo << "\n";
}


bool StreamMidiScheduler::eventWaiting()
{
    return false;
}


MidiEvent StreamMidiScheduler::rx()
{
    return MidiEvent();
}


void StreamMidiScheduler::tx(MidiEvent e)
{
    if (e.data.status == MidiCommand_Invalid) return;
    out << "[StreamMidiScheduler::tx] ";
    outClock(e.time);
    out << " - ";
    outMidiCommand(e.data);
    out << "\n";
}


void StreamMidiScheduler::txSysEx(const unsigned char *, size_t size)
{
    out << "[StreamMidiScheduler::txSysEx] " << size << " bytes\n";
}


void StreamMidiScheduler::outClock(Clock c)
{
    out << std::setw(4) << c / Clock::PPQN
        << "."
        << std::setw(3) << c % Clock::PPQN;
}


void StreamMidiScheduler::outMidiCommand(MidiCommand mc)
{
    out << std::hex;
    switch (mc.status)
    {
        case MidiCommand_Invalid:         out << "[Invalid]......."; break;
        case MidiCommand_TSE_Meta:        out << "[TSE Meta]......"; break;
        case MidiCommand_NoteOn:          out << "Note On........."; break;
        case MidiCommand_NoteOff:         out << "Note Off........"; break;
        case MidiCommand_KeyPressure:     out << "Key Pressure...."; break;
        case MidiCommand_ControlChange:   out << "Control Change.."; break;
        case MidiCommand_ProgramChange:   out << "Program Change.."; break;
        case MidiCommand_ChannelPressure: out << "Channel Pressure"; break;
        case MidiCommand_PitchBend:       out << "Pitch Bend......"; break;
        case MidiCommand_System:          out << "System.........."; break;
    }
    out << " c:" << mc.channel
        << " p:"  << mc.port
        << " d1:" << std::setw(2) << mc.data1;
    if (MidiCommand_NoDataBytes[mc.status] == 2)
        out << " d2:" << std::setw(2) << mc.data2;
    switch (mc.status)
    {
        case MidiCommand_NoteOn:
        case MidiCommand_NoteOff:
        case MidiCommand_KeyPressure:
            {
                std::string note = TSE3::Util::numberToNote(mc.data1);
                out << "  (" << note << ")";
            }
            break;
    }
    out << std::dec;
}


/******************************************************************************
 * NullMidiScheduler class
 *****************************************************************************/

NullMidiScheduler::NullMidiScheduler()
{
}


NullMidiScheduler::~NullMidiScheduler()
{
}


const char *NullMidiScheduler::implementationName()
{
    return "NullMidiScheduler version 0.00 [dev].";
}


size_t NullMidiScheduler::ports() const
{
    return 1;
}


const char *NullMidiScheduler::portName(size_t /*port*/) const
{
    return "NullMidiScheduler port";
}


const char *NullMidiScheduler::portType(size_t /*port*/) const
{
    return "NullMidiScheduler port";
}


bool NullMidiScheduler::portReadable(size_t /*port*/) const
{
    return false;
}


bool NullMidiScheduler::portWriteable(size_t /*port*/) const
{
    return true;
}


void NullMidiScheduler::tx(MidiCommand /*c*/)
{
}


void NullMidiScheduler::start(const Clock start)
{
    startClock = start;
    notify(&MidiSchedulerListener::MidiScheduler_Started);
}


void NullMidiScheduler::stop(const Clock /*stopTime*/)
{
    notify(&MidiSchedulerListener::MidiScheduler_Stopped);
}


void NullMidiScheduler::moveTo(const Clock /*moveTime*/, const Clock newTime)
{
    startClock = newTime;
    notify(&MidiSchedulerListener::MidiScheduler_Moved);
}


Clock NullMidiScheduler::clock()
{
    return startClock;
}


int NullMidiScheduler::msecs()
{
    return 0;
}


void NullMidiScheduler::setTempo(const int newTempo, const Clock /*changeTime*/)
{
    _tempo = newTempo;
}


bool NullMidiScheduler::eventWaiting()
{
    return false;
}


MidiEvent NullMidiScheduler::rx()
{
    return MidiEvent();
}


void NullMidiScheduler::tx(MidiEvent /*e*/)
{
}


void NullMidiScheduler::txSysEx(const unsigned char * /*data*/, size_t /*size*/)
{
}

