/* ==================================================== ======== ======= *
 *
 *  ustr.hh
 *  Ubit Project  [Elc][beta1][2001]
 *  Author: Eric Lecolinet
 *
 *  Part of the Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
 *
 *  (C) 1999-2001 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 *
 * ==================================================== [Elc:01] ======= *
 * ==================================================== ======== ======= */

#ifndef _ustr_hh
#define	_ustr_hh
//pragma ident	"@(#)ustr.hh	ubit:b.11.1"

//shortcuts (see notes below)
class UStr& ustr(const char* s = null);
class UTitle& utitle(const char *s);

/* ==================================================== [Elc:01] ======= */
/* === useful functions =============================== ======== ======= */

// creates a duplicate of 's1' (similar to strdup when this function exists)
// - note: argument can be null
char *u_strdup(const char *s1);

// creates a duplicate of 's1' + 's2'
// - note: arguments can be null
char *u_strdupcat(const char *s1, const char *s2);

// creates a duplicate of 's1' followed by 'sep' followed by 's2'
// - note: 'sep' is not added if either 's1' or 's2' is null
// --> this function is quite useful when dealing with pathames
//
char *u_strdupcat(const char *s1, char sep, const char *s2);

// finds the file extension in the pathname
// - note: the returned value is null or points to the extension
//         its is not duplicated (and thus, can't be freed)
const char *u_strext(const char *pathname);

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */

class UStr: public UItem {
protected:
  char *s;
  int  len;
  class UEditable *model; //test
public:
  static  const UClass  uclass;
  virtual const UClass* getClass() const {return &uclass;}
  virtual class UStr*   strCast()  {return this;}
  
  // goBOL = go to Beginning Of Line (if needed)
  static UStr newline, goBOL;

  // NOTE on UStr callbacks: 
  // the following callbacks are fired when object's value is changed:
  // 1. UOn::change      callbacks of THIS object
  // 2. UOn::changeItem  callbacks of its PARENTS
  // 3. UOn::changeStr   callbacks of its PARENTS

  UStr(const UStr&);
  UStr(const char* = null, u_modes bmodes = 0);
  virtual ~UStr();

  virtual void set(const UStr&, u_bool update = true);
  virtual void set(const UStr*, u_bool update = true);
  virtual void set(const char*, u_bool update = true);
  //resets to empty string
  virtual void reset(u_bool upd = true);

  UStr& operator=(const UStr&_s) {set(_s); return *this;}
  UStr& operator=(const char*_s) {set(_s); return *this;}

  // noms modifies pour eviter ambiguites
  virtual void setInt(long, u_bool update = true);
  virtual void setFloat(double, u_bool update = true);

  u_bool equals(const UStr&) const;
  u_bool equals(const UStr*) const;
  u_bool equals(const char*) const;

  // strings are compared in the lexicographical order
  int compareTo(const UStr& s2) const;
  int compareTo(const UStr* s2) const;
  int compareTo(const char* s2) const;
  friend int compare(const UStr& s1, const UStr& s2);

  // returns the length of the string
  int length() const {return len;}

  // returns the internal char* representation
  // -- should not be freed nor modified!
  const char *chars() const {return s;}

  // returns a duplicate of the internal char* representation
  // -- must be freed by calling free()
  char *get() const;

  // same as get() but returns 'nbchars' characters from offset 'pos'
  char *getN(int pos, int nbchars) const;

  // returns character at offset 'pos' (-1 means last char of string)
  // -- returns 0 if postion does not exist
  char charAt(int pos) const;

  // changes character at offset 'pos' (-1 means last char of string)
  // -- returns false if char can't be inserted (wrong position 
  //    or invalid newchar)
  u_bool setCharAt(int pos, char newchar, u_bool upd = true);

  // appends s2 to the end of this UStr
  // -- s2 is not changed
  virtual void append(const UStr& s2, u_bool update = true);
  virtual void append(const UStr* s2, u_bool update = true);
  virtual void append(const char* s2, u_bool update = true);

  // inserts s2 at the given position in this UStr
  // -- s2 is not changed
  // -- pos== 0 means "beginning of string"
  // -- pos==-1 means 'end of string'
  virtual void insert(int pos, char newchar, u_bool update = true);
  virtual void insert(int pos, const UStr& s2, u_bool update = true);
  virtual void insert(int pos, const UStr* s2, u_bool update = true);
  virtual void insert(int pos, const char* s2, u_bool update = true);

  // removes 'nbchars' characters from offset 'pos'
  virtual void remove(int pos, int nbchars, u_bool update = true);
  //obsolete synonym
  virtual void removeN(int pos, int nbchars, u_bool update = true);

  // replaces 'nbchars' characters from offset 'pos' by 's2'
  // -- s2 is not changed
  // -- this function combines insert() and remove()
  virtual void replace(int pos, int nbchars, const UStr& s2, u_bool upd = true);
  virtual void replace(int pos, int nbchars, const UStr* s2, u_bool upd = true);
  virtual void replace(int pos, int nbchars, const char* s2, u_bool upd = true);

  // splits the UStr into two separate UStr
  // returns a new UStr containing the remaining chars after pos
  // (and null if no chars are found)
  virtual UStr* split(int pos, u_bool update = true);

  // checks input conformance to format model (if any)
  virtual u_bool checkFormat(int pos, char);
  virtual u_bool checkFormat(int pos, const char*);
  // syncrhonizes textual and model values
  virtual void syncVals(char*,int);

  // updates all parents
  virtual void update();

  // updates grahics if arg is true the fires object's UOn::change callbacks 
  // of this object then fires UOn::changeItem AND UOn::changeStr callbacks 
  //of its parents
  virtual void changed(u_bool update_now);

  //==== Ubit Intrinsics

  //!Watch: length must be the exact string length
  virtual u_bool setImpl(const char*, int length, u_bool chgd, u_bool upd);
  virtual u_bool insertImpl(int pos, char newchar, u_bool upd);
  virtual u_bool insertImpl(int pos, const char* s2,  u_bool upd);
  virtual u_bool replaceImpl(int pos, int nbchars, const char* s2, u_bool upd);

  virtual void getSize(UContext*, u_dim *w, u_dim *h) const;
  virtual void paint(UWinGraph&, UContext*, const URegion &r) const;

  //prototypes for warped text
  virtual void getSize(UContext*, u_dim maxwidth, u_dim chw, 
		       class UFlowCell*) const;
  virtual void paint(UWinGraph&, UContext*, const URegion &r,
		     class UFlowCell*) const;
  virtual void paint2(UWinGraph&, UContext*, const URegion &r,
		      int offset, int cellen) const;
};

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */

class UTitle: public UStr {
  friend class UBox;
public:
  static  const UClass  uclass;
  virtual const UClass* getClass() const {return &uclass;}
  
  UTitle(const char* = null);
  virtual void update();

  //package_private: ===[internal implementation]=========================
  virtual void getSize(UContext*, u_dim *w, u_dim *h) const;
  virtual void paint(UWinGraph&, UContext*, const URegion &r) const;
  //prototypes for warped text
  virtual void getSize(UContext*, u_dim maxwidth, u_dim chw, 
		       class UFlowCell*) const;
  virtual void paint(UWinGraph&, UContext*, const URegion&, 
		     class UFlowCell*) const;
};

#endif
/* ==================================================== [TheEnd] ======= */
/* ==================================================== [Elc:01] ======= */
