/* ==================================================== ======== ======= *
 *
 *  uview.hh
 *  Ubit Project  [Elc][beta1][2001]
 *  Author: Eric Lecolinet
 *
 *  Part of the Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
 *
 *  (C) 1999-2001 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 *
 * ==================================================== [Elc:01] ======= *
 * ==================================================== ======== ======= */

#ifndef _uview_hh
#define	_uview_hh
//pragma ident	"@(#)uview.hh	ubit:001.11.6"
#include <uctrl.hh>


class UViewStyle : public UMode {
public:
  static  const UClass  uclass;
  virtual const UClass* getClass() const {return &uclass;}

  UViewStyle(UView* (*)(UBoxLink*, UView*, UWinGraph*), u_modes = 0);
  virtual ~UViewStyle() {clean();}

  //package_private: ====[internal implementation]=======================

  virtual void update();
  virtual void addingTo(ULink *selflink, UGroup *parent);
  //NB: removingFrom() requires a destructor to be defined
  virtual void removingFrom(ULink *selflink, UGroup *parent);

  // pointer to the corresponding U***View::makeView "static" constructor
  UView* (*makeView)(UBoxLink*, UView* parview, UWinGraph*);
};

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */

class UView : public URegion {
  friend class UViewImpl;
  friend class UUpdateImpl;
public:
  enum {HFLEX, HCENTER, LEFT, RIGHT};
  enum {VFLEX, VCENTER, TOP, BOTTOM};
  enum {VERTICAL, HORIZONTAL};
  enum VMODES {
    SHOWN      = 1<<1,
    // this view has been damaged and must be repaint
    DAMAGED      = 1<<2,
    // this is transparent (thus, its parents must be redrawn)
    TRANSPARENT  = 1<<3,
    // this view has a fixed width and/or height
    FIXED_WIDTH  = 1<<4,
    FIXED_HEIGHT = 1<<5,
    // this view is shared (this is actually a UWin view)
    WIN_SHARED   = 1<<6,
    // coordonates are obsolete because this (floating) view is being moved
    // (note that coords are updated by doUpdate/setCoords)
    OBSOLETE_COORDS = 1<<7
   //!BEWARE: no comma after last item!
  };
protected:
  UView     *parview; 		// parent View
  UBoxLink  *boxlink;		// corresponding Box Link
  UWinGraph *wingraph;		// Window Graphics
  // View extensions: used by certain views when useful
  // (for viewMove and viewResize callbacks etc.)
  class UViewExt *ext;
public:
  u_dim     favoriteWidth, favoriteHeight;
  int       vmodes;

  static  UViewStyle style;  // renderer
  virtual UViewStyle* getViewStyle() {return &style;}

  // "static" constructor used by UViewStyle to make a new view
  static UView* makeView(UBoxLink*, UView* parview, UWinGraph*);

  UView(UBoxLink*, UView* parview, UWinGraph*);
  virtual ~UView();

  // changes the View modes
  int getVmodes() const {return vmodes;}
  void setVmodes(int m, u_bool onoff) {
    if (onoff) vmodes = vmodes | m; else vmodes = vmodes & ~m;
  }

  // is ONE of these vmodes verified ?
  u_bool isDef(int some_Vmodes) const {
    return ((vmodes & some_Vmodes) != 0);
  }
  // are ALL these vmodes verified ?
  u_bool isAllDef(u_modes all_Vmodes) const {
    return ((vmodes & all_Vmodes) == all_Vmodes);
  }

  // returns the corresponding Box
  UBox *getBox() const;

  // NOTE: getContainingParent() is meaningless for Window views
  UBox* getContainingParent() const;
  UView* getParentView() const {return parview;}
  UDisp* getDisp()    const;
  UAppli* getAppli()  const;
  u_bool isRealized() const;

  // return the location of this view relatively to its PARENT
  // - Note: return 0 if this view is a WINDOW View.
  u_pos getX() const;
  u_pos getY() const;
  u_bool where(u_pos &x, u_pos &y) const;

  // return the location of this view relatively to its enclosing WINDOW
  // - Note: return 0 if this view is a WINDOW View.
  u_pos getXwin() const {return x;}
  u_pos getYwin() const {return y;}
  u_bool whereInWin(u_pos &x, u_pos &y) const;

  // return the location of this view on the screen
  u_bool whereOnScreen(u_pos &x, u_pos &y) const;

  // size of this View
  u_dim getWidth()  const {return width;}
  u_dim getHeight() const {return height;}
  void  getSize(u_dim &w, u_dim &h) const {w = width; h = height;}

  // favorite size of this View
  u_dim getFavoriteWidth()  const;
  u_dim getFavoriteHeight() const;

  // changes the size of this view (when applicable)
  // note: no immediate effect (will be taken into account when redisplayed)
  void setWidth(u_dim width);
  void setHeight(u_dim height);
  void resize(u_dim width, u_dim height);

  // NOTE: as layout is performed automatically, it is not possible to set 
  // the position of Views directly.
  // Use the 'UPos' brick to locate UBox(es) at a given position

  // convert a location in view coordinates to window coordinates
  u_pos XToXwin(u_pos x_in_view) {return x_in_view + x;}
  u_pos YToYwin(u_pos y_in_view) {return y_in_view + y;}

  // convert a location in window coordinates to view coordinates
  u_pos XwinToX(u_pos x_in_win) {return x_in_win - x;}
  u_pos YwinToY(u_pos y_in_win) {return y_in_win - y;}

  //package_private: ====[Ubit Intrinsics]==============================

  UWin*  getHardwin() const;
  UView* getHardwinView() const;

  // Returns the Window Graphics: 
  // !CAUTION: 
  // -- NEVER delete this object !
  // -- ALWAYS call its begin() method BEFORE drawing graphics 
  //    and its end() method when finished

  UWinGraph* getWinGraph() const {return wingraph;}
  UWinGraph& wg() const {return *wingraph;}

  // update view paint in VIEW coordinates:
  // - 'localRegion' is relative to View coordinates
  //   a null value means: "repaint the whole view"
  // - 'doubleBuffering' : paints in double buffering

  void updatePaint(URegion* localRegion=null, u_bool doubleBuffering=false);

  // update view paint in WINDOW coordinates:
  // - 'windowRegion' is relative to Window coordinates
  // - 'doubleBuffering' : paints in double buffering
  // - 'updateAllLayers' : paints all layers located beneath the view
  //   (this in necessary in case of transparent views)
  // - otherwise, if updateAllLayers is false : paints from 'firstPaintedLayer' 
  //   (if it is not null) then (always) paints 'this' view

  void updateWinPaint(const URegion &windowRegion, u_bool doubleBuffering, 
		      UView *firstPaintedLayer, u_bool updateAllLayers);

  // update internal data but don't repaint (must be called after scrolling
  // and in various cases for initializing the views' content)

  int updateWinData(const URegion &region);

  UBoxLink *getBoxLink() {return boxlink;}
  void _setXwin(u_pos x_in_win);
  void _setYwin(u_pos y_in_win);
  void _setParentView(UView *_parview) {parview = _parview;}

  virtual u_bool doLayout(UContext&, class UViewLayout&);
  virtual void doUpdate(UContext&, URegion r, URegion clip, class UViewUpdate&);

  // locateSource either searches:
  // - 'searchedView'
  // - the visible view where (e->xmouse, e->ymouse) is located
  // this function sets:  e->{source, sourceView, sourceProps, redrawnClip}

  virtual UView* locateSource(UEvent*, UContext*, URegion clip, 
			      UView *searchedView);
  //Note: contains() modifies the mofifies clip
  virtual UView* contains(u_pos xmouse, u_pos y_mouse, URegion &clip);
  virtual UView* contains(UView *searchv, URegion &clip);

  //NB: NOT virtual:
  u_bool locateItemVPos(UContext&, ULink*, class UWinGraph &g, 
				const URegion&, class UViewUpdate&);
  u_bool locateItemHPos(UContext&, ULink*, class UWinGraph &g, 
				const URegion&, class UViewUpdate&);
  u_bool locateItemPtr(UContext&, ULink*, class UWinGraph &g, 
				const URegion&, class UViewUpdate&);
};

#endif
/* ==================================================== [TheEnd] ======= */
/* ==================================================== [Elc:01] ======= */

