/* ==================================================== ======== ======= *
 *
 *  uufinder.cpp  [part of uubrowser]
 *  Ubit Project [Elc][2003]
 *  Author: Eric Lecolinet
 *
 *  Part of the Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
 *
 *  (C) 1999-2003 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 *
 * ==================================================== [Elc:03] ======= *
 * ==================================================== ======== ======= */

//pragma ident	"@(#)uufinder.cpp	ubit:03.06.00"

#include <iostream>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <X11/keysym.h>
#include <ubit/ubit.hpp>
#include <ubit/udir.hpp>
#include <ubit/ext/ubrowser.hpp>
#include <ubit/ext/ufinder.hpp>
#include <ubit/ext/udoc.hpp>
using namespace std;

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

BrwDir::BrwDir(UBrowser* brw, const UStr& _fpath) {
  fpath = _fpath;
  fname = _fpath.getFileName();
  iconbox = null;
  keep_open = false;

  popmenu.addlist
    (
     ualpha(0.5) + UBgcolor::black
     + UColor::white + UFont::bold + UBorder::none
				// read again if already opened
     + ubutton("     ReOpen " + UColor::yellow + fpath
	       + ucall(brw, this, true, &UBrowser::openDir))
     + ubutton("     Iconify Folder"
	       + ucall(brw, this, &UBrowser::iconifyIconbox))
     + ubutton("     Close Folder"
	       + ucall(brw, this, true, &UBrowser::removeIconbox))
     + usepar()
     + ubutton(UPix::cross + " Remove Folder" 
	       + ucall(brw, this, &UBrowser::removeFromHistory))
     );
  popmenu.setSoftwinMode();

  addlist(font + " " + USymbol::right + " " + fname
	  + popmenu  
				// do not read again if already opened
	  + UOn::action / ucall(brw, this, false, &UBrowser::openDir)
				// show info in the status bar
	  + UOn::enter  / ucall(brw, this, &UBrowser::showDirInfo)
	  + UOn::leave  / ucall(brw, (BrwDir*)0, &UBrowser::showDirInfo)
	  );

  // right mouse button opens the menu
  popmenu.autoOpens(*this, UEvent::MButton3, 0);
}
 
BrwDir::~BrwDir() {
}

const UStr& BrwDir::getDir() const 
{return fpath;}

class UIconbox* BrwDir::getIconbox() const 
{return iconbox;}

void BrwDir::setDir(const UStr& _fpath) 
{fpath = _fpath;}

void BrwDir::setIconbox(class UIconbox* _ibox) {
  if (iconbox != _ibox) {
    emph(false);
    iconbox = _ibox;
  }
}

void BrwDir::emph(bool state) {
  if (state) font.set(UFont::bold);
  else font.set(UFont::_bold);
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */
// HISTORY

BrwDir* UBrowser::findDir(const UStr& name) {
  BrwDir* found = null;
  int count;
  UBrick** children = ctrls->history_list.getChildren(count);
  
  if (children) {
    for (int k = 0; k < count; k++) {
      BrwDir* de = dynamic_cast<BrwDir*>(children[k]);
      if (de && de->fpath.equals(name)) found = de;
    }
    delete[] children;
  }

  return found;
}

/*
BrwDir* UBrowser::findDir(const UIconbox* ibox) {
  BrwDir* found = null;
  int count;
  UBrick** children = ctrl->history_list.getChildren(count);
  
  if (children) {
    for (int k = 0; k < count; k++) {
      BrwDir* de = dynamic_cast<BrwDir*>(children[k]);
      if (de && de->iconbox == ibox) found = de;
    }
    delete[] children;
  }

  //setDirMode();
  return found;
}
*/

void UBrowser::removeIconbox(BrwDir* de, bool upd) {
  if (!de) return;
  if (de->iconbox) {
    if (de->iconbox == selected_iconbox) selected_iconbox = null;

    // pour eviter plantages si on se detruit soi-meme:
    // le directory sera detruit au coup suivant
    kill_ring = de->iconbox;
   
    finder.remove(de->iconbox, true, upd);
    de->iconbox = null;    // uptr : automatic deletion
    de->emph(false);
  }
}

void UBrowser::addToHistory(BrwDir* de) {
  // if not already in list
  ctrls->history_list.remove(de, true/*UGroup::QUIET_DEL*/, false);
  ctrls->history_list.insert(0, de);
  last_direntry = de;
}

void UBrowser::removeFromHistory(BrwDir* de) {
  if (last_direntry == de)  last_direntry = null;
  removeIconbox(de, true);
  ctrls->history_list.remove(de, true);
}

void UBrowser::showDirInfo(BrwDir* de) {
  //....
}

void UBrowser::iconifyIconbox(BrwDir* de) {
  if (de->iconbox) {
    de->iconbox->iconify(!de->iconbox->isIconified());
  }
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

void UBrowser::showDoc(const UStr& pathname, UDoc *doc) {
  doc_spane->setScroll(0,0);
  actionNotify(DocShow, pathname, doc);

  if (mode != DocMode) {
    mode = DocMode;
    main_panel.remove(finder, true, false);
    main_panel.add(doc_panel, false);
    ctrls->showDoc();
    //update(); inutile?
  }
  else ctrls->showSelectedIcon();
}

void UBrowser::showDir(const UStr& pathname) {
  if (getFinderSpane()) getFinderSpane()->setScroll(0,0);
  actionNotify(DirShow, pathname, null);

  if (mode != DirMode) {
    mode = DirMode;
    ctrls->showDir();
    main_panel.remove(doc_panel, true, false);
    main_panel.add(finder, false);
    update();  //necessaire
  }
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

int UBrowser::openDoc(const UStr& pathname) {
  int stat = 0;

  showAlert(false);

  actionNotify(DocRequest, pathname, null);
  UDoc::Errors errs;
  UDoc* doc = UDoc::readDoc(pathname, link_callbacks, &errs);
  actionNotify(DocOpen, pathname, doc);
  
  if (doc && errs.stat > 0) {
    docbox.setContent(pathname, *doc);
    showDoc(pathname, doc);
  }
  return stat;
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

int UBrowser::openDir(const UStr& pathname, bool reload) {
  int stat = false;
  UStr fpath = pathname;
  UDir::parseDir(fpath);  // normaliser

  if (fpath.empty()) return false;

  showAlert(false);

  // remove the final slash (except for /)
  if (!fpath.equals("/") && fpath.charAt(-1) == '/') fpath.remove(-1, 1);

  // check if dentry already in the history
  BrwDir* dentry = findDir(fpath);

  if (!dentry) {
    dentry = new BrwDir(this, fpath);
    stat = openDir(dentry, true);  // load dir
  }
  else {
    stat = openDir(dentry, reload);
  }
  
  if (dentry->iconbox) addToHistory(dentry);
  else delete dentry;
  return stat;
}

/* ==================================================== ======== ======= */

int UBrowser::openDir(BrwDir* de, bool reload) {
  if (!de) return false;
  int stat = false;
  // conserver scale courant sinon default=0
  int scale = 0;

  showAlert(false);

  actionNotify(DirRequest, de->fpath, null);
  docbox.clearContent();

  if (last_direntry && last_direntry->iconbox) {
    scale = last_direntry->iconbox->getScale()->getValue();
  }
  
  if (!reload && de->iconbox) {
    de->iconbox->iconify(false);
    stat = true;
    actionNotify(DirOpen, de->fpath, null);
    goto end;
  }
  
  // fermer le precedent (sauf si KeepOpen)
  if (last_direntry && !last_direntry->isKeepOpen())  {
    removeIconbox(last_direntry, false);
    last_direntry = null;
  }

  // detruire l'ancien iconbox s'il existe
  if (de->iconbox) removeIconbox(de, false);
  
  if (!de->fpath.isFileFound(false)) {
    showAlert("Can't Open Folder: " & de->fpath);
    goto end;
  }

  actionNotify(DirOpen, de->fpath, null);

  // create a new iconbox in the navigator
  de->iconbox = new UIconbox(de->fpath);
  de->iconbox->showIconifyButton(false);
  de->iconbox->showCloseButton(false);
  // tout de suite sinon flicke ensuite
  de->iconbox->setScale(scale);
  
  de->iconbox->addlist 	// add iconbox callbacks
    (
     UOn::select / ucall(this, &de->iconbox, &UBrowser::iconSelectCB)
     + UOn::action / ucall(this, &de->iconbox, &UBrowser::iconActionCB)
     
     // called when the iconbox is closed (the user clicked the "x" btn)
     // this will delete the iconbox
     + UOn::close / ucall(this, de, true, &UBrowser::removeIconbox)
     + opts.iconbox_hspacing
     + opts.iconbox_vspacing
     );
  
  stat = de->iconbox->readDir(de->fpath, de->fpath);
  
  if (stat == UFilestat::Opened) {    
    finder.insert(0, de->iconbox, true);  //update = true
    de->emph(true);
    last_direntry = de;
    
    ctrls->preview.clearContent();
    selected_iconbox = de->iconbox;
    //selected_ima = null;

    if (opts.show_icon_previews) {      // charger les imagettes
      pos.reset();
      UTimer* t = UAppli::openTimer(100, -1);
      t->onAction(ucall(this, t, &selected_iconbox, &UBrowser::showIconPreviews));
    }
  }
  else {
    if (stat == UFilestat::CannotOpen) {
      showAlert("Can't Read Folder: " & de->fpath);
    } 
  }

 end:
    if (stat > 0)  {
      de->iconbox->setScale(scale);
      showDir(de->fpath);
    }
  return stat;
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

UScrollpane* UBrowser::getFinderSpane() {
  return selected_iconbox ? selected_iconbox->getSpane() : null;
}

UIconbox* UBrowser::getSelectedIconbox() const {
  return selected_iconbox;
}

UIcon* UBrowser::getSelectedEntry() const {
  return selected_iconbox ? selected_iconbox->getSelectedIcon() : null;
}

void UBrowser::previousEntry() {
  if (!selected_iconbox) return;

  UIcon* current_icon = selected_iconbox->getSelectedIcon();
  UIcon* prev_icon = selected_iconbox->getPreviousIcon();

  if (mode == DocMode) {             // sauter les dirs
    while (prev_icon && prev_icon->isDir()) {

      UIcon* prev_icon2 = selected_iconbox->getPreviousIcon();
      if (prev_icon == prev_icon2) {
        prev_icon = current_icon;         // cas que des dirs
        break;
      }
      else prev_icon = prev_icon2;
    }
  }

  if (prev_icon) {
    selected_iconbox->selectIcon(prev_icon, false);
    if (mode == DocMode) {
      UStr pathname = selected_iconbox->getPath() & "/" & prev_icon->getName();
      open(pathname);
    }
  }
}

void UBrowser::nextEntry() {
  if (!selected_iconbox) return;

  UIcon* current_icon = selected_iconbox->getSelectedIcon();
  UIcon* next_icon = selected_iconbox->getNextIcon();

  if (mode == DocMode) {
    while (next_icon && next_icon->isDir()) {

      UIcon* next_icon2 = selected_iconbox->getNextIcon();
      if (next_icon == next_icon2) {
        next_icon = current_icon;         // cas que des dirs
        break;
      }
      else next_icon = next_icon2;
    }
  }

  if (next_icon) {
    selected_iconbox->selectIcon(next_icon, false);
    if (mode == DocMode) {
      UStr pathname = selected_iconbox->getPath() & "/" & next_icon->getName();
      open(pathname);
    }
  }
}

void UBrowser::openEntry() {
  if (!getSelectedEntry()) return;

  if (open_in_fullwin) showFullWin();  // et si pas un dir
  else {
    UStr pathname =
    selected_iconbox->getPath() & "/" & getSelectedEntry()->getName();
    open(pathname);
  }
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */
// Callbacks

void UBrowser::iconActionCB(UIconbox* ibox) {
  if (selected_iconbox && selected_iconbox != ibox) {
    selected_iconbox->selectIcon(null);
    // must be reloaded
    //selected_ima = null;  // uptr : automatic deletion
  }
  openEntry();
}

/* ==================================================== ======== ======= */

void UBrowser::iconSelectCB(UIconbox* ibox) {
  if (selected_iconbox && selected_iconbox != ibox) {
    selected_iconbox->selectIcon(null);
    // must be reloaded
    //selected_ima = null;  // uptr : automatic deletion
  }

  selected_iconbox = ibox;
  showPreviewRequest(getSelectedEntry());

  if (open_in_fullwin && fullwin->isShown()) showFullWin();
}

/* ==================================================== ======== ======= */

void UBrowser::showPreviewRequest(UIcon* icon) {
  if (!icon) return;
  
  // afficher preview sauf si deja selectionne 
  if (icon != last_preview_request) {
    last_preview_request = icon;

    UTimer* t = UAppli::openTimer(1, 1);
    t->onAction(ucall(this, icon, &UBrowser::showPreview));
  }
}

/* ==================================================== ======== ======= */

void UBrowser::showPreview(UIcon* icon) {
  if (!icon) return; 

  if (icon != last_preview) {
    last_preview = icon;
    UStr pathname = selected_iconbox->getPath() & "/" & icon->getName();
    ctrls->preview.readContent(icon->getName(), pathname);
  }
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

void UBrowser::showIconPreviews(UTimer* t, UIconbox* iconbox) {

  if (!iconbox || iconbox != selected_iconbox)  {
    t->close();
    return;
  }

  UIcon* icon = null;
  while ((icon = iconbox->getIcon(pos))) {
    UStr pathname = iconbox->getPath() & "/" & icon->getName();
    if (icon->readContent(icon->getName(), pathname) >= UFilestat::Opened)
      //icon->update(UUpdate::paint);
    return;  // !ne pas enlever!
  }

  if (!icon) t->close();
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

BrwFullWin::~BrwFullWin() {
}

BrwFullWin::BrwFullWin(UBrowser* _brw) :
  brw(*_brw),
  content(uwidth(), uheight(), brw.opts.background + brw.opts.fgcolor)
{
  winsize = MediumWinSize;
  resizing = false;

  height.setAutoUpdate(false);
  width.setAutoUpdate(false);
  
  UBox& controls = ubar
    (
     uhcenter() + UFont::bold + UBorder::none
     + uhspacing(0)
     + ubutton(UPix::left  + ucall(_brw, &UBrowser::previousEntry))
     + ubutton(UPix::right + ucall(_brw, &UBrowser::nextEntry))
     + "   "
     + ubutton("Small"  + ucall(this, SmallWinSize, &BrwFullWin::setSize))
     + ubutton("Medium" + ucall(this, MediumWinSize,&BrwFullWin::setSize))
     + ubutton("Large"  + ucall(this, LargeWinSize, &BrwFullWin::setSize))
     + ubutton("Screen" + ucall(this, FullWinSize,  &BrwFullWin::setSize))
     + "   "
     + ubutton("Close" + ushow(this, false))
     );
  
  // un double click ferme la fenetre
  content.add(UOn::mbiclick / ushow(this, false));
  content.add(UOn::kpress / ucall(this, &BrwFullWin::keyPressed));
  
  addlist
    (
     utitle("Full View")
     + width + height
     + brw.opts.background + brw.opts.fgcolor
     + UOn::viewResize / ucall(this, &BrwFullWin::resized)
     + uhflex()
     + uvflex()  + content
     + ubottom() + controls
     );
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

void BrwFullWin::setSize(WinSize sz) {
  if (winsize == sz) return;
  resizing = true;
  winsize = sz;
  
  switch(sz) {
  case SmallWinSize: 
    width.set(400);
    height.set(400);
    resize(400,400);
    break;
  case MediumWinSize: 
    width.set(600);
    height.set(600);
    resize(600,600);
    break;
  case LargeWinSize: 
    width.set(800);
    height.set(800);
    resize(800,800);
    break;
  case FullWinSize: 
    UAppli* a = getAppli();
    width.set(a->getScreenWidth());
    height.set(a->getScreenHeight());
    resize(a->getScreenWidth(), a->getScreenHeight());
    moveOnScreen(0, -10);
    break;
  }
  
  rescale();		// and redisplay
  resizing = false;
}

/* ==================================================== ======== ======= */

void UBrowser::showFullWin() {
  /*
  if (!selected_iconbox || !getSelectedIcon()) return;

  if (getSelectedIcon() == selected_fullwin) {
    // image already displayed; just reopen window
    // NB: si on refait show ca flicke par contre ca permet de remettre
    // au 1er plan !
    if (!fullwin->isShown()) fullwin->show(true);
    //BUG cyclage! fullwin.show(true);
  }
  
  else if (!selected_ima)
    iconActionCB(selected_iconbox);   // reload image !!! ABSURDE A REVOIR
  else {
    selected_fullwin = getSelectedIcon();
    fullwin->rescale();
  }
  */
}

/* ==================================================== ======== ======= */

void BrwFullWin::rescale() {
  /*
  if (!brw.getSelectedIma()) return;

  // ceci pour interdire le traitement de callbacks recus par fullviewResized
  // et qui seront provoques par les actions qui suivent
  resizing = true;

  // juste rescale from selected_ima
  UIma* scaled_ima = 
    brw.rescaleIma(brw.getSelectedIma(), width.get(),
		   height.get() - brw.opts.icon_vmargin.get());
  
  // display scaled image and (re)opens window
  content.set(scaled_ima, brw.getSelectedIcon()->getName());

  if (!isShown()) show(true);
  resizing = false;
  */
}

/* ==================================================== ======== ======= */

void BrwFullWin::resized(UEvent& e) {
  if (resizing) return;

  width.set(e.getView()->getWidth());
  height.set(e.getView()->getHeight());

  rescale(); // redisplay
}

/* ==================================================== ======== ======= */

void BrwFullWin::keyPressed(UEvent& e) {
  switch(e.getKeySym()) {
  case XK_Escape:
    show(false);
    break;
  case XK_Left:
  case XK_Page_Up:
    brw.previousEntry();
    break;
  case XK_Right:
  case XK_Page_Down:
    brw.nextEntry();
    break;
  }
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */
