/* ==================================================== ======== ======= *
 *
 *  uuflow.cpp
 *  Ubit Project  [Elc][2003]
 *  Author: Eric Lecolinet
 *
 *  Part of the Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
 *
 *  (C) 1999-2003 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 *
 * ==================================================== [Elc:03] ======= *
 * ==================================================== ======== ======= */

//pragma ident	"@(#)uuflow.cpp		ubit:03.05.03"
#include <iostream>
#include <udefs.hpp>
#include <ubrick.hpp>
#include <ucall.hpp>
#include <uerror.hpp>
#include <uwin.hpp>
#include <umenu.hpp>
#include <umenuImpl.hpp>
#include <uappli.hpp>
#include <uflow.hpp>
#include <ugraph.hpp>
#include <unatdisp.hpp>
#include <utextsel.hpp>
#include <uboxImpl.hpp>
#include <umenu.hpp>
#include <uima.hpp>
#include <uborder.hpp>
#include <ucond.hpp>
using namespace std;


UFlow::UFlow(int flow_id, UAppli& _a, UDisp& _disp) :
  appli(_a),
  disp(_disp),
  id(flow_id),
  lastEnteredEvent(0,null,null,null), 
  lastPressedEvent(0,null,null,null),
  menuCtrl(*new UMenuCtrl(this)),
  textsel(*new UTextsel(_a.conf.default_selection_color, 
			_a.conf.default_selection_bgcolor,
			_a.conf.default_selection_font, 
			true))
{
  lastEnteredView = null;
  lastEnteredBox= null;
  lastPressed   = null;
  lastArmed     = null;
  dragSource    = dropTarget = null;

  lastCursor    = null;
  browsing_group  = null;

  mclickTime    = 0;
  mclickCount   = 0;
  mclickX = mclickY = -5;
}

UFlow::~UFlow() {
  cerr << "UFlow::~UFlow " << this << endl;
  // ici il faudrait notifier le UDisp !!!
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

void UFlow::deleteNotify(UView* deleted_view) {
  if (deleted_view == lastEnteredView) {
    //NB: lastEnteredBox always corresponds to lastEnteredView
    if (lastEnteredBox == lastArmed)   lastArmed  = null;
    if (lastEnteredBox == lastPressed) lastPressed = null;
    if (lastEnteredBox == dragSource)  dragSource = null;
    if (lastEnteredBox == dropTarget)  dropTarget = null;
    if (lastEnteredBox == textsel.getObj()) textsel.setObj(null);
    lastEnteredView = null;
    lastEnteredBox  = null;
  }
}

void UFlow::deleteNotify(UGroup *deleted_group) {
  if (deleted_group == lastEnteredBox) {
    lastEnteredView = null;
    lastEnteredBox  = null;
  }
  if (deleted_group == lastArmed)   lastArmed  = null;
  if (deleted_group == lastPressed) lastPressed = null;
  if (deleted_group == dragSource)  dragSource = null;
  if (deleted_group == dropTarget)  dropTarget = null;
  if (deleted_group == textsel.getObj()) textsel.setObj(null);
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

void UFlow::setCursor(UEvent& e, const UCursor* c) {
  if (id == 0) {            // Cursor pas gere pour iflowID > 0
    if (c != lastCursor) {
      lastCursor = c;
      e.sourceView->wg().setCursor(lastCursor);
    }
  }
}

/* ==================================================== ======== ======= */

static const char* telepointer_xpm[] = {
  /*
" 9 9 2 1",
". c white",
"# c None s None",
".........",
".#######.",
".######..",
".#####...",
".#####...",
".######..",
".##..###.",
".#....##.",
"........."
  */
" 8 8 2 1",
". c white",
"# c None s None",
"........",
".#######",
".######.",
".#####..",
".#####..",
".######.",
".##..##.",
".#.....#",
};
static UPix telepointer_pix(telepointer_xpm, UMode::UCONST);

UWin* UFlow::retrieveRemPointer(UDisp* rem_disp)  {
  unsigned int disp_id = rem_disp->getID();

  if (disp_id < rempointers.size() && rempointers[disp_id])
    return rempointers[disp_id];

  else {
    for (unsigned int k = rempointers.size(); k <= disp_id; k++) 
      rempointers.push_back(null);

    UWin* ptr = 
      new UMenu
      (
       uhmargin(0) + uvmargin(0) + UBorder::none
       + UMode::ignoreEvents 
       + UColor::red
       + UBgcolor::red
       + UOn::select/UBgcolor::green
       + telepointer_pix
       );

    rempointers[disp_id] = ptr;

    ptr->setSoftwinMode(false);
    rem_disp->add(ptr);
    ptr->show();
    return ptr;
  }
}

void UFlow::showRemPointers(UEvent& e, int mode)  {
  UBox* source = e.getSource() ? e.getSource()->boxCast() : null;
  if (!source) return;

  //cerr << "winmm : flow "  << e.getFlow() << " / disp " << e.getDisp()
  //     << " / event pos " << e.getX() << " " << e.getY()<<endl;
  
  for (ULinkLink* ll = source->getParentList().first(); 
       ll != null; ll = ll->getNext()) {
    // UBoxLink* par construction
    UBoxLink *link = static_cast<UBoxLink*>(ll->link());
    
    for (UView* v = link->getViews(); v != null; v = v->getNext())
      if (v->getDisp() != e.getDisp()){
	//cerr <<"move : disp " << v->getDisp() << " / view " << v 
	//     << " / view pos " << v->getXscreen() << " " << v->getYscreen()
	//     << " / event pos " << e.getX() << " " << e.getY()<<endl<<endl;
	//pseudo_mouse->move(v, e.getX(), e.getY());

	UWin* remptr = retrieveRemPointer(v->getDisp());
	if (remptr){
	  if (mode) {
	    if(mode> 0) remptr->select(true);
	    else if(mode< 0) remptr->select(false);
	    remptr->update();
	  }
          remptr->toFront();
	  remptr->moveOnScreen(v->getXscreen() + e.getX(), 
			       v->getYscreen() + e.getY());
	}
      }
  }
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

void UFlow::winDestroy(UWin *win, UEvent& e) {
  win->setCmodes(UMode::CAN_SHOW, false);
  if (menuCtrl.isActive(win))  menuCtrl.closeAllMenus(true);
  delete win;
}

void UFlow::winLeave(UWin* win, UEvent& e) {
  // ATTENTION:
  // le XGrab genere des LeaveWindow qunad on ouvre le menu associe
  // a un bouton et qunad on bouge la souris sur ce bouton une fois
  // que le menu est ouvert. ne me demandez pas pourquoi, mais en tout
  // cas, ce qui est sur, c'est qu'il ne faut pas en tenir compte
  // sinon le bouton ouvrant va osciller entre Enter et Leave qunad
  // on deplace la souris sur le menubar

  if (menuCtrl.isActive(win)) return;

  if (lastEnteredView && (win == lastEnteredView->getBox())) {
    boxLeave(e);
  }
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

void UFlow::winMousePress(UWin *win, UEvent& e) {
  if (getAppli().modalwin) {
    if (win == getAppli().modalwin)	// box inside modalwin
      boxMousePress(e);
    else getAppli().modalwin->highlight(true); // highlight ++ RAISE!
  }

  else if (menuCtrl.isActive()) {
    if (!e.sourceView) {        // mouse outside GUI ==> close menus
      menuCtrl.closeAllMenus(true);
    }
    // mouse inside the grabbedMenu GROUP (including the menubar if any)
    // ==> process mouse events (and don't close menus)
    else if (browsing_group && e.sp.parentBrowsingGroup == browsing_group) {
      boxMousePress(e);
    }
    else { // mouse inside GUI but outside menu ==> close menus
      menuCtrl.closeAllMenus(true);
      
      // ex: boxMousePress(e); pour eviter boxRelease() orphelin
      // mais effet nefaste: si je clique sur un menu opener et que
      // son menu est deja ouvert, je le ferme d'abord par 
      // closeAllMenus() puis je le rouvre par boxMousePress()
      // [or on veut qu'un premier click sur un opener ouvre le menu 
      // [mais que le click suivant le ferme et ainsi de suite]
    }
  }

  else boxMousePress(e);	// cas standard


  if (appli.conf.telepointers && appli.displist.size() > 1)
    showRemPointers(e, +1);
}

/* ==================================================== ======== ======= */

void UFlow::winKeyPress(UWin *win, UEvent& e) {
  if (getAppli().modalwin) {
    if (win == getAppli().modalwin)       // box inside modalwin
      boxKeyPress(e);
    else getAppli().modalwin->highlight(true); // highlight ++ RAISE!
  }

  else if (menuCtrl.isActive()) {
    // mouse outside GUI ==> do nothing
    if (!e.sourceView)
      ;
    // mouse inside the grabbedMenu GROUP (including the menubar if any)
    // ==> process arentmouse events (and don't close menus)
    else if (browsing_group && e.sp.parentBrowsingGroup == browsing_group) {
      //printf("kpress: mouse in BG \n");
      boxKeyPress(e);
    }
    else { // mouse inside GUI but outside menu ==> do nothing
    }
  }

  else boxKeyPress(e);	// cas standard
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

void UFlow::winMouseMove(UWin* win, UEvent& e) {
  boxMouseMove(e);

  if (appli.conf.telepointers && appli.displist.size() > 1)
    showRemPointers(e);
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

void UFlow::winMouseRelease(UWin* win, UEvent& e) {
  if (getAppli().modalwin) {             // box inside modalwin
    if (win == getAppli().modalwin) boxMouseRelease(e);
    else getAppli().modalwin->highlight(false); // unhighlight
  }

  // tous les autres cas (y compris si menuCtrl.isActive())
  // sinon risque de perdre des Release importants (typiquement s'il y a
  // un mdrag a partir d'un browsing group (par ex un menu)
  // ==> il y aura des Release orphelins dans certains cas 
  else boxMouseRelease(e);

  if (appli.conf.telepointers && appli.displist.size() > 1)
    showRemPointers(e, -1);

  /* Note: chie dans la colle pour diverses raisons: 
   * 1) d'abord le Press suffit pour finir le grab : inutile de fermer 
   *    le menu sur Release, 
   * 2) ensuite ca ferme le menu qunad on push un bouton du menu puis
   *    release la souris en dehors du menu, ce qui est carrement penible
   *    (surtout avec menu dialogs),
   * 3) enfin ca va remdre impossible le DND pour la meme raison

   * if (menuCtrl.isActive() && !e.sourceView) {
   *  menuCtrl.closeAllMenus(true);
   * return;
   * }
   */
}

/* ==================================================== ======== ======= */

void UFlow::winKeyRelease(UWin *win, UEvent& e) {
  if (getAppli().modalwin) {
    if (win == getAppli().modalwin)       // box inside modalwin
      boxKeyRelease(e);
    else getAppli().modalwin->highlight(false); // unhighlight
  }

  else if (menuCtrl.isActive()) {
    // mouse outside GUI ==> do nothing
    if (!e.sourceView)
      ;
    else if (browsing_group && e.sp.parentBrowsingGroup == browsing_group) {
      boxKeyRelease(e);
    }
    else { // mouse inside GUI but outside menu ==> do nothing
    }
  }

  else boxKeyRelease(e);	// cas standard
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

void UFlow::boxEnter(UEvent &e, UBox *box, int bstyle) {
  lastEnteredEvent.copy(e);
  lastEnteredEvent.id = UEvent::enter;
  lastEnteredView = e.sourceView;
  lastEnteredBox  = box;
  box->enterBehavior(*this, lastEnteredEvent, bstyle);
}

void UFlow::boxLeave(UEvent &e) {
  lastEnteredEvent.actualize(UEvent::leave, e.xmouse, e.ymouse, e.xev);
  lastEnteredView->getBox()->leaveBehavior(*this, lastEnteredEvent);
  lastEnteredView = null;
  lastEnteredBox  = null;
}

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */

void UFlow::boxKeyPress(UEvent& e) {
  // *** securite ***
  // normalement aurait du etre desactive par boxMotion mais le XEvent
  // correspondant peut eventuellement avoir ete manque
  if (lastEnteredView && lastEnteredView != e.sourceView)
    boxLeave(e);

  if (e.sourceView) {
    UBox *box = e.sourceView->getBox();
    // *** securite ***
    // normalement aurait du etre active par boxMotion mais le XEvent
    // correspondant peut eventuellement avoir ete manque
    if (e.sourceView != lastEnteredView) boxEnter(e, box, UCtrl::STANDARD);
    box->keyPressBehavior(*this, e);
  }
}

void UFlow::boxKeyRelease(UEvent& e) {
  if (e.sourceView) {
    UBox *box = e.sourceView->getBox();
    box->keyReleaseBehavior(*this, e);
  }
}

/* ==================================================== [Elc:00] ======= */
/* ==================================================== ======== ======= */

void UFlow::boxMousePress(UEvent& e) {
  // *** securite ***
  // normalement aurait du etre desactive par boxMotion mais le XEvent
  // correspondant peut eventuellement avoir ete manque
  if (lastEnteredView && lastEnteredView != e.sourceView)
    boxLeave(e);

  if (e.sourceView) {
    // *** securite *** pour le cas ou le "release" aurait ete loupe
    /*
     * code pas correct : ne pend pas bien en compte les "accords" de boutons
     * (ie qd 2 buttons sont appuyes en meme temps)
     * de plus il faudrait stocker no_btn dans UEvent car il varie ds XEvent!
     *
    if (lastPressed	//&& lastPressed->isDef(UMode::PRESS_CB, 0)
	) {
      lastPressedEvent.actualize(UEvent::mrelease, e.xmouse,e.ymouse, e.xev);
      cout << "pseudo boxrel " << e.getButtonNumber() << " " 
	   << lastPressedEvent.getButtonNumber() << "\n";
      lastPressed->fire(lastPressedEvent, UOn::mrelease);
    }
    */
    UBox *box = e.sourceView->getBox();
    // RECOPIE tout le contenu AINSI que celui des champs pointes
    lastPressedEvent.copy(e);
  
    // screen coords. of the View that got the Press event.
    //=> see UEvent.getX(), UEvent.getY() for details
    lastPressedEvent.xdrag_ref = e.getXscreen()- e.getX();
    lastPressedEvent.ydrag_ref = e.getYscreen()- e.getY();
    lastPressed = box;

    // *** securite ***
    // normalement aurait du etre active par boxMotion mais le XEvent
    // correspondant peut eventuellement avoir ete manque
    if (e.sourceView != lastEnteredView) boxEnter(e, box, UCtrl::STANDARD);
    
    // NB: ordre des callbacks
    // -- fire(UOn::press) est remis AVANT armBehavior car c'est l'ordre standard
    // que l'on trouve dans Swing et ailleurs
    // -- si l'on veut dessiner sur l'objet apres le ARM il faut utiliser UOn::arm

    box->fire(e, UOn::mpress);
    box->armBehavior(*this, e, UCtrl::STANDARD);  // means NOT browsing

    if (box->isCmode(UMode::CAN_SELECT_TEXT) 
	&& (e.getButtons() == UEvent::MButton1))    // PARAMETRER!!
      textsel.start(e);
  }
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */
// !NOTE IMPORTANTE:
// C'EST LA Box INITIALE (= celle qui a fait le Press) QUI RECUPERE 
// LES Motions et le Release final

void UFlow::boxMouseRelease(UEvent& e) {
  // NB1: ordre des callbacks
  // -- fire(UOn::release) est remis AVANT armBehavior car c'est l'ordre standard
  // que l'on trouve dans Swing et ailleurs
  // -- si l'on veut dessiner sur l'objet apres le DISARM il faut utiliser
  // UOn::disarm
  
  // NB2: l'evenement Release doit etre relatif a l'objet qui a fait Press
  // en dehors du if (e.sourceView) pour detecter les 'release'
  // en dehors de la fenetre
  // NB: ordre d'appel des callbacks: voir note ci-dessus a propos de UOn::press
  
  if (lastPressed
      /*&& lastPressed->isDef(UMode::PRESS_CB, 0)*/) {
    lastPressedEvent.actualize(UEvent::mrelease, e.xmouse, e.ymouse, e.xev);
    lastPressed->fire(lastPressedEvent, UOn::mrelease);
  }

  if (textsel.beingSelected()) {
    // si on a *effectivement* selectionne qq chose 
    // => obtenir la XSelection (pour desactiver la section dans les
    // autres applis et leur permettre ensuite de savoir que qq chose
    // a ete selectione)
    if (textsel.complete(e)) disp.getNatDisp()->askSelectionOwnership(e);;
  }

  if (e.sourceView) {
    UBox *box = e.sourceView->getBox();

    // -- STANDARD means NOT browsing UCtrl::STANDARD
    // -- returned value: false la vue a ete detruite
    //    ==> sortir alors au plus vite!
    if (box->disarmBehavior(*this, e, UCtrl::STANDARD)) {
      // mrelax is received by the object where the mouse was actually
      // released (otherwise it is similar to mrelease)
      box->fire(e, UOn::mrelax);
    }
  }

  lastPressed = null;   // deplace 7AUG02 (sinon mclick pas detecte)
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

void UFlow::boxMouseMove(UEvent& e) {
  // detection normale des "leave" (si l'evenement n'est pas manque
  // sinon controle de securite dans boxPress)
  if (lastEnteredView && lastEnteredView != e.sourceView) {
    boxLeave(e);
  }

  if (e.sourceView) {
    UBox *box = e.sourceView->getBox();
    bool mbtn_pressed = (e.getButtons() != 0);

    // detection normale des "enter" (si l'evenement n'est pas manque
    // sinon controle de securite dans boxPress)

    if (e.sourceView != lastEnteredView) {
      boxEnter(e, box, (mbtn_pressed ? UCtrl::BROWSING : UCtrl::STANDARD));
    }
    
    // Move callbacks called if MOUSE_MOVE_CB is ON if we are in the
    // postChildEventNotify mode (so that parents will be notified)
    if (e.isPostChildEventNotify()
	|| box->isDef(UMode::MOUSE_MOVE_CB,0)) {
      box->fire(e, UOn::mmove);
    }

    // dans le cas Drag: 
    // NB faire le Press apres le reaffichage provoque par le Enter eventuel
    // l'evenement doit etre relatif a l'objet qui a fait Press
 
    if (mbtn_pressed && lastPressed) {  // Drag

      if (textsel.beingSelected() && box->isDef(0,UMode::CAN_SELECT_TEXT))
	textsel.extend(e);

      // Drag callbacks called if MOUSE_DRAG_CB is ON if we are in the
      // postChildEventNotify mode (so that parents will be notified)
      if (e.isPostChildEventNotify() 
	  || lastPressed->isDef(UMode::MOUSE_DRAG_CB,0)) {
	lastPressedEvent.actualize(UEvent::mdrag, e.xmouse, e.ymouse, e.xev);
	lastPressed->fire(lastPressedEvent, UOn::mdrag);
      }
    }
  }
}

/* ==================================================== [TheEnd] ======= */
/* ==================================================== [Elc:03] ======= */
