/* ==================================================== ======== ======= *
 *
 *  uunatima.cpp : native images and pixmaps
 *  Ubit Project [Elc][2003]
 *  Author: Eric Lecolinet
 *
 *  Part of the Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
 *
 *  (C) 1999-2003 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 *
 * ==================================================== [Elc:03] ======= *
 * ==================================================== ======== ======= */

//pragma ident	"@(#)uunatima.cpp ubit:03.06.04"
#include "config.h"
#include <string.h>
#include <ubrick.hpp>
#include <ucall.hpp>
#include <uerror.hpp>
#include <ucolor.hpp>
#include <ufont.hpp>
#include <uappli.hpp>
#include <unatima.hpp>
#include <unatdisp.hpp>
#include <uima.hpp>
#include <ustr.hpp>
using namespace std; 

const char* const UNatIma::GIF = "gif";
const char* const UNatIma::JPG = "jpg";
const char* const UNatIma::XPM = "xpm";
const char* const UNatIma::XPM_DATA  = "[XPM DATA]";

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */
// ximashape == null ==> opaque image
// ximashape != null ==> transparent background

UNatIma::UNatIma(UNatDisp *nd, UX_Image ima, UX_Image imashape) {
  natdisp   = nd;
  xima      = ima;
  ximashape = imashape;
  lscale    = 0;     //LOGICAL scale
  //next      = null;
#ifdef WITH_GL
  glIma = null;
  imaData = null;;
  createGLTexture(nd);
#endif
}

UNatIma::~UNatIma() {
  if (xima) XDestroyImage(xima);
  if (ximashape) XDestroyImage(ximashape);
  xima = null;
  ximashape = null;
#ifdef WITH_GL
  if (glIma) glDeleteTextures(1, &glIma);
  delete [] imaData;
#endif
}

u_dim UNatIma::getWidth() const {
  if (xima) return xima->width;
  else {
    // affiche des milliards de messages...
    //uerror("UNatIma::getWidth", UError::Unrealized_image);
    return -1;
  }
}

u_dim UNatIma::getHeight() const {
  if (xima) return xima->height;
  else {
    //uerror("UNatIma::getHeight", UError::Unrealized_image);
    return -1;
  }
}

bool UNatIma::isRealized() const {return xima != null;}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

int UNatIma::readImage(UNatDisp* nd, const char* filename_or_data,
		       const char* filetype, UNatIma*& natima) {

  if (!filename_or_data || (!*filename_or_data && filetype != XPM_DATA)) 
    return UFilestat::CannotOpen;

  if (!filetype) filetype = CStr::strext(filename_or_data);

  UNatIma::Reader reader = null;

  if (filetype == null)
    return UFilestat::UnknownType;
#ifdef HAVE_LIBXPM
  else if (filetype == XPM_DATA)             // XPM DATA : must exactly match!
    reader = UNatIma::xpmDataReader;
#endif
#if defined (HAVE_LIBGIF) || defined(HAVE_LIBUNGIF) 
  else if (strcasecmp(filetype, GIF)==0)     // .gif
    reader = UNatIma::gifFileReader;
#endif
#ifdef HAVE_LIBJPEG
  else if (strcasecmp(filetype, JPG)==0)     // .jpg
    reader = UNatIma::jpegFileReader;
  else if (strcasecmp(filetype, "JPEG")==0)     // .jpeg
    reader = UNatIma::jpegFileReader;
#endif
#ifdef HAVE_LIBXPM
  else if (strcasecmp(filetype, XPM)==0)     // .xpm
    reader = UNatIma::xpmFileReader;
#endif
  else return UFilestat::UnknownType;
  
  return (*reader)(nd, filename_or_data, natima);
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */
//NOTE: les NatPix ne sont pas utilises en mode OpenGL

UNatPix::UNatPix(UNatDisp *nd, UX_Pixmap p, UX_Pixmap m,
		 u_dim ww, u_dim hh, int dd) {
  natdisp = nd;
  height    = hh; 
  width     = ww;
  depth     = dd;
  xpix      = p;
  xpixshape = m;
  lscale    = 0;     //LOGICAL scale
}

// creates the pixmaps from the images
// ximashape == null ==> opaque image
// ximashape != null ==> transparent background

void UNatPix::set(UX_Image xima, UX_Image ximashape) {
  lscale    = 0;     //LOGICAL scale
  if (!xima) {
    height    = 0; 
    width     = 0;
    depth     = 0;
    xpix      = None;
    xpixshape = None;
  }

  else if (xima->width == 0 || xima->height == 0) {
    UError::error("warning@UNatPix::set", UError::Null_width_or_height);
    // test important car X plante sur XCreatePixmap si une dimension
    // est nulle
    return;
  }
  else {
    width  = xima->width;
    height = xima->height;
    depth  = xima->depth;
    xpix = XCreatePixmap(natdisp->getXDisplay(), natdisp->getXWindow(),
			 width, height, xima->depth);
    XGCValues values;
    values.foreground = BlackPixelOfScreen(natdisp->getXScreen());
    values.background = WhitePixelOfScreen(natdisp->getXScreen());
    
    GC gc = XCreateGC(natdisp->getXDisplay(), xpix,
		      GCForeground | GCBackground, &values);

    XPutImage(natdisp->getXDisplay(), xpix, gc,
	      xima, 0, 0, 0, 0, width, height);
    XFreeGC(natdisp->getXDisplay(), gc);

    if (!ximashape) xpixshape = None;
    else {
      xpixshape = XCreatePixmap(natdisp->getXDisplay(), 
				natdisp->getXWindow(),
				width, height, ximashape->depth);
      values.foreground = 1;
      values.background = 0;

      //!ATT: le GC est de depth 1 dans ce cas !
      GC gc = XCreateGC(natdisp->getXDisplay(), xpixshape,
			GCForeground | GCBackground, &values);

      XPutImage(natdisp->getXDisplay(), xpixshape, gc, ximashape, 
		0, 0, 0, 0, width, height);
      XFreeGC(natdisp->getXDisplay(), gc);
    }
  }
}

UNatPix::UNatPix(UNatDisp *nd, UX_Image xima, UX_Image ximashape) {
  natdisp = nd;
  xpix = xpixshape = null;
  set(xima, ximashape);
}

UNatPix::UNatPix(UNatDisp *nd, UNatIma *ima) {
  natdisp = nd;
  xpix = xpixshape = null;
  set(ima->xima, ima->ximashape);
}

UNatPix::~UNatPix() {
  if (natdisp
      && natdisp->getXDisplay() 
      //obs: && this != natdisp->getUnknownPixmap()
      ) {
    if (xpix != None) XFreePixmap(natdisp->getXDisplay(), xpix);
    if (xpixshape != None) XFreePixmap(natdisp->getXDisplay(), xpixshape);
  }
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

UX_Image UNatIma::createEmptyImage(UNatDisp* nd, 
				   u_dim width, u_dim height, u_dim depth) {

  if (width <= 0 || height <= 0 || depth <= 0) {
    UError::error("warning@UNatIma::createEmptyImage", 
		  "null or negative size or depth");
    return null;
  }
  if (depth > nd->getDepth()) {
    UError::error("UNatIma::createEmptyImage",
		  "requested depth is too large for this display:", depth);
    return null;
  }

  int bitmap_pad;
  if (depth > 16) bitmap_pad = 32;
  else if (depth > 8) bitmap_pad = 16;
  else bitmap_pad = 8;

  UX_Image ima = 
    XCreateImage(nd->getXDisplay(), nd->getXVisual(), 
		 depth, ZPixmap, 
		 0,              // offset (# of pixels to ignore at the
		                 // beginning of the scanline)
		 NULL,	         // data will be created later
		 width, height,  // requested size
		 bitmap_pad,     // bitmap_pad (quantum of a scanline)
		 0);             // bytes_per_line (0 : continuous scanlines
                                 //  => calculated auto.)
  if (!ima) {
    UError::error("UNatIma::createEmptyImage",
		  "invalid arguments for this display");
    return null;
  }

  // once ima is created, we can get the actual line width via field 
  // 'bytes_per_line' 
  // !CAUTION: note: it is impossible to guess this value before the XImage 
  // is created as X may attribute more bit per pixel than what we requested
  // (for instance a 32bit-per-pix graphics card will probably attribute
  // 32 bits even if we asked for 24 bits!)
  // => !!!WRONG:
  //    datasize = ((float)attr.width * attr.height * nd->getDepth() / 8. + 1);

  ima->data = (char*)malloc(ima->bytes_per_line * height);

  if (!ima->data) {
    UError::error("UNatIma::createEmptyImage",UError::No_more_memory);
    XDestroyImage(ima);
    return null;
  }

  return ima;   // everything OK  
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

UNatIma* UNatIma::createScaledImage(float xscale, float yscale) {
  return createScaledImage(natdisp, xscale, yscale);
}

UNatIma* UNatIma::createScaledImage(UNatDisp* to_nd, 
				    float xscale, float yscale) {
  if (!natdisp) {
    UError::error("UNatIma::createScaledImage","unrealized image","");
    return null;
  }

  UX_Image xima_clone = null, ximashape_clone = null;

  if (xima) 
    xima_clone = createScaledImage(to_nd, natdisp, xima, xscale, yscale);

  if (ximashape)
    ximashape_clone = createScaledImage(to_nd, natdisp, ximashape, xscale, yscale);

  return new UNatIma(to_nd, xima_clone, ximashape_clone);
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */
// NOTE: UNatIma::createScaledImage was adapted from gifrsize.c from 
// the Gif-Lib Written by Gershon Elber (Ver 0.1, Jul. 1989)

struct SimpleCopy {
  UNatDisp *to_nd, *from_nd;
  XImage *to_ima, *from_ima;

  void init() {}

  // copies from 'from_ima' to 'to_ima'
  inline void copy_from2to(int to_x, int to_y, int from_x, int from_y) { 
    XPutPixel(to_ima, to_x, to_y, XGetPixel(from_ima, from_x, from_y));
  }

  // copies in the same image 'to_ima'
  inline void copy_to2to(int to_x, int to_y, int from_x, int from_y) { 
    XPutPixel(to_ima, to_x, to_y, XGetPixel(to_ima, from_x, from_y));
  }
};

/* ==================================================== ======== ======= */

struct RGBConvCopy {
  UNatDisp *to_nd, *from_nd;
  XImage *to_ima, *from_ima;
  int from_red_shift, from_green_shift, from_blue_shift; 
  int to_red_shift, to_green_shift, to_blue_shift; 


  static void init_shift(int& to_shift, int& from_shift, 
			 int to_nd_shift, int to_nd_bits,
			 int from_nd_shift, int from_nd_bits) {
    from_shift = from_nd_shift;
    to_shift = to_nd_shift;

    int delta_bits = from_nd_bits - to_nd_bits;

    if (delta_bits > 0) from_shift += delta_bits;
    else if (delta_bits < 0) to_shift += -delta_bits;
  }


  void init() {
    init_shift(to_red_shift, from_red_shift, 
	       to_nd->getRedShift(), to_nd->getRedBits(), 
	       from_nd->getRedShift(), from_nd->getRedBits());

    init_shift(to_green_shift, from_green_shift, 
	       to_nd->getGreenShift(), to_nd->getGreenBits(), 
	       from_nd->getGreenShift(), from_nd->getGreenBits());

    init_shift(to_blue_shift, from_blue_shift, 
	       to_nd->getBlueShift(), to_nd->getBlueBits(), 
	       from_nd->getBlueShift(), from_nd->getBlueBits());
  }


  inline void copy_from2to(int to_x, int to_y, int from_x, int from_y) { 

    u_long from_pix = XGetPixel(from_ima, from_x, from_y);

    u_long red   = (from_pix & from_nd->getRedMask())   >> from_red_shift ;
    u_long green = (from_pix & from_nd->getGreenMask()) >> from_green_shift;
    u_long blue  = (from_pix & from_nd->getBlueMask())  >> from_blue_shift;
     
    // diviser ou masquer eventuellemnet !!!!

    XPutPixel(to_ima, to_x, to_y, 
	      (red << to_red_shift) 
	      + (green << to_green_shift)
	      + (blue  << to_blue_shift)
	      );
  }

  // copies in the same image 'to_ima'
  inline void copy_to2to(int to_x, int to_y, int from_x, int from_y) { 
    XPutPixel(to_ima, to_x, to_y, XGetPixel(to_ima, from_x, from_y));
  }
};

/* ==================================================== ======== ======= */

template <class copy_type>
struct ResizeAlgo : public copy_type {

  ResizeAlgo(UNatDisp *_to_nd, XImage *_to_ima, 
	     UNatDisp *_from_nd, XImage *_from_ima) {
    to_nd    = _to_nd;
    to_ima   = _to_ima;
    from_nd  = _from_nd;
    from_ima = _from_ima;
    init();
  }

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  inline void resizeLine(int in_y, int out_y, float xscale) {
    int in_w  = from_ima->width;
    int out_w = to_ima->width;
    int in_x, out_x;
    float out_flx = 0.0;
    int last_out_x = 0;

    for (in_x = 0; in_x < in_w; out_flx += xscale, in_x++) {
      
      out_x = (int)out_flx;
      if (out_x > out_w) out_x = out_w;
      
      for (; last_out_x < out_x; last_out_x++) 
	copy_from2to(last_out_x, out_y, in_x, in_y);
    }
    
    // fill the line if needed
    for (in_x = in_w - 1; last_out_x < out_w; last_out_x++)
      copy_from2to(last_out_x, out_y, in_x, in_y);
  }

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  inline void duplicateLine(int to_w, int to_y, int& last_to_y) {
    for (int to_x = 0; to_x < to_w; to_x++) 
      copy_to2to(to_x, last_to_y, to_x, to_y);
  }

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  void resizeImage(float xscale, float yscale) {
    int from_h = from_ima->height, from_y;
    int to_w   = to_ima->width;
    int to_h   = to_ima->height;
    int to_y;
    float y = 0.0;
    int last_to_y = 0;
    
    for (from_y = 0; from_y < from_h; y += yscale, from_y++) {
      to_y = int(y);

      //NB: don't call resizeLine(...,out_y..) if out_y >= out_h !
      if (to_y >= to_h) break;

      if (last_to_y < to_y) {
	resizeLine(from_y, to_y, xscale);
      
	// duplicate additionnal lines 
	for ( ; last_to_y < to_y; last_to_y++)
	  duplicateLine(to_w, to_y, last_to_y);
      }
    }
  
    // add additionnal lines at the end if scale is not dividable
    ++last_to_y;               //skip out_y

    if (last_to_y < to_h) {
      to_y = last_to_y;

      // from_y -> from_h-1 pour ne pas sortir des limites de l'image
      resizeLine(from_h-1, to_y, xscale);

      while (++last_to_y < to_h)
	duplicateLine(to_w, to_y, last_to_y);
    }
  }
};

/* ==================================================== ======== ======= */
/* ==================================================== ======== ======= */
/* NB:
 * - undefined result if the NatDisp is different from those of the
 *   source_ima (invalid colors if the Visual or the depth are different)
 * - image depth must be <= to the NatDisp depth
 */

UX_Image UNatIma::createScaledImage(UNatDisp* to_nd, UNatDisp* from_nd,
				    UX_Image from_ima, 
				    float xscale, float yscale) {

  int to_depth;
  if (from_ima->depth <= to_nd->getDepth()) to_depth = from_ima->depth;
  else to_depth = to_nd->getDepth();

  UX_Image to_ima = createEmptyImage(to_nd,
				     int(from_ima->width  * xscale + 0.5),
				     int(from_ima->height * yscale + 0.5),
				     to_depth);
  if (!to_ima) {
    // inutile: message deja affiche par createEmptyImage()
    // UError::error("UNatIma::createScaledImage", "can't create image");
    return null;
  }

  if (to_nd == from_nd || to_depth == 1) {
    // if to_nd == from_nd or if depth == 1 no convertion is needed
    ResizeAlgo<SimpleCopy> resalgo(to_nd, to_ima, from_nd, from_ima);
    resalgo.resizeImage(xscale, yscale);
  }
  else {
    // RGB values are converted according to the characteristics of
    // to_nd and from_nd
    ResizeAlgo<RGBConvCopy> resalgo(to_nd, to_ima, from_nd, from_ima);
    resalgo.resizeImage(xscale, yscale);
  }
  return to_ima;
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

bool UNatIma::blendImage(UNatDisp *nd, UX_Image xima1, UX_Color pixel2,
			 float alpha) {
  if (xima1->depth > nd->getDepth()) {
    UError::error("UNatIma::blendImage", UError::Incompatible_depths);
    return false;
  }  

  UX_Visual v = nd->getXVisual();
  if (!v || v->c_class != TrueColor) {
    UError::error("UNatIma::blendImage", UError::Blending_requires_TrueColor);
    return false;
  }
  int ww = xima1->width;
  int hh = xima1->height;
  register u_long pix1;
  register float alpha_compl = 1.0 - alpha;

  float 
  pix2_red_alpha   = (pixel2 & v->red_mask)   * alpha,
  pix2_green_alpha = (pixel2 & v->green_mask) * alpha,
  pix2_blue_alpha  = (pixel2 & v->blue_mask)  * alpha;

  u_long 
  red_mask   = v->red_mask, 
  green_mask = v->green_mask,
  blue_mask  = v->blue_mask;
  //red, blue, green;

  for (int y = 0; y < hh; y++)
    for (int x = 0; x < ww; x++) {
      pix1 = XGetPixel(xima1, x, y); 
      /*
      red   = u_long((pix1 & red_mask)   * alpha_compl + pix2_red_alpha);
      green = u_long((pix1 & green_mask) * alpha_compl + pix2_green_alpha);
      blue  = u_long((pix1 & blue_mask)  * alpha_compl + pix2_blue_alpha);
      XPutPixel(xima1, x, y, 
		(red & red_mask) + (green & green_mask) + (blue & blue_mask));
      */
      XPutPixel(xima1, x, y, 
		(u_long((pix1 & red_mask)   * alpha_compl + pix2_red_alpha)   & red_mask)
		+ 
		(u_long((pix1 & green_mask) * alpha_compl + pix2_green_alpha) & green_mask)
		+ 
		(u_long((pix1 & blue_mask)  * alpha_compl + pix2_blue_alpha)  & blue_mask)
		);
    }
  return true;
}

/* ==================================================== ======== ======= */

bool UNatIma::blendImages(UNatDisp *nd, UX_Image xima1, UX_Image xima2,
			  float alpha) {
  if (xima1->depth != xima2->depth || xima1->depth > nd->getDepth()) {
    UError::error("blendImages", UError::Incompatible_depths);
    return false;
  }
  UX_Visual v = nd->getXVisual();
  if (!v || v->c_class != TrueColor) {
    UError::error("blendImages", UError::Blending_requires_TrueColor);
    return false;
  }

  float alpha_compl = 1.0 - alpha;
  int ww = min(xima1->width, xima2->width);
  int hh = min(xima1->height, xima2->height);
  u_long red, blue, green;
  register u_long 
    pix1, pix2,
    red_mask   = v->red_mask, 
    green_mask = v->green_mask,
    blue_mask  = v->blue_mask;
 
  for (int y = 0; y < hh; y++)
    for (int x = 0; x < ww; x++) {
      pix1 = XGetPixel(xima1, x, y); 
      pix2 = XGetPixel(xima2, x, y);

      red   = u_long((pix1 & red_mask)   * alpha_compl + (pix2 & red_mask) * alpha);
      green = u_long((pix1 & green_mask) * alpha_compl + (pix2 & green_mask) * alpha);
      blue  = u_long((pix1 & blue_mask)  * alpha_compl + (pix2 & blue_mask) * alpha);

      XPutPixel(xima1, x, y,
		(red & red_mask) + (green & green_mask) + (blue & blue_mask));
    }
  return true;
}
/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */
#ifdef WITH_GL

bool UNatIma::createGLTexture(UNatDisp* nd) {
  glGenTextures(1, &glIma);
  //cout << "glIma= " << glIma <<"\n";

  // !CF: ima->data = (char*)malloc(ima->bytes_per_line * height);
  int bpp    = xima->bits_per_pixel;
  u_long size = xima->width * xima->height * bpp / 8;
  bool no_shape = (ximashape ? false : true);

  glBindTexture(GL_TEXTURE_2D, glIma);
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);//GL_NEAREST);
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);//GL_NEAREST);
  
  if (xima->depth != 24) {
    UError::error("UNatIma::makeGLtexture","wrong depth:", xima->depth);
    return false;
  }
  else if (bpp != 32) {
    UError::error("UNatIma::makeGLtexture","wrong bpp:", bpp);
    return false;
  }

  UX_Visual v = nd->getXVisual();
  if (!v || v->c_class != TrueColor) {
    UError::error("UNatIma::makeGLtexture", 
		  UError::Blending_requires_TrueColor);
    return false;
  }

  register unsigned long 
    pix1,
    red_mask   = nd->getRedMask(), 
    green_mask = nd->getGreenMask(),
    blue_mask  = nd->getBlueMask(),
    red_shift  = nd->getRedShift(), 
    green_shift= nd->getGreenShift(),
    blue_shift = nd->getBlueShift();

  imaData = new unsigned char[size]; 
  //cout << "bpp"<< bpp<<"\n";

  unsigned long k = 0;

  for (int y = 0; y < xima->height; y++)
    for (int x = 0; x <xima->width; x++) {
      pix1 = XGetPixel(xima, x, y); 
      imaData[k]   = ((pix1 & red_mask)   >> red_shift);
      imaData[k+1] = ((pix1 & green_mask) >> green_shift);
      imaData[k+2] = ((pix1 & blue_mask)  >> blue_shift);
      if (no_shape) imaData[k+3] = 0xff; 
      else {                          // has shape
	if (XGetPixel(ximashape, x, y))
	  imaData[k+3] = 0xff;
	else imaData[k+3] = 0;
      }
      k += 4;
    }
  gluBuild2DMipmaps(GL_TEXTURE_2D, 4, xima->width, xima->height,
		      GL_RGBA, GL_UNSIGNED_BYTE, imaData);
  return true;
}

#endif
/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */
// essaye de tout caser dans la Colormap donnee en arg.
// returns the convtable

unsigned long* UNatIma::allocXColors(UNatDisp *nd, 
				     const XColor* colors, int colorCount) 
{
  UX_Display display = nd->getXDisplay();
  UX_Colormap cmap = nd->getXColormap();
  unsigned long* convtable = new unsigned long[colorCount];
  register int k;
  // on essaye d'allouer les couleurs dans la cmap et on voit ce qui reste
  int todoCount = 0;
  int* todo = new int[colorCount];
  XColor col;
  col.flags = DoRed | DoGreen | DoBlue;

  for (k = 0; k < colorCount; k++) {
    col.red   = colors[k].red;
    col.green = colors[k].green;
    col.blue  = colors[k].blue;

    if (XAllocColor(display, cmap, &col)) 
      convtable[k] = col.pixel;
    else {
      convtable[k] = 0;
      todo[todoCount] = k;
      todoCount++;
    }
  }

  // -- si on a reussi a tout caser : c'est fini!
  if (todoCount == 0) {
    delete[] todo;
    return convtable;
  }

  // -- sinon: on associe aux couleurs non-allouees la couleur la plus proche 
  // dans la colormap

  // pas toujours vrai si Visuel specifique ... !!
  // mais comment retrouver la vraie taille de cmap ???
  int num_xcolors = DisplayCells(display, XScreenNumberOfScreen(nd->getXScreen()));

  XColor* xcolors = new XColor[num_xcolors];
  for (k = 0; k < num_xcolors; k++) xcolors[k].pixel = k;
  XQueryColors(display, cmap, xcolors, num_xcolors);

  unsigned long green, red, blue;

  for (k = 0; k < todoCount; k++) {
    red   = colors[todo[k]].red;
    green = colors[todo[k]].green;
    blue  = colors[todo[k]].blue;

    unsigned long dist, mindist = 0xffffffff;
    int jmin = 0;

    for (int j = 0; j < num_xcolors; j++) {
      long r = red   - xcolors[j].red;
      long g = green - xcolors[j].green;
      long b = blue  - xcolors[j].blue;
      if (r <0) r = -r;
      if (g <0) g = -g;
      if (b <0) b = -b;

      // NB: on pourrait faire nettement mieux: voir par exemple create.c
      // dans la distrib XPM
      dist = r + g + b;
      if (dist < mindist) {mindist = dist; jmin = j;}
    }

    convtable[todo[k]] = jmin;
  }

  delete[] xcolors;
  delete[] todo;
  return convtable;
}

/* ==================================================== ======== ======= */
// version plus sophistiquee qui essaie d'abord d'allouer les couleurs
// les plus frequentes dans l'image
// en pratique: resultat similaire (= aussi mediocre)

/*
static void AllocColors2(Display *display, Colormap cmap,
			int width, int height,
			unsigned char **buffer,
			unsigned long *pixelTable,
			XColor *colors, int colorCount) {
  register int j, k;
  long compteur[256];

  for (k = 0; k < 256; k++) {
    compteur[k] = 0;
    pixelTable[k] = 0;
  }

  // On cherche les couleurs les plus utilisees 
  for (int i = 0; i < height; i++)
    for (j = 0; j < width; j++)
      // valeurs entre 0 et 255
      compteur[buffer[i][j]]++;

  // On reorganise les couleurs par ordre decroissant d'occurence
  // et on essaye de les allouer dans la cmap
  int nb_reste = 0;
  int reste[256];
  XColor col;
  col.flags = DoRed | DoGreen | DoBlue;

  for (k = 0; k < 256; k++) {
    long max_count = 0;
    int jmax;
    for (j = 0; j < 256; j++)
      if (compteur[j] > max_count) { 
	max_count = compteur[j];
	jmax = j;
      }
    compteur[jmax] = 0;	 //cas traite
    col.red   = colors[jmax].red;
    col.green = colors[jmax].green;
    col.blue  = colors[jmax].blue;

    if (XAllocColor(display, cmap, &col)) pixelTable[jmax] = col.pixel;
    else reste[nb_reste++] = jmax;
  }

  printf("colorCount = %d, reste = %d\n", colorCount, nb_reste);

  // On associe aux couleurs non-allouees la couleur la plus proche 
  // dans la colormap
  XColor colors[256];
  unsigned long green, red, blue;

  // 256 --> mumColors = DisplayCells(xdisplay, DefaultScreen(xdisplay));
  for (k = 0; k < 256; k++) colors[k].pixel = k;
  XQueryColors(display, cmap, colors, 256);

  for (k = 0; k < nb_reste; k++) {
    red   = colors[reste[k]].red;
    green = colors[reste[k]].green;
    blue  = colors[reste[k]].blue;

    unsigned long dist, mindist = 0xffffffff;
    int jmin = 0;

    for (j = 0; j < 256; j++) {
      //!att:: le cas (signed long) est necessaire!
      dist = U_ABS((signed long)red - colors[j].red)
	+ U_ABS((signed long)green - colors[j].green)
	+ U_ABS((signed long)blue - colors[j].blue);
      if (dist < mindist) {mindist = dist; jmin = j;}
    }
    pixelTable[reste[k]] = jmin;
  }
}
*/
/* ==================================================== [TheEnd] ======= */
/* ==================================================== [Elc:03] ======= */
