/*
 * Copyright (c) 2003-2005 Erez Zadok
 * Copyright (c) 2003-2005 Charles P. Wright
 * Copyright (c) 2003-2005 Mohammad Nayyer Zubair
 * Copyright (c) 2003-2005 Puja Gupta
 * Copyright (c) 2003-2005 Harikesavan Krishnan
 * Copyright (c) 2003-2005 Stony Brook University
 * Copyright (c) 2003-2005 The Research Foundation of State University of New York
 *
 * For specific licensing information, see the COPYING file distributed with
 * this package.
 *
 * This Copyright notice must be kept intact and distributed with all sources.
 */
/*
 *  $Id: print.c,v 1.48 2005/03/16 14:08:09 cwright Exp $
 */

/* Print debugging functions */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif /* HAVE_CONFIG_H */
#include "fist.h"
#include "unionfs.h"

static int fist_debug_var = 0;

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
#define PageUptodate(page) Page_Uptodate(page)
#endif


/* get value of debugging variable */
int
fist_get_debug_value(void)
{
  return fist_debug_var;
}

/* set debug level variable and return the previous value */
int
fist_set_debug_value(int val)
{
  int prev = fist_debug_var;

  fist_debug_var = val;
  fist_dprint(1, "unionfs: setting debug level to %d\n", val);
  return prev;
}

/*
 * Utilities used by both client and server
 * Standard levels:
 * 0) no debugging
 * 1) hard failures
 * 2) soft failures
 * 3) current test software
 * 4) main procedure entry points
 * 5) main procedure exit points
 * 6) utility procedure entry points
 * 7) utility procedure exit points
 * 8) obscure procedure entry points
 * 9) obscure procedure exit points
 * 10) random stuff
 * 11) all <= 1
 * 12) all <= 2
 * 13) all <= 3
 * ...
 */

static char buf[4096];

void
fist_dprint_internal(const char *file, const char *function, int line, int level, char *str,...)
{
  va_list ap;
  int var = fist_get_debug_value();

  if (level >= 10 || level < 0) {
      printk("<0>fist_dprint_internal: Invalid level passed from %s:%s:%d\n", file, function, line);
  }

  if (var == level || (var > 10 && (var - 10) >= level)) {
    va_start(ap, str);
    vsnprintf(buf, 4096, str, ap);
    printk("%s", buf);
    va_end(ap);
  }
  return;
}


static int num_indents = 0;
char indent_buf[80] = "                                                                               ";
char *
add_indent(void)
{
    indent_buf[num_indents] = ' ';
    num_indents++;
    if (num_indents > 79)
	num_indents = 79;
    indent_buf[num_indents] = '\0';
    return indent_buf;
}

char *
del_indent(void)
{
    if (num_indents <= 0)
	return "<IBUG>";
    indent_buf[num_indents] = ' ';
    num_indents--;
    indent_buf[num_indents] = '\0';
    return indent_buf;
}

void
fist_print_buffer_flags(char *str, struct buffer_head *buffer)
{
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,10)
    /*
     * XXX: 2.4.10 removed buffer_protected state flag.
     */
# define buffer_protected(buffer) (-1)
#endif /* kernel 2.4.10 and newer */

    if (!buffer) {
	printk("PBF %s 0x%p\n", str, buffer);
	return;
    }

    fist_dprint(8, "PBF %s 0x%p: Uptodate:%d Dirty:%d Locked:%d Req:%d Protected:%d\n",
		str,
		buffer,
		buffer_uptodate(buffer),
		buffer_dirty(buffer),
		buffer_locked(buffer),
		buffer_req(buffer),
		buffer_protected(buffer)
		);
}


void
fist_print_page_flags(char *str, page_t *page)
{
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,10)
    /*
     * XXX: 2.4.10 forgot to export the ksym to swapper_space,
     * which is used in PageSwapCache().
     */
# define fist_PageSwapCache(page) (-1)
#else
# define fist_PageSwapCache(page) (PageSwapCache(page) ? 1 : 0)
#endif /* kernel 2.4.10 and newer */

#ifdef PageDecrAfter
#define fist_PageDecrAfter(page)	(PageDecrAfter(page) ? 1 : 0)
#else  /* not PageDecrAfter */
#define fist_PageDecrAfter(page)	(-1)
#endif /* not PageDecrAfter */

    fist_dprint(8, "PPF %s 0x%p/0x%lx: Locked:%d Error:%d Referenced:%d Uptodate:%d DecrAfter:%d Slab:%d SwapCache:%d Reserved:%d\n",
		str,
		page,
		page->index,
		(PageLocked(page) ? 1 : 0),
		(PageError(page) ? 1 : 0),
		(PageReferenced(page) ? 1 : 0),
		(PageUptodate(page) ? 1 : 0),
		fist_PageDecrAfter(page),
		(PageSlab(page) ? 1 : 0),
		fist_PageSwapCache(page),
		(PageReserved(page) ? 1 : 0)
		);
}


void
fist_print_inode(char *str, const inode_t *inode)
{
    inode_t *hidden_inode;

    if (!inode) {
	printk("PI:%s: NULL INODE PASSED!\n", str);
	return;
    }
    fist_dprint(8, "\n");
    fist_dprint(8, "=====Inode Data=====\n");
    fist_dprint(8, "PI:%s: %s=%lu\n", str, "i_ino", inode->i_ino);
    fist_dprint(8, "PI:%s: %s=%u\n", str, "i_count", atomic_read(&inode->i_count));
    fist_dprint(8, "PI:%s: %s=%u\n", str, "i_nlink", inode->i_nlink);
    fist_dprint(8, "PI:%s: %s=%o\n", str, "i_mode", inode->i_mode);
    fist_dprint(8, "PI:%s: %s=%llu\n", str, "i_size", inode->i_size);
    fist_dprint(8, "PI:%s: %s=%p\n", str, "i_op", inode->i_op);
    fist_dprint(8, "PI:%s: %s=%p (%s)\n", str, "i_sb",
		inode->i_sb,
		(inode->i_sb ? sbt(inode->i_sb) : "NullTypeSB")
		);

    if(itopd(inode)) {
	PASSERT(itopd(inode));
	fist_dprint(8, "ibstart=%d, ibend=%d\n", ibstart(inode), ibend(inode));
    }
    fist_dprint(8, "\n");   /* printing underlying inodes, if any */

    if(itopd(inode)) {
	int bindex;

	fist_dprint(8, "Hidden Inodes\n");
	for (bindex = ibstart(inode); bindex <= ibend(inode); bindex++) {

	    hidden_inode = itohi_index(inode, bindex);
	    fist_dprint(8, "---------------------\n");

            if(!hidden_inode) {
		fist_dprint(8, "Hidden Inode # %d = NULL", bindex);
		continue;
	    }

	    fist_dprint(8, "Hidden Inode # %d: %s: %s=%lu\n", bindex, str, "i_ino", hidden_inode->i_ino);
	    fist_dprint(8, "Hidden Inode # %d: %s: %s=%u\n", bindex, str, "i_count", atomic_read(&hidden_inode->i_count));
	    fist_dprint(8, "Hidden Inode # %d: %s: %s=%u\n", bindex, str, "i_nlink", hidden_inode->i_nlink);
	    fist_dprint(8, "Hidden Inode # %d: %s: %s=%o\n", bindex, str, "i_mode", hidden_inode->i_mode);
	    fist_dprint(8, "Hidden Inode # %d: %s: %s=%llu\n", bindex, str, "i_size", hidden_inode->i_size);
	    fist_dprint(8, "Hidden Inode # %d: %s: %s=%p\n", bindex, str, "i_op", hidden_inode->i_op);
	    fist_dprint(8, "Hidden Inode # %d: %s: %s=%p (%s)\n", bindex, str, "i_sb",
			hidden_inode->i_sb,
			(hidden_inode->i_sb ? sbt(hidden_inode->i_sb) : "NullTypeSB")
			);
	}
    }
}


void
fist_print_pte_flags(char *str, const page_t *page)
{
    unsigned long address;

    PASSERT(page);
    address = page->index;
    fist_dprint(8, "PTE-FL:%s index=0x%lx\n", str, address);
}


void
fist_print_vma(char *str, const vm_area_t *vma)
{
    return;
}


void
fist_print_file(char *str, const file_t *file)
{
    file_t *hidden_file;

    if (!file) {
	fist_dprint(8, "PF:%s: NULL FILE PASSED!\n", str);
	return;
    }
    fist_dprint(8, "PF:%s: %s=0x%p\n", str, "f_dentry", file->f_dentry);
    fist_dprint(8, "PF:%s: file's dentry name=%s\n", str, file->f_dentry->d_name.name);
    if (file->f_dentry->d_inode) {
	PASSERT(file->f_dentry->d_inode);
	fist_dprint(8, "PF:%s: %s=%lu\n", str,"f_dentry->d_inode->i_ino", file->f_dentry->d_inode->i_ino);
	fist_dprint(8, "PF:%s: %s=%o\n", str, "f_dentry->d_inode->i_mode", file->f_dentry->d_inode->i_mode);
    }
    fist_dprint(8, "PF:%s: %s=0x%p\n", str, "f_op", file->f_op);
    fist_dprint(8, "PF:%s: %s=0x%x\n", str, "f_mode", file->f_mode);
    fist_dprint(8, "PF:%s: %s=0x%llu\n", str, "f_pos", file->f_pos);
    fist_dprint(8, "PF:%s: %s=%u\n", str, "f_count", atomic_read(&file->f_count));
    fist_dprint(8, "PF:%s: %s=0x%x\n", str, "f_flags", file->f_flags);

    fist_dprint(8, "PF:%s: %s=%lu\n", str, "f_version", file->f_version);
    if(ftopd(file)) {
	fist_dprint(8, "PF:%s: fbstart=%d, fbend=%d\n", str, fbstart(file), fbend(file));
    }
    if(ftopd(file)) {
	int bindex;
	for (bindex = fbstart(file); bindex <= fbend(file); bindex++) {

	    hidden_file = ftohf_index(file, bindex);
	    if(!hidden_file) {
		fist_dprint(8, "PF:%s: HF#%d is NULL\n", str, bindex);
		continue;
	    }

	    fist_dprint(8, "PF:%s: HF#%d %s=0x%p\n", str, bindex, "f_dentry", hidden_file->f_dentry);
	    if (hidden_file->f_dentry->d_inode) {
		PASSERT(hidden_file->f_dentry->d_inode);
	        fist_dprint(8, "PF:%s: HF#%d: %s=%lu\n", str, bindex, "f_dentry->d_inode->i_ino", hidden_file->f_dentry->d_inode->i_ino);
	        fist_dprint(8, "PF:%s: HF#%d: %s=%o\n", str, bindex, "f_dentry->d_inode->i_mode", hidden_file->f_dentry->d_inode->i_mode);
	    }
	    fist_dprint(8, "PF:%s: HF#%d: %s=0x%p\n", str, bindex, "f_op", hidden_file->f_op);
	    fist_dprint(8, "PF:%s: HF#%d: %s=0x%x\n", str, bindex, "f_mode", hidden_file->f_mode);
	    fist_dprint(8, "PF:%s: HF#%d: %s=%llu\n", str, bindex, "f_pos", hidden_file->f_pos);
	    fist_dprint(8, "PF:%s: HF#%d: %s=%u\n", str, bindex, "f_count", atomic_read(&hidden_file->f_count));
	    fist_dprint(8, "PF:%s: HF#%d: %s=0x%x\n", str, bindex,"f_flags", hidden_file->f_flags);
	    fist_dprint(8, "PF:%s: HF#%d: %s=%lu\n", str, bindex, "f_version", hidden_file->f_version);
	}
    }
}


void
fist_print_dentry(char *str, const dentry_t *dentry)
{
    dentry_t *hidden_dentry = NULL;

    if (!dentry) {
	fist_dprint(8, "PD:%s: NULL DENTRY PASSED!\n", str);
	return;
    }
    if (IS_ERR(dentry)) {
	fist_dprint(8, "PD:%s: ERROR DENTRY (%ld)!\n", str, PTR_ERR(dentry));
	return;
    }
    PASSERT(dentry);

    fist_dprint(8, "=====Dentry Data (%p)=====\n", dentry);

    fist_dprint(8, "PD:%s: %s=%d\n", str, "d_count",
		atomic_read(&dentry->d_count));
    fist_dprint(8, "PD:%s: %s=%x\n", str, "d_flags", (int) dentry->d_flags);
    fist_dprint(8, "PD:%s: %s=%p (%s)\n", str, "d_inode",
		dentry->d_inode,
		(dentry->d_inode ? sbt(dentry->d_inode->i_sb) : "nil") );
    fist_dprint(8, "PD:%s: %s=%p (%s)\n", str, "d_parent",
		dentry->d_parent,
		(dentry->d_parent ? sbt(dentry->d_parent->d_sb) : "nil") );
    fist_dprint(8, "PD:%s: %s=\"%s\"\n", str, "d_parent->d_name.name", dentry->d_parent->d_name.name);
    //    fist_dprint(8, "PD:%s: %s=%x\n", str, "d_mounts", (int) dentry->d_mounts);
    //    fist_dprint(8, "PD:%s: %s=%x\n", str, "d_covers", (int) dentry->d_covers);
    //    fist_dprint(8, "PD:%s: %s=%x\n", str, "d_hash", (int) & dentry->d_hash);
    //    fist_dprint(8, "PD:%s: %s=%x\n", str, "d_lru", (int) & dentry->d_lru);
    //    fist_dprint(8, "PD:%s: %s=%x\n", str, "d_child", (int) & dentry->d_child);
    //    fist_dprint(8, "PD:%s: %s=%x\n", str, "d_subdirs", (int) & dentry->d_subdirs);
    //    fist_dprint(8, "PD:%s: %s=%x\n", str, "d_alias", (int) & dentry->d_alias);
    //    fist_dprint(8, "PD:%s: %s=%x\n", str, "d_name", (int) & dentry->d_name);
    //
    fist_dprint(8, "PD:%s: %s=\"%s\"\n", str, "d_name.name", dentry->d_name.name);
    if (dentry->d_inode) {
	char t;
	if (S_ISDIR(dentry->d_inode->i_mode))
		t = 'd';
	else if (S_ISLNK(dentry->d_inode->i_mode))
		t = 'l';
	else if (S_ISCHR(dentry->d_inode->i_mode))
		t = 'c';
	else if (S_ISBLK(dentry->d_inode->i_mode))
		t = 'b';
	else if (S_ISREG(dentry->d_inode->i_mode))
		t = 'f';
	else
		t = '?';
	fist_dprint(8, "PD:%s: dentry->d_inode->i_mode: %c%o\n", str, t, dentry->d_inode->i_mode);
    }
    fist_dprint(8, "PD:%s: %s=%d\n", str, "d_name.len", (int) dentry->d_name.len);
    //    fist_dprint(8, "PD:%s: %s=%x\n", str, "d_name.hash", (int) dentry->d_name.hash);
    //    fist_dprint(8, "PD:%s: %s=%lu\n", str, "d_time", dentry->d_time);
    fist_dprint(8, "PD:%s: %s=%p\n", str, "d_op", dentry->d_op);
    fist_dprint(8, "PD:%s: %s=%p (%s)\n", str, "d_sb",
		dentry->d_sb, sbt(dentry->d_sb));
    //    fist_dprint(8, "PD:%s: %s=%lu\n", str, "d_reftime", dentry->d_reftime);
    fist_dprint(8, "PD:%s: %s=%p\n", str, "d_fsdata", dentry->d_fsdata);
    // don't do this, it's not zero-terminated!!!
    //    fist_dprint(8, "PD:%s: %s=\"%s\"\n", str, "d_iname", dentry->d_iname);
    //    fist_dprint(8, "\n");
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
    fist_dprint(8, "PD:%s:%s=%d\n", str, "list_empty(d_hash)", list_empty(&((dentry_t *)dentry)->d_hash));
#else
    fist_dprint(8, "PD:%s: %s=%d\n", str, "hlist_unhashed(d_hash)", hlist_unhashed(&((dentry_t *)dentry)->d_hash));
#endif

    if(dtopd(dentry)) {
        fist_dprint(8, "PD:%s: dbstart=%d, dbend=%d\n", str, dbstart(dentry), dbend(dentry));
    }
    fist_dprint(8, "\n");

    if(dtopd(dentry) && (dbstart(dentry) != -1)) {
	int bindex;

	PASSERT(dtopd(dentry));

	fist_dprint(8, "PD:%s: Hidden Dentries:\n", str);

	for (bindex = dbstart(dentry); bindex <= dbend(dentry); bindex++) {

	    hidden_dentry = dtohd_index(dentry, bindex);
	    fist_dprint(8, "---------------------\n");
	    if(!hidden_dentry) {
		fist_dprint(8, "%s: HD#%d = NULL\n", str, bindex);
		continue;
	    }

	    fist_dprint(8, "%s: HD#%d: %s=%d\n", str, bindex, "d_count",
		    atomic_read(&hidden_dentry->d_count));
	    fist_dprint(8, "%s: HD#%d: %s=%x\n", str, bindex, "d_flags", (int) hidden_dentry->d_flags);
	    fist_dprint(8, "%s: HD#%d: %s=%p (%s)\n", str, bindex, "d_inode",
		    hidden_dentry->d_inode,
		    (hidden_dentry->d_inode ? sbt(hidden_dentry->d_inode->i_sb) : "nil") );
	    fist_dprint(8, "%s: HD#%d: %s=%p (%s)\n", str, bindex, "d_parent",
		    hidden_dentry->d_parent,
		    (hidden_dentry->d_parent ? sbt(hidden_dentry->d_parent->d_sb) : "nil") );
	    fist_dprint(8, "%s: HD#%d: %s=\"%s\"\n", str, bindex, "d_name.name", hidden_dentry->d_name.name);
	    fist_dprint(8, "%s: HD#%d: %s=\"%s\"\n", str, bindex, "d_parent->d_name.name", hidden_dentry->d_parent->d_name.name);
	    fist_dprint(8, "%s: HD#%d: %s=%d\n", str, bindex, "d_name.len", (int) hidden_dentry->d_name.len);
	    if (hidden_dentry->d_inode) {
		char t;
		if (S_ISDIR(hidden_dentry->d_inode->i_mode))
		    t = 'd';
		else if (S_ISLNK(hidden_dentry->d_inode->i_mode))
		    t = 'l';
		else if (S_ISCHR(hidden_dentry->d_inode->i_mode))
		    t = 'c';
		else if (S_ISBLK(hidden_dentry->d_inode->i_mode))
		    t = 'b';
		else if (S_ISREG(hidden_dentry->d_inode->i_mode))
		    t = '-';
		else
		    t = '?';
		fist_dprint(8, "%s: HD#%d: dentry->d_inode->i_mode: %c%o\n", str, bindex, t, hidden_dentry->d_inode->i_mode);
	    }
	    fist_dprint(8, "%s: HD#%d: %s=%p\n", str, bindex, "d_op", hidden_dentry->d_op);
	    fist_dprint(8, "%s: HD#%d: %s=%p (%s)\n", str, bindex, "d_sb",
		    hidden_dentry->d_sb, sbt(hidden_dentry->d_sb));
	    fist_dprint(8, "%s: HD#%d: %s=%p\n", str, bindex, "d_fsdata", hidden_dentry->d_fsdata);
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
	    fist_dprint(8, "%s: HD#%d: %s=%d\n", str, bindex, "list_empty(d_hash)", list_empty(&((dentry_t *)hidden_dentry)->d_hash));
#else
	    fist_dprint(8, "%s: HD#%d: %s=%d\n", str, bindex, "hlist_unhashed(d_hash)", hlist_unhashed(&((dentry_t *)dentry)->d_hash));
#endif

	}
    }
}

void
fist_checkinode(inode_t *inode, char *msg)
{
    if (!inode) {
	printk(KERN_WARNING "fist_checkinode - inode is NULL! (%s)\n", msg);
	return;
    }
    if (!itopd(inode)) {
	fist_dprint(8, "fist_checkinode(%ld) - no private data (%s)\n", inode->i_ino, msg);
	return;
    }
    if ((itopd(inode)->b_start < 0) || !itohi(inode)) {
	fist_dprint(8, "fist_checkinode(%ld) - underlying is NULL! (%s)\n", inode->i_ino, msg);
	return;
    }
    if (!inode->i_sb) {
	fist_dprint(8, "fist_checkinode(%ld) - inode->i_sb is NULL! (%s)\n", inode->i_ino, msg);
	return;
    }
    fist_dprint(8, "inode->i_sb->s_type %p\n", inode->i_sb->s_type);
    if (!inode->i_sb->s_type) {
	fist_dprint(8, "fist_checkinode(%ld) - inode->i_sb->s_type is NULL! (%s)\n", inode->i_ino, msg);
	return;
    }
    fist_dprint(6, "CI: %s: inode->i_count = %d, hidden_inode->i_count = %d, inode = %lu, sb = %s, hidden_sb = %s\n",
		msg, atomic_read(&inode->i_count),
		itopd(inode)->b_start >= 0 ? atomic_read(&itohi(inode)->i_count): -1, inode->i_ino,
		inode->i_sb->s_type->name, itopd(inode)->b_start >= 0 ? itohi(inode)->i_sb->s_type->name: "(none)");
}

void
fist_print_sb(char *str, const super_block_t *sb)
{
    super_block_t *hidden_superblock;

    if (!sb) {
        fist_dprint(8, "PSB:%s: NULL SB PASSED!\n", str);
        return;
    }

    fist_dprint(8, "PSB:%s: %s=%u\n", str, "s_blocksize", (int) sb->s_blocksize);
    fist_dprint(8, "PSB:%s: %s=%u\n", str, "s_blocksize_bits", (int) sb->s_blocksize_bits);
    fist_dprint(8, "PSB:%s: %s=0x%x\n", str, "s_flags", (int) sb->s_flags);
    fist_dprint(8, "PSB:%s: %s=0x%x\n", str, "s_magic", (int) sb->s_magic);
    fist_dprint(8, "PSB:%s: %s=%llu\n", str, "s_maxbytes", sb->s_maxbytes);
    fist_dprint(8, "PSB:%s: %s=%d\n", str, "s_count", (int) sb->s_count);
    fist_dprint(8, "PSB:%s: %s=%d\n", str, "s_active", (int) atomic_read(&sb->s_active));
    if(stopd(sb))
	fist_dprint(8, "sbstart=%d, sbend=%d\n", sbstart(sb), sbend(sb));
    fist_dprint(8, "\n");

    if(stopd(sb)) {
	int bindex;
	for (bindex = sbstart(sb); bindex <= sbend(sb); bindex++) {
	    hidden_superblock = stohs_index(sb, bindex);
	    if(!hidden_superblock) {
		fist_dprint(8, "PSB:%s: HS#%d is NULL", str, bindex);
		continue;
	    }

	    fist_dprint(8, "PSB:%s: HS#%d: %s=%u\n", str, bindex, "s_blocksize", (int) hidden_superblock->s_blocksize);
	    fist_dprint(8, "PSB:%s: HS#%d: %s=%u\n", str, bindex, "s_blocksize_bits", (int) hidden_superblock->s_blocksize_bits);
	    fist_dprint(8, "PSB:%s: HS#%d: %s=0x%x\n", str, bindex, "s_flags", (int) hidden_superblock->s_flags);
	    fist_dprint(8, "PSB:%s: HS#%d: %s=0x%x\n", str, bindex, "s_magic", (int) hidden_superblock->s_magic);
	    fist_dprint(8, "PSB:%s: HS#%d: %s=%llu\n", str, bindex, "s_maxbytes", hidden_superblock->s_maxbytes);
	    fist_dprint(8, "PSB:%s: HS#%d: %s=%d\n", str, bindex, "s_count", (int) hidden_superblock->s_count);
	    fist_dprint(8, "PSB:%s: HS#%d: %s=%d\n", str, bindex, "s_active", (int) atomic_read(&hidden_superblock->s_active));
	}
    }
}




/*
 * vim:shiftwidth=4
 * Local variables:
 * c-basic-offset: 4
 * End:
 */
