;;; This is an implementation of the Unlambda programming language.
;;; Version 1.92.1 of 1999/10/30
;;; Modified to work with bigloo (tested with bigloo-2.1a)

;;; Copyright (C) 1999 by David A. Madore <david.madore@ens.fr>

;;; This program is free software; you can redistribute it and/or
;;; modify it under the terms of the GNU General Public License
;;; as published by the Free Software Foundation; either version
;;; 2 of the License, or (at your option) any later version.
;;;
;;; This program is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY; without even the implied warranty
;;; of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See
;;; the GNU General Public License for more details.
;;;
;;; You should have received a copy of the GNU General Public License
;;; along with this program; if not, write to the Free Software
;;; Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

;; Data representation: f applied to g is represented as (f . g) (we
;; use a pair rather than a list because application *always* takes
;; *exactly* one argument).  Primitive procedures k, s, and so on are
;; represented as (k), (s) and so on.  Derived procedures (k1, s1, s2
;; and the like) are represented by lists whose first element is k1,
;; s1... and whose subsequent elements give internal data to the
;; derived procedure (data that was set there by whatever created the
;; procedure, to be remember when it is called).

;; I would be very grateful if somebody could explain to me how all
;; this works.  -- The author

;; Bigloo module declaration
(module unlambda
  (main main))

;; Parse the input file and return the representation of it.
(define (parse input-port)
  (define (gobble-comment)
    (if (not (let ((ch (read-char input-port)))
	       (or (eof-object? ch)
		   (eqv? ch #\newline))))
	(gobble-comment)))
  (case (let ((ch (read-char input-port)))
	  (if (eof-object? ch)
	      (error "PARSE" "Unexpected end of file" input-port)
	      ch))
    ((#\`) (let* ((op (parse input-port))
		  (arg (parse input-port)))
	     `(,op . ,arg)))
;   ((#\space #\ht #\cr #\newline) (parse input-port))
    ((#\space #\newline) (parse input-port)) ; #\ht and #\cr are not standard, shit!
    ((#\#) (gobble-comment) (parse input-port))
    ((#\k #\K) '(k)) ; (lambda (x) (lambda (y) x))
    ((#\s #\S) '(s)) ; (lambda (x) (lambda (y) (lambda (z) ((x z) (y z)))))
    ((#\i #\I) '(i)) ; identity (same as ``skk)
    ((#\v #\V) '(v)) ; return v
    ((#\c #\C) '(c)) ; call/cc
    ((#\d #\D) '(d)) ; delay (special form, force at next call)
    ((#\e #\E) '(e)) ; exit immediately
;; The p function has been replaced by the more general . function
;   ((#\p #\P) '(pr #\*)) ; print an asterisk (same as .*)
    ((#\r #\R) '(pr #\newline)) ; print newline
    ((#\.) `(pr ,(let ((ch (read-char input-port)))
		   (if (eof-object? ch)
		       (error "PARSE" "Unexpected end of file" input-port)
		       ch)))) ; print given char
    ((#\@) '(rd)) ; read next input char
    ((#\?) `(rc ,(let ((ch (read-char input-port)))
		   (if (eof-object? ch)
		       (error "PARSE" "Unexpected end of file" input-port)
		       ch)))) ; compare character under reading head
    ((#\|) '(pc)) ; call arg with dot function for current char
    (else (error "PARSE" "Character not understood" input-port))))

;; Unparse (display) an object.
(define (unparse exp)
  (cond
   ((pair? (car exp))
    (write-char #\`) (unparse (car exp)) (unparse (cdr exp)))
   (else (case (car exp)
	   ((k) (write-char #\k))
	   ((k1) (write-char #\`) (write-char #\k) (unparse (cadr exp)))
	   ((s) (write-char #\s))
	   ((s1) (write-char #\`) (write-char #\s) (unparse (cadr exp)))
	   ((s2) (write-char #\`) (write-char #\`) (write-char #\s)
	    (unparse (cadr exp)) (unparse (caddr exp)))
	   ((i) (write-char #\i))
	   ((v) (write-char #\v))
	   ((c) (write-char #\c))
	   ((c1) (display "<continuation>"))
	   ((d) (write-char #\d))
	   ((d1) (write-char #\`) (write-char #\d) (unparse (cadr exp)))
	   ((e) (write-char #\e))
	   ((pr) (if (eqv? (cadr exp) #\newline) (write-char #\r)
		     (begin (write-char #\.) (write-char (cadr exp)))))
	   ((rd) (write-char #\@))
	   ((rc) (begin (write-char #\?) (write-char (cadr exp))))
	   ((pc) (write-char #\|))
	   (else
	    (error "UNPARSE" "Internal error: unexpected type to unparse!"
		   exp))))))

;; The eval function
(define (ev exp)
  (cond
   ((pair? (car exp))
    (let ((op (ev (car exp))))
      (if (eqv? (car op) 'd)
	  `(d1 ,(cdr exp))
	  (ap op (ev (cdr exp))))))
   (else exp)))

;; The exit continuation (makes the e function work).
(define (exit-cnt v)
  (error "EXIT-CNT" "Please start with entry-ev and not ev" v))
;; The first eval function (begins by capturing a continuation so that
;; the exit function works).
(define (entry-ev exp)
  (call-with-current-continuation
   (lambda (cnt)
     (set! exit-cnt cnt)
     (ev exp))))

;; Character under the reading head (``current character'')
(define current-char #f)

;; The apply function
(define (ap exp arg)
;; Uncomment the following lines to enable debugging
; (display "Debug:")
; (display "Expression: ") (unparse exp)
; (display "; applied to: ") (unparse arg) (newline)
  (case (car exp)
    ((k) `(k1 ,arg))
    ((k1) (cadr exp))
    ((s) `(s1 ,arg))
    ((s1) `(s2 ,(cadr exp) ,arg))
    ((s2) (ev `((,(cadr exp) . ,arg) . (,(caddr exp) . ,arg))))
    ((i) arg)
    ((v) '(v))
    ((c) (call-with-current-continuation
	  (lambda (cont) (ev `(,arg . (c1 ,cont))))))
    ((c1) ((cadr exp) arg))
;   ((d) 'impossible)
    ((d1) (ev `(,(cadr exp) . ,arg)))
    ((e) (exit-cnt arg))
    ((pr) (display (cadr exp)) arg)
    ((rd) (set! current-char (read-char))
     (ev `(,arg . ,(if (eof-object? current-char) '(v) '(i)))))
    ((rc) (ev `(,arg . ,(if (eqv? current-char (cadr exp))
			    '(i)
			    '(v)))))
    ((pc) (ev `(,arg . ,(if (and current-char
				 (not (eof-object? current-char)))
			    `(pr ,current-char)
			    '(v)))))
    (else (error "AP" "Internal error: unexpected type to apply!" exp))))

 
;; Main program function
(define (main argv)
  (let parse-args
      ((args (cdr argv))
       (arg-fileport #f)
       (opt-unparse #f)
       (options-done #f))
    (if (null? args)
	((if opt-unparse
	     (lambda (exp) (unparse exp) (newline))
	     (lambda (exp) #t))
	 (entry-ev (parse (if arg-fileport
			      arg-fileport
			      (current-input-port)))))
	(if (and (not options-done)
		 (eqv? (string-ref (car args) 0) #\-))
	    (cond ((equal? (car args) "-")
		   (parse-args (cdr args) (current-input-port) opt-unparse #t))
		  ((equal? (car args) "--")
		   (parse-args (cdr args) arg-fileport opt-unparse #t))
		  ((equal? (car args) "-e")
		   (if (null? (cdr args))
		       (error "MAIN" "Argument to -e missing" (cdr args))
		       (parse-args (cddr args) (open-input-string (cadr args))
				   opt-unparse options-done)))
		  ((equal? (car args) "-u")
		   (parse-args (cdr args) arg-fileport #t options-done))
		  ((equal? (car args) "-h")
		   (display "Syntax is unlambda [opts] [filename]") (newline)
		   (display "Options include:") (newline)
		   (display "  -h  this help") (newline)
		   (display "  -v  display version number") (newline)
		   (display "  -e  evaluate following unlambda expr") (newline)
		   (display "  -u  print resulting expression after eval") (newline)
		   (display "  --  terminate option line") (newline)
		   (display "A filename of - or none at all means read stdin")(newline))
		  ((equal? (car args) "-v")
		   (display "Unlambda Scheme Interpreter") (newline)
		   (display "Bigloo-compiled version 1.92.1") (newline))
		  (else
		   (error "MAIN" "Option not recognized (try -h)" (car args))))
	    (if arg-fileport
		(error "MAIN" "Multiple filenames specified" (car args))
		(parse-args (cdr args) (open-input-file (car args))
			    opt-unparse #t))))))
