
/*
 * GLX Server Extension
 * Copyright (C) 1996  Steven G. Parker  (sparker@cs.utah.edu)
 * Copyright (C) 1998, 1999  Terence Ripperda (ripperda@sgi.com)
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * STEPHEN PARKER, TERENCE RIPPERDA, OR ANY OTHER CONTRIBUTORS BE LIABLE FOR 
 * ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, 
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE 
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

/*
 * GL Non-Rendering Commands
 * modified July 14, 1998 by Terence Ripperda (ripperda@engr.sgi.com)
 *
 * to help clean up some of the code and files I am splitting them
 * up along more distinguished lines. This file corresponds to 
 * section 9.2 of the GLX Extension for OpenGL Protocol Specification
 * Version 1.2. This section describes the protocol for OpenGL calls
 * that do not render and therefore are not includable in a display
 * list. 
 */

#include "glxlib.h"
#include "misc.h"
#include "glxcommon.h"
#include "extensions/Xext.h"
#include "extensions/extutil.h"
#include "GL/GLXproto.h"
#include <stdio.h>
#include <assert.h>
#include "buffer_macros.h"

extern GLContext* GLCurrent;
extern XExtDisplayInfo * __gl_find_display(Display*);

static char* gl_extension_name = "GLX";

#define GLCheckExtension(dpy, i, val) \
   XextCheckExtension(dpy, i, gl_extension_name, val)

#define GLSimpleCheckExtension(dpy, i) \
   XextSimpleCheckExtension(dpy, i, gl_extension_name)



GLboolean __glx_AreTexturesResident(GLsizei n, const GLuint *textureNames,
    GLboolean *residences)
{
    XExtDisplayInfo* info = __gl_find_display(GLCurrent->dpy);
    xGLXAreTexturesResidentReq* req;
    xGLXAreTexturesResidentReply reply;
    Display* dpy=GLCurrent->dpy;
    unsigned int num_bytes = 0;
    unsigned int tmp = 0;
    Bool reply_flag = True;

    assert(textureNames);
    assert(residences);

    num_bytes = n * 4;

    GLXRenderFlush();
    LockDisplay(dpy);
    GetReqExtra(GLXAreTexturesResident, num_bytes, req);

    req->reqType = info->codes->major_opcode;
    req->glx_opcode = X_GLXAreTexturesResident;
    req->context_tag = GLCurrent->contextTag;
    req->n = n;

    memcpy( (void *) ( ((char *) req) + sz_xGLXAreTexturesResidentReq), 
       (void *) textureNames, num_bytes);

    reply_flag = _XReply(dpy, (xReply *) &reply, 0, xFalse);
    if (reply_flag == False) {
        UnlockDisplay(dpy);
        SyncHandle();
        return GL_FALSE;
    }

    num_bytes = (n + 3) >> 2;
    tmp = num_bytes & 3;
    _XRead(dpy, (char *) residences, num_bytes);
    if (tmp) _XEatData(dpy, 4 - tmp);
    UnlockDisplay(dpy);
    SyncHandle();

    return reply.value;
}


void __glx_DeleteLists( GLuint list, GLsizei range)
{
    XExtDisplayInfo* info = __gl_find_display(GLCurrent->dpy);
    xGLXDeleteListsReq* req;
    Display* dpy=GLCurrent->dpy;

    GLSimpleCheckExtension(GLCurrent->dpy, info);

    GLXRenderFlush();
    LockDisplay(dpy);
    GetReq(GLXDeleteLists, req);
    req->reqType = info->codes->major_opcode;
    req->glx_opcode = X_GLXDeleteLists;
    req->context_tag = GLCurrent->contextTag;

    req->list=list;
    req->range=range;

    UnlockDisplay(GLCurrent->dpy);
    SyncHandle();
}


void __glx_DeleteTextures(GLsizei n, const GLuint *textures)
{
    XExtDisplayInfo* info = __gl_find_display(GLCurrent->dpy);
    Display* dpy=GLCurrent->dpy;
    unsigned int num_bytes = 4 * n;

    GLSimpleCheckExtension(GLCurrent->dpy, info);

    GLXRenderFlush();
    LockDisplay(dpy);
    if (GLCurrent->SGIhost) {
       xGLXVendorPrivateReq* req;

       GetReq(GLXVendorPrivate, req);
       req->reqType = info->codes->major_opcode;
       req->glx_opcode = X_GLXVendorPrivate;
       req->opcode = X_GLXDeleteTexturesSGI; /* SGI's vendorPrivate opcode */
       req->context_tag = GLCurrent->contextTag;
       req->length += num_bytes >> 2;
       _XSend(dpy, (char *) textures,  num_bytes);
    } else {
       xGLXDeleteTexturesReq* req;

       GetReq(GLXDeleteTextures, req);

       req->reqType = info->codes->major_opcode;
       req->glx_opcode = X_GLXDeleteTextures;
       req->context_tag = GLCurrent->contextTag;
       req->n = n;
       req->length += GLX_pad(num_bytes) >> 2;
       _XSend(dpy, (char *) textures, num_bytes);
    }
    UnlockDisplay(dpy);
    SyncHandle();
    return;
}

void __glx_EndList()
{
    XExtDisplayInfo* info = __gl_find_display(GLCurrent->dpy);
    xGLXEndListReq* req;
    Display* dpy=GLCurrent->dpy;

    GLSimpleCheckExtension(GLCurrent->dpy, info);

    GLXRenderFlush();
    LockDisplay(dpy);
    GetReq(GLXEndList, req);
    req->reqType = info->codes->major_opcode;
    req->glx_opcode = X_GLXEndList;
    req->context_tag = GLCurrent->contextTag;
    UnlockDisplay(GLCurrent->dpy);
    SyncHandle();
}


void __glx_FeedbackBuffer(GLsizei size, GLenum type, GLfloat* buffer)
{
    XExtDisplayInfo* info = __gl_find_display(GLCurrent->dpy);
    xGLXFeedbackBufferReq* req;
    Display* dpy=GLCurrent->dpy;

    GLSimpleCheckExtension(GLCurrent->dpy, info);

    GLXRenderFlush();
    LockDisplay(dpy);
    GetReq(GLXFeedbackBuffer, req);
    req->reqType = info->codes->major_opcode;
    req->glx_opcode = X_GLXFeedbackBuffer;
    req->context_tag = GLCurrent->contextTag;

    req->size=size;
    req->type=type;
    GLCurrent->feedback_buffer=buffer;
    GLCurrent->feedback_buffer_size=size;

    UnlockDisplay(GLCurrent->dpy);
    SyncHandle();
}

void __glx_Finish()
{
    XExtDisplayInfo* info = __gl_find_display(GLCurrent->dpy);
    xGLXFinishReq* req;
    xGLXFinishReply reply;
    Display* dpy=GLCurrent->dpy;

    GLSimpleCheckExtension(GLCurrent->dpy, info);

    GLXRenderFlush();
    LockDisplay(dpy);
    GetReq(GLXFinish, req);
    req->reqType = info->codes->major_opcode;
    req->glx_opcode = X_GLXFinish;
    req->context_tag = GLCurrent->contextTag;

    _XReply(dpy, (xReply*)&reply, 0, xTrue);
    UnlockDisplay(dpy);
    SyncHandle();
}

void __glx_Flush()
{
    XExtDisplayInfo* info = __gl_find_display(GLCurrent->dpy);
    xGLXFlushReq* req;
    Display* dpy=GLCurrent->dpy;

    GLSimpleCheckExtension(GLCurrent->dpy, info);

    GLXRenderFlush();
    LockDisplay(dpy);
    GetReq(GLXFlush, req);
    req->reqType = info->codes->major_opcode;
    req->glx_opcode = X_GLXFlush;
    req->context_tag = GLCurrent->contextTag;
    UnlockDisplay(GLCurrent->dpy);
    SyncHandle();
    XFlush(dpy);
}

GLuint __glx_GenLists(GLsizei range)
{
    XExtDisplayInfo* info = __gl_find_display(GLCurrent->dpy);
    xGLXGenListsReq* req;
    xGLXGenListsReply reply;
    Display* dpy=GLCurrent->dpy;

    GLCheckExtension(GLCurrent->dpy, info, 0);

    GLXRenderFlush();
    LockDisplay(dpy);
    GetReq(GLXGenLists, req);

    req->reqType = info->codes->major_opcode;
    req->glx_opcode = X_GLXGenLists;
    req->context_tag = GLCurrent->contextTag;
    req->data=range;

    _XReply(dpy, (xReply*)&reply, 0, xTrue);
    UnlockDisplay(dpy);
    SyncHandle();
    return reply.data;
}

void __glx_GenTextures(GLsizei n, GLuint *textureNames)
{
    XExtDisplayInfo* info = __gl_find_display(GLCurrent->dpy);
    Display* dpy=GLCurrent->dpy;
    unsigned int num_bytes = 4 * n;

    GLSimpleCheckExtension(GLCurrent->dpy, info);

    GLXRenderFlush();
    LockDisplay(dpy);
    
    if (GLCurrent->SGIhost) {
       xGLXVendorPrivateReq* req;
       xGLXVendorPrivateReply reply;

       GetReq(GLXVendorPrivate, req);

       req->reqType = info->codes->major_opcode;
       req->glx_opcode = X_GLXVendorPrivate;
       req->opcode = X_GLXGenTextures;
       req->length += GLX_pad(sizeof(GLsizei)) >> 2;
       _XSend(dpy, (char *) &n,  GLX_pad(sizeof(GLsizei)));
       _XReply(dpy, (xReply*)&reply, 0, xFalse);
       _XRead(dpy, (char *) textureNames, n * sizeof(GLsizei));

    } else {
       xGLXGenTexturesReq* req;
       xGLXGenTexturesReply reply;

       GetReq(GLXGenTextures, req);

       req->reqType = info->codes->major_opcode;
       req->glx_opcode = X_GLXGenTextures;
       req->context_tag = GLCurrent->contextTag;
       req->n = n;

       _XReply(dpy, (xReply *) &reply, 0, xTrue);
       _XRead(dpy, (char *) textureNames, num_bytes);
    }
    UnlockDisplay(dpy);
    SyncHandle();
    return;
}

void __glx_GetBooleanv( GLenum pname, GLboolean *params )
{
    XExtDisplayInfo* info = __gl_find_display(GLCurrent->dpy);
    xGLXGetBooleanvReq* req;
    xGLXGetBooleanvReply reply;
    Display* dpy=GLCurrent->dpy;
    int length;

    GLSimpleCheckExtension(GLCurrent->dpy, info);

    GLXRenderFlush();
    LockDisplay(dpy);
    GetReq(GLXGetBooleanv, req);

    req->reqType = info->codes->major_opcode;
    req->glx_opcode = X_GLXGetBooleanv;
    req->context_tag = GLCurrent->contextTag;
    req->data=pname;

    _XReply(dpy, (xReply*)&reply, 0, xFalse);

    length = reply.length;
    if(length==0){
	params[0]=reply.singlebool;
    } else {
	if((reply.n%4) == 0)
	    _XRead(dpy, (char*)params, 4L*length);
	else {
	    /* Allocate temporary space to avoid over-running the buffer.... */
	    int i;
	    GLboolean *tmp_bools = NULL;
	    tmp_bools=(GLboolean*)malloc(4L*length);
	    _XRead(dpy, (char*) tmp_bools, 4L*length);
	    for(i=0;i<reply.n;i++)
		params[i]=tmp_bools[i];
	    free(tmp_bools);
	}
    }
    UnlockDisplay(dpy);
    SyncHandle();
}


void __glx_GetClipPlane( GLenum plane, GLdouble *equation )
{
    XExtDisplayInfo* info = __gl_find_display(GLCurrent->dpy);
    xGLXGetClipPlaneReq* req;
    xGLXGetClipPlaneReply reply;
    Display* dpy=GLCurrent->dpy;
    int length;

    GLSimpleCheckExtension(GLCurrent->dpy, info);

    GLXRenderFlush();
    LockDisplay(dpy);
    GetReq(GLXGetClipPlane, req);

    req->reqType = info->codes->major_opcode;
    req->glx_opcode = X_GLXGetClipPlane;
    req->context_tag = GLCurrent->contextTag;
    req->data=plane;

    _XReply(dpy, (xReply*)&reply, 0, xFalse);

    length = reply.length;
    _XRead(dpy, (char*)equation, 4L*length);
    UnlockDisplay(dpy);
    SyncHandle();
}


void __glx_GetDoublev(GLenum pname, GLdouble* params)
{
    XExtDisplayInfo* info = __gl_find_display(GLCurrent->dpy);
    xGLXGetDoublevReq* req;
    xGLXGetDoublevReply reply;
    Display* dpy=GLCurrent->dpy;
    int length;

    GLSimpleCheckExtension(GLCurrent->dpy, info);

    GLXRenderFlush();
    LockDisplay(dpy);
    GetReq(GLXGetDoublev, req);

    req->reqType = info->codes->major_opcode;
    req->glx_opcode = X_GLXGetDoublev;
    req->context_tag = GLCurrent->contextTag;
    req->data=pname;

    _XReply(dpy, (xReply*)&reply, 0, xFalse);

    length = reply.length;
    if(length==0){
	params[0]=reply.singledata;
    } else {
	_XRead(dpy, (char*)params, 4L*length);
    }
    UnlockDisplay(dpy);
    SyncHandle();
}

GLenum __glx_GetError()
{
    XExtDisplayInfo* info = __gl_find_display(GLCurrent->dpy);
    xGLXGetErrorReq* req;
    xGLXGetErrorReply reply;
    Display* dpy=GLCurrent->dpy;

    GLCheckExtension(GLCurrent->dpy, info, GL_FALSE);

    GLXRenderFlush();
    LockDisplay(dpy);
    GetReq(GLXGetError, req);

    req->reqType = info->codes->major_opcode;
    req->glx_opcode = X_GLXGetError;
    req->context_tag = GLCurrent->contextTag;

    _XReply(dpy, (xReply*)&reply, 0, xFalse);
    UnlockDisplay(dpy);
    SyncHandle();
    return reply.data;
}

void __glx_GetFloatv(GLenum pname, GLfloat* params)
{
    XExtDisplayInfo* info = __gl_find_display(GLCurrent->dpy);
    xGLXGetFloatvReq* req;
    xGLXGetFloatvReply reply;
    Display* dpy=GLCurrent->dpy;
    int length;

    GLSimpleCheckExtension(GLCurrent->dpy, info);

    GLXRenderFlush();
    LockDisplay(dpy);
    GetReq(GLXGetFloatv, req);

    req->reqType = info->codes->major_opcode;
    req->glx_opcode = X_GLXGetFloatv;
    req->context_tag = GLCurrent->contextTag;
    req->data=pname;

    _XReply(dpy, (xReply*)&reply, 0, xFalse);

    length = reply.length;
    if(length==0){
	params[0]=reply.singledata;
    } else {
	_XRead(dpy, (char*)params, 4L*length);
    }
    UnlockDisplay(dpy);
    SyncHandle();
}


void __glx_GetIntegerv(GLenum pname, GLint* params)
{
    XExtDisplayInfo* info = __gl_find_display(GLCurrent->dpy);
    xGLXGetIntegervReq* req;
    xGLXGetIntegervReply reply;
    Display* dpy=GLCurrent->dpy;
    int length;

    GLSimpleCheckExtension(GLCurrent->dpy, info);

    /* check for client-side info */
    /* a round trip to server is slow anyways, so I'm not worried about speed */
    switch (pname) {
       case GL_UNPACK_SWAP_BYTES:
          *params = GLCurrent->Unpack.SwapBytes;
          return;
       case GL_UNPACK_LSB_FIRST:
          *params = GLCurrent->Unpack.LsbFirst;
          return;
       case GL_UNPACK_ROW_LENGTH:
          *params = GLCurrent->Unpack.RowLength;
          return;
       case GL_UNPACK_SKIP_PIXELS:
          *params = GLCurrent->Unpack.SkipPixels;
          return;
       case GL_UNPACK_SKIP_ROWS:
          *params = GLCurrent->Unpack.SkipRows;
          return;
       case GL_UNPACK_ALIGNMENT:
          *params = GLCurrent->Unpack.Alignment;
          return;
       default:
    }

    GLXRenderFlush();
    LockDisplay(dpy);
    GetReq(GLXGetIntegerv, req);

    req->reqType = info->codes->major_opcode;
    req->glx_opcode = X_GLXGetIntegerv;
    req->context_tag = GLCurrent->contextTag;
    req->data=pname;

    _XReply(dpy, (xReply*)&reply, 0, xFalse);

    length = reply.length;
    if(length==0){
	params[0]=reply.singledata;
    } else {
	_XRead(dpy, (char*)params, 4L*length);
    }
    UnlockDisplay(dpy);
    SyncHandle();
}




void __glx_GetLightfv( GLenum light, GLenum pname, GLfloat *params )
{
    XExtDisplayInfo* info = __gl_find_display(GLCurrent->dpy);
    xGLXGetLightfvReq* req;
    xGLXGetLightfvReply reply;
    Display* dpy=GLCurrent->dpy;
    int length;

    GLSimpleCheckExtension(GLCurrent->dpy, info);

    GLXRenderFlush();
    LockDisplay(dpy);
    GetReq(GLXGetLightfv, req);

    req->reqType = info->codes->major_opcode;
    req->glx_opcode = X_GLXGetLightfv;
    req->context_tag = GLCurrent->contextTag;
    req->light=light;
    req->pname=pname;

    _XReply(dpy, (xReply*)&reply, 0, xFalse);

    length = reply.length;
    if(length==0){
	params[0]=reply.singledata;
    } else {
	_XRead(dpy, (char*)params, 4L*length);
    }
    UnlockDisplay(dpy);
    SyncHandle();
}

void __glx_GetLightiv( GLenum light, GLenum pname, GLint *params )
{
    XExtDisplayInfo* info = __gl_find_display(GLCurrent->dpy);
    xGLXGetLightivReq* req;
    xGLXGetLightivReply reply;
    Display* dpy=GLCurrent->dpy;
    int length;

    GLSimpleCheckExtension(GLCurrent->dpy, info);

    GLXRenderFlush();
    LockDisplay(dpy);
    GetReq(GLXGetLightiv, req);

    req->reqType = info->codes->major_opcode;
    req->glx_opcode = X_GLXGetLightiv;
    req->context_tag = GLCurrent->contextTag;
    req->light=light;
    req->pname=pname;

    _XReply(dpy, (xReply*)&reply, 0, xFalse);

    length = reply.length;
    if(length==0){
	params[0]=reply.singledata;
    } else {
	_XRead(dpy, (char*)params, 4L*length);
    }
    UnlockDisplay(dpy);
    SyncHandle();
}


void __glx_GetMapdv( GLenum target, GLenum query, GLdouble *v )
{
    XExtDisplayInfo* info = __gl_find_display(GLCurrent->dpy);
    xGLXGetMapdvReq* req;
    xGLXGetMapdvReply reply;
    Display* dpy=GLCurrent->dpy;
    int length;

    GLSimpleCheckExtension(GLCurrent->dpy, info);

    GLXRenderFlush();
    LockDisplay(dpy);
    GetReq(GLXGetMapdv, req);

    req->reqType = info->codes->major_opcode;
    req->glx_opcode = X_GLXGetMapdv;
    req->context_tag = GLCurrent->contextTag;
    req->target=target;
    req->query=query;

    _XReply(dpy, (xReply*)&reply, 0, xFalse);

    length = reply.length;
    if(length==0){
	v[0]=reply.singledata;
    } else {
	_XRead(dpy, (char*)v, 4L*length);
    }
    UnlockDisplay(dpy);
    SyncHandle();
}


void __glx_GetMapfv( GLenum target, GLenum query, GLfloat *v )
{
    XExtDisplayInfo* info = __gl_find_display(GLCurrent->dpy);
    xGLXGetMapfvReq* req;
    xGLXGetMapfvReply reply;
    Display* dpy=GLCurrent->dpy;
    int length;

    GLSimpleCheckExtension(GLCurrent->dpy, info);

    GLXRenderFlush();
    LockDisplay(dpy);
    GetReq(GLXGetMapfv, req);

    req->reqType = info->codes->major_opcode;
    req->glx_opcode = X_GLXGetMapfv;
    req->context_tag = GLCurrent->contextTag;
    req->target=target;
    req->query=query;

    _XReply(dpy, (xReply*)&reply, 0, xFalse);

    length = reply.length;
    if(length==0){
	v[0]=reply.singledata;
    } else {
	_XRead(dpy, (char*)v, 4L*length);
    }
    UnlockDisplay(dpy);
    SyncHandle();
}


void __glx_GetMapiv( GLenum target, GLenum query, GLint *v )
{
    XExtDisplayInfo* info = __gl_find_display(GLCurrent->dpy);
    xGLXGetMapivReq* req;
    xGLXGetMapivReply reply;
    Display* dpy=GLCurrent->dpy;
    int length;

    GLSimpleCheckExtension(GLCurrent->dpy, info);

    GLXRenderFlush();
    LockDisplay(dpy);
    GetReq(GLXGetMapiv, req);

    req->reqType = info->codes->major_opcode;
    req->glx_opcode = X_GLXGetMapiv;
    req->context_tag = GLCurrent->contextTag;
    req->target=target;
    req->query=query;

    _XReply(dpy, (xReply*)&reply, 0, xFalse);

    length = reply.length;
    if(length==0){
	v[0]=reply.singledata;
    } else {
	_XRead(dpy, (char*)v, 4L*length);
    }
    UnlockDisplay(dpy);
    SyncHandle();
}

void __glx_GetMaterialfv( GLenum face, GLenum pname, GLfloat *params )
{
    XExtDisplayInfo* info = __gl_find_display(GLCurrent->dpy);
    xGLXGetMaterialfvReq* req;
    xGLXGetMaterialfvReply reply;
    Display* dpy=GLCurrent->dpy;
    int length;

    GLSimpleCheckExtension(GLCurrent->dpy, info);

    GLXRenderFlush();
    LockDisplay(dpy);
    GetReq(GLXGetMaterialfv, req);

    req->reqType = info->codes->major_opcode;
    req->glx_opcode = X_GLXGetMaterialfv;
    req->context_tag = GLCurrent->contextTag;
    req->face=face;
    req->pname=pname;

    _XReply(dpy, (xReply*)&reply, 0, xFalse);

    length = reply.length;
    if(length==0){
	params[0]=reply.singledata;
    } else {
	_XRead(dpy, (char*)params, 4L*length);
    }
    UnlockDisplay(dpy);
    SyncHandle();
}

void __glx_GetMaterialiv( GLenum face, GLenum pname, GLint *params )
{
    XExtDisplayInfo* info = __gl_find_display(GLCurrent->dpy);
    xGLXGetMaterialivReq* req;
    xGLXGetMaterialivReply reply;
    Display* dpy=GLCurrent->dpy;
    int length;

    GLSimpleCheckExtension(GLCurrent->dpy, info);

    GLXRenderFlush();
    LockDisplay(dpy);
    GetReq(GLXGetMaterialiv, req);

    req->reqType = info->codes->major_opcode;
    req->glx_opcode = X_GLXGetMaterialiv;
    req->context_tag = GLCurrent->contextTag;
    req->face=face;
    req->pname=pname;

    _XReply(dpy, (xReply*)&reply, 0, xFalse);

    length = reply.length;
    if(length==0){
	params[0]=reply.singledata;
    } else {
	_XRead(dpy, (char*)params, 4L*length);
    }
    UnlockDisplay(dpy);
    SyncHandle();
}

void __glx_GetPixelMapfv( GLenum map, GLfloat *values )
{
    XExtDisplayInfo* info = __gl_find_display(GLCurrent->dpy);
    xGLXGetPixelMapfvReq* req;
    xGLXGetPixelMapfvReply reply;
    Display* dpy=GLCurrent->dpy;
    int length;

    GLSimpleCheckExtension(GLCurrent->dpy, info);

    GLXRenderFlush();
    LockDisplay(dpy);
    GetReq(GLXGetPixelMapfv, req);

    req->reqType = info->codes->major_opcode;
    req->glx_opcode = X_GLXGetPixelMapfv;
    req->context_tag = GLCurrent->contextTag;
    req->data=map;

    _XReply(dpy, (xReply*)&reply, 0, xFalse);

    length = reply.length;
    if(length==0){
	values[0]=reply.singledata;
    } else {
	_XRead(dpy, (char*)values, 4L*length);
    }
    UnlockDisplay(dpy);
    SyncHandle();
}

void __glx_GetPixelMapuiv( GLenum map, GLuint *values )
{
    XExtDisplayInfo* info = __gl_find_display(GLCurrent->dpy);
    xGLXGetPixelMapuivReq* req;
    xGLXGetPixelMapuivReply reply;
    Display* dpy=GLCurrent->dpy;
    int length;

    GLSimpleCheckExtension(GLCurrent->dpy, info);

    GLXRenderFlush();
    LockDisplay(dpy);
    GetReq(GLXGetPixelMapuiv, req);

    req->reqType = info->codes->major_opcode;
    req->glx_opcode = X_GLXGetPixelMapuiv;
    req->context_tag = GLCurrent->contextTag;
    req->data=map;

    _XReply(dpy, (xReply*)&reply, 0, xFalse);

    length = reply.length;
    if(length==0){
	values[0]=reply.singledata;
    } else {
	_XRead(dpy, (char*)values, 4L*length);
    }
    UnlockDisplay(dpy);
    SyncHandle();
}

void __glx_GetPixelMapusv( GLenum map, GLushort *values )
{
    XExtDisplayInfo* info = __gl_find_display(GLCurrent->dpy);
    xGLXGetPixelMapusvReq* req;
    xGLXGetPixelMapusvReply reply;
    Display* dpy=GLCurrent->dpy;
    int length;

    GLSimpleCheckExtension(GLCurrent->dpy, info);

    GLXRenderFlush();
    LockDisplay(dpy);
    GetReq(GLXGetPixelMapusv, req);

    req->reqType = info->codes->major_opcode;
    req->glx_opcode = X_GLXGetPixelMapusv;
    req->context_tag = GLCurrent->contextTag;
    req->data=map;

    _XReply(dpy, (xReply*)&reply, 0, xFalse);

    length = reply.length;
    if(length==0){
	values[0]=reply.singledata;
    } else {
	if((reply.n%2) == 0)
	    _XRead(dpy, (char*)values, 4L*length);
	else {
	    /* Allocate temporary space to avoid over-running the buffer.... */
	    int i;
	    unsigned short* tmp_values;
	    tmp_values=(unsigned short*)malloc(4L*length);
	    _XRead(dpy, (char*)tmp_values, 4L*length);
	    for(i=0;i<reply.n;i++)
		values[i]=tmp_values[i];
	    free(tmp_values);
	}
    }
    UnlockDisplay(dpy);
    SyncHandle();
}

void __glx_GetTexGendv( GLenum coord, GLenum pname, GLdouble *params )
{
    XExtDisplayInfo* info = __gl_find_display(GLCurrent->dpy);
    xGLXGetTexGendvReq* req;
    xGLXGetTexGendvReply reply;
    Display* dpy=GLCurrent->dpy;
    int length;

    GLSimpleCheckExtension(GLCurrent->dpy, info);

    GLXRenderFlush();
    LockDisplay(dpy);
    GetReq(GLXGetTexGendv, req);

    req->reqType = info->codes->major_opcode;
    req->glx_opcode = X_GLXGetTexGendv;
    req->context_tag = GLCurrent->contextTag;
    req->coord=coord;
    req->pname=pname;

    _XReply(dpy, (xReply*)&reply, 0, xFalse);

    length = reply.length;
    if(length==0){
	params[0]=reply.singledata;
    } else {
	_XRead(dpy, (char*)params, 4L*length);
    }
    UnlockDisplay(dpy);
    SyncHandle();
}

const GLubyte *__glx_GetString( GLenum name )
{
    XExtDisplayInfo* info = __gl_find_display(GLCurrent->dpy);
    xGLXGetStringReq* req;
    xGLXGetStringReply reply;
    Display* dpy=GLCurrent->dpy;
    int length;
    static GLubyte* string = 0;
    static GLint slength=0;

    GLCheckExtension(GLCurrent->dpy, info, NULL);

    GLXRenderFlush();
    LockDisplay(dpy);
    GetReq(GLXGetString, req);

    req->reqType = info->codes->major_opcode;
    req->glx_opcode = X_GLXGetString;
    req->context_tag = GLCurrent->contextTag;
    req->data=name;

    _XReply(dpy, (xReply*)&reply, 0, xFalse);

    length = reply.length*4L;
    if(length > slength || !string){
	if (string) free(string);
	string = malloc(length);
	slength = length;
    }
    _XRead(dpy, (char*)string, length);
    UnlockDisplay(dpy);
    SyncHandle();
    /* XXX verify this works */
    /* return (unsigned char *) strdup( (char *) string); */
    return string;
}

void __glx_GetTexEnvfv( GLenum target, GLenum pname, GLfloat *params )
{
    XExtDisplayInfo* info = __gl_find_display(GLCurrent->dpy);
    xGLXGetTexEnvfvReq* req;
    xGLXGetTexEnvfvReply reply;
    Display* dpy=GLCurrent->dpy;
    int length;

    GLSimpleCheckExtension(GLCurrent->dpy, info);

    GLXRenderFlush();
    LockDisplay(dpy);
    GetReq(GLXGetTexEnvfv, req);

    req->reqType = info->codes->major_opcode;
    req->glx_opcode = X_GLXGetTexEnvfv;
    req->context_tag = GLCurrent->contextTag;
    req->target=target;
    req->pname=pname;

    _XReply(dpy, (xReply*)&reply, 0, xFalse);

    length = reply.length;
    if(length==0){
	params[0]=reply.singledata;
    } else {
	_XRead(dpy, (char*)params, 4L*length);
    }
    UnlockDisplay(dpy);
    SyncHandle();
}


void __glx_GetTexEnviv( GLenum target, GLenum pname, GLint *params )
{
    XExtDisplayInfo* info = __gl_find_display(GLCurrent->dpy);
    xGLXGetTexEnvivReq* req;
    xGLXGetTexEnvivReply reply;
    Display* dpy=GLCurrent->dpy;
    int length;

    GLSimpleCheckExtension(GLCurrent->dpy, info);

    GLXRenderFlush();
    LockDisplay(dpy);
    GetReq(GLXGetTexEnviv, req);

    req->reqType = info->codes->major_opcode;
    req->glx_opcode = X_GLXGetTexEnviv;
    req->context_tag = GLCurrent->contextTag;
    req->target=target;
    req->pname=pname;

    _XReply(dpy, (xReply*)&reply, 0, xFalse);

    length = reply.length;
    if(length==0){
	params[0]=reply.singledata;
    } else {
	_XRead(dpy, (char*)params, 4L*length);
    }
    UnlockDisplay(dpy);
    SyncHandle();
}

void __glx_GetTexGenfv( GLenum coord, GLenum pname, GLfloat *params )
{
    XExtDisplayInfo* info = __gl_find_display(GLCurrent->dpy);
    xGLXGetTexGenfvReq* req;
    xGLXGetTexGenfvReply reply;
    Display* dpy=GLCurrent->dpy;
    int length;

    GLSimpleCheckExtension(GLCurrent->dpy, info);

    GLXRenderFlush();
    LockDisplay(dpy);
    GetReq(GLXGetTexGenfv, req);

    req->reqType = info->codes->major_opcode;
    req->glx_opcode = X_GLXGetTexGenfv;
    req->context_tag = GLCurrent->contextTag;
    req->coord=coord;
    req->pname=pname;

    _XReply(dpy, (xReply*)&reply, 0, xFalse);

    length = reply.length;
    if(length==0){
	params[0]=reply.singledata;
    } else {
	_XRead(dpy, (char*)params, 4L*length);
    }
    UnlockDisplay(dpy);
    SyncHandle();
}

void __glx_GetTexGeniv( GLenum coord, GLenum pname, GLint *params )
{
    XExtDisplayInfo* info = __gl_find_display(GLCurrent->dpy);
    xGLXGetTexGenivReq* req;
    xGLXGetTexGenivReply reply;
    Display* dpy=GLCurrent->dpy;
    int length;

    GLSimpleCheckExtension(GLCurrent->dpy, info);

    GLXRenderFlush();
    LockDisplay(dpy);
    GetReq(GLXGetTexGeniv, req);

    req->reqType = info->codes->major_opcode;
    req->glx_opcode = X_GLXGetTexGeniv;
    req->context_tag = GLCurrent->contextTag;
    req->coord=coord;
    req->pname=pname;

    _XReply(dpy, (xReply*)&reply, 0, xFalse);

    length = reply.length;
    if(length==0){
	params[0]=reply.singledata;
    } else {
	_XRead(dpy, (char*)params, 4L*length);
    }
    UnlockDisplay(dpy);
    SyncHandle();
}


void __glx_GetTexLevelParameterfv( GLenum target, GLint level,
			      GLenum pname, GLfloat *params )
{
    XExtDisplayInfo* info = __gl_find_display(GLCurrent->dpy);
    xGLXGetTexLevelParameterfvReq* req;
    xGLXGetTexLevelParameterfvReply reply;
    Display* dpy=GLCurrent->dpy;
    int length;

    GLSimpleCheckExtension(GLCurrent->dpy, info);

    GLXRenderFlush();
    LockDisplay(dpy);
    GetReq(GLXGetTexLevelParameterfv, req);

    req->reqType = info->codes->major_opcode;
    req->glx_opcode = X_GLXGetTexLevelParameterfv;
    req->context_tag = GLCurrent->contextTag;
    req->target=target;
    req->level=level;
    req->pname=pname;

    _XReply(dpy, (xReply*)&reply, 0, xFalse);

    length = reply.length;
    if(length==0){
	params[0]=reply.singledata;
    } else {
	_XRead(dpy, (char*)params, 4L*length);
    }
    UnlockDisplay(dpy);
    SyncHandle();
}


void __glx_GetTexLevelParameteriv( GLenum target, GLint level,
			      GLenum pname, GLint *params )
{
    XExtDisplayInfo* info = __gl_find_display(GLCurrent->dpy);
    xGLXGetTexLevelParameterivReq* req;
    xGLXGetTexLevelParameterivReply reply;
    Display* dpy=GLCurrent->dpy;
    int length;

    GLSimpleCheckExtension(GLCurrent->dpy, info);

    GLXRenderFlush();
    LockDisplay(dpy);
    GetReq(GLXGetTexLevelParameteriv, req);

    req->reqType = info->codes->major_opcode;
    req->glx_opcode = X_GLXGetTexLevelParameteriv;
    req->context_tag = GLCurrent->contextTag;
    req->target=target;
    req->level=level;
    req->pname=pname;

    _XReply(dpy, (xReply*)&reply, 0, xFalse);

    length = reply.length;
    if(length==0){
	params[0]=reply.singledata;
    } else {
	_XRead(dpy, (char*)params, 4L*length);
    }
    UnlockDisplay(dpy);
    SyncHandle();
}


void __glx_GetTexParameterfv( GLenum target, GLenum pname, GLfloat *params)
{
    XExtDisplayInfo* info = __gl_find_display(GLCurrent->dpy);
    xGLXGetTexParameterfvReq* req;
    xGLXGetTexParameterfvReply reply;
    Display* dpy=GLCurrent->dpy;
    int length;

    GLSimpleCheckExtension(GLCurrent->dpy, info);

    GLXRenderFlush();
    LockDisplay(dpy);
    GetReq(GLXGetTexParameterfv, req);

    req->reqType = info->codes->major_opcode;
    req->glx_opcode = X_GLXGetTexParameterfv;
    req->context_tag = GLCurrent->contextTag;
    req->target=target;
    req->pname=pname;

    _XReply(dpy, (xReply*)&reply, 0, xFalse);

    length = reply.length;
    if(length==0){
	params[0]=reply.singledata;
    } else {
	_XRead(dpy, (char*)params, 4L*length);
    }
    UnlockDisplay(dpy);
    SyncHandle();
}

void __glx_GetTexParameteriv( GLenum target, GLenum pname, GLint *params )
{
    XExtDisplayInfo* info = __gl_find_display(GLCurrent->dpy);
    xGLXGetTexParameterivReq* req;
    xGLXGetTexParameterivReply reply;
    Display* dpy=GLCurrent->dpy;
    int length;

    GLSimpleCheckExtension(GLCurrent->dpy, info);

    GLXRenderFlush();
    LockDisplay(dpy);
    GetReq(GLXGetTexParameteriv, req);

    req->reqType = info->codes->major_opcode;
    req->glx_opcode = X_GLXGetTexParameteriv;
    req->context_tag = GLCurrent->contextTag;
    req->target=target;
    req->pname=pname;

    _XReply(dpy, (xReply*)&reply, 0, xFalse);

    length = reply.length;
    if(length==0){
	params[0]=reply.singledata;
    } else {
	_XRead(dpy, (char*)params, 4L*length);
    }
    UnlockDisplay(dpy);
    SyncHandle();
}

GLboolean __glx_IsEnabled(GLenum cap)
{
    XExtDisplayInfo* info = __gl_find_display(GLCurrent->dpy);
    xGLXIsEnabledReq* req;
    xGLXIsEnabledReply reply;
    Display* dpy=GLCurrent->dpy;

    GLCheckExtension(GLCurrent->dpy, info, GL_FALSE);

    GLXRenderFlush();
    LockDisplay(dpy);
    GetReq(GLXIsEnabled, req);

    req->reqType = info->codes->major_opcode;
    req->glx_opcode = X_GLXIsEnabled;
    req->context_tag = GLCurrent->contextTag;
    req->data=cap;

    _XReply(dpy, (xReply*)&reply, 0, xFalse);
    UnlockDisplay(dpy);
    SyncHandle();
    return reply.data;
}

GLboolean __glx_IsList(GLuint list)
{
    XExtDisplayInfo* info = __gl_find_display(GLCurrent->dpy);
    xGLXIsListReq* req;
    xGLXIsListReply reply;
    Display* dpy=GLCurrent->dpy;

    GLCheckExtension(GLCurrent->dpy, info, GL_FALSE);

    GLXRenderFlush();
    LockDisplay(dpy);
    GetReq(GLXIsList, req);

    req->reqType = info->codes->major_opcode;
    req->glx_opcode = X_GLXIsList;
    req->context_tag = GLCurrent->contextTag;
    req->data=list;

    _XReply(dpy, (xReply*)&reply, 0, xFalse);
    UnlockDisplay(dpy);
    SyncHandle();
    return reply.data;
}

GLboolean __glx_IsTexture(GLuint textureName)
{
    XExtDisplayInfo* info = __gl_find_display(GLCurrent->dpy);
    xGLXIsTextureReq* req;
    xGLXIsTextureReply reply;
    Display* dpy=GLCurrent->dpy;

    GLCheckExtension(GLCurrent->dpy, info, GL_FALSE);

    GLXRenderFlush();
    LockDisplay(dpy);
    GetReq(GLXIsTexture, req);

    req->reqType = info->codes->major_opcode;
    req->glx_opcode = X_GLXIsTexture;
    req->context_tag = GLCurrent->contextTag;
    req->texture=textureName;

    _XReply(dpy, (xReply*)&reply, 0, xFalse);
    UnlockDisplay(dpy);
    SyncHandle();
    return reply.value;
}


void __glx_NewList( GLuint list, GLenum mode)
{
    XExtDisplayInfo* info = __gl_find_display(GLCurrent->dpy);
    xGLXNewListReq* req;
    Display* dpy=GLCurrent->dpy;

    GLSimpleCheckExtension(GLCurrent->dpy, info);

    GLXRenderFlush();
    LockDisplay(dpy);
    GetReq(GLXNewList, req);
    req->reqType = info->codes->major_opcode;
    req->glx_opcode = X_GLXNewList;
    req->context_tag = GLCurrent->contextTag;

    req->list=list;
    req->mode=mode;

    UnlockDisplay(GLCurrent->dpy);
    SyncHandle();
}

/* check the current PixelStore options to see if unpacking is needed */
void __glx_check_unpacking() 
{
   GLCurrent->need_unpacking = GL_FALSE;

   if (GLCurrent->Unpack.SwapBytes != DEFAULT_UNPACK_SWAP_BYTES) {
      GLCurrent->need_unpacking = GL_TRUE;
      return;
   }

#if 0
   /* we're not unpacking by lsb, so we don't care */
   if (GLCurrent->Unpack.LsbFirst != DEFAULT_UNPACK_LSB_FIRST) {
      GLCurrent->need_unpacking = GL_TRUE;
      return;
   }
#endif

   if (GLCurrent->Unpack.RowLength != DEFAULT_UNPACK_ROW_LENGTH) {
      GLCurrent->need_unpacking = GL_TRUE;
      return;
   }

   if (GLCurrent->Unpack.SkipRows != DEFAULT_UNPACK_SKIP_ROWS) {
      GLCurrent->need_unpacking = GL_TRUE;
      return;
   }

   if (GLCurrent->Unpack.SkipPixels != DEFAULT_UNPACK_SKIP_PIXELS) {
      GLCurrent->need_unpacking = GL_TRUE;
      return;
   }

#if 0
   if (GLCurrent->Unpack.Alignment != DEFAULT_UNPACK_ALIGNMENT) {
      GLCurrent->need_unpacking = GL_TRUE;
      return;
   }
#endif

}

void __glx_PixelStoref( GLenum pname, GLfloat param)
{
    XExtDisplayInfo* info = __gl_find_display(GLCurrent->dpy);

    GLSimpleCheckExtension(GLCurrent->dpy, info);

    if(pname == GL_PACK_SWAP_BYTES){
	GLCurrent->Pack.SwapBytes=(int)param;
    } else if(pname == GL_PACK_LSB_FIRST){
	GLCurrent->Pack.LsbFirst=(int)param;
    } else if(pname == GL_PACK_ROW_LENGTH){
	GLCurrent->Pack.RowLength=(int)param;
    } else if(pname == GL_PACK_SKIP_PIXELS){
	GLCurrent->Pack.SkipPixels=(int)param;
    } else if(pname == GL_PACK_SKIP_ROWS){
	GLCurrent->Pack.SkipRows=(int)param;
    } else if(pname == GL_PACK_ALIGNMENT){
	GLCurrent->Pack.Alignment=(int)param;
    } else if(pname == GL_UNPACK_SWAP_BYTES){
	GLCurrent->Unpack.SwapBytes=(int)param;
    } else if(pname == GL_UNPACK_LSB_FIRST){
	GLCurrent->Unpack.LsbFirst=(int)param;
    } else if(pname == GL_UNPACK_ROW_LENGTH){
	GLCurrent->Unpack.RowLength=(int)param;
    } else if(pname == GL_UNPACK_SKIP_PIXELS){
	GLCurrent->Unpack.SkipPixels=(int)param;
    } else if(pname == GL_UNPACK_SKIP_ROWS){
	GLCurrent->Unpack.SkipRows=(int)param;
    } else if(pname == GL_UNPACK_ALIGNMENT){
	GLCurrent->Unpack.Alignment=(int)param;
    }

    __glx_check_unpacking();

}


void __glx_PixelStorei( GLenum pname, GLint param)
{
    XExtDisplayInfo* info = __gl_find_display(GLCurrent->dpy);

    GLSimpleCheckExtension(GLCurrent->dpy, info);

    if(pname == GL_PACK_SWAP_BYTES){
	GLCurrent->Pack.SwapBytes=param;
    } else if(pname == GL_PACK_LSB_FIRST){
	GLCurrent->Pack.LsbFirst=param;
    } else if(pname == GL_PACK_ROW_LENGTH){
	GLCurrent->Pack.RowLength=param;
    } else if(pname == GL_PACK_SKIP_PIXELS){
	GLCurrent->Pack.SkipPixels=param;
    } else if(pname == GL_PACK_SKIP_ROWS){
	GLCurrent->Pack.SkipRows=param;
    } else if(pname == GL_PACK_ALIGNMENT){
	GLCurrent->Pack.Alignment=param;
    } else if(pname == GL_UNPACK_SWAP_BYTES){
	GLCurrent->Unpack.SwapBytes=param;
    } else if(pname == GL_UNPACK_LSB_FIRST){
	GLCurrent->Unpack.LsbFirst=param;
    } else if(pname == GL_UNPACK_ROW_LENGTH){
	GLCurrent->Unpack.RowLength=param;
    } else if(pname == GL_UNPACK_SKIP_PIXELS){
	GLCurrent->Unpack.SkipPixels=param;
    } else if(pname == GL_UNPACK_SKIP_ROWS){
	GLCurrent->Unpack.SkipRows=param;
    } else if(pname == GL_UNPACK_ALIGNMENT){
	GLCurrent->Unpack.Alignment=param;
    }

    __glx_check_unpacking();

}

GLint __glx_RenderMode(GLenum mode)
{
    GLint n=0;
    XExtDisplayInfo* info = __gl_find_display(GLCurrent->dpy);
    xGLXRenderModeReq* req;
    xGLXRenderModeReply reply;
    Display* dpy=GLCurrent->dpy;

    GLCheckExtension(GLCurrent->dpy, info, 0);

    GLXRenderFlush();
    LockDisplay(dpy);
    GetReq(GLXRenderMode, req);

    req->reqType = info->codes->major_opcode;
    req->glx_opcode = X_GLXRenderMode;
    req->context_tag = GLCurrent->contextTag;
    req->mode=mode;

    if(GLCurrent->rendermode == GL_RENDER){
	/* No reply... */
    } else {
	_XReply(dpy, (xReply*)&reply, 0, xFalse);
	n=reply.n;
	if(GLCurrent->rendermode == GL_FEEDBACK){
	    if(n > GLCurrent->feedback_buffer_size)
		n=GLCurrent->feedback_buffer_size;
	    _XRead(dpy, (char*)GLCurrent->feedback_buffer, 4L*n);
	} else if(GLCurrent->rendermode == GL_SELECT){
	    if(n > GLCurrent->selection_buffer_size)
		n=GLCurrent->selection_buffer_size;
	    _XRead(dpy, (char*)GLCurrent->selection_buffer, 4L*n);
	}
	n=reply.retval;
    }
    GLCurrent->rendermode=mode;
    UnlockDisplay(dpy);
    SyncHandle();
    return n;
}


void __glx_SelectBuffer(GLsizei size, GLuint* buffer)
{
    XExtDisplayInfo* info = __gl_find_display(GLCurrent->dpy);
    xGLXSelectBufferReq* req;
    Display* dpy=GLCurrent->dpy;

    GLSimpleCheckExtension(GLCurrent->dpy, info);

    GLXRenderFlush();
    LockDisplay(dpy);
    GetReq(GLXSelectBuffer, req);
    req->reqType = info->codes->major_opcode;
    req->glx_opcode = X_GLXSelectBuffer;
    req->context_tag = GLCurrent->contextTag;

    req->size=size;
    GLCurrent->selection_buffer=buffer;
    GLCurrent->selection_buffer_size=size;

    UnlockDisplay(GLCurrent->dpy);
    SyncHandle();
}


void __glx_GetPolygonStipple( GLubyte *mask )
{
    XExtDisplayInfo* info = __gl_find_display(GLCurrent->dpy);
    xGLXGetPolygonStippleReq* req;
    xGLXGetPolygonStippleReply reply;
    Display* dpy=GLCurrent->dpy;
    int length;

    GLSimpleCheckExtension(GLCurrent->dpy, info);

    GLXRenderFlush();
    LockDisplay(dpy);
    GetReq(GLXGetPolygonStipple, req);

    req->reqType = info->codes->major_opcode;
    req->glx_opcode = X_GLXGetPolygonStipple;
    req->context_tag = GLCurrent->contextTag;
    req->lsb_first=GLCurrent->Pack.LsbFirst;

    _XReply(dpy, (xReply*)&reply, 0, xFalse);

    length = reply.length;
    _XRead(dpy, (char*)mask, 4L*length);
    UnlockDisplay(dpy);
    SyncHandle();
}



void __glx_GetTexImage( GLenum target, GLint level, GLenum format,
		   GLenum type, GLvoid *pixels )
{
    int w;
    XExtDisplayInfo* info = __gl_find_display(GLCurrent->dpy);
    xGLXGetTexImageReq* req;
    xGLXGetTexImageReply reply;
    Display* dpy=GLCurrent->dpy;
    int length;

    GLSimpleCheckExtension(GLCurrent->dpy, info);

    GLXRenderFlush();
    LockDisplay(dpy);
    GetReq(GLXGetTexImage, req);

    req->reqType = info->codes->major_opcode;
    req->glx_opcode = X_GLXGetTexImage;
    req->context_tag = GLCurrent->contextTag;
    req->target=target;
    req->level=level;
    req->format=format;
    req->type=type;
    req->swap_bytes=GLCurrent->Pack.SwapBytes;

    _XReply(dpy, (xReply*)&reply, 0, xFalse);

    length = reply.length;
    w=GLCurrent->Pack.RowLength?reply.width:GLCurrent->Pack.RowLength;
    if(w == reply.width && GLCurrent->Pack.Alignment == 4){
	/* Contiguous... */
	_XRead(dpy, (char*)pixels, 4L*length);
    } else {
	char* p=(char*)malloc(4L*length);
	_XRead(dpy, p, 4L*length);
	free(p);
    }
    UnlockDisplay(dpy);
    SyncHandle();
    
}

void __glx_ReadPixels( GLint x, GLint y, GLsizei width, GLsizei height,
		  GLenum format, GLenum type, GLvoid *pixels )
{
    XExtDisplayInfo* info = __gl_find_display(GLCurrent->dpy);
    xGLXReadPixelsReq* req;
    xGLXReadPixelsReply reply;
    Display* dpy=GLCurrent->dpy;
    int length;

    GLSimpleCheckExtension(GLCurrent->dpy, info);

    printf("glReadPixels: 0x%x 0x%x\n", format, type);
    GLXRenderFlush();
    LockDisplay(dpy);
    GetReq(GLXReadPixels, req);

    req->reqType = info->codes->major_opcode;
    req->glx_opcode = X_GLXReadPixels;
    req->context_tag = GLCurrent->contextTag;
    req->x=x;
    req->y=y;
    req->width=width;
    req->height=height;
    req->format=format;
    req->type=type;
    req->swap_bytes=GLCurrent->Pack.SwapBytes;
    req->lsb_first=GLCurrent->Pack.LsbFirst;

    _XReply(dpy, (xReply*)&reply, 0, xFalse);

    length = reply.length<<2;
    __GLX_GET_BUFFER(pixels, length, width, height, format, type);
    UnlockDisplay(dpy);
    SyncHandle();
    
}


static struct gl_attrib_node *new_attrib_node( GLbitfield kind )
{
   struct gl_attrib_node *an;

   an = (struct gl_attrib_node *) malloc( sizeof(struct gl_attrib_node) );
   if (an) {
      an->kind = kind;
   }
   return an;
}


void __glx_PushClientAttrib( GLbitfield mask )
{
   struct gl_attrib_node *newnode;
   struct gl_attrib_node *head;

   if (GLCurrent->ClientAttribStackDepth>=MAX_CLIENT_ATTRIB_STACK_DEPTH) {
      __glx_error( GL_STACK_OVERFLOW, "glPushClientAttrib" );
      return;
   }

   /* Build linked list of attribute nodes which save all attribute */
   /* groups specified by the mask. */
   head = NULL;

   if (mask & GL_CLIENT_PIXEL_STORE_BIT) {
      struct gl_pixelstore_attrib *attr;
      /* packing attribs */
      attr = MALLOC_STRUCT( gl_pixelstore_attrib );
      MEMCPY( attr, &GLCurrent->Pack, sizeof(struct gl_pixelstore_attrib) );
      newnode = new_attrib_node( GL_CLIENT_PACK_BIT );
      newnode->data = attr;
      newnode->next = head;
      head = newnode;
      /* unpacking attribs */
      attr = MALLOC_STRUCT( gl_pixelstore_attrib );
      MEMCPY( attr, &GLCurrent->Unpack, sizeof(struct gl_pixelstore_attrib) );
      newnode = new_attrib_node( GL_CLIENT_UNPACK_BIT );
      newnode->data = attr;
      newnode->next = head;
      head = newnode;
   }
   if (mask & GL_CLIENT_VERTEX_ARRAY_BIT) {
      struct gl_array_attrib *attr;
      attr = MALLOC_STRUCT( gl_array_attrib );
      MEMCPY( attr, &GLCurrent->Array, sizeof(struct gl_array_attrib) );
      newnode = new_attrib_node( GL_CLIENT_VERTEX_ARRAY_BIT );
      newnode->data = attr;
      newnode->next = head;
      head = newnode;
   }

   /* etc... */

   GLCurrent->ClientAttribStack[GLCurrent->ClientAttribStackDepth] = head;
   GLCurrent->ClientAttribStackDepth++;
}


void __glx_PopClientAttrib( )
{
   struct gl_attrib_node *attr, *next;

   if (GLCurrent->ClientAttribStackDepth==0) {
      __glx_error( GL_STACK_UNDERFLOW, "glPopClientAttrib" );
      return;
   }

   GLCurrent->ClientAttribStackDepth--;
   attr = GLCurrent->ClientAttribStack[GLCurrent->ClientAttribStackDepth];

   while (attr) {
      switch (attr->kind) {
         case GL_CLIENT_PACK_BIT:
            MEMCPY( &GLCurrent->Pack, attr->data,
                    sizeof(struct gl_pixelstore_attrib) );
            break;
         case GL_CLIENT_UNPACK_BIT:
            MEMCPY( &GLCurrent->Unpack, attr->data,
                    sizeof(struct gl_pixelstore_attrib) );
            break;
         case GL_CLIENT_VERTEX_ARRAY_BIT:
            MEMCPY( &GLCurrent->Array, attr->data,
		    sizeof(struct gl_array_attrib) );
            break;
         default:
            __glx_problem( "Bad attrib flag in PopClientAttrib");
            break;
      }

      next = attr->next;
      free( (void *) attr->data );
      free( (void *) attr );
      attr = next;
   }

   GLCurrent->NewState = NEW_ALL;
}

void __glx_GetPointerv( GLenum pname, GLvoid **params )
{
   GLuint texSet = 0;
   switch (pname) {
      case GL_VERTEX_ARRAY_POINTER:
         *params = GLCurrent->Array.VertexPtr;
         break;
      case GL_NORMAL_ARRAY_POINTER:
         *params = GLCurrent->Array.NormalPtr;
         break;
      case GL_COLOR_ARRAY_POINTER:
         *params = GLCurrent->Array.ColorPtr;
         break;
      case GL_INDEX_ARRAY_POINTER:
         *params = GLCurrent->Array.IndexPtr;
         break;
      case GL_TEXTURE_COORD_ARRAY_POINTER:
         *params = GLCurrent->Array.TexCoordPtr[texSet];
         break;
      case GL_EDGE_FLAG_ARRAY_POINTER:
         *params = GLCurrent->Array.EdgeFlagPtr;
         break;
#if 0
      case GL_FEEDBACK_BUFFER_POINTER:
         *params = GLCurrent->Feedback.Buffer;
         break;
      case GL_SELECTION_BUFFER_POINTER:
         *params = GLCurrent->Select.Buffer;
         break;
#endif
      default:
         __glx_error( GL_INVALID_ENUM, "glGetPointerv" );
         return;
   }
}

