/*
 * GLX Hardware Device Driver for Matrox G200
 * Copyright (C) 1999 Keith Whitwell
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * KEITH WHITWELL, OR ANY OTHER CONTRIBUTORS BE LIABLE FOR ANY CLAIM, 
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR 
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE 
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * Keith Whitwell <keithw@precisioninsight.com>
 */

#include <stdlib.h>
#include <signal.h>

#include "context.h"
#include "depth.h"
#include "macros.h"
#include "texstate.h"
#include "triangle.h"
#include "vb.h"
#include "types.h"

#include "xsmesaP.h"
#include "glx_log.h"
#include "mesaglx/context.h"
#include "mesaglx/matrix.h"
#include "mesaglx/types.h"

#define GC XXGC
#include "gcstruct.h"
#include "pixmapstr.h"
#include "servermd.h" /* PixmapBytePad */
#include "scrnintstr.h"
#include "regionstr.h"
#include "windowstr.h"
#undef GC

#include "mm.h"
#include "mgadd.h"
#include "g200_mac.h"
#include "mgalib.h"
#include "hwlog.h"
#include "mgadirect.h"
#include "mgadma.h"
#include "mgaglx.h"
#include "mgasymbols.h"

#include "glx_clients.h"
#include "glx_init.h"

#ifndef NO_MTRR
#define MTRR_NEED_STRINGS
#include <errno.h>
#include <asm/mtrr.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <unistd.h>
int mtrr;
void mgaCloseMTRR();
#endif

#include "xf86_OSproc.h"



static void mgaClientDMAFlush( int wait );
static void mgaClientReleaseHardware( void );

/* 
 */
#define GLX_START                X_GLXDirectPrivateStart

#define X_GLXMgaSwapBuffers   (GLX_START+2)
#define X_GLXMgaDmaFlush      (GLX_START+3)
#define X_GLXMgaFBGeom        (GLX_START+4)


/* Nice thing about direct rendering is that the server *is* the
 * client, in the sense that they should both be the same executable
 * image 'glx.so'.  This means that most of the protocol structs can
 * be private to this file.
 */
typedef struct {   
   char init_fn[20];		/* The first element is always the
				 * name of the function for the client
				 * to call to initialize glx.so.  */

   int screenNum;		/* not respected - assumed to be zero */

   int vgaInfoVirtualX;
   int vgaInfoVirtualY;
   int vgaInfoDepth;
   int vgaInfoDisplayWidth;
   int vgaInfoVideoRam;
   int vgaInfoChipID;
   unsigned long vgaInfoPhysBase;

   char vgaInfoChipset[80];   
   int vgaBitsPerPixel;
   int vgaLinearSize;

   int MGAdacIsHWCursor;
   int MGAchipset;
   int MGAydstorg;
   pciTagRec MGAPciTag;
   Bool MGAUsePCIRetry;

   int xf86PixmapCacheMemoryStart;
   int xf86PixmapCacheMemoryEnd;
   int xf86PCIFlags;

   int mgaglx_dma;
   int mgaglx_dmaadr;
   int mgaglx_dmasize;
   int mgaglx_systemtexture;
   int mgaglx_cardcmds;
   int mgaglx_cmdsize;
   int mgaglx_issdram;

   int mgaActiveDmaBuffer;
   int glx_first_visual;
   int glx_chipset;

} mgaDirectHWInfo;


extern int __glXNumClients();

static int (*send_vendor_private)( int opcode,
				   char *vp, 
				   size_t vp_sz,
				   xReply *reply,
				   char *data,
				   size_t data_sz ) = 0;


ClientPtr mgaGLXClientInit( mgaDirectHWInfo *hw, 
			    void *display,
			    int screen,
			    int (*send_func)(),
			    void (**release_hook)( void ))
{
   if (!mgaInstallLocalSymbols())
      return 0;
   
   send_vendor_private = send_func;
   mgaDoDmaFlush = mgaClientDMAFlush;

   memset(&GLXSYM(vga256InfoRec), 0, sizeof(GLXSYM(vga256InfoRec)));
   memset(&GLXSYM(MGAdac), 0, sizeof(GLXSYM(MGAdac)));
   memset(&GLXSYM(xf86AccelInfoRec), 0, sizeof(GLXSYM(xf86AccelInfoRec)));
   
   GLXSYM(vgaLinearSize) = hw->vgaLinearSize;
   GLXSYM(vga256InfoRec).virtualX = hw->vgaInfoVirtualX;
   GLXSYM(vga256InfoRec).virtualY = hw->vgaInfoVirtualY;
   GLXSYM(vga256InfoRec).depth = hw->vgaInfoDepth;
   GLXSYM(vga256InfoRec).displayWidth = hw->vgaInfoDisplayWidth;
   GLXSYM(vga256InfoRec).videoRam = hw->vgaInfoVideoRam;
   GLXSYM(vga256InfoRec).chipID = hw->vgaInfoChipID;
   GLXSYM(vga256InfoRec).physBase = hw->vgaInfoPhysBase;
   GLXSYM(vga256InfoRec).chipset = hw->vgaInfoChipset;
   GLXSYM(vgaBitsPerPixel) = hw->vgaBitsPerPixel;
   GLXSYM(xf86AccelInfoRec).PixmapCacheMemoryStart = hw->xf86PixmapCacheMemoryStart;
   GLXSYM(xf86AccelInfoRec).PixmapCacheMemoryEnd = hw->xf86PixmapCacheMemoryEnd;
   GLXSYM(xf86AccelInfoRec).ServerInfoRec = &GLXSYM(vga256InfoRec);
   GLXSYM(MGAdac).isHwCursor = hw->MGAdacIsHWCursor;   
   GLXSYM(MGAPciTag) = hw->MGAPciTag;
   GLXSYM(MGAchipset) = hw->MGAchipset;
   GLXSYM(MGAydstorg) = hw->MGAydstorg;
   GLXSYM(MGAUsePCIRetry) = hw->MGAUsePCIRetry;
   GLXSYM(xf86PCIFlags) = hw->xf86PCIFlags;
   mgaglx.dmaDriver = hw->mgaglx_dma;
   mgaglx.dmaAdr = hw->mgaglx_dmaadr;
   mgaglx.dmaSize = hw->mgaglx_dmasize;
   mgaglx.cmdSize = hw->mgaglx_cmdsize;
   mgaglx.cardCmds = hw->mgaglx_cardcmds;
   mgaglx.systemTexture = hw->mgaglx_systemtexture;
   mgaglx.isSdram = hw->mgaglx_issdram;
   mgaActiveDmaBuffer = hw->mgaActiveDmaBuffer;

   GLXSYM(vgaLinearBase) = xf86MapVidMem(hw->screenNum, 
				 LINEAR_REGION,
				 (pointer) GLXSYM(vga256InfoRec).physBase,
				 GLXSYM(vgaLinearSize));

   if (GLXSYM(vgaLinearBase) == (pointer) -1) {
      hwError("failed to map vga linear region");
      mgaClientReleaseHardware();
      return 0;
   }

   *release_hook = mgaClientReleaseHardware;

   return glxInitDirectClient( display, screen, hw->glx_first_visual,
			       XF_SVGA, hw->glx_chipset );
}

static void mgaClientReleaseHardware( void )
{
   if (GLXSYM(vgaLinearBase) != (pointer) -1) {
      xf86UnMapVidMem(0,
		      LINEAR_REGION,
		      (pointer) GLXSYM(vga256InfoRec).physBase,
		      GLXSYM(vgaLinearSize));
      GLXSYM(vgaLinearBase) = 0;
	CloseGART();
   }
}


static int mgaGLXGoDirect( ClientPtr client )
{
   xGLXGetStringReply reply;
   mgaDirectHWInfo *hw;
   int l;

   /* Allow only one direct client, which must be the only active
    * client, and must be on the local server.  
    */
   if (direct_client || !__glx_is_server || __glXNumClients() != 1 || 
       !LocalClient(client)) 
      return BadAccess;

   if (mgaglx.dmaDriver < 2) {
      hwError("Direct clients only allowed with real dma");
      return BadMatch;
   }

   mgaDmaFlush();

   direct_client = client;
   hw = (mgaDirectHWInfo *)malloc( sizeof(*hw) );

   /* Name of function to call to revive the struct.  Client will
    * receive buf in reply to its go_direct request.  Client will map
    * glx.so, and use buf as a string argument to dlsym(), to retreive
    * a function to call, namely mgaGLXClientInit.  It will then call
    * that function passing hw as its argument.
    *
    * hw should contain all the information necessary to get glx.so
    * talking to hardware from the client.
    */
   strcpy(hw->init_fn, "mgaGLXClientInit");   

   hw->screenNum = 0;		/* oh really? */
   hw->vgaInfoVirtualX = GLXSYM(vga256InfoRec).virtualX;
   hw->vgaInfoVirtualY = GLXSYM(vga256InfoRec).virtualY;
   hw->vgaInfoDepth = GLXSYM(vga256InfoRec).depth;
   hw->vgaInfoDisplayWidth = GLXSYM(vga256InfoRec).displayWidth;
   hw->vgaInfoVideoRam = GLXSYM(vga256InfoRec).videoRam;
   hw->vgaInfoChipID = GLXSYM(vga256InfoRec).chipID;
   hw->vgaInfoPhysBase = GLXSYM(vga256InfoRec).physBase;

   strncpy(hw->vgaInfoChipset, GLXSYM(vga256InfoRec).chipset, 80);
   hw->vgaInfoChipset[79] = 0;

   hw->vgaBitsPerPixel = GLXSYM(vgaBitsPerPixel);
   hw->vgaLinearSize = GLXSYM(vgaLinearSize);
   hw->xf86PixmapCacheMemoryStart = GLXSYM(xf86AccelInfoRec).PixmapCacheMemoryStart;
   hw->xf86PixmapCacheMemoryEnd = GLXSYM(xf86AccelInfoRec).PixmapCacheMemoryEnd;   
   hw->xf86PCIFlags = GLXSYM(xf86PCIFlags);
   hw->MGAdacIsHWCursor = GLXSYM(MGAdac).isHwCursor;
   hw->MGAPciTag = GLXSYM(MGAPciTag);
   hw->MGAchipset = GLXSYM(MGAchipset);
   hw->MGAydstorg = GLXSYM(MGAydstorg);
   hw->MGAUsePCIRetry = GLXSYM(MGAUsePCIRetry);
   hw->mgaglx_dma = mgaglx.dmaDriver;
   hw->mgaglx_dmaadr = mgaglx.dmaAdr;
   hw->mgaglx_dmasize = mgaglx.dmaSize;
   hw->mgaglx_cmdsize = mgaglx.cmdSize;
   hw->mgaglx_cardcmds = mgaglx.cardCmds;
   hw->mgaglx_systemtexture = mgaglx.systemTexture;
   hw->mgaglx_issdram = mgaglx.isSdram;

   hw->mgaActiveDmaBuffer = mgaActiveDmaBuffer;
   hw->glx_first_visual = __glx_first_visual;
   hw->glx_chipset = glx_chipset;

   reply.type=X_Reply;
   reply.sequenceNumber = client->sequence;
   reply.length=0;
   reply.n=l=(sizeof(*hw)+3)/4;

   WriteToClient(client, sizeof(xGLXGetStringReply), (char*)&reply);
   WriteToClient(client, sizeof(int)*l, (char*)hw);

   return client->noClientException;
}



static int mgaGLXReleaseDirect( ClientPtr client )
{
   if (__glx_is_server && direct_client && direct_client == client) {
      direct_client = 0;
      return Success;
   }

   return BadAccess;
}




/* =======================================================================
 * Dma flush
 */
typedef struct  {
   CARD32 dma_primaryDwords;
   CARD32 dma_buffer_id;	/* validation only */
   CARD32 dma_sync;
} DmaFlushReq;


typedef struct {
    BYTE type;
    CARD8 xpad;
    CARD16 sequenceNumber;
    CARD32 length;
    CARD32 dma_buffer_id;	/* new active id */   
} DmaFlushReply;



static int mgaGLXDirectDMAFlush( ClientPtr client, 
				 xGLXVendorPrivateReq *stuff )
{
   DmaFlushReq *vp = (DmaFlushReq *)(stuff + 1);
   xReply reply;
   DmaFlushReply *rep = (DmaFlushReply *)&reply;	
   extern void mgaServerDmaFlush( int );


   if (client != direct_client) 
      return BadAccess;

   if ( mgaActiveDmaBuffer != vp->dma_buffer_id ) 
   {
      hwError("someone's been playing with dma on the server\n");
      return BadImplementation;
   }

   dma_buffer->primaryDwords = vp->dma_primaryDwords;

   mgaServerDmaFlush( vp->dma_sync );

   rep->type = X_Reply;
   rep->length = 0;	
   rep->sequenceNumber = client->sequence;
   rep->dma_buffer_id = mgaActiveDmaBuffer;
   
   WriteToClient( client, sizeof(xReply), (char *) &reply );
	
   return client->noClientException;
}


/*
 * mgaClientDMAFlush
 * When direct rendering, this is called instead of starting
 * the dma from user space.  This only happens for overflows,
 * explicit flush/finish, and software rendering.  The swapbuffers
 * interface handles flushing in normal cases.
 */
static void mgaClientDMAFlush( int wait ) 
{
   DmaFlushReq vp;
   xReply reply;

   vp.dma_buffer_id = mgaActiveDmaBuffer;
   vp.dma_primaryDwords = dma_buffer->primaryDwords;
   vp.dma_sync = wait;

   if (send_vendor_private( X_GLXMgaDmaFlush,
			    (char *)&vp, sizeof(vp), 
			    &reply, 0, 0 ))
   {
      DmaFlushReply *rep = (DmaFlushReply *)&reply;
      mgaActiveDmaBuffer = rep->dma_buffer_id;
   }

   mgaDmaResetBuffer();
}


/* =======================================================================
 * Geometry request 
 *   - could just use XGetGeom, but this is simpler, really.
 */
typedef struct  {
   CARD32 drawable;
} FBGeomReq;


typedef struct {
   BYTE type;
   CARD8 xpad;
   CARD16 sequenceNumber;
   CARD32 length;
   CARD16 width;
   CARD16 height;
} FBGeomReply;


static int mgaGLXDirectGetGeometry( ClientPtr client, 
				    xGLXVendorPrivateReq *stuff )
{
   xReply reply;
   DrawablePtr frontbuf;
   FBGeomReq *vp = (FBGeomReq *)(stuff + 1);
   FBGeomReply *rep = (FBGeomReply *)&reply;	

   if (client != direct_client) 
      return BadAccess;

   frontbuf = LookupIDByClass( vp->drawable, RC_DRAWABLE );   
   if (!frontbuf) 
      return GLXBadDrawable;

   rep->type = X_Reply;
   rep->length = 0;	
   rep->sequenceNumber = client->sequence;
   rep->width = frontbuf->width;
   rep->height = frontbuf->height;
   
   WriteToClient( client, sizeof(xReply), (char *) &reply );
	
   return client->noClientException;
}


void mgaClientGetGeometry( DrawablePtr d ) 
{
   FBGeomReq vp;
   xReply reply;

   vp.drawable = d->id;    
   
   if (send_vendor_private( X_GLXMgaFBGeom,
			    (char *)&vp, sizeof(vp), 
			    &reply, 0, 0 ))
   {
      FBGeomReply *rep = (FBGeomReply *)&reply;
      d->width = rep->width;
      d->height = rep->height;
   }
}




/* ===========================================================================
 * Buffer swap
 *   - Combines dma flush, buffer swap and get geometry.
 */

/* Need to issue the swap commands from the server as the client
 * doesn't have the clip rects.  This isn't so bad as we would want to
 * do a dma flush immediately afterwards anyhow.
 */
typedef struct {
   mgaBuffer	buffer;
   
   CARD32 front_drawable;
   CARD32 dma_primaryDwords;
   CARD32 dma_secondaryDwords;	/* only needed for performance boxes */
   CARD32 dma_buffer_id;	/* validation only */
   CARD32 attrib;
   CARD32 flag;
} SwapBufferReq;

typedef struct {
   BYTE type;
   CARD8 xpad;
   CARD16 sequenceNumber;
   CARD32 length;
   CARD32 dma_buffer_id;	/* new active id */   
   CARD16 width;
   CARD16 height;
   CARD16	hardwareWentIdle;
} SwapBufferReply;

#define FLG_TEX_SWAP 0x1

static int mgaGLXDirectSwapBuffers( ClientPtr client, 
				    xGLXVendorPrivateReq *stuff )
{
   SwapBufferReq *vp = (SwapBufferReq *)(stuff+1);
   xReply reply;
   SwapBufferReply *rep = (SwapBufferReply *)&reply;	
   DrawablePtr frontbuf;

   if (client != direct_client) 
      return BadAccess;

   if ( mgaActiveDmaBuffer != vp->dma_buffer_id ) 
   {
      fprintf(stderr, "somebody's been playing with dma on the server %d %ld\n",
	      mgaActiveDmaBuffer, vp->dma_buffer_id);

      return BadImplementation;
   }

   dma_buffer->primaryDwords = vp->dma_primaryDwords;

   frontbuf = LookupIDByClass( vp->front_drawable, RC_DRAWABLE );   

   if (!frontbuf) {
      return GLXBadDrawable;
   }

   mgaBackToFront(frontbuf, &vp->buffer);   
   mgaDmaFlush();
   	
   rep->type = X_Reply;
   rep->length = 0;	
   rep->sequenceNumber = client->sequence;
   rep->dma_buffer_id = mgaActiveDmaBuffer;
   rep->width = frontbuf->width;
   rep->height = frontbuf->height;
   rep->hardwareWentIdle = mgaglx.hardwareWentIdle;
   	
   WriteToClient( client, sizeof(xReply), (char *) &reply );

   return client->noClientException;
}

void mgaClientSwapBuffers( XSMesaBuffer b ) 
{
   SwapBufferReq vp;
   xReply reply;
   mgaBufferPtr buf;

	buf = (mgaBufferPtr)b->backimage->devPriv;
	
#if 0
   if (!b->db_state || 
       !b->backimage ||
       !(buf = (mgaBufferPtr)b->backimage->devPriv))
   {
      fprintf(stderr, "client swap buffers: wtf???\n");
      return;
   }
#endif

#if 0   
   if (mgaCtx && mgaCtx->gl_ctx) {
      FLUSH_VB(mgaCtx->gl_ctx, "mga client swap buffers");
   }
#endif
   mgaglx.swapBuffersCount++;

   vp.front_drawable = b->frontbuffer->id;    
   vp.buffer = *buf;

   vp.dma_buffer_id = mgaActiveDmaBuffer;
   vp.dma_primaryDwords = dma_buffer->primaryDwords;
   vp.dma_secondaryDwords = dma_buffer->secondaryDwords;
   vp.flag = 0;

   if (send_vendor_private( X_GLXMgaSwapBuffers,
			    (char *)&vp, sizeof(vp), &reply, 0, 0 ))
   {
      SwapBufferReply *rep = (SwapBufferReply *)&reply;

      b->frontbuffer->width = rep->width;
      b->frontbuffer->height = rep->height;

	mgaglx.hardwareWentIdle = rep->hardwareWentIdle;
      mgaActiveDmaBuffer = rep->dma_buffer_id;
      mgaDmaResetBuffer();
   }
   else
      FatalError("clientSwapBuffers failed");
}







/* ===========================================================================
 */

int mgaGLXVendorPrivate( ClientPtr client,
			 XSMesaContext ctx, 
			 xGLXVendorPrivateReq *stuff )
{
   if (!__glx_is_server) 
      return GLXUnsupportedPrivateRequest;

   switch (stuff->opcode) {
   case X_GLXDirectGoDirect:
      return mgaGLXGoDirect( client );
   case X_GLXDirectRelease:
      return mgaGLXReleaseDirect( client );
   case X_GLXMgaSwapBuffers:
      return mgaGLXDirectSwapBuffers( client, stuff );
   case X_GLXMgaDmaFlush:
      return mgaGLXDirectDMAFlush( client, stuff );
   case X_GLXMgaFBGeom:
      return mgaGLXDirectGetGeometry( client, stuff );
   default:
      hwError("not-handled case in mgaGLXVendorPrivate");
      return GLXUnsupportedPrivateRequest;
   }
}


int mgaGLXAllowDirect( ClientPtr client )
{
   if (mgaglx.dmaDriver < 2) {
      hwError("Don't allow direct clients with pseudo-dma");
      return 0;
   }

   return 1;
}

