/*
 * GLX Hardware Device Driver for S3 ViRGE DX/GX/GX2
 * Copyright (C) 1999 Jim Duchek
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * WITTAWAT YAMWONG, OR ANY OTHER CONTRIBUTORS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * Based on Mach64 driver: mach64state.c
 *
 *    Jim Duchek <jimduchek@ou.edu>
 */

#include <stdlib.h>
#include <stdio.h>

#include "xsmesaP.h"

#include "types.h"
#include "pb.h"
#include "dd.h"

#include "glx_log.h"

#include "s3virgeglx.h"
#include "glx_symbols.h"


/* FIXME: GH */
#define S3VIRGE_CONTEXT(ctx)	((s3virgeContextPtr)(((XSMesaContext)(ctx)->DriverCtx)->hw_ctx))

/*
 * s3virgeIsTexturingEnabled
 * FIXME: i'm sure there is an easier way of checking this
 */
int s3virgeIsTexturingEnabled( const GLcontext *ctx ) 
{
	struct gl_texture_object	*tObj;

	if ( ctx->Texture.Enabled != TEXTURE0_2D ) {
		s3virgeMsg(1, "Not Tex Enabled.\n");
		return 0;
	}

	tObj = ctx->Texture.Unit[0].Current;
	if ( !tObj ) {
		s3virgeMsg(1, "No tObj.\n");
		return 0;
	}
	
	/* check for weird mesa state where it looks like we
	are enabled but the texture object's image is an invalid pointer.
	This should probably be fixed in mesa someplace */
	if ( tObj != ctx->Texture.Unit[0].CurrentD[2] ) {
		s3virgeMsg(1, "Funky tObj.\n");
		return 0;
	}
	
	return 1;
}


/* =============================================================
 * Alpha blending
 */


static int s3virgeCalcAlpha_CNTL( const GLcontext *ctx) 
{
	int ret = 0;
	int lit, mode;
	int src, dst, emode;
	int salph;
	s3virgeTextureObject_t *t = ctx->Texture.Unit[0].Current->DriverData;	

	if ( !s3virgeIsTexturingEnabled( ctx ) ) {
		if ( ctx->Color.AlphaEnabled ) {
			ret |= (0x3 << 18);		
		}
	} else {
		salph = 2;
		mode = 0; lit = 0;
	
		emode = ctx->Texture.Unit[0].EnvMode;
		
		/* Unfortunately we don't really HAVE blending modes,		 * so we've got to do some creative crap here.  
		 * Grr.  The following should make Q2 look decent :) */
		 
		src = ctx->Color.BlendSrcRGB;  // I'm too lazy to type
		dst = ctx->Color.BlendDstRGB;  // these out more than once ;)

					
		if (emode == GL_REPLACE && src == GL_ZERO && dst == GL_SRC_COLOR) {
			lit = 0;
			salph = 0;
			s3virgeglx.triHack |= S3HACK_ZERO_SRCCOL;
			if (!s3virgeglx.lightmapHack)
				lit = 2;
		} else {
			switch (emode) {
			case GL_REPLACE:
				lit = 0;
				salph = 2;
				s3virgeglx.triHack |= S3HACK_WHITE_TRI_ALPHA;
				break;
			case GL_DECAL:
				mode = (0x2);
				lit = 1;
				break;
			case GL_BLEND: // FIXME Correct ?
				mode = (0x2);
				lit = 1;
				s3virgeMsg(0, "Blend!?!?!\n");
				break;
			case GL_MODULATE:
//				if (t->hasAlpha) {
//					salph = 2;
//				} else {
//					salph = 3;
//				}
				mode = (0x1);
				salph = 2;
				lit = 1;
				s3virgeglx.triHack |= S3HACK_WHITE_TRI;
				break;
			default:
				s3virgeMsg(1,  "s3virgeUpdateAlphaMode(): not supported texture mode %d\n",
					ctx->Texture.Unit[0].EnvMode );
			}
			if (src == GL_SRC_ALPHA && dst == GL_ONE_MINUS_SRC_ALPHA) {
				// generic, we're fine.
			} else {
				s3virgeMsg(1, "Odd mode %04x %04x and %04x\n", emode, src, dst);
			}

		}

		if (t->texelBytes == 1) {
			lit = 0;
			mode = 0x2;
			if (salph)
				salph = 0;
		}
			
		switch (lit) {
		case 0:
			ret |= (0x6 << 27);
			break;
		case 1:
			ret |= (0x5 << 27);
			break;
		case 2:
		default:
			ret |= (0xF << 27);
			break;
		}

		ret |= (mode << 15);

		ret |= (salph << 18);

	}
	return ret;
}

static int s3virgeCalcZ_CNTL( const GLcontext *ctx ) 
{
	int		zcntl;

	if ( !ctx->Depth.Test) {
	        s3virgeglx.triHack |= S3HACK_WHITE_TRI; 
		return (3 << 24);
	}
	
	switch( ctx->Depth.Func )
	{
	case GL_NEVER:
	    zcntl = 0; break;
	case GL_GREATER:
	    zcntl = 1; break;
	case GL_EQUAL:
	    zcntl = 2; break;
	case GL_GEQUAL:
	    zcntl = 3; break;
	case GL_LESS:
	    zcntl = 4; break;
	case GL_NOTEQUAL:
	    zcntl = 5; break;
	case GL_LEQUAL:
	    zcntl = 6; break;
	case GL_ALWAYS:
	    zcntl = 7; 
	    break;
	default:
	    break;
	}
	zcntl = zcntl << 20;
	
	if ( ctx->Depth.Mask ) {
		zcntl |= (1 << 23);	/* enable writes to depth buffer */
	}
	
	return zcntl;
}


static int s3virgeCalcScale( const GLcontext *ctx ) 
{
	switch( ctx->Texture.Unit[0].Current->MinFilter ) {
		case GL_LINEAR:
			return (0x6 << 12);
		case GL_LINEAR_MIPMAP_NEAREST:
			return (0x1 << 12);
		case GL_LINEAR_MIPMAP_LINEAR:
			return (0x3 << 12);
		case GL_NEAREST:
			return (0x4 << 12);
		case GL_NEAREST_MIPMAP_NEAREST:
			return (0x0 << 12);
		case GL_NEAREST_MIPMAP_LINEAR:
			return (0x2 << 12);
		default: 
			s3virgeError("Unknown filter\n"); 
	}
	return 0;
}




/* =============================================================
 * Hardware clipping
 */

static void s3virgeUpdateClipping( const GLcontext *ctx )
{
	int		x1,x2,y1,y2;

	if (ctx->Scissor.Enabled) {
	    	x1 = ctx->Scissor.X;
		x2 = ctx->Scissor.X + ctx->Scissor.Width - 1;
		y1 = s3virgeDB->height - ctx->Scissor.Y - ctx->Scissor.Height + 1;
		y2 = s3virgeDB->height - ctx->Scissor.Y - 1;
		if ( x1 < 0 ) x1 = 0;
		if ( y1 < 0 ) y1 = 0;
		if ( x2 >= s3virgeDB->width ) x2 = s3virgeDB->width - 1;
		if ( y2 >= s3virgeDB->height ) y2 = s3virgeDB->height - 1;

		if ( ( x1 > x2 ) || ( y1 > y2 ) ) {
			x1 = 0; x2 = 0;
			y2 = 0; y1 = 1;
		}
	} else {
		x1 = 0;
		x2 = s3virgeDB->width;
		y1 = 0;
		y2 = s3virgeDB->height;
	}
	
	if (s3virgeglx.dmaDriver == 0) {
		OUTREG((S3VIRGE_3DTRI_REG | S3VIRGE_CLIP_L_R), ((x1 << 16) | x2));
		OUTREG((S3VIRGE_3DTRI_REG | S3VIRGE_CLIP_T_B), ((y1 << 16) | y2));
	} else {
		DMAOUT( ((x1 << 16) | x2) );
		DMAOUT( ((y1 << 16) | y2) );
	}
		
}



/* =============================================================
 */


/*
 * s3virgeDDUpdateState
 * This will be called before any beginrender if state has changed
 */
void s3virgeDDUpdateState( GLcontext *ctx ) {
	int		textureFormat;
	struct gl_texture_object *tObj;
	s3virgeTextureObject_t *t = NULL;	
	
	s3virgeglx.c_setupPointers++;

	/* Make sure our DMA is on */
	EnsureDMAOn();
	

	/* for now, we are going to set every register we care about */

	if ( !s3virgeDB ) {
		ctx->Driver.TriangleFunc = NULL;
		ctx->Driver.LineFunc = NULL;
		return;
	}

	/* no acceleration if our back buffer isn't on card */
	if ( !s3virgeDB->backBufferBlock ) {
		ctx->Driver.TriangleFunc = NULL;
		ctx->Driver.LineFunc = NULL;
		return;	
	}
	
	/* no acceleration if our z buffer isn't on card */
	if ( ctx->Depth.Test && !s3virgeDB->depthBufferBlock ) {
		ctx->Driver.TriangleFunc = NULL;
		ctx->Driver.LineFunc = NULL;
		return;	
	}
	
	/* no acceleration if any stencil operations are enabled */
	if ( ctx->Stencil.Enabled ) {
		ctx->Driver.TriangleFunc = NULL;
		ctx->Driver.LineFunc = NULL;
		return;	
	}

	/* no acceleration if stipple is on */
	if ( ctx->Polygon.StippleFlag ) {
		ctx->Driver.TriangleFunc = s3virgeGouraudTriangle;
		ctx->Driver.LineFunc = s3virgeGouraudTriangle;
		return;	
	}

//	ctx->Driver.TriangleFunc = s3virgeGouraudTriangle;
//	return;


//	s3virgeSaveState();
	
	s3virgeglx.triHack = 0;  

	if (!s3virgeIsTexturingEnabled(ctx)) {
		s3virgeglx.currenttri_cmd = (0x80000005 |
			(0x1 << 1) | 
			(0x3 << 18) | /* alpha blending */
			s3virgeCalcZ_CNTL( ctx ));
		ctx->Driver.TriangleFunc = s3virgeGouraudTriangle;
	} else {
		int ret; 
				
		tObj = ctx->Texture.Unit[0].Current;
		
		if (!tObj->DriverData) {
			s3virgeCreateTexObj( s3virgeCtx, tObj );
		}

		t = (s3virgeTextureObject_t *)tObj->DriverData;
		
		t->age = s3virgeglx.swapBuffersCount;

		ret = s3virgeCalcAlpha_CNTL( ctx );
		
		if (t->texelBytes == 1) {
			ret |= (0x6 << 5);
		} else if (t->texelBytes == 2) {
			if (t->hasAlpha)
				ret |= (0x1 << 5);
			else
				ret |= (0x2 << 5);
		}

		
		s3virgeglx.currenttri_cmd = (0x80000005 |
			(0x1 << 1) | 
			ret |
			((t->widthLog2 & 0xF) << 8) |
			s3virgeCalcScale( ctx ) |
			(0x1 << 26) | /* Texture wrap */
			s3virgeCalcZ_CNTL( ctx ));

		if (s3virgeglx.triHack & S3HACK_WHITE_TRI_ALPHA) {
			s3virgeglx.whitetri_cmd = (0x80000005 |
				(0x1 << 1) |
				(0x2 << 18) |
				((t->hasAlpha) ? (0x1 << 5) : (0x2 << 5)) |
				((t->widthLog2 & 0xF) << 8) |
				s3virgeCalcScale( ctx ) |
				(0x1 << 26) |
				(0x5 << 27) |
				s3virgeCalcZ_CNTL( ctx ));
		} else {			
			s3virgeglx.whitetri_cmd = (0x80000005 |
				(0x1 << 1) |
				s3virgeCalcZ_CNTL( ctx ));
		}
			
		ctx->Driver.TriangleFunc = s3virgeTextureTriangle;
	}

	if (s3virgeglx.dmaDriver == 0) {	
		WAITFIFOEMPTY(8);
		
		/* Depth source */
		if ( !s3virgeDB->depthBufferBlock ) {
			OUTREG( (S3VIRGE_3DTRI_REG | S3VIRGE_Z_BASE), 0 );
		} else {
			OUTREG( (S3VIRGE_3DTRI_REG | S3VIRGE_Z_BASE), (s3virgeDB->depthBufferBlock->ofs & 0x003FFFFF));
		}
	
		if ((s3virgeglx.triHack & S3HACK_ZERO_SRCCOL) && s3virgeglx.lightmapHack) {
			OUTREG( S3VIRGE_3DTRI_REG | S3VIRGE_DEST_BASE, (s3virgeDB->lightmapBufferBlock->ofs) & 0x003FFFFF ); /* Mem location of dest */
		} else {
			OUTREG( S3VIRGE_3DTRI_REG | S3VIRGE_DEST_BASE, (s3virgeDB->backBufferBlock->ofs) & 0x003FFFFF ); /* Mem location of dest */
		}

		s3virgeUpdateClipping(ctx);
	
		/* src & dest stride */
		if (!s3virgeIsTexturingEnabled(ctx)) {
			OUTREG( (S3VIRGE_3DTRI_REG | S3VIRGE_DEST_SRC_STRIDE), 
				((s3virgeDB->pitch * 2) << 16));
		} else {
			OUTREG( (S3VIRGE_3DTRI_REG | S3VIRGE_DEST_SRC_STRIDE), 
				((s3virgeDB->pitch * 2) << 16) 
				| (tObj->Image[0]->Width * t->texelBytes));
		}
		
		/* Z stride */
		OUTREG( (S3VIRGE_3DTRI_REG | S3VIRGE_Z_STRIDE), s3virgeDB->pitch * 2 );
		
		if (t)
			OUTREG( (S3VIRGE_3DTRI_REG | S3VIRGE_TEX_BASE), t->memBlock->ofs);

	
		OUTREG( (S3VIRGE_3DTRI_REG | S3VIRGE_CMDSET), s3virgeglx.currenttri_cmd);
	} else {
		DMAGETPTR(12);

		DMAOUTREG( (S3VIRGE_3DTRI_REG | S3VIRGE_Z_BASE), 12);
		
		/* Depth source */
		if ( !s3virgeDB->depthBufferBlock ) {
			DMAOUT( 0 );
		} else {
			DMAOUT((s3virgeDB->depthBufferBlock->ofs & 0x003FFFFF));
		}
	
		if ((s3virgeglx.triHack & S3HACK_ZERO_SRCCOL) && s3virgeglx.lightmapHack) {
			DMAOUT((s3virgeDB->lightmapBufferBlock->ofs) & 0x003FFFFF ); /* Mem location of dest */
		} else {
			DMAOUT((s3virgeDB->backBufferBlock->ofs) & 0x003FFFFF ); /* Mem location of dest */
		}

		s3virgeUpdateClipping(ctx);
	
		/* src & dest stride */
		if (!t) {
			DMAOUT(((s3virgeDB->pitch * 2) << 16));
		} else {
			DMAOUT(((s3virgeDB->pitch * 2) << 16) 
				| (tObj->Image[0]->Width * t->texelBytes));
		}
		
		/* Z stride */
		DMAOUT((s3virgeDB->pitch * 2 ));
		
		if (t)
			DMAOUT(t->memBlock->ofs);
		else 
			DMAOUT(0);
		
		DMAOUT(0);
		DMAOUT(0); /* Fog color */
		DMAOUT(0);
		DMAOUT(0);

		DMAOUT(s3virgeglx.currenttri_cmd);

		DMAFINISH();	
	}
}	


void s3virgeDDInitStatePointers( GLcontext *ctx )
{
    ctx->Driver.Enable = 0;
    ctx->Driver.LightModelfv = 0;
    ctx->Driver.AlphaFunc = 0;
    ctx->Driver.BlendEquation = 0;
    ctx->Driver.BlendFunc = 0;
    ctx->Driver.BlendFuncSeparate = 0;
    ctx->Driver.DepthFunc = 0;
    ctx->Driver.DepthMask = 0;
    ctx->Driver.Fogfv = 0;
    ctx->Driver.Scissor = 0;
    ctx->Driver.CullFace = 0;
    ctx->Driver.FrontFace = 0;
    ctx->Driver.ShadeModel = 0;
    ctx->Driver.ColorMask = 0;
    ctx->Driver.ReducedPrimitiveChange = 0;
    ctx->Driver.RenderStart = s3virgeDDUpdateState;
    ctx->Driver.RenderFinish = 0;
    ctx->Driver.UpdateState = 0;

#if 0 /* FIXME: GH */
    ctx->Driver.Viewport = s3virgeDDViewport;
    ctx->Driver.DepthRange = s3virgeDDDepthRange;
    ctx->Driver.RegisterVB = s3virgeDDRegisterVB;
    ctx->Driver.UnregisterVB = s3virgeDDUnregisterVB;
    ctx->Driver.TexEnv = s3virgeTexEnv;
    ctx->Driver.TexParameter = s3virgeTexParameter;
#endif
    ctx->Driver.UpdateTexturePalette = s3virgeUpdateTexturePalette;
    ctx->Driver.TexImage = s3virgeTexImage;
    ctx->Driver.TexSubImage = s3virgeTexSubImage;
    ctx->Driver.DeleteTexture = s3virgeDeleteTexture;
    ctx->Driver.IsTextureResident = s3virgeIsTextureResident;
}



/*
 * Local Variables:
 * mode: c
 * c-basic-offset: 4
 * End:
 */
