
/*
 * GLX Hardware Device Driver for SiS 6326
 * Copyright (C) 2000 Jim Duchek
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * WITTAWAT YAMWONG, OR ANY OTHER CONTRIBUTORS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * Based on Mach64 driver: mach64glx.c
 *
 *    Jim Duchek <jim@linuxpimps.com>
 */

#include <stdlib.h>

#include "context.h"
#include "depth.h"
#include "macros.h"
#include "texstate.h"
#include "triangle.h"
#include "vb.h"
#include "types.h"
#include "compiler.h"

#include "xsmesaP.h"
#include "glx_log.h"
#include "glx_config.h"
#include "mesaglx/context.h"
#include "mesaglx/matrix.h"
#include "mesaglx/types.h"

#define GC XXGC
#include "gcstruct.h"
#include "pixmapstr.h"
#include "servermd.h" /* PixmapBytePad */
#include "scrnintstr.h"
#include "regionstr.h"
#include "windowstr.h"
#undef GC

#include "sis6326glx.h"

#include "glx_symbols.h"

/* our collected globals */
sis6326Glx_t		sis6326glx;
sis6326BufferPtr	sis6326DB = NULL;
sis6326ContextPtr	sis6326Ctx = NULL;

memHeap_t	*cardHeap;

memHeap_t	*textureHeap;

hwUI32		sysmemBytes;		/* size of memory block */
hwUI32		sysmemPhysical;		/* 0 if we don't have a physical mapping */
unsigned char	*sysmemVirtual;


/*
 * ForceSoftwareBuffers
 * Free any hardware buffers and reallocate only in
 * virtual memory, so hardware acceleration can't be used.
 */
static void ForceSoftwareBuffers( sis6326BufferPtr buf ) {
	if ( buf->backBufferBlock ) {
		/* free the hw buffer */
		mmFreeMem( buf->backBufferBlock );
		buf->backBufferBlock = NULL;
	}

	/* allocate a main memory buffer */
	hwMsg( 1, "moving back buffer to system memory to disable hw accel.\n" );
	buf->backBuffer = malloc( buf->pitch * buf->height * sis6326glx.bytesPerPixel );
	if ( !buf->backBuffer ) {
		FatalError( "Malloc for back buffer failed" );
	}

	if ( buf->depthBufferBlock ) {
		/* free the hw buffer */
		mmFreeMem( buf->depthBufferBlock );
		buf->depthBufferBlock = NULL;
	}

	/* allocate a main memory buffer */
	hwMsg( 1, "moving depth buffer to system memory to disable hw accel.\n" );
	buf->depthBuffer = malloc( buf->pitch * buf->height * 2 );
	if ( !buf->depthBuffer ) {
		FatalError( "Malloc for depth buffer failed" );
	}
}


void sis6326GLXCreateDepthBuffer( GLcontext* ctx ) {
	/* we create the depth buffer when creating the image,
	 * so there is nothing to do here.
	 */
}

/*
 * sis6326GLXDestroyImage
 */
void sis6326GLXDestroyImage( GLXImage* image ) {
	sis6326BufferPtr	buf;

	hwMsg( 1, "sis6326GLXDestroyImage( %p )\n", image->devPriv );

	buf = (sis6326BufferPtr)image->devPriv;

	/* free memory either in card space or VM */
	if ( buf->backBufferBlock ) {
		mmFreeMem( buf->backBufferBlock );
	} else {
		free( buf->backBuffer );
	}

	if ( buf->depthBufferBlock ) {
		mmFreeMem( buf->depthBufferBlock );
	} else {
		free( buf->depthBuffer );
	}

	buf->magic = 0;
	free( buf );

	if ( hwGetLogLevel() >= 1 )  {
		mmDumpMemInfo( cardHeap );
	}

	xfree( image );
}

/*
 * sis6326GLXCreateImage
 *
 * This will always succeed, but if the buffer couldn't be allocated in
 * card memory, HW accel won't be usable.
 */
GLXImage *sis6326GLXCreateImage( WindowPtr pwindow, int depth,
				int width, int height, GLXImage *old_image )
{
	sis6326BufferPtr		buf;
	GLXImage		*image;

	hwMsg( 1, "sis6326GLXCreateImage( %i, %i )\n", width, height );

	/* free the old image */
	if ( old_image && old_image->devPriv ) {
		sis6326GLXDestroyImage( old_image );
	}

	buf = (sis6326BufferPtr)calloc(1,sizeof(sis6326Buffer));
	if ( !buf ) {
		FatalError( "Malloc for buf failed\n" );
	}

	buf->magic = sis6326BufferMagic;

	buf->width = width;
	buf->height = height;
	buf->pitch = ( width + 63 ) & ~63;

	image = (GLXImage *) xalloc( sizeof(GLXImage) );
	if ( !image ) {
		FatalError( "Malloc for back ximage failed" );
	}
	
	image->pwin = pwindow;
	image->width = width;
	image->height = height;
	image->bits_per_pixel = depth;
	image->data = buf->backBuffer;
	image->bytes_per_line = buf->pitch * sis6326glx.bytesPerPixel;
	image->devPriv = buf;

	/* How can I draw XImage with any line pad? */
	/* e.g. bytes_per_line != PixmapBytePad */
	/* My work around: adjust width instead */
	image->width = buf->pitch;

	/* allocate the back buffer */
	buf->backBufferBlock = mmAllocMem( cardHeap, buf->pitch * height *
					   sis6326glx.bytesPerPixel, 7, 0 );

	/* always allocate a depth buffer, rather than waiting for
	 * createDepthBuffer, because some weird conditions can arise
	 * where drawing functions are called after a resize, but before
	 * depth buffer re-creation.  All buffers should be created in one
	 * place anyway...
	 */
 	buf->depthBufferBlock = mmAllocMem( cardHeap, buf->pitch * buf->height * 2, 7, 0 );


	/* fall back to software rendering if buffers didn't fit on card.
	 * it is possible to accelerate some operations with a back
	 * buffer in hardware and a depth buffer in software, but it
	 * will often be slower due to no caching on the framebuffer,
	 * so just force both of them to software
	 */
	if ( buf->backBufferBlock == 0 || buf->depthBufferBlock == 0 ) {
		ForceSoftwareBuffers( buf );
	} else {
		buf->backBuffer = (unsigned char *)
			sis6326glx.linearBase + buf->backBufferBlock->ofs;
		buf->depthBuffer = (unsigned char *)
			sis6326glx.linearBase + buf->depthBufferBlock->ofs;
		if ( hwGetLogLevel() >= 1 )  {
			mmDumpMemInfo( cardHeap );
		}
	}

	return image;
}


/*
 * DoMakeCurrent
 * Changing either the context or the buffer will force this.
 */
static void DoMakeCurrent( XSMesaContext c, XSMesaBuffer b )
{
	sis6326ContextPtr	ctx;
	sis6326BufferPtr		buf;

	hwMsg( 10, "DoMakeCurrent( %p, %p )\n", c, b );

	sis6326Ctx = NULL;
	sis6326DB = NULL;

	if ( !c || !b ) {
		return;
	}

	ctx = c->hw_ctx;
	if ( !VALID_SIS6326_CONTEXT( ctx ) ) {
		FatalError( "DoMakeCurrent: !VALID_SIS6326_CONTEXT" );
	}

	/* if we don't have a backimage, create one now.  This happens
	 * because of the way software GLX doesn't allocate a
	 * buffer when front buffer rendering, but we need one
	 * anyway
	 */
	if ( !b->backimage ) {
		b->backimage = sis6326GLXCreateImage( (WindowPtr)b->frontbuffer,
						     16, b->width, b->height, NULL );
	}

	buf = (sis6326BufferPtr) b->backimage->devPriv;
	if ( buf->magic != sis6326BufferMagic ) {
		FatalError( "DoMakeCurrent: != sis6326BufferMagic" );
	}

	if ( !ctx || !buf ) {
		return;
	}

	/* for development, we can turn off hardware accel on a per-context basis
	 * the most global way to prevent any hardware acceleration is to force the
	 * back buffer into main memory instead of card memory.
	 */
	if ( c->no_accel && ( buf->backBufferBlock || buf->depthBufferBlock ) ) {
		ForceSoftwareBuffers( buf );
	}

	/* valid conditions to render with */
	buf->xsBuf = b;	/* save the xsbuffer so we can get the window for front swaps */
	ctx->DB = buf;
	sis6326Ctx = ctx;
	sis6326DB = ctx->DB;
}


/*
 * sis6326GLXMakeCurrent
 * Note that GLX calls this every batch of commands,
 * so it will show up several times a frame even if
 * the client application isn't issuing makeCurrent calls.
 */
GLboolean sis6326GLXMakeCurrent( XSMesaContext c )
{
	if ( c == XSMesa ) {
		return GL_TRUE;
	}

	/* mesa 3.1 will sometimes leave unflushed vertexes */
	if ( XSMesa ) {
		glFlush();
	}

	hwMsg( 10, "sis6326GLXMakeCurrent( %p )\n", c );
	XSMesa = c;
	if ( c ) {
		DoMakeCurrent( c, c->xsm_buffer );
		gl_make_current( c->gl_ctx,
				 ( c->xsm_buffer ) ? c->xsm_buffer->gl_buffer : NULL );
	} else {
		/* detach */
		DoMakeCurrent( NULL, NULL );
		gl_make_current( NULL, NULL );
	}

	return GL_TRUE;
}

/*
 * Bind buffer b to context c and make c the current rendering context.
 */
GLboolean sis6326GLXBindBuffer( XSMesaContext c, XSMesaBuffer b )
{
	hwMsg( 10, "sis6326GLXBindBuffer( %p, %p )\n", c, b );

	DoMakeCurrent( c, b );
	XSMesaBindBuffer( c, b );

	if ( sis6326Ctx ) {
		sis6326_setup_DD_pointers( sis6326Ctx->gl_ctx );
	}

	return GL_TRUE;
}



/*
 * Create a new XSMesaContext.
 * Input:  v - XSMesaVisual
 *         share_list - another XSMesaContext with which to share display
 *                      lists or NULL if no sharing is wanted.
 * Return:  an XSMesaContext or NULL if error.
 */
XSMesaContext sis6326GLXCreateContext( XSMesaVisual v,
				      XSMesaContext share_list )
{
	XSMesaContext	c;

	hwMsg( 0, "### Creating new xsmesa context for SiS 6326...\n" );

	c = (XSMesaContext) calloc( 1, sizeof(struct xsmesa_context) );
	if ( !c ) {
		return NULL;
	}

	c->gl_ctx = gl_create_context( v->gl_visual,
				       ( share_list ) ? share_list->gl_ctx : NULL,
				       (void *) c,
				       GL_TRUE /* direct rendering */ );
	if ( !c->gl_ctx ) {
		free( c );
		return NULL;
	}

	c->xsm_visual = v;
	c->xsm_buffer = NULL;			/* set later by XSMesaMakeCurrent */
	c->pixelformat = v->dithered_pf;	/* Dithering is enabled by default */
	c->hw_ctx = (sis6326ContextPtr) calloc( 1, sizeof(sis6326Context) );
	if ( !c->hw_ctx ) {
		hwError( "Cannot create hardware specific context.\n" );
		return NULL;
	}

	((sis6326Context *)c->hw_ctx)->magic = sis6326ContextMagic;
	((sis6326Context *)c->hw_ctx)->gl_ctx = c->gl_ctx;

	/* clear extensions we don't support */
	sis6326DDExtensionsInit( c->gl_ctx );

	((sis6326ContextPtr)(c->hw_ctx))->refcount++;
	c->gl_ctx->Driver.UpdateState = sis6326_setup_DD_pointers;

	/* Ask mesa to clip fog coordinates for us. */
	c->gl_ctx->TriangleCaps |= DD_CLIP_FOG_COORD;

	/* modify pipeline for fog / cva / etc */
	if ( c->gl_ctx->NrPipelineStages ) {
		c->gl_ctx->NrPipelineStages =
			sis6326DDRegisterPipelineStages( c->gl_ctx->PipelineStage,
							c->gl_ctx->PipelineStage,
							c->gl_ctx->NrPipelineStages );
	}

	hwMsg( 1, "sis6326GLXCreateContext succeeded: %p\n", c );

	return c;
}



void sis6326GLXDestroyContext( XSMesaContext c )
{
	hwMsg( 1, "sis6326GLXDestroyContext( %p )\n", c );

	if ( c->hw_ctx == sis6326Ctx ) {
		sis6326Ctx = NULL;
	}
	if ( XSMesa == c ) {
		XSMesa = 0;
	}

	((sis6326Context *)c->hw_ctx)->magic = 0;
	free( c->hw_ctx );

	XSMesaDestroyContext( c );
}


/*
 * Local Variables:
 * mode: c
 * tab-width: 8
 * c-basic-offset: 8
 * End:
 */


//#ifndef NO_MTRR
#if 0
static int IsPowerOfTwo( int val )
{
	int		i;

	for ( i = 0 ; i < 32 ; i++ ) {
		if ( val == ( 1 << i ) ) {
			return 1;
		}
	}
	return 0;
}

static void sis6326CloseMTRR( void )
{
	close( mtrr );
}

static void sis6326OpenMTRR( void )
{
	if ( ( mtrr = open( "/proc/mtrr", O_WRONLY, 0 ) ) == -1 )
	{
		if ( errno == ENOENT ) {
			hwError( "/proc/mtrr not found: MTRR not enabled\n" );
		}  else {
			hwError( "Error opening /proc/mtrr: %s\n", strerror( errno ) );
			hwError( "MTRR not enabled\n" );
		}
		return;
	}
	atexit( sis6326CloseMTRR );
}


static int CoverRangeWithMTRR( int base, int range, int type )
{
	int		count;
	int		size;

	count = 0;
	hwMsg( 1,"CoverRangeWithMTRR( 0x%x, 0x%x, %i )\n", base, range, type );

	while ( range )
	{
		/* see how big of an mtrr we can make */

		/* size must be a power of 2 */
		for ( size = 2048 ; ; )
		{
			size *= 2;

			/* the base must be a multiple of the size */
			if ( base != size * ( base / size ) ) {
				size >>= 1;
				break;
			}

			if ( size > range ) {
				size >>= 1;
				break;
			}
		}

		/* set it if we aren't just checking the number */
		if ( type != -1 ) {
			struct mtrr_sentry sentry;

			sentry.base = base;
			sentry.size = size;
			sentry.type = type;

			hwMsg( 1, "MTRR fragment added: addr=0x%x size=0x%x type=%i\n",
			       sentry.base, sentry.size, sentry.type );
			if ( ioctl( mtrr, MTRRIOC_SET_ENTRY, &sentry ) == -1 ) {
				hwError( "Error doing ioctl(2) on /proc/mtrr: %s\n",
					 strerror( errno ) );
			}
		}

		base += size;
		range -= size;
		count++;
	}

	hwMsg( 1, "------\n" );

	return count;
}

/*
 * SetWriteCombining
 *
 * Checks for unsetting an existing MTRR if needed
 */
static void SetWriteCombining( long physical, int bytes )
{
	struct mtrr_sentry sentry;
	struct mtrr_gentry gentry;
	int i;

	if ( !mtrr ) {
		return;
	}

	/* remove any MTRR that conver the range */
	for ( i = 0 ; i < 128 ; i++ )
	{
		gentry.regnum = i;
		if ( ioctl( mtrr, MTRRIOC_GET_ENTRY, &gentry ) == -1 ) {
			break;
		}
		hwMsg( 1, "MTRR reg %i: addr=0x%x size=0x%x type=%i\n",
		       i, gentry.base, gentry.size, gentry.type );
		if ( gentry.base >= physical + bytes ) {
			continue;
		}
		if ( gentry.base + gentry.size <= physical ) {
			continue;
		}

		/* we must delete this entry */
		sentry.base = gentry.base;
		sentry.size = gentry.size;
		if ( ioctl( mtrr, MTRRIOC_DEL_ENTRY, &sentry ) == -1 ) {
			hwError( "Error doing MTRRIOC_DEL_ENTRY on /proc/mtrr: %s\n",
				     strerror( errno ) );
		} else {
			hwMsg( 1, "MTRRIOC_DEL_ENTRY succeeded\n" );
		}

		/* recreate fragments around the new region if necessary */
		if ( gentry.base < physical ) {
			CoverRangeWithMTRR( gentry.base,
					    physical - sentry.base,
					    gentry.type );
		}
		if ( gentry.base + gentry.size > physical + bytes ) {
			CoverRangeWithMTRR( physical + bytes,
					    gentry.base + gentry.size - sentry.base,
					    gentry.type );
		}

		/* because we deleted an entry, we need to check this index again */
		i--;
	}

	/* set this range to write combining */
	sentry.base = physical;
	sentry.size = bytes;
	sentry.type = MTRR_TYPE_WRCOMB; /* write-combining */

	if ( ioctl( mtrr, MTRRIOC_SET_ENTRY, &sentry ) == -1 ) {
		hwError( "Error doing ioctl(2) on /proc/mtrr: %s\n",
			 strerror( errno ) );
		hwError( "MTRR not enabled\n" );
	} else {
		hwMsg( 1, "MTRR enabled: write-combining, addr=0x%x size=0x%x\n",
		       sentry.base, sentry.size );
	}
}
#endif

/*=============================================*/


/*
 * This function should only verify that the current hardware is supported.
 * It should do no setup.
 */
static GLboolean det_hwGfx()
{
	if ( sis6326glx.depth == 24 ) {
		sis6326glx.depth = 32;	// FIXME
	}

	if( ( sis6326glx.depth != 15 ) && ( sis6326glx.depth != 16 ) && ( sis6326glx.depth != 32 ) ) {
		hwError( "Unsupported depth: %d, only 15, 16, and 32 bpp are supported right now\n",
			 sis6326glx.depth );
		return GL_FALSE;
	}

	return GL_TRUE;
}

/*
 * sis6326InitLogging
 *
 */
void sis6326InitLogging( void )
{
	char	*logName;

	/* open the logfile and set loglevel */
	logName = glx_getvar_secure( "hw_logfile" );
	if ( __glx_is_server ) {
		hwOpenLog( logName, "[sis6326] " );
	} else {
		/* direct rendering clients use a different file
		   so they don't stomp on the server's log */
		if ( logName ) {
			char	newName[1024];

			strcpy( newName, logName );
			strcat( newName, "_direct" );
			hwOpenLog( newName, "[sis6326] " );
		}
		else {
			/* just set logging prefix */
			hwOpenLog( NULL, "[sis6326] " );
		}
	}

	if ( glx_getvar( "hw_loglevel" ) ) {
		hwSetLogLevel( glx_getint( "hw_loglevel" ) );
	} else {
		hwSetLogLevel( DBG_LEVEL_BASE );
	}
}


/*
 *
 * GetXServerInfo
 */
static int GetXServerInfo( void ) 
{

	sis6326glx.linearPhysical = GLXSYM(vga256InfoRec).MemBase;
	sis6326glx.linearBase = (char *)GLXSYM(vgaLinearBase);
	sis6326glx.MMIOBase = (char *)GLXSYM(sisMMIOBase);
	sis6326glx.depth = GLXSYM(vga256InfoRec).depth;
	sis6326glx.virtualX = GLXSYM(vga256InfoRec).virtualX;
	sis6326glx.virtualY = GLXSYM(vga256InfoRec).virtualY;
	sis6326glx.displayWidth = GLXSYM(vga256InfoRec).displayWidth;
	sis6326glx.videoRam = GLXSYM(vga256InfoRec).videoRam;
	sis6326glx.bytesPerPixel = ( GLXSYM(vga256InfoRec).bitsPerPixel +
				    7 ) / 8;

	/* under some conditions the fbdev server reports
	   displayWidth == -1 for some reason */
	if ( sis6326glx.displayWidth < sis6326glx.virtualX ) {
		sis6326glx.displayWidth = sis6326glx.virtualX;
	}

	hwMsg( 1, "width: %d\n", sis6326glx.virtualX );
	hwMsg( 1, "height: %d\n", sis6326glx.virtualY );
	hwMsg( 1, "pitch: %d\n", sis6326glx.displayWidth );
	hwMsg( 1, "depth: %d\n", sis6326glx.depth );
	hwMsg( 1, "bytesPerPixel: %d\n", sis6326glx.bytesPerPixel );
	hwMsg( 1, "videoRam: %dk\n", sis6326glx.videoRam );
	hwMsg( 1, "memBase: 0x%08x\n", sis6326glx.linearBase );
	hwMsg( 1, "ioBase: 0x%08x\n", sis6326glx.MMIOBase );
	hwMsg( 1, "memPhysical: 0x%08x\n", sis6326glx.linearPhysical );
	hwMsg( 1, "apertureOffset: 0x%08x\n", sis6326glx.apertureOffset );

	return 1;
}

void
sis6326DumpRegisters()
{
	int i;

	hwMsg(1, "SRs\n");
	for (i=0; i<=0x3C; i++) {
		outb(0x3c4, i);
		hwMsg(1, "%02x = %02x\n", i, inb(0x3c5));
	}
	
	hwMsg(1, "Config Regs\n");
	for (i=0; i<=0x5C; i+=4) {
		hwMsg(1, "%02x = %08x\n", i, INREG(i));
	}

	hwMsg(1, "2D Regs.\n");
	for (i=0x8280; i<=0x82FC; i+=4) {
		hwMsg(1, "%02x = %08x\n", i, INREG(i));
	}


	hwMsg(1, "3D Regs.\n");
	for (i=0x8800; i<=0x8B7C; i+=4) {
		hwMsg(1, "%02x = %08x\n", i, INREG(i));
	}

}



/*
 * sis6326InitGLX
 * This is the initial entry point for the sis6326 hardware driver,
 * called at X server module load time, or libGL direct rendering
 * init time.
 */
GLboolean sis6326InitGLX( void )
{
	int tmp;
	int i;
	sis6326InitLogging();

	if ( __glx_is_server ) {
		if ( !GetXServerInfo() ) {
			hwMsg( 1, "not a 6326!\n" );
			return GL_FALSE;
		}

    		/* check to make sure that we are on an apropriate chip and not
      	 		running in 8bpp mode */
    		if ( !det_hwGfx() ) {
			return GL_FALSE;
    		}
		/* make sure the user has turned off the pixmap and font cache */
		/* if the don't, then rendering will write all over their fonts */
		if ( !OFLG_ISSET( OPTION_NO_PIXMAP_CACHE, &GLXSYM(vga256InfoRec).options ) ) {
			hwError( "Pixmap caches must be disabled to use the GLX module.\n" );
			hwError( "Make sure you have the following in your XF86config file:\n" );
			hwError( "Section \"Device\"\n" );
 	   		hwError( "	Option	\"no_pixmap_cache\"\n" );
			return GL_FALSE;
		}
		
		/* Lets turn the son of a bitch on... */
		
		outb(0x3c4, 0x05);
		outb(0x3c5, 0x86);
		tmp = inb(0x3c5);

		if (tmp != 0xA1)
			hwMsg(1, "Problem unlocking registers...\n");


		sis6326DumpRegisters();

		// enable Turbo Queue & reg enable
		outb(0x3c4, 0x27);
		tmp = inb(0x3c5);
		outb(0x3c5, 0xC0);
				
		// set Turbo Queue to second-last 32k segment of RAM		
		outb(0x3c4, 0x2C);
		outb(0x3c5, ((sis6326glx.videoRam - 64) / 32));

		// turn off 3D engine
		outb(0x3c4, 0x39);
		tmp = inb(0x3c5);
		outb(0x3c5, (tmp | 0x4));

		// set TQ to save room for mostly 3D		
		outb(0x3c4, 0x3C);
		tmp = inb(0x3c5);
		outb(0x3c5, (tmp | 0x3));

		// i/o gating wihle write-buffer is not empty
		outb(0x3c4, 0xB);
		tmp = inb(0x3c5);
		outb(0x3c5, (tmp | 0x4));

	} else {
		/* we are starting up as a direct rendering client */
	}

	/* start up our card memory manager */
	cardHeap = mmInit( 0, sis6326glx.videoRam * 1024 );
	if ( !cardHeap ) {
		hwMsg( 1, "cardHeap creation failed, exiting!\n" );
		return GL_FALSE;	/* really shouldn't happen */
	}

	/* reserve memory for the front buffer */
	mmReserveMem( cardHeap, 0, sis6326glx.displayWidth * sis6326glx.virtualY * sis6326glx.bytesPerPixel );

	/* reserve memory for the Turbo Queue (patent pending ;) and hw cursor */
	mmReserveMem( cardHeap, (sis6326glx.videoRam-64) * 1024, 64 * 1024 );

	/* the remaining memory is available for back buffers, depth
	   buffers, and textures */
	mmDumpMemInfo( cardHeap );

	if ( !sis6326glx.agpTextures ) {
		/* fall back to local card memory for textures */
		textureHeap = cardHeap;
	}

	/* FIXME: what other GLXProcs pointers should we change? */
	GLXProcs.CreateContext = sis6326GLXCreateContext;
	GLXProcs.DestroyContext = sis6326GLXDestroyContext;
	GLXProcs.SwapBuffers = sis6326GLXSwapBuffers;
	GLXProcs.CreateImage = sis6326GLXCreateImage;
	GLXProcs.DestroyImage = sis6326GLXDestroyImage;
	GLXProcs.CreateDepthBuffer = sis6326GLXCreateDepthBuffer;
	GLXProcs.MakeCurrent = sis6326GLXMakeCurrent;
	GLXProcs.BindBuffer = sis6326GLXBindBuffer;
	GLXProcs.VendorPrivate = sis6326GLXVendorPrivate;
	GLXProcs.AllowDirect = sis6326GLXAllowDirect;

	/* these vars can be changed between invocations of direct clients */
	if ( glx_getint( "sis6326_nullprims" ) ) {
		hwMsg( 1, "enabling sis6326_nullprims\n" );
		sis6326glx.nullprims = 1;
	}
	if ( glx_getint( "sis6326_skipdma" ) ) {
		hwMsg( 1, "enabling sis6326_skipdma\n" );
		sis6326glx.skipDma = 1;
	}
	if ( glx_getint( "hw_boxes" ) ) {
		hwMsg( 1, "enabling hw_boxes\n" );
		sis6326glx.boxes = 1;
	}
	if ( glx_getint( "sis6326_nofallback" ) ) {
		hwMsg( 1, "enabling sis6326_nofallback\n" );
		sis6326glx.noFallback = 1;
	}

	if ( glx_getint( "sis6326_finish" ) ) {
		hwMsg( 1, "enabling sis6326_finish\n" );
		sis6326glx.enforceFinish = 1;
	}

    	/* see if we can draw our basic primitives */
    	if ( __glx_is_server && glx_getint( "sis6326_drawtest" ) ) {
		hwMsg( 1, "enabling sis6326_drawtest\n" );
	}

	hwError( "sis6326InitGLX completed\n" );
	return GL_TRUE;
}

void sis6326Finish( void )
{
	int timeout = 0;
	while(!(INREG(SIS6326_TRI_FIRE) & 0x3)) { 
		timeout++; 
		if (timeout > 0xFFFFF) { 
			hwMsg(1, "Finish timed out..\n"); 
			break; 
		}
	}
}