/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include "opengl.h"

#include "visu_extension.h"
#include "visu_tools.h"
#include "visu_object.h"
#include "visu_configFile.h"
#include "openGLFunctions/renderingMode.h"
#include "openGLFunctions/interactive.h"
#include "openGLFunctions/view.h"
#include "openGLFunctions/objectList.h"
#include "renderingBackend/visu_windowInterface.h"
#include "coreTools/toolConfigFile.h"
#include "coreTools/toolColor.h"

/******************************************************************************/

/* static int stereo; */
static LightEnvironnement *currentLights;

#define FLAG_PARAMETER_OPENGL_IMMEDIATE   "opengl_immediateDrawing"
#define DESC_PARAMETER_OPENGL_IMMEDIATE   "If true, changes of parameters means immediate redrawing ; boolean 0 or 1"
#define PARAMETER_OPENGL_IMMEDIATE_DEFAULT 1
static int opengl_immediate;
static gboolean readOpenGLImmediate(gchar **lines, int nbLines, int position,
				       VisuData *dataObj, GError **error);

#define FLAG_PARAMETER_OPENGL_ANTIALIAS "opengl_antialias"
#define DESC_PARAMETER_OPENGL_ANTIALIAS "If true, lines are drawn smoother ; boolean 0 or 1"
#define PARAMETER_OPENGL_ANTIALIAS_DEFAULT 0
static int antialias;
static gboolean readOpenGLAntialias(gchar **lines, int nbLines, int position,
				       VisuData *dataObj, GError **error);

#define FLAG_PARAMETER_OPENGL_FAKEBS "opengl_fakeBackingStore"
#define DESC_PARAMETER_OPENGL_FAKEBS "If true, V_Sim catches the Expose event from the X server and calls a redraw ; boolean 0 or 1"
#define PARAMETER_OPENGL_FAKEBS_DEFAULT 0
static int fakeBackingStore;

#define FLAG_PARAMETER_OPENGL_TRANS "opengl_trueTransparency"
#define DESC_PARAMETER_OPENGL_TRANSS "If true, the transparency rendering is enhanced ; boolean 0 or 1"
static gboolean trueTransparency = FALSE;
static gboolean readOpenGLTransparency(gchar **lines, int nbLines, int position,
				       VisuData *dataObj, GError **error);

#define FLAG_PARAMETER_OPENGL_ANGLE "opengl_stereoAngle"
#define DESC_PARAMETER_OPENGL_ANGLE "Give the angle of the two receivers in stereo output ; float positive"
static float stereoAngles[2];
#define PARAMETER_OPENGL_ANGLE_DEFAULT 5.f
static gboolean readOpenGLStereoAngle(gchar **lines, int nbLines, int position,
				       VisuData *dataObj, GError **error);

#define FLAG_PARAMETER_OPENGL_STEREO "opengl_stereo"
#define DESC_PARAMETER_OPENGL_STEREO "If true, try to draw in stereo ; boolean 0 or 1"
static gboolean stereoStatus = FALSE;
static gboolean readOpenGLStereo(gchar **lines, int nbLines, int position,
				       VisuData *dataObj, GError **error);

/* Local methods. */
static void exportParametersOpenGL(GString *data, VisuData *dataObj);

void glSetAntiAlias();
/* Private - This initialise the opengl redering (light...) */
void gl_new_data(GenericRenderingWindow window, VisuData *data);

/******************************************************************************/

int initOpengl()
{
  VisuConfigFileEntry *resourceEntry;

  DBG_fprintf(stderr, "OpenGl: initialization.\n");

/*   glViewport(0, 0, width, height); */

  /* Deals with lights. */
  currentLights = lightEnvironnement_new();
  lightEnvironnementAdd_light(currentLights, light_newDefault());

  /* Parameters */
  resourceEntry = visuConfigFileAdd_entry(VISU_CONFIGFILE_PARAMETER,
					  FLAG_PARAMETER_OPENGL_TRANS,
					  DESC_PARAMETER_OPENGL_TRANSS,
					  1, readOpenGLTransparency);
  visuConfigFileSet_version(resourceEntry, 3.4f);
  resourceEntry = visuConfigFileAdd_entry(VISU_CONFIGFILE_PARAMETER,
					  FLAG_PARAMETER_OPENGL_ANTIALIAS,
					  DESC_PARAMETER_OPENGL_ANTIALIAS,
					  1, readOpenGLAntialias);
  resourceEntry = visuConfigFileAdd_entry(VISU_CONFIGFILE_PARAMETER,
					  FLAG_PARAMETER_OPENGL_IMMEDIATE,
					  DESC_PARAMETER_OPENGL_IMMEDIATE,
					  1, readOpenGLImmediate);
  resourceEntry = visuConfigFileAdd_entry(VISU_CONFIGFILE_PARAMETER,
					  FLAG_PARAMETER_OPENGL_ANGLE,
					  DESC_PARAMETER_OPENGL_ANGLE,
					  1, readOpenGLStereoAngle);
  visuConfigFileSet_version(resourceEntry, 3.4f);
  resourceEntry = visuConfigFileAdd_entry(VISU_CONFIGFILE_PARAMETER,
					  FLAG_PARAMETER_OPENGL_STEREO,
					  DESC_PARAMETER_OPENGL_STEREO,
					  1, readOpenGLStereo);
  visuConfigFileSet_version(resourceEntry, 3.4f);
  visuConfigFileAdd_exportFunction(VISU_CONFIGFILE_PARAMETER,
				   exportParametersOpenGL);
  
  /* Initialise others sub-modules dealing with OpenGL. */
  openGLObjectList_init();
  openGLInit_renderingMode();
  OpenGLViewInit();
  visuInteractive_get_type();

  antialias = PARAMETER_OPENGL_ANTIALIAS_DEFAULT;
  fakeBackingStore = PARAMETER_OPENGL_FAKEBS_DEFAULT;
  opengl_immediate = PARAMETER_OPENGL_IMMEDIATE_DEFAULT;
  stereoAngles[0] = +PARAMETER_OPENGL_ANGLE_DEFAULT;
  stereoAngles[1] = -PARAMETER_OPENGL_ANGLE_DEFAULT;

  return 1;
}

LightEnvironnement* openGLGet_currentLights()
{
  return currentLights;
}

/******************************************************************************/

void openGLSet_color(float* material, float* rgba)
{
  float mm[4] = {0.0f, 0.0f, 0.0f, 0.0f};

  glColor4fv(rgba);

  mm[3] = rgba[3];
  mm[0] = material[material_amb] * rgba[0];
  mm[1] = material[material_amb] * rgba[1];
  mm[2] = material[material_amb] * rgba[2];
  glMaterialfv(GL_FRONT, GL_AMBIENT, mm);
  mm[0] = material[material_dif] * rgba[0];
  mm[1] = material[material_dif] * rgba[1];
  mm[2] = material[material_dif] * rgba[2];
  glMaterialfv(GL_FRONT, GL_DIFFUSE, mm);
  glMaterialf(GL_FRONT, GL_SHININESS, material[material_shi] * 128.0f);
  mm[0] = material[material_spe];
  mm[1] = material[material_spe];
  mm[2] = material[material_spe];
  glMaterialfv(GL_FRONT, GL_SPECULAR, mm);
  mm[0] = material[material_emi] * rgba[0];
  mm[1] = material[material_emi] * rgba[1];
  mm[2] = material[material_emi] * rgba[2];
  glMaterialfv(GL_FRONT, GL_EMISSION, mm);
}
void openGLSet_highlightColor(float material[5], float rgb[3], float alpha)
{
  float rgba[4], hsl[3];
  
  color_RGBtoHSL(hsl, rgb);
  hsl[0] = fModulo(0.5f + hsl[0], 1);
  hsl[1] = 1.f;
  hsl[2] = .5f;
  color_HSLtoRGB(rgba, hsl);
  rgba[3] = alpha;
  openGLSet_color(material, rgba);
}

void openGLInit_context()
{
  /* Set the openGL flags. */
  glEnable(GL_DEPTH_TEST);
  glDepthFunc(GL_LEQUAL);
  glDepthRange(0.0, 1.0);
  glClearDepth(1.0);

  glEnable(GL_CULL_FACE);
  glCullFace(GL_BACK);

  glEnable(GL_NORMALIZE);

  glEnable(GL_BLEND);
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

  lightEnvironnementApply(currentLights);
  openGLApply_renderingMode(openGLGet_globalRenderingOption());

  glSetAntiAlias();

/*   OpenGLViewSet_windowSize(view, wi, he); */
}

/******************************************************************************/

void openGL_reDraw(const char **lists, VisuData *data)
{
  int i, i_stereo, stereo;
  static int stereo_buf[2] = {GL_BACK_LEFT, GL_BACK_RIGHT};
  OpenGLView *view;
  GLboolean glStereo;

#if DEBUG == 1
  GTimer *timer;
  gulong fractionTimer;
#endif

#if DEBUG == 1
  timer = g_timer_new();
  g_timer_start(timer);
#endif
  
  g_return_if_fail(IS_VISU_DATA_TYPE(data));

  view = visuDataGet_openGLView(VISU_DATA(data));

  glGetBooleanv(GL_STEREO, &glStereo);
  stereo = (view && glStereo && stereoStatus)?1:0;
  for(i_stereo = 0; i_stereo <= stereo; i_stereo++)
    {
      
      if (stereo == 1)
	{   
	  glRotatef(stereoAngles[i_stereo],
		    view->camera->up[0],
		    view->camera->up[1],
		    view->camera->up[2]);
	  glDrawBuffer(stereo_buf[i_stereo]);
	  DBG_fprintf(stderr, "OpenGL: draw on buffer %d.\n", i_stereo);
	}
      else
	glDrawBuffer(GL_BACK);

      glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

      glEnable(GL_DEPTH_TEST);
      if (trueTransparency)
	{
	  glDepthMask(1);
	  glEnable(GL_ALPHA_TEST);
	  glAlphaFunc(GL_EQUAL, 1.);
	}
      else
	glDisable(GL_ALPHA_TEST);

      glPushMatrix();
      glTranslated(-view->box->dxxs2, -view->box->dyys2, -view->box->dzzs2);
      if (lists)
	for (i = 0; lists[i]; i++)
	  OpenGLExtensionCall_list(lists[i], FALSE);
      else
	OpenGLExtensionCall_allLists();
      if (trueTransparency)
	{
	  DBG_fprintf(stderr, "OpenGL: second pass for transparency.\n");
	  glAlphaFunc(GL_LESS, 1.);
	  glDepthMask(0);
	  if (lists)
	    for (i = 0; lists[i]; i++)
	      OpenGLExtensionCall_list(lists[i], FALSE);
	  else
	    OpenGLExtensionCall_allLists();
	  glDepthMask(1);
	  glAlphaFunc(GL_ALWAYS, 1.);
	}
      if (lists)
	for (i = 0; lists[i]; i++)
	  OpenGLExtensionCall_list(lists[i], TRUE);
      else
	OpenGLExtensionCall_lastLists();
      glPopMatrix();  

    }

#if DEBUG == 1
  glFlush();
  g_timer_stop(timer);
  fprintf(stderr, "OpenGL: lists drawn in %g micro-s.\n", g_timer_elapsed(timer, &fractionTimer)/1e-6);
  g_timer_destroy(timer);
#endif
}

void openGL_drawToEmpty(const char **lists _U_, VisuData *data _U_)
{
  gpointer image;
  int widthImg, heightImg;
  guchar *imageData;
  gboolean alpha;
  int x, y;
  float rgb[3];
  int viewport[4];

  DBG_fprintf(stderr, "Opengl: Rebuild default background.\n");
  glPushAttrib(GL_ENABLE_BIT);
  visuRenderingWindowGet_backgroundColor(visuRenderingWindowGet_current(), rgb);
  DBG_fprintf(stderr, "Opengl: set background color to (%f,%f,%f).\n",
	      rgb[0], rgb[1], rgb[2]);
  glClearColor(rgb[0], rgb[1], rgb[2], 0.25f);
  glDrawBuffer(GL_BACK);
  glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
  image = visuRenderingWindowGet_backgroundImage(visuRenderingWindowGet_current(),
						 &imageData, &alpha,
						 &widthImg, &heightImg);
  if (image)
    {
      glDisable(GL_FOG);
      glEnable(GL_BLEND);
      glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

      /* We can use fixed value for widthImg and heightImg since we project then
	 in a 2D ortho with these values. */
      glGetIntegerv(GL_VIEWPORT, viewport);
      DBG_fprintf(stderr, "Opengl: Get viewport %dx%d.\n", viewport[2], viewport[3]);
      x = (viewport[2] - widthImg) / 2;
      y = (viewport[3] + heightImg) / 2;
      DBG_fprintf(stderr, "Opengl: draw at %dx%d.\n", x, y);
      glMatrixMode(GL_PROJECTION);
      glPushMatrix();
      glLoadIdentity();
      gluOrtho2D(0.0, (float)viewport[2], 0., (float)viewport[3]);
   
      glMatrixMode(GL_MODELVIEW);
      glPushMatrix();
      glLoadIdentity();

      glRasterPos2i(x, y);

      glPixelZoom(1., -1.);
      if (alpha)
	glDrawPixels(widthImg, heightImg, GL_RGBA, GL_UNSIGNED_BYTE, imageData);
      else
	glDrawPixels(widthImg, heightImg, GL_RGB, GL_UNSIGNED_BYTE, imageData);
      glPixelZoom(1., 1.);

      glPopMatrix();
      glMatrixMode(GL_PROJECTION);
      glPopMatrix();
      glMatrixMode(GL_MODELVIEW);
    }
  glPopAttrib();
} 

/******************************************************************************/

/* To set the antialiasing on lines. */
void glSetAntiAlias()
{
  if (antialias)
    {
      /* Set blend if not present */
      glEnable(GL_BLEND);
      glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

      /* Antialias */
      glEnable(GL_LINE_SMOOTH);
      glHint(GL_LINE_SMOOTH_HINT, GL_DONT_CARE);
    }
  else
    {
      glDisable(GL_LINE_SMOOTH);
/*       glDisable(GL_BLEND); */
    }
}
int setAntialias(int value)
{
  if (value == antialias)
    return 0;

  antialias = value;
  glSetAntiAlias();
  
  return 1;
}
/* Get the value of the antialiasing parameter. */
int getAntialias()
{
  return antialias;
}

gboolean openGLSet_trueTransparency(gboolean status)
{
  if (status == trueTransparency)
    return FALSE;

  trueTransparency = status;  
  return TRUE;
}
gboolean openGLGet_trueTransparency()
{
  return trueTransparency;
}

/* If true all changes are applied only when the refresh button
   is pushed. */
void openGLSet_immediate(int bool)
{
  if (bool == opengl_immediate)
    return;

  opengl_immediate = bool;
}
int openGLGet_immediate()
{
  return opengl_immediate;
}
gboolean openGLSet_stereoAngle(float angle)
{
  g_return_val_if_fail(angle > 0.f, FALSE);

  if (stereoAngles[0] == angle)
    return FALSE;

  stereoAngles[0] = angle;
  stereoAngles[1] = -angle;

  return stereoStatus;
}
float openGLGet_stereoAngle()
{
  return stereoAngles[0];
}
gboolean openGLGet_stereoCapability()
{
  GLboolean glStereo;

  glGetBooleanv(GL_STEREO, &glStereo);

  return (gboolean)glStereo;
}
void openGLSet_stereo(gboolean status)
{
  if (stereoStatus == status)
    return;
  stereoStatus = status;
}
gboolean openGLGet_stereo()
{
  return stereoStatus;
}


/***************************/
/* Dealing with parameters */
/***************************/

static gboolean readOpenGLAntialias(gchar **lines, int nbLines, int position,
				    VisuData *dataObj _U_, GError **error)
{
  gboolean val;
  
  g_return_val_if_fail(nbLines == 1, FALSE);

  if (!configFileRead_boolean(lines[0], position, &val, 1, error))
    return FALSE;
  setAntialias((int)val);

  return TRUE;
}

static gboolean readOpenGLImmediate(gchar **lines, int nbLines, int position,
				    VisuData *dataObj _U_, GError **error)
{
  gboolean val;
  
  g_return_val_if_fail(nbLines == 1, FALSE);

  if (!configFileRead_boolean(lines[0], position, &val, 1, error))
    return FALSE;
  openGLSet_immediate((int)val);

  return TRUE;
}
static gboolean readOpenGLTransparency(gchar **lines, int nbLines, int position,
				       VisuData *dataObj _U_, GError **error)
{
  gboolean val;
  
  g_return_val_if_fail(nbLines == 1, FALSE);

  if (!configFileRead_boolean(lines[0], position, &val, 1, error))
    return FALSE;
  openGLSet_trueTransparency(val);

  return TRUE;
}
static gboolean readOpenGLStereo(gchar **lines, int nbLines, int position,
				 VisuData *dataObj _U_, GError **error)
{
  gboolean val;
  
  g_return_val_if_fail(nbLines == 1, FALSE);

  if (!configFileRead_boolean(lines[0], position, &val, 1, error))
    return FALSE;
  openGLSet_stereo(val);

  return TRUE;
}
static gboolean readOpenGLStereoAngle(gchar **lines, int nbLines, int position,
				      VisuData *dataObj _U_, GError **error)
{
  float val;
  
  g_return_val_if_fail(nbLines == 1, FALSE);

  if (!configFileRead_float(lines[0], position, &val, 1, error))
    return FALSE;
  if (val <= 0.f)
    {
      *error = g_error_new(CONFIG_FILE_ERROR, CONFIG_FILE_ERROR_VALUE,
			   _("Parse error at line %d: angle must be positive.\n"),
			   position);
      return FALSE;
    }
  openGLSet_stereoAngle(val);

  return TRUE;
}
static void exportParametersOpenGL(GString *data, VisuData *dataObj _U_)
{
  g_string_append_printf(data, "# %s\n", DESC_PARAMETER_OPENGL_ANTIALIAS);
  g_string_append_printf(data, "%s: %d\n\n", FLAG_PARAMETER_OPENGL_ANTIALIAS,
	  antialias);
  g_string_append_printf(data, "# %s\n", DESC_PARAMETER_OPENGL_IMMEDIATE);
  g_string_append_printf(data, "%s: %d\n\n", FLAG_PARAMETER_OPENGL_IMMEDIATE,
	  opengl_immediate);
  g_string_append_printf(data, "# %s\n", DESC_PARAMETER_OPENGL_TRANSS);
  g_string_append_printf(data, "%s: %d\n\n", FLAG_PARAMETER_OPENGL_TRANS,
	  trueTransparency);
  g_string_append_printf(data, "# %s\n", DESC_PARAMETER_OPENGL_STEREO);
  g_string_append_printf(data, "%s: %d\n\n", FLAG_PARAMETER_OPENGL_STEREO,
	  stereoStatus);
  g_string_append_printf(data, "# %s\n", DESC_PARAMETER_OPENGL_ANGLE);
  g_string_append_printf(data, "%s: %f\n\n", FLAG_PARAMETER_OPENGL_ANGLE,
	  stereoAngles[0]);
}
