/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD, Damien
	CALISTE, Olivier D'Astier, laboratoire L_Sim, (2001-2005)
  
	Adresses ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.
	D'ASTIER, dastier AT iie P cnam P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD and Damien
	CALISTE and Olivier D'Astier, laboratoire L_Sim, (2001-2005)

	E-mail addresses :
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.
	D'ASTIER, dastier AT iie P cnam P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#ifndef RENDERINGSPIN_H
#define RENDERINGSPIN_H

#include <visu_rendering.h>
#include <visu_data.h>
#include <visu_object.h>
#include <visu_tools.h>

/**
 * SpinDrawingPolicy:
 * @policyAlwaysSpin: Arrows are drawn whatever the modulus value.
 * @policyHideNullSpin: Spin with a null modulus are hidden.
 * @policyAtomicNullSpin: Follow atomic rendering for null modulus.
 * @policyNbModes: a flag to count the number of modes.
 *
 * Different policy to render the spin when the modulus is null. This
 * policy is applied for all #VisuElement.
 */
typedef enum
  {
    policyAlwaysSpin,    
    policyHideNullSpin,  
    policyAtomicNullSpin,
    policyNbModes
  } SpinDrawingPolicy;
/**
 * SpinModulusPolicy:
 * @policyNoneModulus: arrows have all the same size, whatever the
 * modulus value.
 * @policyPerTypeModulus: arrows are scaled per node type.
 * @policyGlobalModulus: arrows are scaled globaly.
 * @policyNbModulusModes: a flag to count the number of modes.
 *
 * Different policy to render the spin depending on the modulus.
 */
typedef enum
  {
    policyNoneModulus,    
    policyPerTypeModulus,  
    policyGlobalModulus,
    policyNbModulusModes
  } SpinModulusPolicy;

/* An identifier for the different shapes to draw elements. */
typedef enum
  {
    arrow_smooth,
    arrow_sharp,
    arrow_ellipsoid,
    arrow_torus,
    arrow_shape_nb
  } SpinArrowShape;

/**
 * initSpin:
 *
 * Create the whole method and initialise its values. 
 */
void rspinInit();

/**
 * These are keys for the storing of spin files in a VisuData object. 
 */
enum
  {
    FILE_KIND_POSITION,
    FILE_KIND_SPIN
  };

/**
 * SPINVALUES_ID:
 *
 * This flag should be used when creating a new spin rendering method
 * as the flag for the spin values as node properties (see
 * visuNodeNew_pointerProperty()).
 */
#define SPINVALUES_ID     "spinRendering_values"
/**
 * SPINMAXMODULUS_ID:
 *
 * This flag should be used when creating a new spin rendering method
 * as the flag for the max modulus values as a #VisuData property (see
 * visuDataSet_property()).
 */
#define SPINMAXMODULUS_ID "spinRendering_maxModulus"
/**
 * SPIN_THETA:
 *
 * The offset to read theta in a float[3] array.
 */
#define SPIN_THETA   0
/**
 * SPIN_PHI:
 *
 * The offset to read phi in a float[3] array.
 */
#define SPIN_PHI     1
/**
 * SPIN_MODULUS:
 *
 * The offset to read the modulus in a float[3] array.
 */
#define SPIN_MODULUS 2

/**
 * rspin_load:
 * @data: a #VisuData object ;
 * @format: a pointer on a format (can be NULL if format is to be guessed) ;
 * @nSet: an integer ;
 * @error: a pointer to store a possible error.
 *
 * Try to load the spin file declared in the @data. It tries all formats added with
 * rspin_addLoadMethod().
 *
 * Returns: FALSE if the file can't be loaded.
 */
gboolean rspin_load(VisuData *data, FileFormat *format, int nSet, GError **error);

/**
 * rspin_shape_number_to_translated_name:
 * @n: an id for spin shape.
 *
 * This routine returnes the translated name in UTF-8 corresponding to
 * the given shape id.
 *
 * Returns: a string owned by V_Sim.
 */
const char* rspin_shape_number_to_translated_name(SpinArrowShape n);
/**
 * rspin_hiding_name_to_number:
 * @name: a string.
 *
 * In the config file, the hiding policy resource is stored with its name (untranslated).
 * This method is used to retrieve the id from the name.
 *
 * Returns: -1 if the name is invalid.
 */
int rspin_hiding_name_to_number(const char *name);
/**
 * rspin_hiding_number_to_name:
 * @n: an id for hiding policy.
 *
 * Transform ids to untranslated names.
 *
 * Returns: the name associated to the id.
 */
const char* rspin_hiding_number_to_name(int n);
/**
 * rspin_hiding_number_to_translated_name:
 * @n: an id for hiding policy.
 *
 * Transform ids to translated names.
 *
 * Returns: the name associated to the id in UTF-8.
 */
const char* rspin_hiding_number_to_translated_name(int n);

/**
 * returnOneF:
 *
 * Return value: the incredible value of 1.0f
 */
float returnOneF();

/**
 * SpinElementResources:
 * @spin_elementHatLength: the length of the pointing element ;
 * @spin_elementTailLength: the length of the tail ;
 * @spin_elementHatRadius: the raidus of the pointing element ;
 * @spin_elementTailRadius: the radius of the tail ;
 * @spin_elementHatColor: if TRUE, the pointing part use the color of the element ;
 * @spin_elementTailColor: if TRUE, the tail uses the color of the element ;
 * @spin_elementAAxis: the size of the A axis (elipsoid shape) ;
 * @spin_elementBAxis: the size of the B axis (elipsoid shape) ;
 * @spin_elementElipsoidColor: if TRUE, the elipsoid uses the color of the element ;
 * @spin_elementShape: an id to defined the shape (rounded arrow, elipsoid...) ;
 * @spin_nbElementResources: number of resources per element.
 * 
 * These are resources defined for each element. They can be accessed with
 * rspin_getElementResource() or rspin_getElementResource_boolean() and other
 * methods of the same kind.
 */
typedef enum
  {
    spin_elementHatLength,
    spin_elementTailLength,
    spin_elementHatRadius,
    spin_elementTailRadius,
    spin_elementHatColor,
    spin_elementTailColor,
    spin_elementAAxis,
    spin_elementBAxis,
    spin_elementElipsoidColor,
    spin_elementShape,
    spin_nbElementResources
  } SpinElementResources;

/**
 * SpinGlobalResources:
 * @spin_globalConeTheta: the theta angle to set the color cone ;
 * @spin_globalConePhi: the phi angle to set the color cone ;
 * @spin_globalColorWheel: an angle to set the color origin around the z axis of the cone ;
 * @spin_globalHidingMode: an id used to define the policy used to draw spin
 *                         with null modulus (see #SpinDrawingPolicy) ;
 * @spin_globalAtomic: if TRUE, atoms are drawn with spins ;
 * @spin_globalModulus: an id used to define the policy used to draw spin
 *                      depending on modulus (see #SpinModulusPolicy) ;
 * @spin_nbGlobalResources: number of global resources.
 *
 * These are resources that impact the entire rendering spin method. They can be accessed
 * through rspin_getGlobalResource() or rspin_setGlobalResource_boolean() and other
 * similar methods.
 */
typedef enum
  {
    spin_globalConeTheta,
    spin_globalConePhi,
    spin_globalColorWheel,
    spin_globalHidingMode,
    spin_globalAtomic,
    spin_globalModulus,
    spin_nbGlobalResources
  } SpinGlobalResources;


/** 
 * rspin_getGlobalResource:
 * @property: the name of the resource ;
 * @type: a location to store the type.
 *
 * This is a generic method to access global resources. Use rspin_setGlobalResource_boolean()
 * is favored if the type of the value is known. If the returned value is not NULL
 * the argument @type contains the type of the returned value.
 *
 * Return value: a pointer to the location where the value for the given global 
 * resource is stored.
 */
gpointer rspin_getGlobalResource(SpinGlobalResources property, GType *type);
/**
 * rspin_setGlobalResource_boolean:
 * @property: the id of the property to set ;
 * @value: its value.
 *
 * This method is used to change global resources that are boolean.
 *
 * Returns: TRUE if the value was changed.
 */
gboolean rspin_setGlobalResource_boolean(SpinGlobalResources property,
					 gboolean value);
/**
 * rspin_setGlobalResource_uint:
 * @property: the id of the property to set ;
 * @value: its value.
 *
 * This method is used to change global resources that are unsigned int.
 *
 * Returns: TRUE if the value was changed.
 */
gboolean rspin_setGlobalResource_uint(SpinGlobalResources property,
				      guint value);
/**
 * rspin_setGlobalResource_float:
 * @property: the id of the property to set ;
 * @value: its value.
 *
 * This method is used to change global resources that are floating point.
 *
 * Returns: TRUE if the value was changed.
 */
gboolean rspin_setGlobalResource_float(SpinGlobalResources property,
				       gfloat value);
/**
 * rspin_getGlobalResource_boolean:
 * @property: the id of the property to get.
 *
 * This is the specific method to retrieve value of boolean global resources.
 *
 * Returns: the boolean value.
 */
gboolean rspin_getGlobalResource_boolean(SpinGlobalResources property);
/**
 * rspin_getGlobalResource_uint:
 * @property: the id of the property to get.
 *
 * This is the specific method to retrieve value of unsigned int global resources.
 *
 * Returns: the unsigned int value.
 */
guint rspin_getGlobalResource_uint(SpinGlobalResources property);
/**
 * rspin_getGlobalResource_float:
 * @property: the id of the property to get.
 *
 * This is the specific method to retrieve value of floating point global resources.
 *
 * Returns: the floating point value.
 */
gfloat rspin_getGlobalResource_float(SpinGlobalResources property);




/**
 * rspin_getElementResource:
 * @ele: a pointer to a #VisuElement object ;
 * @property: the id of the resource (see #SpinElementResources) ;
 * @type: a location to store the type.
 *
 * This is a generic method to access resources per element. Use rspin_setElementResource_boolean()
 * is favored if the type of the value is known (boolean in this exemple).
 * If the returned value is not NULL the argument @type contains the type of the returned value.
 *
 * Return value: a pointer to the location where the value for the given global 
 * resource is stored.
 */
gpointer rspin_getElementResource(VisuElement *ele, SpinElementResources property, GType *type);
/**
 * rspin_setElementResource_boolean:
 * @ele: a pointer to a #VisuElement object ;
 * @property: the id of the property to set ;
 * @value: its value.
 *
 * This method is used to change element resources that are boolean.
 *
 * Returns: TRUE if the value was changed.
 */
gboolean rspin_setElementResource_boolean(VisuElement *ele, SpinElementResources property,
					  gboolean value);
/**
 * rspin_setElementResource_uint:
 * @ele: a pointer to a #VisuElement object ;
 * @property: the id of the property to set ;
 * @value: its value.
 *
 * This method is used to change element resources that are unsigned int.
 *
 * Returns: TRUE if the value was changed.
 */
gboolean rspin_setElementResource_uint(VisuElement *ele, SpinElementResources property,
				       guint value);
/**
 * rspin_setElementResource_float:
 * @ele: a pointer to a #VisuElement object ;
 * @property: the id of the property to set ;
 * @value: its value.
 *
 * This method is used to change element resources that are floating point.
 *
 * Returns: TRUE if the value was changed.
 */
gboolean rspin_setElementResource_float(VisuElement *ele, SpinElementResources property,
					gfloat value);
/**
 * rspin_getElementResource_boolean:
 * @ele: a pointer to a #VisuElement object ;
 * @property: the id of the property to get.
 *
 * This is the specific method to retrieve value of boolean element resources.
 *
 * Returns: the boolean value.
 */
gboolean rspin_getElementResource_boolean(VisuElement *ele, SpinElementResources property);
/**
 * rspin_getElementResource_uint:
 * @ele: a pointer to a #VisuElement object ;
 * @property: the id of the property to get.
 *
 * This is the specific method to retrieve value of unsigned int element resources.
 *
 * Returns: the unsigned int value.
 */
guint rspin_getElementResource_uint(VisuElement *ele, SpinElementResources property);
/**
 * rspin_getElementResource_float:
 * @ele: a pointer to a #VisuElement object ;
 * @property: the id of the property to get.
 *
 * This is the specific method to retrieve value of floating point element resources.
 *
 * Returns: the floating point value.
 */
gfloat rspin_getElementResource_float(VisuElement *ele, SpinElementResources property);


/**
 * rspin_createShapeSpin:
 * @visuData: a #VisuData object ;
 * @ele: a visu element.
 *
 * Creates the shape to be drawed by opengl for the @ele VisuElement.
 * Keep in mind that it does NOT redraw the scene. You must explicitly
 * send a redraw signal for that task to be done.
 *
 * Returns: an id representing an OpenGL list where the shape has been
 *          created.
 */
int rspin_createShapeSpin(VisuData *visuData, VisuElement* ele);

/**
 * rspin_placeNodeSpin
 * @visuData: a #VisuData object ;
 * @node: a visu node ;
 * @ele: the element matching the @node.
 *
 * Apply a particular drawing treatment (coloration) on the given @node 
 * according to its orientation and current set module parameters.
 */
void rspin_placeNodeSpin(VisuData *visuData, VisuNode *node, VisuElement* ele);

/**
 * rspin_addLoadMethod:
 * @meth: a new load method.
 *
 * This routine is used to register a new load method with its description
 * (file formats, name...). The list of all known load methods is automatically
 * resorted after a call to this method to reflect the priorities.
 */
void rspin_addLoadMethod(RenderingFormatLoad* meth);

#define VISU_RENDERING_SPIN "Spin visualisation"

#endif
