/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#include "visu_pickMesure.h"

#include <GL/gl.h>
#include <GL/glu.h> 

#include "opengl.h"
#include "visu_tools.h"
#include "visu_data.h"
#include "visu_object.h"
#include "visu_extension.h"
#include "visu_rendering.h"
#include "openGLFunctions/text.h"
#include "openGLFunctions/objectList.h"
#include "openGLFunctions/interactive.h"
#include "coreTools/toolColor.h"
#include "renderingBackend/visu_windowInterface.h"

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <glib.h>

/**
 * SECTION:visu_pickMesure
 * @short_description: Some methods to treat a #VisuPick informations.
 *
 * <para>The pick mesures work on static variables, thus only one pick
 * session is possible at a time. Each time a node is selected,
 * marksAndMesures() should be called. To retrieve then the results of
 * the measurements, getPickMesureInfos() and getPickMesureErrors()
 * are available.</para>
 */

int openGlListMarksId;

#define PICK_MESURE_MARK_BIG_SQUARE_SIZE 8
#define PICK_MESURE_MARK_SMALL_SQUARE_SIZE 4
guchar pickMesureMark_bigSquare[PICK_MESURE_MARK_BIG_SQUARE_SIZE *
				PICK_MESURE_MARK_BIG_SQUARE_SIZE * 4];
guchar pickMesureMark_smallSquare[PICK_MESURE_MARK_SMALL_SQUARE_SIZE *
				  PICK_MESURE_MARK_SMALL_SQUARE_SIZE * 4];

typedef enum
  {
    PICK_MESURE_MARK_BIG_SQUARE,
    PICK_MESURE_MARK_SMALL_SQUARE,
    PICK_MESURE_MARK_HIGHLIGHT,
    PICK_MESURE_MARK_DISTANCE,
    PICK_MESURE_MARK_ANGLE,
    PICK_MESURE_MARK_LINE
  } PickMesureMarkType;

struct MarkInfo_struct
{
  /* Mark type. */
  PickMesureMarkType type;

  /* Id used to address the VisuNode when the
     pointer node1 has changed, for example when a new VisuData
     is loaded with the same geometry and we want to apply this mark. */
  guint idNode1;
  /* Idem for a second node. */
  guint idNode2;
  /* Idem for a third node. */
  guint idNode3;

  /* List of nodes */
  GList *nodes;
  unsigned int size;
  GLfloat *coord;
  unsigned int *nIds;

  /* Text not used. */
  gchar *text;
};
/**
 * _PickMesure:
 * @selected: a #VisuNode corresponding to the normal selected node (if set) ;
 * @ref1: a #VisuNode corresponding to reference 1 (if set) ;
 * @ref2: a #VisuNode corresponding to reference 2 (if set) ;
 * @info: a string containing the formated pick informations ;
 * @error: a string containing an error message (if any) ;
 * @newValuesAvailable: a flag to set if the pick action has canged anything ;
 * @formatFlag: if set, the errors and info GString are used and strings
 *              output for the pick action are created.
 *
 * This structure reflects the state of a current pick session used for
 * distance and angles measurements.
 */
struct _PickMesure
{
  VisuData *data;

  gint idSelected;
  gint idRef1;
  gint idRef2;
  GList *idRegion;
  gint idHighlighted;

  gboolean formatFlag;
  GString *info;
  GString *errors;

  gboolean newValuesAvailable;
  PickMesureType typeOfNews;

  float drag[3];

  /* A list of MarkInfo_struct elements. */
  GList *storedMarks;
  gboolean storeMeasures;
};

typedef struct color_struct {float rgba[4];} Color_;
#define N_BRIGHT_COLORS 20
static Color_ brightColors[N_BRIGHT_COLORS] =
  {{{1,0,0, 1.f}},
   {{0,1,0, 1.f}},
   {{0,0,1, 1.f}},
   {{1,0,1, 1.f}},
   {{0,1,1, 1.f}},
   {{1,1,0, 1.f}},
   {{1,0.5,0, 1.f}},
   {{0.5,0,0.5, 1.f}},
   {{0,0.5,0.5, 1.f}},
   {{0.5,0.5,0, 1.f}},
   {{0.5,0,0, 1.f}},
   {{0,0.5,0, 1.f}},
   {{0,0,0.5, 1.f}},
   {{0.5,0.5,0.5, 1.f}},
   {{1,0,0.5, 1.f}},
   {{0.5,1,0, 1.f}},
   {{0.5,0,1, 1.f}},
   {{0,1,0.5, 1.f}},
   {{0,0.5,1, 1.f}},
   {{0,0,0, 1.f}}};

/* Local methods. */
static void pickMesureFree(gpointer data);
static void pickMesureRebuild_colourInvList(VisuData *dataObj);

/**************************/
/* Local drawing methods. */
/**************************/
/* OpenGL routines. */
static void drawMarkList(VisuData *data, GList *list, int listType);
static void drawMarkDot(VisuData *data, VisuNode *node, PickMesureMarkType type);
static void putMarkDot(VisuData *data, guint nodeId, PickMesureMarkType type);
static void drawMarkDistance(VisuData *data, VisuNode *nodeRef,
			     VisuNode *node, PickMesureMarkType type);
static void putMarkDistance(VisuData *data, guint nodeRefId,
			    guint nodeId, PickMesureMarkType type);
static void drawMarkAngle(VisuData *data, VisuNode *nodeRef,
			  VisuNode *nodeRef2, VisuNode *node, guint id);
static void drawMarkLine(VisuData *data, GLfloat *coord,
			 unsigned int size, PickMesureMarkType type);
/* static void putMarkLine(VisuData *data, GLfloat *coord, unsigned int size, PickMesureMarkType type); */

/* Manage mark lists. */
static struct MarkInfo_struct* markNew(PickMesureMarkType type);
static void markFree(struct MarkInfo_struct *mark);
static struct MarkInfo_struct* markCopy(struct MarkInfo_struct* mark);

typedef gboolean (*MatchFunc)(struct MarkInfo_struct *markInfo,
			      guint nodeId, gpointer data);
static struct MarkInfo_struct* addMarkDotToList(PickMesure *mesureData, guint idNode,
						PickMesureMarkType type);
static void toggleMarkDistanceInList(PickMesure *mesureData, guint nodeRefId,
				     guint nodeId, gboolean set);
/* static struct MarkInfo_struct* updateMarkLineInList(PickMesure *mesureData, GList *region, */
/* 					PickMesureMarkType type); */
/*static void updateMarkLineInList1(PickMesure *mesureData, unsigned int *region,
					PickMesureMarkType type);*/
static GList* lookupMarkInList(PickMesure *mesureData, MatchFunc match,
			       guint nodeId, gpointer data);
static void removeMark(PickMesure *mesureData, GList *list);
static void removeMarkFromList(PickMesure *mesureData, MatchFunc match,
			       guint nodeId, gpointer data);


/* Local callbacks. */
static void createPickMesureOnNewData(GObject *visu, VisuData *dataObj,
				      gpointer data);
static void updateListOnNodeChange(VisuData *dataObj, gpointer data);
static void updateListOnElementChange(VisuData *dataObj, VisuElement *ele,
				      gpointer data);
static void updateListOnPopulationChange(VisuData *dataObj, int *nodes,
					 gpointer data);
static void updateListOnCameraChange(VisuData *obj, OpenGLView *view, gpointer data);


void initPick_module()
{
  OpenGLExtension *ext1, *ext2;
  int i;

  /* Get an id for the OpenGL list used to draw marks. */
  openGlListMarksId = openGLObjectList_new(2);
  /* Create the marks. */
  for (i = 0; i < PICK_MESURE_MARK_BIG_SQUARE_SIZE *
	 PICK_MESURE_MARK_BIG_SQUARE_SIZE * 4 ; i++)
    pickMesureMark_bigSquare[i] = 0xff;
  for (i = 0; i < PICK_MESURE_MARK_SMALL_SQUARE_SIZE *
	 PICK_MESURE_MARK_SMALL_SQUARE_SIZE * 4; i++)
    pickMesureMark_smallSquare[i] = 0xff;

  /* We listen to the dataReadyForRendering signal to create a pickMesure
     object and attach it to an object when created. */
  g_signal_connect(VISU_INSTANCE, "dataNew",
		   G_CALLBACK(createPickMesureOnNewData), (gpointer)0);

  /* Initialize an extension to draw the marks. */
  ext1 = OpenGLExtension_new("MarksInv", _("Marks - inverse color"),
			     _("Draw some marks on element in video inverse."),
			     openGlListMarksId, pickMesureRebuild_colourInvList);
  OpenGLExtensionRegister(ext1);
  OpenGLExtensionSet_priority(ext1, OPENGL_EXTENSION_PRIORITY_LAST);
  OpenGLExtensionSet_saveOpenGLState(ext1, TRUE);
  ext1->used = 1;

  ext2 = OpenGLExtension_new("Marks", _("Marks - classical"),
			     _("Draw some marks on element."),
			     openGlListMarksId + 1, pickMesureRebuild_classicalList);
  OpenGLExtensionRegister(ext2);
  OpenGLExtensionSet_priority(ext2, OPENGL_EXTENSION_PRIORITY_LOW);
/*   OpenGLExtensionSet_saveOpenGLState(ext2, TRUE); */
  ext2->used = 1;
}

static void createPickMesureOnNewData(GObject *visu _U_, VisuData *dataObj,
				      gpointer data _U_)
{
  PickMesure *mesureData;

  if (!dataObj)
    return;

  /* By default, we need no extended output but we
     draw measurements on screen. */
  mesureData                     = g_malloc(sizeof(PickMesure));
  mesureData->data               = dataObj;
  mesureData->idRef1             = -99;
  mesureData->idRef2             = -99;
  mesureData->idSelected         = -99;
  mesureData->idHighlighted      = -99;
  mesureData->idRegion           = (GList*)0;
  mesureData->info               = (GString*)0;
  mesureData->errors             = (GString*)0;
  mesureData->newValuesAvailable = FALSE;
  mesureData->formatFlag         = FALSE;
  mesureData->storedMarks        = (GList*)0;
  mesureData->storeMeasures      = TRUE;
  DBG_fprintf(stderr, "Visu PickMesure: create a new object, %p.\n",
	      (gpointer)mesureData);

  if (mesureData->storeMeasures)
    openGLText_initFontList();

  /* Connect a signal on file change to remove everything. */
  g_signal_connect(G_OBJECT(dataObj), "NodePopulationDecrease",
		   G_CALLBACK(updateListOnPopulationChange), (gpointer)mesureData);
  g_signal_connect(G_OBJECT(dataObj), "NodePositionChanged",
		   G_CALLBACK(updateListOnNodeChange), (gpointer)mesureData);
  g_signal_connect(G_OBJECT(dataObj), "NodeRenderedChanged",
		   G_CALLBACK(updateListOnNodeChange), (gpointer)mesureData);
  g_signal_connect(G_OBJECT(dataObj), "ElementRenderedChanged",
		   G_CALLBACK(updateListOnElementChange), (gpointer)mesureData);
  g_signal_connect(G_OBJECT(dataObj), "OpenGLThetaPhiOmega",
		   G_CALLBACK(updateListOnCameraChange), (gpointer)mesureData);
  g_signal_connect(G_OBJECT(dataObj), "OpenGLXsYs",
		   G_CALLBACK(updateListOnCameraChange), (gpointer)mesureData);
  g_signal_connect(G_OBJECT(dataObj), "OpenGLGross",
		   G_CALLBACK(updateListOnCameraChange), (gpointer)mesureData);
  g_signal_connect(G_OBJECT(dataObj), "OpenGLPersp",
		   G_CALLBACK(updateListOnCameraChange), (gpointer)mesureData);
  
  g_object_set_data_full(G_OBJECT(dataObj), "pickMesure_data",
			 (gpointer)mesureData, pickMesureFree);

  /* Empty the OpenGL list. */
  glDeleteLists(openGlListMarksId, 2);
}

static void pickMesureFree(gpointer data)
{
  PickMesure *mesureData;
  GList *list;

  DBG_fprintf(stderr, "Visu PickMesure: freeing object %p.\n", data);
  mesureData = (PickMesure*)data;

  if (mesureData->idRegion)
    g_list_free(mesureData->idRegion);
  if (mesureData->info)
    g_string_free(mesureData->info, TRUE);
  if (mesureData->errors)
    g_string_free(mesureData->errors, TRUE);
  list = mesureData->storedMarks;
  while (list)
    {
      markFree((struct MarkInfo_struct*)list->data);
      list = g_list_next(list);
    }
  g_list_free(mesureData->storedMarks);
  g_free(mesureData);
}

static struct MarkInfo_struct* markNew(PickMesureMarkType type)
{
  struct MarkInfo_struct *mark;

  mark          = g_malloc(sizeof(struct MarkInfo_struct));
  mark->type    = type;
  mark->idNode1 = -1;
  mark->idNode2 = -1;
  mark->idNode3 = -1;

  mark->nodes   = (GList*)0;
  mark->size    = 0;
  mark->coord   = (GLfloat*)0;
  mark->nIds    = (unsigned int*)0;

  mark->text    = (gchar*)0;

  return (struct MarkInfo_struct*)mark;
}
static void markFree(struct MarkInfo_struct *mark)
{
  if (mark->nodes)
    g_list_free(mark->nodes);

  if (mark->coord)
    g_free(mark->coord);
  if (mark->nIds)
    g_free(mark->nIds);

  g_free(mark);
}
static struct MarkInfo_struct* markCopy(struct MarkInfo_struct* mark)
{
  struct MarkInfo_struct *newMark;

  g_return_val_if_fail(mark, (struct MarkInfo_struct*)0);

  newMark = markNew(mark->type);
  newMark->idNode1 = mark->idNode1;
  newMark->idNode2 = mark->idNode2;
  newMark->idNode3 = mark->idNode3;

  /* TODO, copy the list information. */

  return newMark;
}


/****************************/
/* OpenGL drawing routines. */
/****************************/
/* This method recreate the opengl list associated to marks with
   all the marks given as argument. */
static void drawMarkList(VisuData *data, GList *list, int listType)
{
  struct MarkInfo_struct *mark;
  VisuNode *node1;
  GList *tmpLst;
  guint id;

  DBG_fprintf(stderr, "Visu pickMesure: update the list %p"
	      " of all marks.\n", (gpointer)list);
  if (listType == 0 || listType < 0)
    glDeleteLists(openGlListMarksId, 1);
  if (listType == 1 || listType < 0)
    glDeleteLists(openGlListMarksId + 1, 1);
  if (!list)
    return;
  if (listType == 0 || listType < 0)
    {
      id = 0;
      /* Draw the colour inverse list. */
      glNewList(openGlListMarksId, GL_COMPILE);
      glEnable(GL_DEPTH_TEST);
      glClear(GL_DEPTH_BUFFER_BIT);
/*       glDisable(GL_DEPTH_TEST); */
      glDisable(GL_LIGHTING);
      glDisable(GL_FOG);
      glDisable(GL_CULL_FACE);
      /*   glDisable(GL_DITHER); */
      glEnable(GL_BLEND);
      glBlendFunc(GL_ONE_MINUS_DST_COLOR, GL_ONE_MINUS_SRC_COLOR);
      for (tmpLst = list; tmpLst; tmpLst = g_list_next(tmpLst))
	{
	  mark = (struct MarkInfo_struct*)tmpLst->data;
	  DBG_fprintf(stderr, " | draw mark of type %d.\n", mark->type);
	  node1 = visuDataGet_nodeFromNumber(data, mark->idNode1);
	  switch (mark->type)
	    {
	    case PICK_MESURE_MARK_BIG_SQUARE:
	    case PICK_MESURE_MARK_SMALL_SQUARE:
	      drawMarkDot(data, node1, mark->type);
	      break;
	    case PICK_MESURE_MARK_DISTANCE:
	      drawMarkDistance(data, node1,
			       visuDataGet_nodeFromNumber(data, mark->idNode2),
			       mark->type);
	      break;
	    case PICK_MESURE_MARK_LINE:
	      drawMarkLine(data, mark->coord, mark->size, mark->type);
	      break;
	    case PICK_MESURE_MARK_ANGLE:
	      drawMarkAngle(data, node1,
			    visuDataGet_nodeFromNumber(data, mark->idNode2),
			    visuDataGet_nodeFromNumber(data, mark->idNode3), id);
	      id += 1;
	      break;
	    default:
	      break;
	    }
	}
      glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
      glEndList();
    }
  if (listType == 1 || listType < 0)
    {
      /* Draw the classical list. */
      glNewList(openGlListMarksId + 1, GL_COMPILE);
      glEnable(GL_LIGHTING);
      for (tmpLst = list; tmpLst; tmpLst = g_list_next(tmpLst))
	{
	  mark = (struct MarkInfo_struct*)tmpLst->data;
	  if (mark->type == PICK_MESURE_MARK_HIGHLIGHT)
	    {
	      DBG_fprintf(stderr, " | draw mark of type %d.\n", mark->type);
	      node1 = visuDataGet_nodeFromNumber(data, mark->idNode1);
	      drawMarkDot(data, node1, mark->type);
	    }
	}
      glEndList();
    }

}
static void drawMarkDot(VisuData *data, VisuNode *node, PickMesureMarkType type)
{
  float xyz[3], material[5] = {1.f, 1.f, 1.f, 0.f, 0.f};
  VisuElement *ele;
  int nlat;
  float eleSize;
  VisuRendering *method;
  GLUquadricObj *obj;

  g_return_if_fail(node);

  /* If the node is masked, then we don't draw the dot. */
  if (!node->rendered)
    return;
  /* If one of the element is masked, then we don't draw the distance. */
  ele = data->fromIntToVisuElement[node->posElement];
  if (!ele->rendered)
    return;

  DBG_fprintf(stderr, "Visu PickMesure: draw a mark (%d) on node %d.\n",
	      (int)type, node->number);

  visuDataGet_nodePosition(data, node, xyz);
  switch (type)
    {
    case PICK_MESURE_MARK_BIG_SQUARE:
      glRasterPos3f(xyz[0], xyz[1], xyz[2]);
      glDrawPixels(PICK_MESURE_MARK_BIG_SQUARE_SIZE,
		   PICK_MESURE_MARK_BIG_SQUARE_SIZE,
		   GL_RGBA, GL_UNSIGNED_BYTE, pickMesureMark_bigSquare);
      break;
    case PICK_MESURE_MARK_SMALL_SQUARE:
      glRasterPos3f(xyz[0], xyz[1], xyz[2]);
      glDrawPixels(PICK_MESURE_MARK_SMALL_SQUARE_SIZE,
		   PICK_MESURE_MARK_SMALL_SQUARE_SIZE,
		   GL_RGBA, GL_UNSIGNED_BYTE, pickMesureMark_smallSquare);
      break;
    case PICK_MESURE_MARK_HIGHLIGHT:
      obj = gluNewQuadric();
      method = visuRenderingClassGet_current();
      eleSize = visuRenderingGet_sizeOfElement(method, ele);
      openGLSet_highlightColor(material, ele->rgb, 0.5f);
      nlat = OpenGLViewGet_numberOfFacettes(visuDataGet_openGLView(data),
					    eleSize * 1.25);
      glPushMatrix();
      glTranslated(xyz[0], xyz[1], xyz[2]);
      gluSphere(obj, (double)eleSize * 1.25, 2 * nlat, 2 * nlat);
      glPopMatrix();
      gluDeleteQuadric(obj);
      break;
    default:
      break;
    }
}
/* Directly draw the dot on the front buffer. */
static void putMarkDot(VisuData *data, guint nodeId, PickMesureMarkType type)
{
  VisuElement *ele;
  VisuNode *node;
  OpenGLView *view;

  node = visuDataGet_nodeFromNumber(data, nodeId);
  if (!node)
    return;
  /* If the node is masked, then we don't draw the dot. */
  if (!node->rendered)
    return;
  /* If one of the element is masked, then we don't draw the distance. */
  ele = data->fromIntToVisuElement[node->posElement];
  if (!ele->rendered)
    return;

  view = visuDataGet_openGLView(data);

  DBG_fprintf(stderr, "Visu pickMesure: draw directly a mark on front buffer.\n");
  glDrawBuffer(GL_FRONT);

  glPushAttrib(GL_ENABLE_BIT);
  glDisable(GL_DEPTH_TEST);
  glDisable(GL_LIGHTING);
  glDisable(GL_FOG);
  glEnable(GL_BLEND);
  glBlendFunc(GL_ONE_MINUS_DST_COLOR, GL_ONE_MINUS_SRC_COLOR);
  glPushMatrix();
  glTranslated(-view->box->dxxs2, -view->box->dyys2, -view->box->dzzs2);
  drawMarkDot(data, node, type);
  glPopMatrix();
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
  glPopAttrib();

  glDrawBuffer(GL_BACK);

  glFlush();
}
static void drawMarkDistance(VisuData *data, VisuNode *nodeRef,
			     VisuNode *node, PickMesureMarkType type _U_)
{
  float xyzRef[3], xyz[3], dist;
  int i;
  char distStr[8];
  VisuElement *ele;

  g_return_if_fail(node && nodeRef);

  /* If one of the node is masked, then we don't draw the distance. */
  if (!nodeRef->rendered || !node->rendered)
    return;

  /* If one of the element is masked, then we don't draw the distance. */
  ele = data->fromIntToVisuElement[nodeRef->posElement];
  if (!ele->rendered)
    return;
  ele = data->fromIntToVisuElement[node->posElement];
  if (!ele->rendered)
    return;

  visuDataGet_nodePosition(data, nodeRef, xyzRef);
  visuDataGet_nodePosition(data, node, xyz);

  dist = 0.;
  for (i = 0; i < 3; i++)
    dist += (xyzRef[i] - xyz[i]) * (xyzRef[i] - xyz[i]);
  sprintf(distStr, "%7.3f", sqrt(dist));
  distStr[7] = '\0';
  DBG_fprintf(stderr, "Visu pickMesure: draw a link with distance %s.\n", distStr);

  glLineWidth(1);
  glColor4f(1., 1., 1., 0.);
  glBegin(GL_LINES);
  glVertex3fv(xyzRef);
  glVertex3fv(xyz);
  glEnd();
  glPointSize(8.);
  glBegin(GL_POINTS);
  glVertex3fv(xyzRef);
  glVertex3fv(xyz);
  glEnd();
  glRasterPos3f((xyzRef[0] + xyz[0])/2., (xyzRef[1] + xyz[1])/2, (xyzRef[2] + xyz[2])/2.); 
  openGLText_drawChars(distStr, TEXT_NORMAL); 
/*   fprintf(stderr, "----> '%s'\n", distStr); */
  
/*   drawMarkDot(data, node, PICK_MESURE_MARK_SMALL_SQUARE); */
}
/* Directly draw the distance on the front buffer. */
static void putMarkDistance(VisuData *data, guint nodeRefId,
			    guint nodeId, PickMesureMarkType type)
{
  VisuElement *ele;
  VisuNode *nodeRef, *node;
  OpenGLView *view;

  node    = visuDataGet_nodeFromNumber(data, nodeId);
  nodeRef = visuDataGet_nodeFromNumber(data, nodeRefId);
  if (!node || !nodeRef)
    return;
  /* If one of the node is masked, then we don't draw the distance. */
  if (!nodeRef->rendered || !node->rendered)
    return;
  /* If one of the element is masked, then we don't draw the distance. */
  ele = data->fromIntToVisuElement[nodeRef->posElement];
  if (!ele->rendered)
    return;
  ele = data->fromIntToVisuElement[node->posElement];
  if (!ele->rendered)
    return;

  view = visuDataGet_openGLView(data);

  DBG_fprintf(stderr, "Visu pickMesure: draw directly a distance on front buffer.\n");
  glDrawBuffer(GL_FRONT);

  glPushAttrib(GL_ENABLE_BIT);
  glDisable(GL_DEPTH_TEST);
  glDisable(GL_LIGHTING);
  glDisable(GL_FOG);
  glEnable(GL_BLEND);
  glBlendFunc(GL_ONE_MINUS_DST_COLOR, GL_ONE_MINUS_SRC_COLOR);
  glPushMatrix();
  glTranslated(-view->box->dxxs2, -view->box->dyys2, -view->box->dzzs2);
  drawMarkDistance(data, nodeRef, node, type);
  glPopMatrix();
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
  glPopAttrib();

  glDrawBuffer(GL_BACK);

  glFlush();
}
static void drawMarkAngle(VisuData *data, VisuNode *nodeRef,
			  VisuNode *nodeRef2, VisuNode *node, guint id)
{
  float xyzRef[3], xyzRef2[3], xyz[3], dist, u[3], v[3], M[3], c, s, theta, thetaM, l;
  int i, iM;
  char distStr[8];
  VisuElement *ele;

  g_return_if_fail(node && nodeRef && nodeRef2);

  /* If one of the node is masked, then we don't draw the angle. */
  if (!nodeRef->rendered || !nodeRef2->rendered || !node->rendered)
    return;

  /* If one of the element is masked, then we don't draw the angle. */
  ele = data->fromIntToVisuElement[nodeRef->posElement];
  if (!ele->rendered)
    return;
  ele = data->fromIntToVisuElement[nodeRef2->posElement];
  if (!ele->rendered)
    return;
  ele = data->fromIntToVisuElement[node->posElement];
  if (!ele->rendered)
    return;

  DBG_fprintf(stderr, "Visu PickMesure: draw an angle mark on nodes %d/%d/%d.\n",
	      nodeRef->number, nodeRef2->number, node->number);

  visuDataGet_nodePosition(data, nodeRef, xyzRef);
  visuDataGet_nodePosition(data, nodeRef2, xyzRef2);
  visuDataGet_nodePosition(data, node, xyz);

  /* We calculate the new basis set. */
  u[0] = xyz[0] - xyzRef[0];
  u[1] = xyz[1] - xyzRef[1];
  u[2] = xyz[2] - xyzRef[2];
  l = sqrt(u[0] * u[0] + u[1] * u[1] + u[2] * u[2]);
  dist = 1.f / l;
  u[0] *= dist;
  u[1] *= dist;
  u[2] *= dist;

  v[0] = xyzRef2[0] - xyzRef[0];
  v[1] = xyzRef2[1] - xyzRef[1];
  v[2] = xyzRef2[2] - xyzRef[2];
  dist = sqrt(v[0] * v[0] + v[1] * v[1] + v[2] * v[2]);
  l = MIN(l, dist);
  dist = 1.f / dist;
  v[0] *= dist;
  v[1] *= dist;
  v[2] *= dist;
  l /= 4.f;

  dist = u[0] * v[0] + u[1] * v[1] + u[2] * v[2];
  thetaM = acos(dist);
  iM     = MAX((int)(thetaM / G_PI * 20.f), 2);
  sprintf(distStr, "%5.1f", thetaM * 180.f / G_PI);
  distStr[7] = '\0';
  DBG_fprintf(stderr, "Visu PickMesure: the angle is %g.\n", thetaM * 180.f / G_PI);

  v[0] -= dist * u[0];
  v[1] -= dist * u[1];
  v[2] -= dist * u[2];
  dist = sqrt(v[0] * v[0] + v[1] * v[1] + v[2] * v[2]);
  if (dist < 1e-3)
    {
      v[0] = 1.f;
      v[1] = 1.f;
      v[2] = 1.f;
      i = 0;
      if (u[0] != 0.f)
	i = 0;
      else if (u[1] != 0.f)
	i = 1;
      else if (u[2] != 0.f)
	i = 2;
      else
	g_warning("Selected node and reference are the same.");
      v[i] = 1.f - (u[0] + u[1] + u[2]) / u[i];
      dist = 1.f / sqrt(v[0] * v[0] + v[1] * v[1] + v[2] * v[2]);
    }
  else
    dist = 1.f / dist;
  v[0] *= dist;
  v[1] *= dist;
  v[2] *= dist;

  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

  glColor4fv(brightColors[id % N_BRIGHT_COLORS].rgba);
  theta = 0.5f * thetaM;
  c = cos(theta);
  s = sin(theta);
  M[0] = xyzRef[0] + (u[0] * c + v[0] * s) * l * 1.33f;
  M[1] = xyzRef[1] + (u[1] * c + v[1] * s) * l * 1.33f;
  M[2] = xyzRef[2] + (u[2] * c + v[2] * s) * l * 1.33f;
  glRasterPos3fv(M);
  openGLText_drawChars(distStr, TEXT_SMALL); 

  glLineWidth(1);
  glBegin(GL_LINES);
  glVertex3fv(xyzRef);
  glVertex3fv(xyzRef2);
  glVertex3fv(xyzRef);
  glVertex3fv(xyz);
  glEnd();

  glEnable(GL_POLYGON_OFFSET_FILL);
  glPolygonOffset(1.0, 1.0);

  glBegin(GL_POLYGON);
  glVertex3fv(xyzRef);
  for (i = 0; i < iM; i++)
    {
      theta = (float)i / ((float)iM - 1.f) * thetaM;
      c = cos(theta);
      s = sin(theta);
      M[0] = xyzRef[0] + (u[0] * c + v[0] * s) * l;
      M[1] = xyzRef[1] + (u[1] * c + v[1] * s) * l;
      M[2] = xyzRef[2] + (u[2] * c + v[2] * s) * l;
      glVertex3fv(M);
    }
  glEnd();

  glPointSize(4.);
  glBegin(GL_POINTS);
  glVertex3fv(xyzRef2);
  glVertex3fv(xyz);
  glEnd();

  glDisable(GL_POLYGON_OFFSET_FILL);
  glColor3f (0.0, 0.0, 0.0);
  glLineWidth(1.1);
  glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
  glBegin(GL_POLYGON);
  glVertex3fv(xyzRef);
  for (i = 0; i < iM; i++)
    {
      theta = (float)i / ((float)iM - 1.f) * thetaM;
      c = cos(theta);
      s = sin(theta);
      M[0] = xyzRef[0] + (u[0] * c + v[0] * s) * l;
      M[1] = xyzRef[1] + (u[1] * c + v[1] * s) * l;
      M[2] = xyzRef[2] + (u[2] * c + v[2] * s) * l;
      glVertex3fv(M);
    }
  glEnd();
  glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);

/*   glColor4f(0., 0., 0., 1.); */
/*   glBegin(GL_LINE_LOOP); */
/*   glVertex3fv(xyzRef); */
/*   for (i = 0; i < iM; i++) */
/*     { */
/*       theta = (float)i / ((float)iM - 1.f) * thetaM; */
/*       c = cos(theta); */
/*       s = sin(theta); */
/*       M[0] = xyzRef[0] + (u[0] * c + v[0] * s) * l; */
/*       M[1] = xyzRef[1] + (u[1] * c + v[1] * s) * l; */
/*       M[2] = xyzRef[2] + (u[2] * c + v[2] * s) * l; */
/*       glVertex3fv(M); */
/*     } */
/*   glEnd(); */

  glBlendFunc(GL_ONE_MINUS_DST_COLOR, GL_ONE_MINUS_SRC_COLOR);
}

static void drawMarkLine(VisuData *data _U_, GLfloat *coord, unsigned int size, PickMesureMarkType type _U_)
{
  #define NEWLINE 1
  #define NEWNODE 2
  int i = 0, j;
  int elem = 0;
  int nNodes, nTotalVert;
  /***********************************/
  /* values must be selected by user */
  int nVert = 30;
  double radius = 30;
  /***********************************/
  double *vertices, *nodeVertices, *verticesKept;
  double link[2] = {0};
  double distance;
  gboolean visible = TRUE;
  gboolean li = FALSE;

  nNodes = size / 2;
  nTotalVert = nNodes * nVert;
  glMatrixMode(GL_PROJECTION);
  glPushMatrix();
  glLoadIdentity();
  gluOrtho2D(0.0, 600, 0., 600);
   
  glMatrixMode(GL_MODELVIEW);
  glPushMatrix();
  glLoadIdentity();
  vertices     = g_malloc(sizeof(double) * (nTotalVert * 2));
  nodeVertices = g_malloc(sizeof(double) * (nVert * 2));
  verticesKept = g_malloc(sizeof(double) * (nTotalVert * 2) + sizeof(int) * nTotalVert * 2);

  /* For each node, compute and add its vertices coordinates (nodeVertices) 
     to the vertices global array (vertices) */
  for (i = 0; i < nNodes; i++){
    defineNodeVertices(nVert, radius, coord[2 * i], coord[2 * i + 1], nodeVertices);
    addVerticesToGlobalArray(nVert, nodeVertices, vertices, i);
  }
  g_free(nodeVertices);
  verticesKept[elem] = NEWNODE;
  elem++;

  /* For each vertex */
  for (i = 0; i < nTotalVert; i++){
    /* We notify with a marker each new node */
    if (i / nVert != (i - 1) / nVert){
      verticesKept[elem] = NEWNODE;
      elem++;
    }
    /* For each node */
    for (j = 0; j < nNodes; j++){
      /* If vertex i doesn't belong to node j */
      if (j != i / nVert){
        /* Distance between vertex i and node j */
        distance = sqrt( (pow ((coord[2 * j + 1] - vertices[2 * i + 1]), 2)) +
                         (pow ((coord[2 * j]     - vertices[2 * i]),     2)) );
        if (distance < radius)
          visible = FALSE;
      }
    }
    if (visible){
      verticesKept[elem] = vertices[2 * i];
      verticesKept[elem + 1] = vertices[2 * i + 1];
      elem += 2;
    }
    else {
      if (verticesKept[elem - 1] != NEWLINE){
        verticesKept[elem] = NEWLINE;
        elem++;
      }
      visible = TRUE;
    }
  }

  glLineWidth(4.0);
  glColor3f(1.0, 1.0, 1.0);
  i = 0;

  while (i < elem){
    if (verticesKept[i] == NEWLINE)
      i++;
    else if (verticesKept[i] == NEWNODE){
      i++;
      if (verticesKept[i] != NEWNODE && verticesKept[i] != NEWLINE){
        link[0] = verticesKept[i];
        link[1] = verticesKept[i + 1];
        li = TRUE;
      }
      else {
        li = FALSE;
        while (verticesKept[i] == NEWNODE || verticesKept[i] == NEWLINE)
          i++;
      }
    }
    else {
      glBegin(GL_LINES);
      while (verticesKept[i + 2] != NEWNODE && verticesKept[i + 2] != NEWLINE && i + 2 < elem){
        glVertex2f(verticesKept[i], verticesKept[i + 1]);
        glVertex2f(verticesKept[i + 2], verticesKept[i + 3]);
        i += 2;
      }
      if (i + 2 >= elem){
        if (li){
          glVertex2f(verticesKept[i], verticesKept[i + 1]);
          glVertex2f(link[0], link[1]);
        }
        i = elem;
      }
      else if (verticesKept[i + 2] == NEWLINE)
        i += 3;
      else if (verticesKept[i + 2] == NEWNODE){
        if (li){
          glVertex2f(verticesKept[i], verticesKept[i + 1]);
          glVertex2f(link[0], link[1]);
        }
        i += 2;
      }
      glEnd();
    }
  }

  g_free(verticesKept);
  g_free(vertices);
  
  glPopMatrix();
  glMatrixMode(GL_PROJECTION);
  glPopMatrix();
  glMatrixMode(GL_MODELVIEW);
}
/* Directly draw the line on the front buffer. */
/* static void putMarkLine(VisuData *data, GLfloat *coord, unsigned int size, PickMesureMarkType type) */
/* { */
/*   OpenGLView *view; */
/*   view = visuDataGet_openGLView(data); */
/*   DBG_fprintf(stderr, "Visu pickMesure: draw directly a line on front buffer.\n"); */
/*   glDrawBuffer(GL_FRONT); */
/*   glPushAttrib(GL_ENABLE_BIT); */
/*   glDisable(GL_DEPTH_TEST); */
/*   glDisable(GL_LIGHTING); */
/*   glDisable(GL_FOG); */
/*   glEnable(GL_BLEND); */
/*   glBlendFunc(GL_ONE_MINUS_DST_COLOR, GL_ONE_MINUS_SRC_COLOR); */
/*   drawMarkLine(data, coord, size, type); */
/*   glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA); */
/*   glPopAttrib(); */
/*   glDrawBuffer(GL_BACK); */
/*   glFlush(); */
/* } */

/**********************/
/* Manage mark lists. */
/**********************/
/* static gboolean matchDot(struct MarkInfo_struct *markInfo, int nodeId, */
/* 			 gpointer data _U_) */
/* { */
/*   return (markInfo->idNode1 == nodeId && */
/* 	  markInfo->type < PICK_MESURE_END_MARK_DOT); */
/* } */
/* static gboolean matchType(struct MarkInfo_struct *markInfo, */
/* 			  int nodeId _U_, gpointer data) */
/* { */
/*   return (markInfo->type ==(unsigned int)GPOINTER_TO_INT(data)); */
/* } */
static gboolean matchDotType(struct MarkInfo_struct *markInfo,
			     guint nodeId, gpointer data)
{
  return (markInfo->idNode1 == nodeId &&
	  markInfo->type == (unsigned int)GPOINTER_TO_INT(data));
}
/* static gboolean matchDistance(struct MarkInfo_struct *markInfo, */
/* 			      int nodeId, gpointer data _U_) */
/* { */
/*   return ((markInfo->idNode1 == nodeId || markInfo->idNode2 == nodeId) && */
/* 	  (markInfo->type > PICK_MESURE_END_MARK_DOT && */
/* 	   markInfo->type < PICK_MESURE_END_MARK_LINK)); */
/* } */
static gboolean matchAny(struct MarkInfo_struct *markInfo, guint nodeId,
			 gpointer data _U_)
{
  return (markInfo->idNode1 == nodeId || markInfo->idNode2 == nodeId ||
	  markInfo->idNode3 == nodeId);
}
/* static gboolean matchFirst(struct MarkInfo_struct *markInfo, guint nodeId, */
/* 			   gpointer data _U_) */
/* { */
/*   return (markInfo->idNode1 == nodeId); */
/* } */
static GList* lookupMarkInList(PickMesure *mesureData, MatchFunc match,
			       guint nodeId, gpointer data)
{
  GList *list, *rtList;

  g_return_val_if_fail(mesureData, (GList*)0);

  list = mesureData->storedMarks;
  rtList = (GList*)0;
  while (list)
    {
      if (match((struct MarkInfo_struct*)list->data, nodeId, data))
	rtList = g_list_append(rtList, list);
      list = g_list_next(list);
    }
  return rtList;
}
gboolean pickMesureGet_active(PickMesure *mesureData, guint nodeId)
{
  GList *list;
  struct MarkInfo_struct *mark;

  g_return_val_if_fail(mesureData, FALSE);

  for (list = mesureData->storedMarks; list; list = g_list_next(list))
    {
      mark = (struct MarkInfo_struct*)list->data;
      if ((mark->type == PICK_MESURE_MARK_DISTANCE &&
	   mark->idNode1 == nodeId) ||
	  (mark->type == PICK_MESURE_MARK_ANGLE &&
	   mark->idNode1 == nodeId))
	return TRUE;
    }
  return FALSE;
}
static void removeMark(PickMesure *mesureData, GList *list)
{
  struct MarkInfo_struct *mark;

  g_return_if_fail(mesureData && list);

  mark = (struct MarkInfo_struct *)list->data;

  DBG_fprintf(stderr, "Visu PickMesure: remove mark of type %d, node %d/%d.\n",
	      mark->type, mark->idNode1, mark->idNode2);
  markFree(mark);

  if (list->prev)
    list->prev->next = list->next;
  if (list->next)
    list->next->prev = list->prev;
  if (list == mesureData->storedMarks)
    mesureData->storedMarks = list->next;
  g_list_free_1(list);
}
static void removeMarkFromList(PickMesure *mesureData, MatchFunc match,
			       guint nodeId, gpointer data)
{
  GList *list, *rmList;

  g_return_if_fail(mesureData);

  rmList = lookupMarkInList(mesureData, match, nodeId, data);
  list = rmList;
  while(list)
    {
      removeMark(mesureData, (GList*)list->data);
      list = g_list_next(list);
    }
  g_list_free(rmList);
}
static struct MarkInfo_struct* addMarkDotToList(PickMesure *mesureData, guint idNode,
						PickMesureMarkType type)
{
  struct MarkInfo_struct *mark;

  g_return_val_if_fail((type == PICK_MESURE_MARK_BIG_SQUARE ||
			type == PICK_MESURE_MARK_SMALL_SQUARE ||
			type == PICK_MESURE_MARK_HIGHLIGHT) && mesureData,
		       (struct MarkInfo_struct*)0);

  DBG_fprintf(stderr, "Visu PickMesure: add a new mark of type %d.\n", type);
  mark          = markNew(type);
  mark->idNode1 = idNode;

  mesureData->storedMarks = g_list_prepend(mesureData->storedMarks, (gpointer)mark);
  
  return (struct MarkInfo_struct*)mark;
}
/* Method that add/remove a distance mark to the list of drawn marks. */
static void toggleMarkDistanceInList(PickMesure *mesureData, guint nodeRefId,
				     guint nodeId, gboolean set)
{
  struct MarkInfo_struct *mark;
  GList *tmpLst;

  g_return_if_fail(mesureData);

  /* Look for the mark. */
  tmpLst = mesureData->storedMarks;
  while (tmpLst)
    {
      mark = (struct MarkInfo_struct*)tmpLst->data;
      if (mark->type == PICK_MESURE_MARK_DISTANCE &&
	  mark->idNode1 == nodeRefId && mark->idNode2 == nodeId)
	{
	  DBG_fprintf(stderr, "Visu PickMesure: found a distance mark.\n");
	  if (!set)
	    removeMark(mesureData, tmpLst);
	  return;
	}
      tmpLst = g_list_next(tmpLst);
    }
  /* Found none, create a new one. */
  mark          = markNew(PICK_MESURE_MARK_DISTANCE);
  mark->idNode1 = nodeRefId;
  mark->idNode2 = nodeId;
  DBG_fprintf(stderr, "Visu PickMesure: add a distance mark.\n");
  mesureData->storedMarks = g_list_append(mesureData->storedMarks, (gpointer)mark);
}

static void toggleMarkAngleInList(PickMesure *mesureData, guint nodeRefId,
				  guint nodeRef2Id, guint nodeId, gboolean set)
{
  struct MarkInfo_struct *mark;
  GList *tmpLst;

  g_return_if_fail(mesureData);

  /* Look for the mark. */
  for (tmpLst = mesureData->storedMarks; tmpLst; tmpLst = g_list_next(tmpLst))
    {
      mark = (struct MarkInfo_struct*)tmpLst->data;
      if (mark->type == PICK_MESURE_MARK_ANGLE &&
	  mark->idNode1 == nodeRefId &&
	  ((mark->idNode2 == nodeRef2Id &&
	    mark->idNode3 == nodeId) ||
	   (mark->idNode3 == nodeRef2Id &&
	    mark->idNode2 == nodeId)))
	{
	  DBG_fprintf(stderr, "Visu PickMesure: found an angle mark.\n");
	  if (!set)
	    removeMark(mesureData, tmpLst);
	  return;
	}
    }
  /* Found none, create a new one. */
  mark          = markNew(PICK_MESURE_MARK_ANGLE);
  mark->idNode1 = nodeRefId;
  mark->idNode2 = nodeRef2Id;
  mark->idNode3 = nodeId;
  DBG_fprintf(stderr, "Visu PickMesure: add an angle mark.\n");
  mesureData->storedMarks = g_list_append(mesureData->storedMarks, (gpointer)mark);
}

static gboolean setInformation(PickMesure *mesureData, guint node, float range)
{
  VisuNodeInfo *infos;
  float min;
  GList *lst, *tmpLst, *tmpLst2;
  int i, n;
  gboolean redraw;
  float *dists;
  guint *ids;
  float xyz1[3], xyz2[3];

  g_return_val_if_fail(mesureData, FALSE);

  DBG_fprintf(stderr, "Visu PickMesure: compute and print distances"
	      " and angles for %d.\n", node);
  infos = visuDataGet_distanceList(mesureData->data, node, &min);

  lst = (GList*)0;
  redraw = FALSE;
  for (i = 0; infos[i].id != node; i+= 1)
    if (infos[i].dist < range * min)
      {
	toggleMarkDistanceInList(mesureData, node, infos[i].id,
				 TRUE);
	lst = g_list_prepend(lst, GINT_TO_POINTER(infos[i].id));
	redraw = TRUE;
      }
  g_free(infos);

  n = g_list_length(lst);
  if (n > 1)
    {
      n = n * (n - 1 ) / 2;
      DBG_fprintf(stderr, "Visu PickMesure: there are %d angles at max.\n", n);
      ids = g_malloc(sizeof(guint) * n * 2);
      dists = g_malloc(sizeof(float) * n);
      i = 0;
      min = G_MAXFLOAT;
      for (tmpLst = lst; tmpLst; tmpLst = g_list_next(tmpLst))
	for (tmpLst2 = tmpLst->next; tmpLst2; tmpLst2 = g_list_next(tmpLst2))
	  {
	    ids[i * 2 + 0] = (guint)GPOINTER_TO_INT(tmpLst->data);
	    ids[i * 2 + 1] = (guint)GPOINTER_TO_INT(tmpLst2->data);
	    visuDataGet_nodePosition(mesureData->data,
				     visuDataGet_nodeFromNumber(mesureData->data,
								ids[i * 2 + 0]),
				     xyz1);
	    visuDataGet_nodePosition(mesureData->data,
				     visuDataGet_nodeFromNumber(mesureData->data,
								ids[i * 2 + 1]),
				     xyz2);
	    dists[i] =
	      (xyz1[0] - xyz2[0]) * (xyz1[0] - xyz2[0]) +
	      (xyz1[1] - xyz2[1]) * (xyz1[1] - xyz2[1]) +
	      (xyz1[2] - xyz2[2]) * (xyz1[2] - xyz2[2]);
	    DBG_fprintf(stderr, " | %d (%d %d) -> %g\n", i, ids[i * 2 + 0],
			ids[i * 2 + 1], dists[i]);
	    min = MIN(min, dists[i]);
	    i += 1;
	  }
  
      for (i = 0; i < n; i++)
	if (dists[i] < (2.75f * min))
	  {
	    DBG_fprintf(stderr, " | %d (%d %d) -> %g OK\n", i, ids[i * 2 + 0],
			ids[i * 2 + 1], dists[i]);
	    toggleMarkAngleInList(mesureData, node, ids[i * 2 + 0],
				  ids[i * 2 + 1], TRUE);
	  }
      g_free(ids);
      g_free(dists);
    }
  g_list_free(lst);

  if (redraw)
    drawMarkList(mesureData->data, mesureData->storedMarks, 0);

  return redraw;
}

/* Update PickMesure with new nodes selection */
/* static struct MarkInfo_struct* updateMarkLineInList(PickMesure *mesureData, GList *region, PickMesureMarkType type) */
/* { */
/*   struct MarkInfo_struct *mark; */
/*   int len, i; */
/*   GList *tmpLst; */

/*   g_return_val_if_fail(type == PICK_MESURE_MARK_LINE, (struct MarkInfo_struct*)0); */

/*   mark		= g_malloc(sizeof(struct MarkInfo_struct)); */
/*   mark->type	= type; */
/*   mark->nodes	= g_list_copy(region); */

/*   len = g_list_length(region); */
/*   mark->coord	= g_malloc(sizeof(GLfloat) * len * 2); */
/*   mark->nIds	= g_malloc(sizeof(unsigned int) * len); */
/*   i = 0; */
/*   for (tmpLst = region; tmpLst; tmpLst = g_list_next(tmpLst)) */
/*   { */
/*     mark->nIds[i] = GPOINTER_TO_INT(tmpLst->data); */
/*     i++; */
/*   } */

/*   getNodes2DCoordinates(mesureData->data, mark->nIds, len, mark->coord, &mark->size); */
/*   DBG_fprintf(stderr, "Visu PickMesure: add a line mark.\n"); */
/*   mesureData->storedMarks = g_list_append(mesureData->storedMarks, (gpointer)mark); */

/*   return mark; */
/* } */

/* Method that removes all distance marks from the list of drawn marks. */
gboolean pickMesureRemove_allMeasures(PickMesure *mesureData)
{
  GList *list, *rmList;
  struct MarkInfo_struct *mark;

  g_return_val_if_fail(mesureData, FALSE);

  list = mesureData->storedMarks;
  rmList = (GList*)0;
  for (list = mesureData->storedMarks; list; list = g_list_next(list))
    {
      mark = (struct MarkInfo_struct*)list->data;
      if (mark->type == PICK_MESURE_MARK_DISTANCE ||
	  mark->type == PICK_MESURE_MARK_ANGLE)
	rmList = g_list_append(rmList, list);
    }
  if (!rmList)
    return FALSE;

  for (list = rmList; list; list = g_list_next(list))
    removeMark(mesureData, (GList*)list->data);
  g_list_free(rmList);

  drawMarkList(mesureData->data, mesureData->storedMarks, 0);
  return TRUE;
}
gboolean pickMesureRemove_nodeMeasures(PickMesure *mesureData, guint nodeId)
{
  GList *list, *rmList;
  struct MarkInfo_struct *mark;

  g_return_val_if_fail(mesureData, FALSE);

  list = mesureData->storedMarks;
  rmList = (GList*)0;
  for (list = mesureData->storedMarks; list; list = g_list_next(list))
    {
      mark = (struct MarkInfo_struct*)list->data;
      if ((mark->type == PICK_MESURE_MARK_DISTANCE ||
	   mark->type == PICK_MESURE_MARK_ANGLE) &&
	  mark->idNode1 == nodeId)
	rmList = g_list_append(rmList, list);
    }
  if (!rmList)
    return FALSE;

  for (list = rmList; list; list = g_list_next(list))
    removeMark(mesureData, (GList*)list->data);
  g_list_free(rmList);

  drawMarkList(mesureData->data, mesureData->storedMarks, 0);
  return TRUE;
}
gboolean pickMesureRemove_allMarks(PickMesure *mesureData)
{
  gboolean redraw;

  g_return_val_if_fail(mesureData, FALSE);

  if (mesureData->idRef1 >= 0)
    removeMarkFromList(mesureData, matchDotType, mesureData->idRef1,
		       GINT_TO_POINTER(PICK_MESURE_MARK_BIG_SQUARE));
  if (mesureData->idRef2 >= 0)
    removeMarkFromList(mesureData, matchDotType, mesureData->idRef2,
		       GINT_TO_POINTER(PICK_MESURE_MARK_SMALL_SQUARE));
  mesureData->idRef1 = -1;
  mesureData->idRef2 = -1;
  mesureData->idSelected = -1;
  redraw = pickMesureRemove_allMeasures(mesureData);
  
  return redraw;
}
gboolean pickMesureRemove_allHighlights(PickMesure *mesureData)
{
  GList *list, *rmList;

  g_return_val_if_fail(mesureData, FALSE);

  list = mesureData->storedMarks;
  rmList = (GList*)0;
  while (list)
    {
      if (((struct MarkInfo_struct*)list->data)->type == PICK_MESURE_MARK_HIGHLIGHT)
	rmList = g_list_append(rmList, list);
      list = g_list_next(list);
    }
  if (!rmList)
    return FALSE;

  list = rmList;
  while (list)
    {
      removeMark(mesureData, (GList*)list->data);
      list = g_list_next(list);
    }  
  g_list_free(rmList);
  drawMarkList(mesureData->data, mesureData->storedMarks, 1);
  return TRUE;
}


gboolean pickMesureSet_highlight(PickMesure *mesureData, guint idNode,
				 gboolean status)
{
  GList *mark;
  struct MarkInfo_struct *st;

  DBG_fprintf(stderr, "Visu PickMesure: set an highlight (%d) for the node %d.\n",
	      status, idNode);
  
  mark = lookupMarkInList(mesureData, matchDotType, idNode,
			  GINT_TO_POINTER(PICK_MESURE_MARK_HIGHLIGHT));
  DBG_fprintf(stderr, "Visu PickMesure: already existing highlight %p.\n",
	      (gpointer)mark);  

  if (mark)
    {
      /* We check that we only have one result. */
      g_return_val_if_fail(!mark->next, FALSE);

      st = (struct MarkInfo_struct*)((GList*)mark->data)->data;
      if (!status)
	removeMark(mesureData, (GList*)mark->data);
      g_list_free(mark);

      return !status;
    }
  if (!mark && status)
    {
      st = addMarkDotToList(mesureData, idNode, PICK_MESURE_MARK_HIGHLIGHT);
      return TRUE;
    }
  return FALSE;
}

gboolean pickMesureSet_nodeMeasurements(PickMesure *mesureData,
					guint nodeId, float range,
					gboolean set)
{
  gboolean redraw;

  g_return_val_if_fail(mesureData, FALSE);

  if (set)
    redraw = setInformation(mesureData, nodeId, range);
  else
    redraw = pickMesureRemove_nodeMeasures(mesureData, nodeId);

  return redraw;
}

void pickMesureSet_pickNode(PickMesure *mesureData, gint nodeId, PickMesureType type)
{
  double dx, dy, dz, dr, dx1, dy1, dz1, dr1,  dx2, dy2, dz2, dr2;
  double ang;
  float posRef1[3], posRef2[3], posSelect[3];
  VisuNode *ref1, *ref2, *selected;
  GList *mark;
  struct MarkInfo_struct *st;

  g_return_if_fail(mesureData && (type == PICK_SELECTED ||
				  type == PICK_REFERENCE_1 ||
				  type == PICK_REFERENCE_2 ||
				  type == PICK_HIGHLIGHT ||
				  type == PICK_INFORMATION));

  /* No changes. */
  if (/* (type == PICK_SELECTED && nodeId == mesureData->idSelected) || */
      (type == PICK_REFERENCE_1 && nodeId == mesureData->idRef1) ||
      (type == PICK_REFERENCE_2 && nodeId == mesureData->idRef2))
    return;

  DBG_fprintf(stderr, "Pick Mesure: set pick info for node %d (%d).\n",
	      nodeId, (int)type);

  /* Prepare the values for the strings. */
  mesureData->newValuesAvailable = TRUE;
  mesureData->typeOfNews = type;
  if (mesureData->formatFlag)
    {
      if (!mesureData->info)
	mesureData->info = g_string_new("");
      else
	mesureData->info = g_string_erase(mesureData->info, 0, -1);
      if (!mesureData->errors)
	mesureData->errors = g_string_new("");
      else
	mesureData->errors = g_string_erase(mesureData->errors, 0, -1);
    }
  /* Cancel a selected region, if any. */
  if (mesureData->idRegion)
    g_list_free(mesureData->idRegion);
  mesureData->idRegion = (GList*)0;

  /* Update idSelected, idRef1 and idRef2, depending on @type. */
  switch (type)
    {
    case (PICK_SELECTED):
      if (nodeId >= 0)
	{
	  mesureData->idSelected = nodeId;
	  if (mesureData->idRef1 >= 0 &&  mesureData->storeMeasures)
	    {
	      if (mesureData->idRef2 < 0)
		{
		  toggleMarkDistanceInList(mesureData,
					   mesureData->idRef1, mesureData->idSelected,
					   FALSE);
		  putMarkDistance(mesureData->data,
				  mesureData->idRef1, mesureData->idSelected,
				  PICK_MESURE_MARK_DISTANCE);
		  drawMarkList(mesureData->data, mesureData->storedMarks, 0);
		}
	      else
		{
		  toggleMarkAngleInList(mesureData,
					mesureData->idRef1,
					mesureData->idRef2,
					mesureData->idSelected, FALSE);
		  drawMarkList(mesureData->data, mesureData->storedMarks, 0);
		  g_idle_add(visuObjectRedraw, GINT_TO_POINTER(TRUE));
		}
	    }
	}
      else
	{
	  if (mesureData->formatFlag)
	    mesureData->errors = g_string_append(mesureData->errors,
						 _("You picked nothing.\n"));
	  mesureData->idSelected = -1;
	}
      break;
    case (PICK_REFERENCE_1):
      if (nodeId >= 0)
	{
	  if (mesureData->idRef2 == nodeId)
	    {
	      if (mesureData->formatFlag)
		mesureData->errors =
		  g_string_append(mesureData->errors,
				  _("1st reference node should be distinct\n"
				    "from 2nd one.\n"));
	    }
	  else
	    {
	      if (mesureData->idRef1 >= 0)
		{
		  /* Remove old one. */
		  removeMarkFromList(mesureData, matchDotType,
				     mesureData->idRef1,
				     GINT_TO_POINTER(PICK_MESURE_MARK_BIG_SQUARE));
		  putMarkDot(mesureData->data, mesureData->idRef1,
			     PICK_MESURE_MARK_BIG_SQUARE);
		}
	      mesureData->idRef1 = nodeId;
	      addMarkDotToList(mesureData, mesureData->idRef1,
			       PICK_MESURE_MARK_BIG_SQUARE);
	      putMarkDot(mesureData->data, mesureData->idRef1,
			 PICK_MESURE_MARK_BIG_SQUARE);
	    }
	}
      else
	{
	  if (mesureData->formatFlag)
	    mesureData->errors = g_string_append(mesureData->errors,
						 _("Unset reference 1.\n"));
	  if (mesureData->idRef1 >= 0)
	    {
	      removeMarkFromList(mesureData, matchDotType,
				 mesureData->idRef1,
				 GINT_TO_POINTER(PICK_MESURE_MARK_BIG_SQUARE));
	      putMarkDot(mesureData->data, mesureData->idRef1,
			 PICK_MESURE_MARK_BIG_SQUARE);
	    }
	  mesureData->idRef1 = -1;
	  if (mesureData->idRef2 >= 0)
	    {
	      removeMarkFromList(mesureData, matchDotType,
				 mesureData->idRef2,
				 GINT_TO_POINTER(PICK_MESURE_MARK_SMALL_SQUARE));
	      putMarkDot(mesureData->data, mesureData->idRef2,
			 PICK_MESURE_MARK_SMALL_SQUARE);
	    }
	  mesureData->idRef2 = -1;
	}
      drawMarkList(mesureData->data, mesureData->storedMarks, 0);
      break;
    case (PICK_REFERENCE_2):
      if (mesureData->idRef2 >= 0)
	{
	  removeMarkFromList(mesureData, matchDotType,
			     mesureData->idRef2,
			     GINT_TO_POINTER(PICK_MESURE_MARK_SMALL_SQUARE));
	  putMarkDot(mesureData->data, mesureData->idRef2,
		     PICK_MESURE_MARK_SMALL_SQUARE);
	}
      DBG_fprintf(stderr, "Visu PickMesure: set reference 2 to %d (%d).\n",
		  nodeId, mesureData->idRef2);
      mesureData->idRef2 = -1;
      if (nodeId >= 0)
	{
	  if (mesureData->idRef1 < 0)
	    {
	      if (mesureData->formatFlag)
		mesureData->errors =
		  g_string_append(mesureData->errors,
				  _("Unable to set reference 2\n"
				    "without existing reference 1, use\n"
				    "<shift> middle button to set reference 1.\n"));
	    }
	  else if (mesureData->idRef1 == nodeId)
	    {
	      if (mesureData->formatFlag)
		mesureData->errors =
		  g_string_append(mesureData->errors,
				  _("2nd reference node should be distinct\n"
				    "from 1st one.\n"));
	    }
	  else
	    {
	      mesureData->idRef2 = nodeId;
	      addMarkDotToList(mesureData, mesureData->idRef2,
			       PICK_MESURE_MARK_SMALL_SQUARE);
	      putMarkDot(mesureData->data, mesureData->idRef2,
			 PICK_MESURE_MARK_SMALL_SQUARE);
	    }
	}
      else
	{
	  if (mesureData->formatFlag)
	    mesureData->errors =
	      g_string_append(mesureData->errors,
			      _("Unset reference 2.\n"));
	}
      drawMarkList(mesureData->data, mesureData->storedMarks, 0);
      break;
    case PICK_HIGHLIGHT:
      DBG_fprintf(stderr, "Visu PickMesure: toggle an highlight for the node %d.\n",
		  nodeId);
      if (nodeId >= 0)
	{
	  mark = lookupMarkInList(mesureData, matchDotType, nodeId,
				  GINT_TO_POINTER(PICK_MESURE_MARK_HIGHLIGHT));

	  if (mark)
	    {
	      g_return_if_fail(!mark->next);
	      /* We remove unconditionaly. */
	      removeMark(mesureData, (GList*)mark->data);
	      mesureData->idHighlighted = nodeId;
	      mesureData->typeOfNews = PICK_UNHIGHLIGHT;
	    }
	  else
	    {
	      st = addMarkDotToList(mesureData, nodeId, PICK_MESURE_MARK_HIGHLIGHT);
	      mesureData->idHighlighted = nodeId;
	    }

	  drawMarkList(mesureData->data, mesureData->storedMarks, 1);
	}
      else
	{
	  mesureData->newValuesAvailable = FALSE;
	  mesureData->idHighlighted = -99;
	}
      g_idle_add(visuObjectRedraw, GINT_TO_POINTER(TRUE));
      break;
    case PICK_INFORMATION:
      mesureData->idSelected = nodeId;
      if (nodeId >= 0)
	{
	  setInformation(mesureData, nodeId, 1.44f);
	  g_idle_add(visuObjectRedraw, GINT_TO_POINTER(TRUE));
	}
      break;
    default:
      g_error("Can't be here!");
    }

  if (!mesureData->formatFlag)
    return;

  /* Create a string with informations. */
  selected = (mesureData->idSelected < 0)?(VisuNode*)0:
    visuDataGet_nodeFromNumber(mesureData->data, mesureData->idSelected);
  ref1 = (mesureData->idRef1 < 0)?(VisuNode*)0:
    visuDataGet_nodeFromNumber(mesureData->data, mesureData->idRef1);
  ref2 = (mesureData->idRef2 < 0)?(VisuNode*)0:
    visuDataGet_nodeFromNumber(mesureData->data, mesureData->idRef2);
  if (ref1)
    {
      g_string_append_printf(mesureData->info,
			     _("Reference node\t"
			       " <span font_desc=\"monospace\"><b>#%d</b></span>\n"
			       "<span font_desc=\"monospace\" size=\"small\">"
			       "  ( x  = %7.3f ;  y  = %7.3f ;  z  = %7.3f)\n"
			       "</span>"),
			     ref1->number + 1, ref1->xyz[0],
			     ref1->xyz[1],     ref1->xyz[2]);
    }
  if (ref2)
    {
      visuDataGet_nodePosition(mesureData->data, ref1, posRef1);
      visuDataGet_nodePosition(mesureData->data, ref2, posRef2);
      dx = posRef2[0] - posRef1[0];
      dy = posRef2[1] - posRef1[1];
      dz = posRef2[2] - posRef1[2];
      dr = sqrt(dx*dx + dy*dy + dz*dz);
      g_string_append_printf(mesureData->info,
			     _("2nd Reference node\t"
			       " <span font_desc=\"monospace\"><b>#%d</b>\t"
			       " (<i>i.e.</i> dr = %7.3f)</span>\n"
			       "<span font_desc=\"monospace\" size=\"small\">"
			       "  ( x  = %7.3f ;  y  = %7.3f ;  z  = %7.3f)\n"
			       "  (\316\264x  = %7.3f ; \316\264y  = %7.3f ; \316\264z  = %7.3f)\n"
			       "</span>"),
			     ref2->number + 1, dr,
			     ref2->xyz[0], ref2->xyz[1], ref2->xyz[2],
			     dx, dy, dz);
    }
  if (selected)
    {
      if (!ref2)
        {
	  if (ref1)
	    {
	      visuDataGet_nodePosition(mesureData->data, selected, posSelect);
	      visuDataGet_nodePosition(mesureData->data, ref1, posRef1);
	      dx = posSelect[0] - posRef1[0];
	      dy = posSelect[1] - posRef1[1];
	      dz = posSelect[2] - posRef1[2];
	      dr = sqrt(dx*dx + dy*dy + dz*dz);
	      g_string_append_printf(mesureData->info,
				     _("Newly picked node\t"
				       " <span font_desc=\"monospace\"><b>#%d</b>\t"
				       " (<i>i.e.</i> dr = %7.3f)</span>\n"
				       "<span font_desc=\"monospace\" size=\"small\">"
				       "  ( x  = %7.3f ;  y  = %7.3f ;  z  = %7.3f)\n"
				       "  (\316\264x  = %7.3f ; \316\264y  = %7.3f ; \316\264z  = %7.3f)\n"
				       "</span>"),
				     selected->number + 1, dr,
				     selected->xyz[0], selected->xyz[1],
				     selected->xyz[2], dx, dy, dz);
	    }
	  else
	    g_string_append_printf(mesureData->info,
				   _("Newly picked node\t"
				     " <span font_desc=\"monospace\"><b>#%d</b></span>\n"
				     "<span font_desc=\"monospace\" size=\"small\">"
				     "  ( x  = %7.3f ;  y  = %7.3f ;  z  = %7.3f)\n"
				     "</span>"),
				   selected->number +1,
				   selected->xyz[0],
				   selected->xyz[1],
				   selected->xyz[2]
				   );
	}
      else
        {
	  visuDataGet_nodePosition(mesureData->data, selected, posSelect);
	  visuDataGet_nodePosition(mesureData->data, ref1, posRef1);
	  visuDataGet_nodePosition(mesureData->data, ref2, posRef2);
	  dx1 = posSelect[0] - posRef1[0];
	  dy1 = posSelect[1] - posRef1[1];
	  dz1 = posSelect[2] - posRef1[2];
	  dx2 = posRef2[0] - posRef1[0];
	  dy2 = posRef2[1] - posRef1[1];
	  dz2 = posRef2[2] - posRef1[2];
	  dr1 = sqrt(dx1*dx1 + dy1*dy1 + dz1*dz1);
	  g_string_append_printf(mesureData->info,
				 _("Newly picked node\t"
				   " <span font_desc=\"monospace\"><b>#%d</b>\t"
				   " (<i>i.e.</i> dr = %7.3f)</span>\n"
				   "<span font_desc=\"monospace\" size=\"small\">"
				   "  ( x  = %7.3f ;  y  = %7.3f ;  z  = %7.3f)\n"
				   "  (\316\264x1 = %7.3f ; \316\264y1 = %7.3f ; \316\264z1 = %7.3f)\n"
				   "  (\316\264x2 = %7.3f ; \316\264y2 = %7.3f ; \316\264z2 = %7.3f)\n"),
				 selected->number + 1, dr1,
				 selected->xyz[0], selected->xyz[1],
				 selected->xyz[2], dx1, dy1, dz1, dx2, dy2, dz2);
	  dr2 = sqrt(dx2*dx2 + dy2*dy2 + dz2*dz2);
	  ang = acos((dx2*dx1+dy2*dy1+dz2*dz1)/(dr2*dr1))/PI180;
	  g_string_append_printf(mesureData->info,
				 _("  angle (Ref-Ref2, Ref-New) = %5.2f degrees"
				   "</span>"), ang);
        }
    }

  DBG_fprintf(stderr, "Pick mesure info: %s", mesureData->info->str);
  DBG_fprintf(stderr, "Pick mesure error: %s", mesureData->errors->str);
}
void pickMesureSet_pickRegion(PickMesure *mesureData, GList *nodeIds)
{
/*   struct MarkInfo_struct *mark; */
  
  g_return_if_fail(mesureData);
  /* Prepare the values for the strings. */
  mesureData->newValuesAvailable = TRUE;
  mesureData->typeOfNews = PICK_REGION;
  if (mesureData->formatFlag)
    {
      if (mesureData->info)
	g_string_free(mesureData->info, TRUE);
      if (mesureData->errors)
	g_string_free(mesureData->errors, TRUE);
      mesureData->info   = (GString*)0;
      mesureData->errors = (GString*)0;
    }

  /* Copy the region list. */
  if (mesureData->idRegion)
    g_list_free(mesureData->idRegion);
  mesureData->idRegion = g_list_copy(nodeIds);

/*   mark = updateMarkLineInList(mesureData, nodeIds, PICK_MESURE_MARK_LINE); */

/*   putMarkLine(mesureData->data, mark->coord, mark->size, PICK_MESURE_MARK_LINE); */
/*   drawMarkList(mesureData->data, mesureData->storedMarks, 0); */
}
void pickMesureSet_dragStart(PickMesure *mesureData, guint nodeId)
{
#if DEBUG == 1
  VisuNode *node;
  float xyz[3];
#endif

  g_return_if_fail(mesureData);
  
  mesureData->newValuesAvailable = TRUE;
  mesureData->typeOfNews = PICK_DRAG_START;
  mesureData->idSelected = nodeId;

#if DEBUG == 1
  fprintf(stderr, "Pick Mesure: start drag for node %d.\n", nodeId);
  node = visuDataGet_nodeFromNumber(mesureData->data, nodeId);
  visuDataGet_nodePosition(mesureData->data, node, xyz);
  fprintf(stderr, " | %gx%gx%g\n", xyz[0], xyz[1], xyz[2]);
#endif
}
void pickMesureSet_dragStop(PickMesure *mesureData)
{
  g_return_if_fail(mesureData);
  
  mesureData->newValuesAvailable = TRUE;
  mesureData->typeOfNews = PICK_DRAG_STOP;
}
void pickMesureSet_dragMove(PickMesure *mesureData, float dx, float dy, float dz)
{
  g_return_if_fail(mesureData);
  
  mesureData->newValuesAvailable = TRUE;
  mesureData->typeOfNews = PICK_DRAG_MOVE;
  mesureData->drag[0] = dx;
  mesureData->drag[1] = dy;
  mesureData->drag[2] = dz;
}
float* pickMesureGet_drag(PickMesure *mesureData)
{
  g_return_val_if_fail(mesureData, (float*)0);
  
  return (float*)mesureData->drag;
}
gboolean pickMesureGet_newsAvailable(PickMesure *mesureData, PickMesureType *type)
{
  g_return_val_if_fail(mesureData, FALSE);

  if (type)
    *type = mesureData->typeOfNews;
  return mesureData->newValuesAvailable;
}

gchar* pickMesureGet_infos(PickMesure *mesureData)
{
  g_return_val_if_fail(mesureData, (gchar*)0);

  if (!mesureData->info || !mesureData->formatFlag)
    return (gchar*)0;
  return mesureData->info->str;
}
gchar* pickMesureGet_errors(PickMesure *mesureData)
{
  g_return_val_if_fail(mesureData, (gchar*)0);

  if (!mesureData->errors || !mesureData->formatFlag)
    return (gchar*)0;
  return mesureData->errors->str;
}
VisuNode* pickMesureGet_highlightedNode(PickMesure *mesureData)
{
  g_return_val_if_fail(mesureData, (VisuNode*)0);

  if (mesureData->idHighlighted >= 0)
    return visuDataGet_nodeFromNumber(mesureData->data, mesureData->idHighlighted);
  else
    return (VisuNode*)0;
}
VisuNode* pickMesureGet_selectedNode(PickMesure *mesureData)
{
  g_return_val_if_fail(mesureData, (VisuNode*)0);

  if (mesureData->idSelected >= 0)
    return visuDataGet_nodeFromNumber(mesureData->data, mesureData->idSelected);
  else
    return (VisuNode*)0;
}
VisuNode* pickMesureGet_firstReference(PickMesure *mesureData)
{
  g_return_val_if_fail(mesureData, (VisuNode*)0);

  if (mesureData->idRef1 >= 0)
    return visuDataGet_nodeFromNumber(mesureData->data, mesureData->idRef1);
  else
    return (VisuNode*)0;
}
VisuNode* pickMesureGet_secondReference(PickMesure *mesureData)
{
  g_return_val_if_fail(mesureData, (VisuNode*)0);

  if (mesureData->idRef2 >= 0)
    return visuDataGet_nodeFromNumber(mesureData->data, mesureData->idRef2);
  else
    return (VisuNode*)0;
}
GList* pickMesureGet_regionNodes(PickMesure *mesureData)
{
  GList *lst, *tmpLst;
  VisuNode *node;

  g_return_val_if_fail(mesureData, (GList*)0);
  
  lst = (GList*)0;
  tmpLst = mesureData->idRegion;
  while (tmpLst)
    {
      node = visuDataGet_nodeFromNumber(mesureData->data,
					GPOINTER_TO_INT(tmpLst->data));
      if (node)
	lst = g_list_prepend(lst, (gpointer)node);
      tmpLst = g_list_next(tmpLst);
    }
  return lst;
}
int pickMesureGet_regionNNodes(PickMesure *mesureData)
{
  g_return_val_if_fail(mesureData, -1);

  return g_list_length(mesureData->idRegion);
}


void pickMesureUpdate(VisuData *newData, VisuData *oldData)
{
  GList *list;
  PickMesure *newPick, *oldPick;
  struct MarkInfo_struct* mark, *newMark;
  gboolean remove;

  if (!newData || !oldData)
    return;

  DBG_fprintf(stderr, "Visu PickMesure: updating list of marks.\n");

  newPick = (PickMesure*)g_object_get_data(G_OBJECT(newData), "pickMesure_data");
  oldPick = (PickMesure*)g_object_get_data(G_OBJECT(oldData), "pickMesure_data");
  g_return_if_fail(newPick && oldPick);

  /* Try to match previous marks on new VisuData. */
  list = oldPick->storedMarks;
  while (list)
    {
      mark = (struct MarkInfo_struct*)list->data;
      DBG_fprintf(stderr, " | old mark %p of type %d.\n", (gpointer)mark, mark->type);
      remove = FALSE;
      /* We remove mark if one of the node can't be matched. */
      if (mark->type == PICK_MESURE_MARK_BIG_SQUARE ||
	  mark->type == PICK_MESURE_MARK_SMALL_SQUARE ||
	  mark->type == PICK_MESURE_MARK_HIGHLIGHT)
	remove = remove || !visuDataGet_nodeFromNumber(newData, mark->idNode1);
      else if (mark->type == PICK_MESURE_MARK_DISTANCE)
	remove = remove || !visuDataGet_nodeFromNumber(newData, mark->idNode1) ||
	  !visuDataGet_nodeFromNumber(newData, mark->idNode2);
      else if (mark->type == PICK_MESURE_MARK_ANGLE)
	remove = remove || !visuDataGet_nodeFromNumber(newData, mark->idNode1) ||
	  !visuDataGet_nodeFromNumber(newData, mark->idNode2) ||
	  !visuDataGet_nodeFromNumber(newData, mark->idNode3);
      /* TODO add the other possibilities. */
      if (!remove && newPick != oldPick)
	{
	  /* We keep it, and add it in the list of newPick. */
	  newMark = markCopy(mark);
	  newPick->storedMarks = g_list_prepend(newPick->storedMarks, newMark);
	  DBG_fprintf(stderr, " | adding new mark %p of type %d.\n",
		      (gpointer)newMark, newMark->type);
	}
      if (remove && newPick == oldPick)
	{
	  /* We remove it, from newPick. */
	  DBG_fprintf(stderr, " | delete old mark %p of type %d.\n",
		      (gpointer)mark, mark->type);
	  newPick->storedMarks = g_list_delete_link(newPick->storedMarks, list);
	}
      list = g_list_next(list);
    }
  DBG_fprintf(stderr, " | New mark list is pointing to %p.\n",
	      (gpointer)newPick->storedMarks);
  if (oldPick->idRef1 >= 0)
    {
      DBG_fprintf(stderr, "Visu PickMesure : trying to match ref1 node (id: %d).\n",
		  oldPick->idRef1);
      if (visuDataGet_nodeFromNumber(newData, oldPick->idRef1))
	newPick->idRef1 = oldPick->idRef1;
    }
  if (oldPick->idRef2 >= 0)
    {
      DBG_fprintf(stderr, "Visu PickMesure : trying to match ref2 node (id: %d).\n",
	      oldPick->idRef2);
      if (visuDataGet_nodeFromNumber(newData, oldPick->idRef2))
	newPick->idRef2 = oldPick->idRef2;
    }
  if (oldPick->idSelected >= 0)
    {
      DBG_fprintf(stderr, "Visu PickMesure : trying to match selected node (id: %d).\n",
	      oldPick->idSelected);
      if (visuDataGet_nodeFromNumber(newData, oldPick->idSelected))
	newPick->idSelected = oldPick->idSelected;
    }
  newPick->storeMeasures = oldPick->storeMeasures;
  newPick->formatFlag = oldPick->formatFlag;
  drawMarkList(newPick->data, newPick->storedMarks, -1);
}
static void updateListOnPopulationChange(VisuData *dataObj, int *nodes,
					 gpointer data)
{
  int i;
  GList *list, *rmList;

  g_return_if_fail(data);

  DBG_fprintf(stderr, "Visu PickMesure: caught 'NodePopulationDecrease'"
	      " signal (%p).\n", data);
  
  /* Run through the mark list to get all nodeId and look into nodes
     to find if mark must be removed or not. */
  DBG_fprintf(stderr, " | list contains %d elements\n",
	      g_list_length(((PickMesure*)data)->storedMarks));
  rmList = (GList*)0;
  for (list = ((PickMesure*)data)->storedMarks; list; list = g_list_next(list))
    {
      for (i = 0; nodes[i] >= 0; i++)
	if (matchAny((struct MarkInfo_struct*)list->data, nodes[i], (gpointer)0))
	  {
	    rmList = g_list_append(rmList, list);
	    break;
	  }
    }

  if (rmList)
    {
      DBG_fprintf(stderr, " | need to remove %d elements\n", g_list_length(rmList));
      list = rmList;
      while (list)
	{
	  removeMark((PickMesure*)data, (GList*)list->data);
	  list = g_list_next(list);
	}  
      g_list_free(rmList);
      DBG_fprintf(stderr, " | rebuild list.\n");
      drawMarkList(dataObj, ((PickMesure*)data)->storedMarks, -1);
    }

  DBG_fprintf(stderr, " | remove the fixed elements.\n");
  /* Remove also ref1, ref2 and selected from the PickMesure object. */
  for (i = 0; nodes[i] >= 0; i++)
    {
      if (((PickMesure*)data)->idRef1 == nodes[i])
	((PickMesure*)data)->idRef1 = -1;
      if (((PickMesure*)data)->idRef2 == nodes[i])
	((PickMesure*)data)->idRef2 = -1;
      if (((PickMesure*)data)->idSelected == nodes[i])
	((PickMesure*)data)->idSelected = -1;
    }
}
static void updateListOnNodeChange(VisuData *dataObj, gpointer data)
{
  g_return_if_fail(data);

  DBG_fprintf(stderr, "Visu PickMesure: caught a node rendering"
	      " changing signal (%p).\n", data);
  drawMarkList(dataObj, ((PickMesure*)data)->storedMarks, -1);
}
static void updateListOnElementChange(VisuData *dataObj, VisuElement *ele,
				      gpointer data)
{
  g_return_if_fail(data);

  DBG_fprintf(stderr, "Visu PickMesure: caught an element rendering"
	      " changing signal (%s-%p).\n", ele->name, data);
  drawMarkList(dataObj, ((PickMesure*)data)->storedMarks, -1);
}
static void updateListOnCameraChange(VisuData *obj, OpenGLView *view _U_,
				     gpointer data _U_)
{
  GList *tmpLst;
  PickMesure *mesureData;
  struct MarkInfo_struct* mark;
  gboolean redraw;

  g_return_if_fail(data);

  mesureData = (PickMesure*)data;
  redraw = FALSE;
  for (tmpLst = mesureData->storedMarks; tmpLst; tmpLst = g_list_next(tmpLst))
    {
      mark = (struct MarkInfo_struct*)(tmpLst->data);
      if (mark->type == PICK_MESURE_MARK_LINE)
        {
          getNodes2DCoordinates(mesureData->data, mark->nIds, g_list_length(mark->nodes), mark->coord, &mark->size);
          redraw = TRUE;
        }
    }
  if (redraw)
    drawMarkList(obj, mesureData->storedMarks, 0);
}
void pickMesureSet_storeMeasures(PickMesure *mesureData, gboolean storeMeasures)
{
  g_return_if_fail(mesureData);

  DBG_fprintf(stderr, "Visu PickMesure: set store distance to %d.\n",
	      (int)storeMeasures);
  mesureData->storeMeasures = storeMeasures;
  if (storeMeasures)
    openGLText_initFontList();
}
void pickMesureSet_formatedOutput(PickMesure *mesureData, gboolean formatedOutput)
{
  g_return_if_fail(mesureData);

  DBG_fprintf(stderr, "Visu PickMesure: set formated output to %d.\n",
	      (int)formatedOutput);
  mesureData->formatFlag = formatedOutput;
}

GList* pickMesureGet_highlightedList(PickMesure *mesureData)
{
  GList *lst, *tmpLst;
  struct MarkInfo_struct* mark;

  g_return_val_if_fail(mesureData, (GList*)0);

  lst = (GList*)0;
  for (tmpLst = mesureData->storedMarks; tmpLst; tmpLst = g_list_next(tmpLst))
    {
      mark = (struct MarkInfo_struct*)(tmpLst->data);
      if (mark->type == PICK_MESURE_MARK_HIGHLIGHT)
	lst = g_list_prepend(lst, GINT_TO_POINTER(mark->idNode1));
    }
  return lst;
}
gboolean pickMesureGet_typeNode(PickMesure *mesureData, guint nodeId,
				PickMesureType *type)
{
  GList *tmpLst;
  struct MarkInfo_struct* mark;

  g_return_val_if_fail(mesureData && type, FALSE);
  
  for (tmpLst = mesureData->storedMarks; tmpLst; tmpLst = g_list_next(tmpLst))
    {
      mark = (struct MarkInfo_struct*)(tmpLst->data);
      if (mark->idNode1 == nodeId)
	switch (mark->type)
	  {
	  case PICK_MESURE_MARK_HIGHLIGHT:
	    *type = PICK_HIGHLIGHT;
	    return TRUE;
	  case PICK_MESURE_MARK_BIG_SQUARE:
	    *type = PICK_REFERENCE_1;
	    return TRUE;
	  case PICK_MESURE_MARK_SMALL_SQUARE:
	    *type = PICK_REFERENCE_2;
	    return TRUE;
	  default:
	    break;
	  }
    }
  return FALSE;
}

void pickMesureRebuild_classicalList(VisuData *dataObj)
{
  PickMesure *pickMesure;

  pickMesure = (PickMesure*)g_object_get_data(G_OBJECT(dataObj), "pickMesure_data");
  if (!pickMesure)
    return;

  DBG_fprintf(stderr, "Visu PickMesure: rebuilding classical object"
	      " list for PickMesure %p.\n", (gpointer)pickMesure);

  drawMarkList(dataObj, pickMesure->storedMarks, 1);
}
static void pickMesureRebuild_colourInvList(VisuData *dataObj)
{
  PickMesure *pickMesure;

  pickMesure = (PickMesure*)g_object_get_data(G_OBJECT(dataObj), "pickMesure_data");
  if (!pickMesure)
    return;

  DBG_fprintf(stderr, "Visu PickMesure: rebuilding colour inverse object"
	      " list for PickMesure %p.\n", (gpointer)pickMesure);

  drawMarkList(dataObj, pickMesure->storedMarks, 0);
}
gboolean pickMesureHide_highlight(PickMesure *mesure,
				  VisuData *dataObj, gboolean status)
{
  GList *tmpLst;
  struct MarkInfo_struct *mark;
  guint nHl, *hl, j;
  guint min, max;
  VisuDataIter iter;
  gboolean redraw, hide;

  g_return_val_if_fail(mesure, FALSE);

  redraw = FALSE;
  if (status)
    {
      /* We switch off all nodes except the highlighted ones. To avoid
	 to run the list of highlighted nodes to many times, we create
	 a temporary array and store the min and max indexes. */
      nHl = g_list_length(mesure->storedMarks);
      if (nHl == 0)
	return FALSE;

      visuDataIter_new(dataObj, &iter);

      hl = g_malloc(sizeof(gint) * nHl);
      nHl = 0;
      min = iter.nAllStoredNodes;
      max = 0;
      for (tmpLst = mesure->storedMarks; tmpLst; tmpLst = g_list_next(tmpLst))
	{
	  mark = (struct MarkInfo_struct*)(tmpLst->data);
	  if (mark->type == PICK_MESURE_MARK_HIGHLIGHT)
	    {
	      hl[nHl] = mark->idNode1;
	      min = MIN(min, (guint)mark->idNode1);
	      max = MAX(max, (guint)mark->idNode1);
	      nHl += 1;
	    }
	}

      visuDataIter_new(dataObj, &iter);
      for (visuDataIter_start(dataObj, &iter); iter.node;
	   visuDataIter_next(dataObj, &iter))
	if (iter.node->number >= min && iter.node->number <= max)
	  {
	    hide = TRUE;
	    for (j = 0; hide && j < nHl; j++)
	      hide = (iter.node->number != hl[j]);
	    if (hide)
	      redraw = visuNodeSet_visibility(iter.node, FALSE) || redraw;
	  }
	else
	  redraw = visuNodeSet_visibility(iter.node, FALSE) || redraw;

      g_free(hl);
    }
  else
    {
      /* We switch off only highlighted nodes. */
      for (tmpLst = mesure->storedMarks; tmpLst; tmpLst = g_list_next(tmpLst))
	{
	  mark = (struct MarkInfo_struct*)(tmpLst->data);
	  if (mark->type == PICK_MESURE_MARK_HIGHLIGHT)
	    redraw = visuNodeSet_visibility
	      (visuDataGet_nodeFromNumber(dataObj, mark->idNode1), FALSE) || redraw;
	}
    }
  return redraw;
}

/*****************************/
/* XML files for iso-values. */
/*****************************/

/*
   <pick data-mode="selected" data-info="">
     <node id="23" />
     <node id="16" />
     <distance ref="45" id="23" />
   </pick>
*/

/* Known elements. */
#define PICK_PARSER_ELEMENT_PICK  "pick"
#define PICK_PARSER_ELEMENT_NODE  "node"
#define PICK_PARSER_ELEMENT_DIST  "distance"
#define PICK_PARSER_ELEMENT_ANGL  "angle"
/* Known attributes. */
#define PICK_PARSER_ATTRIBUTES_MODE "info-mode"
#define PICK_PARSER_ATTRIBUTES_INFO "info-data"
#define PICK_PARSER_ATTRIBUTES_ID   "id"
#define PICK_PARSER_ATTRIBUTES_REF  "ref"
#define PICK_PARSER_ATTRIBUTES_REF2 "ref2"
#define PICK_PARSER_ATTRIBUTES_HLT  "highlight"

static gboolean startPick;
static DrawItem mode;
static guint info;

/* This method is called for every element that is parsed.
   The user_data must be a GList of _pick_xml. When a 'surface'
   element, a new struct instance is created and prepend in the list.
   When 'hidden-by-planes' or other qualificative elements are
   found, the first surface of the list is modified accordingly. */
static void pickXML_element(GMarkupParseContext *context _U_,
			    const gchar         *element_name,
			    const gchar        **attribute_names,
			    const gchar        **attribute_values,
			    gpointer             user_data,
			    GError             **error)
{
  GList **pickList;
  int i, n;
  guint val, val2, val3;
  gboolean highlight;

  g_return_if_fail(user_data);
  pickList = (GList **)user_data;

  DBG_fprintf(stderr, "Pick parser: found '%s' element.\n", element_name);
  if (!strcmp(element_name, PICK_PARSER_ELEMENT_PICK))
    {
      /* Initialise the pickList. */
      if (*pickList)
	{
	  g_set_error(error, G_MARKUP_ERROR, G_MARKUP_ERROR_PARSE,
		      _("DTD error: element '%s' should appear only once."),
		      PICK_PARSER_ELEMENT_PICK);
	  return;
	}
      *pickList = (GList*)0;
      startPick = TRUE;
      /* Should have 2 mandatory attributes. */
      for (i = 0; attribute_names[i]; i++)
	{
	  if (!strcmp(attribute_names[i], PICK_PARSER_ATTRIBUTES_MODE))
	    {
	      if (!strcmp(attribute_values[i], "never"))
		mode = DRAW_NEVER;
	      else if (!strcmp(attribute_values[i], "selected"))
		mode = DRAW_SELECTED;
	      else if (!strcmp(attribute_values[i], "always"))
		mode = DRAW_ALWAYS;
	      else
		{
		  g_set_error(error, G_MARKUP_ERROR, G_MARKUP_ERROR_PARSE,
			      _("DTD error: attribute '%s' has an unknown value '%s'."),
			      PICK_PARSER_ATTRIBUTES_MODE, attribute_values[i]);
		  return;
		}
	    }
	  else if (!strcmp(attribute_names[i], PICK_PARSER_ATTRIBUTES_INFO))
	    {
	      n = sscanf(attribute_values[i], "%u", &info);
	      if (n != 1 || info < 1)
		{
		  g_set_error(error, G_MARKUP_ERROR, G_MARKUP_ERROR_PARSE,
			      _("DTD error: attribute '%s' has an unknown value '%s'."),
			      PICK_PARSER_ATTRIBUTES_INFO, attribute_values[i]);
		  return;
		}
	    }
	}
    }
  else if (!strcmp(element_name, PICK_PARSER_ELEMENT_NODE))
    {
      if (!startPick)
	{
	  g_set_error(error, G_MARKUP_ERROR, G_MARKUP_ERROR_INVALID_CONTENT,
		      _("DTD error: parent element '%s' of element '%s' is missing."),
		      PICK_PARSER_ELEMENT_PICK, PICK_PARSER_ELEMENT_NODE);
	  return;
	}
      highlight = FALSE;
      /* We parse the attributes. */
      for (i = 0; attribute_names[i]; i++)
	{
	  if (!strcmp(attribute_names[i], PICK_PARSER_ATTRIBUTES_ID))
	    {
	      n = sscanf(attribute_values[i], "%u", &val);
	      if (n != 1)
		{
		  g_set_error(error, G_MARKUP_ERROR, G_MARKUP_ERROR_PARSE,
			      _("DTD error: attribute '%s' has an unknown value '%s'."),
			      PICK_PARSER_ATTRIBUTES_ID, attribute_values[i]);
		  return;
		}
	    }
	  else if (!strcmp(attribute_names[i], PICK_PARSER_ATTRIBUTES_HLT))
	    highlight = (!strcmp(attribute_values[i], "yes") ||
			 !strcmp(attribute_values[i], "Yes"));
	}
      if (highlight)
	*pickList = g_list_prepend(*pickList, GINT_TO_POINTER(PICK_HIGHLIGHT));
      else
	*pickList = g_list_prepend(*pickList, GINT_TO_POINTER(PICK_SELECTED));
      *pickList = g_list_prepend(*pickList, GINT_TO_POINTER(val));
    }
  else if (!strcmp(element_name, PICK_PARSER_ELEMENT_DIST))
    {
      if (!startPick)
	{
	  g_set_error(error, G_MARKUP_ERROR, G_MARKUP_ERROR_INVALID_CONTENT,
		      _("DTD error: parent element '%s' of element '%s' is missing."),
		      PICK_PARSER_ELEMENT_PICK, PICK_PARSER_ELEMENT_DIST);
	  return;
	}

      /* We parse the attributes. */
      for (i = 0; attribute_names[i]; i++)
	{
	  if (!strcmp(attribute_names[i], PICK_PARSER_ATTRIBUTES_ID))
	    {
	      n = sscanf(attribute_values[i], "%u", &val);
	      if (n != 1)
		{
		  g_set_error(error, G_MARKUP_ERROR, G_MARKUP_ERROR_PARSE,
			      _("DTD error: attribute '%s' has an unknown value '%s'."),
			      PICK_PARSER_ATTRIBUTES_ID, attribute_values[i]);
		  return;
		}
	    }
	  else if (!strcmp(attribute_names[i], PICK_PARSER_ATTRIBUTES_REF))
	    {
	      n = sscanf(attribute_values[i], "%u", &val2);
	      if (n != 1)
		{
		  g_set_error(error, G_MARKUP_ERROR, G_MARKUP_ERROR_PARSE,
			      _("DTD error: attribute '%s' has an unknown value '%s'."),
			      PICK_PARSER_ATTRIBUTES_REF, attribute_values[i]);
		  return;
		}
	    }
	}
      *pickList = g_list_prepend(*pickList, GINT_TO_POINTER(PICK_REFERENCE_1));
      *pickList = g_list_prepend(*pickList, GINT_TO_POINTER(val2));
      *pickList = g_list_prepend(*pickList, GINT_TO_POINTER(val));
    }
  else if (!strcmp(element_name, PICK_PARSER_ELEMENT_ANGL))
    {
      if (!startPick)
	{
	  g_set_error(error, G_MARKUP_ERROR, G_MARKUP_ERROR_INVALID_CONTENT,
		      _("DTD error: parent element '%s' of element '%s' is missing."),
		      PICK_PARSER_ELEMENT_PICK, PICK_PARSER_ELEMENT_ANGL);
	  return;
	}

      /* We parse the attributes. */
      for (i = 0; attribute_names[i]; i++)
	{
	  if (!strcmp(attribute_names[i], PICK_PARSER_ATTRIBUTES_ID))
	    {
	      n = sscanf(attribute_values[i], "%u", &val);
	      if (n != 1)
		{
		  g_set_error(error, G_MARKUP_ERROR, G_MARKUP_ERROR_PARSE,
			      _("DTD error: attribute '%s' has an unknown value '%s'."),
			      PICK_PARSER_ATTRIBUTES_ID, attribute_values[i]);
		  return;
		}
	    }
	  else if (!strcmp(attribute_names[i], PICK_PARSER_ATTRIBUTES_REF))
	    {
	      n = sscanf(attribute_values[i], "%u", &val2);
	      if (n != 1)
		{
		  g_set_error(error, G_MARKUP_ERROR, G_MARKUP_ERROR_PARSE,
			      _("DTD error: attribute '%s' has an unknown value '%s'."),
			      PICK_PARSER_ATTRIBUTES_REF, attribute_values[i]);
		  return;
		}
	    }
	  else if (!strcmp(attribute_names[i], PICK_PARSER_ATTRIBUTES_REF2))
	    {
	      n = sscanf(attribute_values[i], "%u", &val3);
	      if (n != 1)
		{
		  g_set_error(error, G_MARKUP_ERROR, G_MARKUP_ERROR_PARSE,
			      _("DTD error: attribute '%s' has an unknown value '%s'."),
			      PICK_PARSER_ATTRIBUTES_REF2, attribute_values[i]);
		  return;
		}
	    }
	}
      *pickList = g_list_prepend(*pickList, GINT_TO_POINTER(PICK_REFERENCE_2));
      *pickList = g_list_prepend(*pickList, GINT_TO_POINTER(val3));
      *pickList = g_list_prepend(*pickList, GINT_TO_POINTER(val2));
      *pickList = g_list_prepend(*pickList, GINT_TO_POINTER(val));
    }
  else if (startPick)
    {
      /* We silently ignore the element if pickList is unset, but
	 raise an error if pickList has been set. */
      g_set_error(error, G_MARKUP_ERROR, G_MARKUP_ERROR_UNKNOWN_ELEMENT,
		  _("Unexpected element '%s'."), element_name);
    }
}

/* Check when a element is closed that everything required has been set. */
static void pickXML_end(GMarkupParseContext *context _U_,
			const gchar         *element_name,
			gpointer             user_data _U_,
			GError             **error _U_)
{
  if (!strcmp(element_name, PICK_PARSER_ELEMENT_PICK))
    startPick = FALSE;
}

/* What to do when an error is raised. */
static void pickXML_error(GMarkupParseContext *context _U_,
			  GError              *error,
			  gpointer             user_data)
{
  DBG_fprintf(stderr, "Pick parser: error raised '%s'.\n", error->message);
  g_return_if_fail(user_data);

  /* We free the current list of pick. */
  g_list_free(*(GList**)user_data);
}

PickMesure* pickMesureParse_XMLFile(const gchar* filename, VisuData *data,
				    GList **infos, DrawItem *drawingMode,
				    guint *drawingInfos, GError **error)
{
  GMarkupParseContext* xmlContext;
  GMarkupParser parser;
  gboolean status;
  gchar *buffer;
  gsize size;
  PickMesure *mesure;
  GList *tmpLst;
  int id;

  g_return_val_if_fail(IS_VISU_DATA_TYPE(data) && filename, (PickMesure*)0);
  g_return_val_if_fail(infos && drawingMode && drawingInfos, (PickMesure*)0);

  mesure = g_object_get_data(G_OBJECT(data), "pickMesure_data");
  g_return_val_if_fail(mesure, (PickMesure*)0);

  buffer = (gchar*)0;
  if (!g_file_get_contents(filename, &buffer, &size, error))
    return FALSE;

  /* Create context. */
  *infos = (GList*)0;
  parser.start_element = pickXML_element;
  parser.end_element   = pickXML_end;
  parser.text          = NULL;
  parser.passthrough   = NULL;
  parser.error         = pickXML_error;
  xmlContext = g_markup_parse_context_new(&parser, 0, infos, NULL);

  /* Parse data. */
  startPick = FALSE;
  status = g_markup_parse_context_parse(xmlContext, buffer, size, error);

  /* Free buffers. */
  g_markup_parse_context_free(xmlContext);
  g_free(buffer);

  if (!status)
    return (PickMesure*)0;

  if (!*infos)
    {
      *error = g_error_new(G_MARKUP_ERROR, G_MARKUP_ERROR_EMPTY,
			  _("No picked node found."));
      return (PickMesure*)0;
    }

  /* Need to reverse the list since elements have been prepended. */
  *infos        = g_list_reverse(*infos);
  *drawingMode  = mode;
  *drawingInfos = info;

  /* Update the current pickMesure. */
  tmpLst = *infos;
  while(tmpLst)
    {
      if (GPOINTER_TO_INT(tmpLst->data) == PICK_SELECTED)
	tmpLst = g_list_next(tmpLst);
      else if (GPOINTER_TO_INT(tmpLst->data) == PICK_HIGHLIGHT)
	{
	  tmpLst = g_list_next(tmpLst);
	  id = GPOINTER_TO_INT(tmpLst->data) - 1;
	  /* We silently ignore out of bound values. */
	  if (visuDataGet_nodeFromNumber(data, id))
	    pickMesureSet_highlight(mesure, GPOINTER_TO_INT(tmpLst->data) - 1, TRUE);
	}
      else if (GPOINTER_TO_INT(tmpLst->data) == PICK_REFERENCE_1)
	{
	  tmpLst = g_list_next(tmpLst);
	  id  = GPOINTER_TO_INT(tmpLst->data) - 1;
	  /* We silently ignore out of bound values. */
	  if (visuDataGet_nodeFromNumber(data, id))
	    pickMesureSet_pickNode(mesure, id, PICK_REFERENCE_1);
	  tmpLst = g_list_next(tmpLst);
	  id = GPOINTER_TO_INT(tmpLst->data) - 1;
	  /* We silently ignore out of bound values. */
	  if (visuDataGet_nodeFromNumber(data, id))
	    pickMesureSet_pickNode(mesure, id, PICK_SELECTED);
	}
      else if (GPOINTER_TO_INT(tmpLst->data) == PICK_REFERENCE_2)
	{
	  tmpLst = g_list_next(tmpLst);
	  id  = GPOINTER_TO_INT(tmpLst->data) - 1;
	  /* We silently ignore out of bound values. */
	  if (visuDataGet_nodeFromNumber(data, id))
	    pickMesureSet_pickNode(mesure, id, PICK_REFERENCE_2);
	  tmpLst = g_list_next(tmpLst);
	  id  = GPOINTER_TO_INT(tmpLst->data) - 1;
	  /* We silently ignore out of bound values. */
	  if (visuDataGet_nodeFromNumber(data, id))
	    pickMesureSet_pickNode(mesure, id, PICK_REFERENCE_1);
	  tmpLst = g_list_next(tmpLst);
	  id = GPOINTER_TO_INT(tmpLst->data) - 1;
	  /* We silently ignore out of bound values. */
	  if (visuDataGet_nodeFromNumber(data, id))
	    pickMesureSet_pickNode(mesure, id, PICK_SELECTED);
	}
      else
	{
	  g_error("Should not be here!");
	}
      
      tmpLst = g_list_next(tmpLst);
    }
  pickMesureRebuild_classicalList(data);

  return mesure;
}

gboolean pickMesureExport_XMLFile(const gchar* filename, VisuData *data,
				  int *nodes, DrawItem drawingMode,
				  guint drawingInfos, GError **error)
{
  gboolean valid, set;
  GString *output;
  int i;
  PickMesure *mesure;
  char *modes[] = {"never", "selected", "always"};
  GList *tmpLst;
  struct MarkInfo_struct *mark;

  g_return_val_if_fail(IS_VISU_DATA_TYPE(data) && filename, FALSE);

  mesure = g_object_get_data(G_OBJECT(data), "pickMesure_data");
  g_return_val_if_fail(mesure, FALSE);

  /*
   <pick data-mode="selected" data-info="">
     <node id="23" />
     <node id="16" highlight="yes" />
     <distance ref="45" id="23" />
     <angle ref="45" ref2="65" id="23" />
   </pick>
  */

  output = g_string_new("  <pick");
  g_string_append_printf(output, " data-mode=\"%s\" data-infos=\"%d\">\n",
			 modes[drawingMode], drawingInfos);
  if (nodes)
    for (i = 0; nodes[i] >= 0; i++)
      {
	set = FALSE;
	/* We print them only if they are not part of marks. */
	tmpLst = mesure->storedMarks;
	while(tmpLst)
	  {
	    mark = (struct MarkInfo_struct*)(tmpLst->data);
	    set = set || ( (mark->type == PICK_MESURE_MARK_DISTANCE &&
			    (guint)nodes[i] == mark->idNode2) ||
			   (mark->type == PICK_MESURE_MARK_HIGHLIGHT &&
			    (guint)nodes[i] == mark->idNode1) );
	    tmpLst = g_list_next(tmpLst);
	  }
	if (!set)
	  g_string_append_printf(output, "    <node id=\"%d\" />\n", nodes[i] + 1);
      }
  tmpLst = mesure->storedMarks;
  while(tmpLst)
    {
      mark = (struct MarkInfo_struct*)(tmpLst->data);
      if (mark->type == PICK_MESURE_MARK_DISTANCE)
	g_string_append_printf(output, "    <distance ref=\"%d\" id=\"%d\" />\n",
			       mark->idNode1 + 1, mark->idNode2 + 1);
      else if (mark->type == PICK_MESURE_MARK_ANGLE)
	g_string_append_printf(output,
			       "    <angle ref=\"%d\" ref2=\"%d\" id=\"%d\" />\n",
			       mark->idNode1 + 1, mark->idNode2 + 1,
			       mark->idNode3 + 1);
      else if (mark->type == PICK_MESURE_MARK_HIGHLIGHT)
	g_string_append_printf(output, "    <node id=\"%d\" highlight=\"yes\" />\n",
			       mark->idNode1 + 1);

      tmpLst = g_list_next(tmpLst);
    }
  g_string_append(output, "  </pick>");

  valid = visuToolsSubstitute_XML(output, filename, "pick", error);
  if (!valid)
    {
      g_string_free(output, TRUE);
      return FALSE;
    }
  
  valid = g_file_set_contents(filename, output->str, -1, error);
  g_string_free(output, TRUE);
  return valid;
}
