/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#ifndef VISU_PICKMESURE_H
#define VISU_PICKMESURE_H

#include <glib.h>
#include "visu_nodes.h"
#include "visu_data.h"
#include "extensions/extInfos.h"

/* Pick methods to mesure distance and angles. */

/**
 * PickMesure:
 *
 *Short way to address #PickMesure_struct objects.
 */
typedef struct _PickMesure PickMesure;

/**
 * PickMesureType:
 * @PICK_SELECTED: a node has been selected with simple clic ;
 * @PICK_HIGHLIGHT: a node has been highlighted ;
 * @PICK_UNHIGHLIGHT: a node has been unhighlighted ;
 * @PICK_REFERENCE_1: a node has been selected has a first reference ;
 * @PICK_REFERENCE_2: a node has been selected has a second reference
 * ;
 * @PICK_REGION: a list of nodes as been selected ;
 * @PICK_DRAG_START: the selected node has been picked for a drag ;
 * @PICK_DRAG_MOVE: the selected node is moving ;
 * @PICK_DRAG_STOP: the selected node has just been released from a drag.
 *
 * These values are used with pickMesureSet_pickNode() to specify that
 * the selected node is a simple one or a reference.
 */
typedef enum
  {
    PICK_SELECTED,
    PICK_HIGHLIGHT,
    PICK_UNHIGHLIGHT,
    PICK_REFERENCE_1,
    PICK_REFERENCE_2,
    PICK_INFORMATION,
    PICK_REGION,
    PICK_DRAG_START,
    PICK_DRAG_MOVE,
    PICK_DRAG_STOP
  } PickMesureType;

/**
 * pickMesureSet_pickNode:
 * @mesureData: a #PickMesure structure ;
 * @nodeId: a node id (or -1 to unselect current one) ;
 * @type: a value in #PickMesureType.
 *
 * Update the information of measurement when the @nodeId is
 * selected. See #PickMesureType for further information on the
 * different kind of possible selection. For this method, only
 * #PICK_SELECTED, #PICK_REFERENCE_1 and #PICK_REFERENCE_2 are
 * possible. See pickMesureGet_newsAvailable() to catch possible
 * changes in a callback.
 *
 * Strings are created to represent
 * the return of the mesure (if this has been activated using
 * pickMesureSet_formatedOutput()). These strings are available with a
 * call to pickMesureGet_infos() and pickMesureGet_errors().
 */
void pickMesureSet_pickNode(PickMesure *mesureData, gint nodeId, PickMesureType type);
/**
 * pickMesureSet_pickRegion:
 * @mesureData: a #PickMesure structure ;
 * @nodeIds: a list of node ids.
 *
 * Update the information of measurement when a region containing
 * @nodeIds is selected. See pickMesureGet_newsAvailable() to catch possible
 * changes in a callback. 
 */
void pickMesureSet_pickRegion(PickMesure *mesureData, GList *nodeIds);
/**
 * pickMesureSet_dragStart:
 * @mesureData: a #PickMesure structure ;
 * @nodeId: a node id.
 *
 * Update the information of measurement when drag is begun on node @nodeId. 
 */
void pickMesureSet_dragStart(PickMesure *mesureData, guint nodeId);
/**
 * pickMesureSet_dragStop:
 * @mesureData: a #PickMesure structure.
 *
 * Update the information of measurement when drag is stopped.
 */
void pickMesureSet_dragStop(PickMesure *mesureData);
/**
 * pickMesureSet_dragMove:
 * @mesureData: a #PickMesure structure ;
 * @dx: a variation in cartesian coordinates ;
 * @dy: a variation in cartesian coordinates ;
 * @dz: a variation in cartesian coordinates.
 *
 * Update the information of measurement when drag is done with the
 * given translation vector. 
 */
void pickMesureSet_dragMove(PickMesure *mesureData, float dx, float dy, float dz);
/**
 * pickMesureGet_drag:
 * @mesureData: a #PickMesure structure.
 *
 * Retrieve the translation vector in a drag session.
 *
 * Returns: a three float vector owned by V_Sim.
 */
float* pickMesureGet_drag(PickMesure *mesureData);
/**
 * pickMesureGet_newsAvailable:
 * @mesureData: a #PickMesure structure that contains the informations
 *              about picked nodes.
 * @type: if not NULL, contains on output the kind of news available.
 * 
 * When a pick is done, maybe nothing interesting happen, get informed
 * with this method.
 *
 * Returns: TRUE if values stored in the given @mesureData have changed.
 */
gboolean pickMesureGet_newsAvailable(PickMesure *mesureData, PickMesureType *type);



/**
 * pickMesureSet_highlight:
 * @mesureData: a #PickMesure structure ;
 * @idNode: the node to highlight ;
 * @status: switch on or off the highlight.
 *
 * Put an highlight (or remove it) on a given node. The OpenGL area
 * should be redraw after the call. After a call to this routine, the
 * calling one should call pickMesureRebuild_classicalList().
 *
 * Returns: TRUE if the status of highlighting for the given node has changed.
 */
gboolean pickMesureSet_highlight(PickMesure *mesureData, guint idNode,
				 gboolean status);
/**
 * pickMesureRebuild_classicalList:
 * @data: a #VisuData object.
 *
 * Rebuild the list of normal coloured marks.
 */
void pickMesureRebuild_classicalList(VisuData *dataObj);
/**
 * pickMesureHide_highlight:
 * @mesure: a #PickMesure structure ;
 * @dataObj: a #VisuData object ;
 * @satatus: a boolean.
 *
 * Change the visibility status of the highlighted nodes listed in
 * @mesure to the @status value.
 *
 * Returns: TRUE if some visibility status have been changed.
 */
gboolean pickMesureHide_highlight(PickMesure *mesure,
				  VisuData *dataObj, gboolean status);

/**
 * pickMesureGet_infos:
 * @mesureData: a #PickMesure structure that contains the informations
 *              about picked nodes.
 *
 * When a pick session is running with the format flag (see
 * pickMesureSet_formatedOutput() for deatils), calls to
 * pickMesureSet_selection() creates strings that represent the values of the mesure.
 * These values are retrievable with this method.
 *
 * Returns: a string (own by V_Sim, should not be freed).
 */
gchar* pickMesureGet_infos(PickMesure *mesureData);
/**
 * pickMesureGet_errors:
 * @mesureData: a #PickMesure structure that contains the informations
 *              about picked nodes.
 *
 * When a pick session is running with the format flag (see
 * pickMesureSet_formatedOutput() for deatils), calls to
 * pickMesureSet_selection() creates strings that represent the values of the mesure.
 * If error occurs, they are retrievable with this method.
 *
 * Returns: a string (own by V_Sim, should not be freed).
 */
gchar* pickMesureGet_errors(PickMesure *mesureData);
/**
 * pickMesureRemove_allMarks:
 * @mesureData: a #PickMesure structure that contains the informations
 *              about Nodes with marks that need to be erased.
 *
 * Erase all marks corresponding to given object.
 *
 * Returns: TRUE if the "OpenGLAskForReDraw" signal show be emitted.
 */
gboolean pickMesureRemove_allMarks(PickMesure *mesureData);
/**
 * pickMesureRemove_allHighlights:
 * @mesureData: a #PickMesure structure.
 *
 * It empties the list of all highlights.
 *
 * Returns: TRUE if the redraw signal should be emitted.
 */
gboolean pickMesureRemove_allHighlights(PickMesure *mesureData);
/**
 * pickMesureGet_selectedNode:
 * @mesureData: a #PickMesure structure that contains the informations
 *              about picked nodes.
 * 
 * The selected node is a node selected by a normal pick, i.e. neither
 * a first nor a second reference.
 *
 * Returns: the last selected node, or NULL if none.
 */
VisuNode* pickMesureGet_selectedNode(PickMesure *mesureData);
/**
 * pickMesureGet_highlightedNode:
 * @mesureData: a #PickMesure structure that contains the informations
 *              about picked nodes.
 * 
 * The highlighted node is a node selected by a highlight pick.
 *
 * Returns: the last highlighted node, or NULL if none.
 */
VisuNode* pickMesureGet_highlightedNode(PickMesure *mesureData);
/**
 * pickMesureGet_highlightedList:
 * @mesureData: a #PickMesure structure that contains the informations
 *              about picked nodes.
 * 
 * The highlighted nodes are nodes selected by a highlight pick. This
 * routine can get the list of stored highlighted nodes.
 *
 * Returns: all highlighted nodes as a list of guint, or NULL if none.
 */
GList* pickMesureGet_highlightedList(PickMesure *mesureData);
/**
 * pickMesureGet_typeNode:
 * @mesureData: a #PickMesure structure that contains the informations
 *              about picked nodes.
 * @nodeId: a node number.
 * @type: a location to store the type.
 * 
 * The nodes can have differen marks. The available return values are
 * restricted to #PICK_HIGHLIGHT, #PICK_SELECTED, #PICK_REFERENCE_1
 * and #PICK_REFERENCE_2.
 *
 * Returns: TRUE if the node has indeed a mark.
 */
gboolean pickMesureGet_typeNode(PickMesure *mesureData, guint nodeId,
				PickMesureType *type);
/**
 * pickMesureGet_firstReference:
 * @mesureData: a #PickMesure structure that contains the informations
 *              about picked nodes.
 * 
 * The first reference is a store node.
 *
 * Returns: the node that has been picked as first reference, or NULL if none.
 */
VisuNode* pickMesureGet_firstReference(PickMesure *mesureData);
VisuNode* pickMesureGet_secondReference(PickMesure *mesureData);
/**
 * pickMesureGet_regionNodes:
 * @mesureData: a #PickMesure structure that contains the informations
 *              about picked nodes.
 * 
 * Create a list of #VisuNode from a selected region. There should be
 * a selected region, see pickMesureGet_newsAvailable().
 *
 * Returns: a newly created list of selected nodes. Should be freed
 * with g_list_free() after use.
 */
GList* pickMesureGet_regionNodes(PickMesure *mesureData);
/**
 * pickMesureGet_regionNNodes:
 * @mesureData: a #PickMesure structure that contains the informations
 *              about picked nodes.
 * 
 * Return only the number of selected nodes in a region, without
 * actually creating the list. See pickMesureGet_regionNodes() to get
 * this list.
 *
 * Returns: the number of selected nodes in a region.
 */
int pickMesureGet_regionNNodes(PickMesure *mesureData);
/**
 * pickMesureRemove_allMeasures:
 * @mesureData: a #PickMesure structure that contains the informations
 *              about picked nodes.
 * 
 * Call this method to erase all previously marked distances.
 *
 * Returns: TRUE if the "OpenGLAskForReDraw" signal show be emitted.
 */
gboolean pickMesureRemove_allMeasures(PickMesure *mesureData);
/**
 * pickMesureSet_storeMeasures:
 * @mesureData: a #PickMesure structure that contains the informations
 *              about picked nodes ;
 * @storeMeasures: a boolean.
 *
 * If this flag is TRUE, distance marks are used. If FALSE, only future distance
 * marks are disable. Previously stored distance marks are kept, use
 * pickMesureRemove_allMeasures() to remove them all.
 */
void pickMesureSet_storeMeasures(PickMesure *mesureData, gboolean storeMeasures);
/**
 * pickMesureSet_formatedOutput:
 * @mesureData: a #PickMesure structure that contains the informations
 *              about picked nodes ;
 * @formatedOutput: a boolean.
 *
 * If this flag is TRUE, one can retrieve some formatted output using the getPickMesureInfos()
 * and getPickMesureErrors() methods.
 */
void pickMesureSet_formatedOutput(PickMesure *mesureData, gboolean formatedOutput);
/**
 * pickMesureUpdate:
 * @newData: a #VisuData object (can be NULL) ;
 * @oldData: a #VisuData object (can be NULL).
 *
 * This method try to retrieve a #PickMesure object associated to each 
 * given #VisuData object and to update the marks and distances on
 * the new from the values of the old.
 */
void pickMesureUpdate(VisuData *newData, VisuData *oldData);

PickMesure* pickMesureParse_XMLFile(const gchar* filename, VisuData *data,
				    GList **infos, DrawItem *drawingMode,
				    guint *drawingInfos, GError **error);

gboolean pickMesureExport_XMLFile(const gchar* filename, VisuData *data,
				  int *nodes, DrawItem drawingMode,
				  guint drawingInfos, GError **error);

gboolean pickMesureSet_nodeMeasurements(PickMesure *mesureData,
					guint nodeId, float range,
					gboolean set);
gboolean pickMesureGet_active(PickMesure *mesureData, guint nodeId);

/**
 * initPick_module:
 *
 * Call by V_Sim on startup, don't use it.
 */
void initPick_module();


#endif
