/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#ifndef VISU_DATA_H
#define VISU_DATA_H

#include <glib.h>
#include <glib-object.h>

#include "visu_tools.h"
#include "visu_elements.h"
#include "visu_nodes.h"
#include "openGLFunctions/view.h"
#include "coreTools/toolPhysic.h"
#include "coreTools/toolFileFormat.h"

G_BEGIN_DECLS

/**
 * VISU_DATA_TYPE:
 *
 * return the type of #VisuData.
 */
#define VISU_DATA_TYPE	     (visu_data_get_type ())
/**
 * VISU_DATA:
 * @obj: a #GObject to cast.
 *
 * Cast the given @obj into #VisuData type.
 */
#define VISU_DATA(obj)	     (G_TYPE_CHECK_INSTANCE_CAST(obj, VISU_DATA_TYPE, VisuData))
/**
 * VISU_DATA_CLASS:
 * @klass: a #GObjectClass to cast.
 *
 * Cast the given @klass into #VisuDataClass.
 */
#define VISU_DATA_CLASS(klass)    (G_TYPE_CHECK_CLASS_CAST(klass, VISU_DATA_TYPE, VisuDataClass))
/**
 * IS_VISU_DATA_TYPE:
 * @obj: a #GObject to test.
 *
 * Test if the given @ogj is of the type of #VisuData object.
 */
#define IS_VISU_DATA_TYPE(obj)    (G_TYPE_CHECK_INSTANCE_TYPE(obj, VISU_DATA_TYPE))
/**
 * IS_VISU_DATA_CLASS:
 * @klass: a #GObjectClass to test.
 *
 * Test if the given @klass is of the type of #VisuDataClass class.
 */
#define IS_VISU_DATA_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE(klass, VISU_DATA_TYPE))
/**
 * VISU_DATA_GET_CLASS:
 * @obj: a #GObject to get the class of.
 *
 * It returns the class of the given @obj.
 */
#define VISU_DATA_GET_CLASS(obj)  (G_TYPE_INSTANCE_GET_CLASS(obj, VISU_DATA_TYPE, VisuDataClass))

/**
 * VisuDataPrivate:
 *
 * Private fields for #VisuData objects.
 */
typedef struct _VisuDataPrivate VisuDataPrivate;
/**
 * VisuData:
 * @parent: an object to inherit from (NULL here).
 * @privateDt: a pointer to the private data.
 * @ntype: number of #VisuElement used in this object.
 * @fromVisuElementToInt: give the number of the #VisuElement when
 *                        its pointer is known.
 * @fromIntToVisuElement: (array) (element-type VisuElement*): give the pointer of a #VisuElement when
 *                        its number is known.
 *
 * This structure describes a #VisuData object.
 */
struct _VisuData
{
  GObject parent;

  /* Number of VisuElements used by these data. */
  unsigned int ntype;
  /* This hashtable gives access to the good array  of VisuNodes
     when one has the VisuElement. */
  GHashTable *fromVisuElementToInt;
  /* This is a corresponding table from the nodes array indice
     to the associated VisuElement. */
  VisuElement **fromIntToVisuElement;

  VisuDataPrivate *privateDt;
};
typedef struct _VisuData VisuData;

/**
 * VisuDataClass:
 *
 * A short way to identify #_VisuDataClass structure.
 */
typedef struct _VisuDataClass VisuDataClass;

/**
 * VisuDataColorFunc:
 * @visuData: a pointer to the calling object ;
 * @ele: a #VisuElement ;
 * @node: a #VisuNode ;
 * @rgba: (in) (array fixed-size=4): an 4 allocated float area to store the return values.
 *
 * This prototype is used to specify an optional method
 * to associate a color with external values to each node.
 */
typedef void (*VisuDataColorFunc)(VisuData *visuData, float rgba[4], VisuElement *ele, VisuNode* node);
/**
 * VisuDataScalingFunc:
 * @visuData: a pointer to the calling object ;
 * @node: a #VisuNode ;
 *
 * Interface for routine that need to rescale @node before drawing
 * them.
 *
 * Returns: the scaling factor.
 */
typedef float (*VisuDataScalingFunc)(VisuData *visuData, VisuNode* node);

/**
 * visu_data_get_type:
 *
 * This method returns the type of #VisuData, use VISU_DATA_TYPE instead.
 *
 * Returns: the type of #VisuData.
 */
GType visu_data_get_type(void);

VisuData* visu_data_new(void);
VisuData* visu_data_newWithSize(guint w, guint h);
VisuData* visu_data_newWithView(VisuOpenGLView *view, gboolean autoAdjust);



int visu_data_getObjectList(VisuData *data);

/**
 * EXT_VISU_DATA_ID:
 *
 * The id used to identify this extension, see
 * visuExtensions_rebuildList() for instance.
 */
#define EXT_VISU_DATA_ID "AllElements"


enum visu_data_boxVector
  {
    VISU_DATA_BOX_DXX,
    VISU_DATA_BOX_DYX,
    VISU_DATA_BOX_DYY,
    VISU_DATA_BOX_DZX,
    VISU_DATA_BOX_DZY,
    VISU_DATA_BOX_DZZ
  };

/**
 * VisuDataBoxBoundaries:
 * @VISU_DATA_BOX_PERIODIC: the full 3D periodicity ;
 * @VISU_DATA_BOX_SURFACE_XY: the Z axis is a free axis ;
 * @VISU_DATA_BOX_SURFACE_YZ: the X axis is a free axis ;
 * @VISU_DATA_BOX_SURFACE_ZX: the Y axis is a free axis ;
 * @VISU_DATA_BOX_FREE: the system is isolated.
 *
 * This describes the periodicity of the bounding box in the three directions.
 */
typedef enum
  {
    VISU_DATA_BOX_PERIODIC,
    VISU_DATA_BOX_SURFACE_XY,
    VISU_DATA_BOX_SURFACE_YZ,
    VISU_DATA_BOX_SURFACE_ZX,
    VISU_DATA_BOX_FREE
  } VisuDataBoxBoundaries;



int visu_data_setPopulation(VisuData *data, unsigned int nbOfTypes,
			     unsigned int* nbOfNodesPerVisuElement,
			     VisuElement **visu_element_Used);
gboolean visu_data_allocatePopulation(VisuData *data,
                                      GArray *nNodesPerElement,
                                      GArray *elements);
gboolean visu_data_allocatePopulationByNames(VisuData *data,
                                             GArray *nNodesPerElement,
                                             GArray *elementNames);

void visu_data_freePopulation(VisuData *data);

void visu_data_addFile(VisuData *data, gchar* file, int kind, ToolFileFormat *format);

void visu_data_removeAllFiles(VisuData *data);

gchar* visu_data_getFile(VisuData *data, int kind, ToolFileFormat **format);
gchar* visu_data_getFilesAsLabel(VisuData *data);

void visu_data_setFileCommentary(VisuData *data, gchar* commentary, gint iSet);

gchar* visu_data_getFileCommentary(VisuData *data, gint iSet);

void visu_data_setNSubset(VisuData *data, int nSet);

int visu_data_getNSubset(VisuData *data);

void visu_data_setISubset(VisuData *data, int iSet);

int visu_data_getISubset(VisuData *data);

gboolean visu_data_compareElements(VisuData *data1, VisuData *data2);

void visu_data_setChangeElementFlag(VisuData *data, gboolean changeElement);

gboolean visu_data_getChangeElementFlag(VisuData *data);

float visu_data_getAllElementExtens(VisuData *dataObj);

VisuNode* visu_data_addNodeFromElement(VisuData *data, VisuElement *ele,
                                       float xyz[3], gboolean reduced,
                                       gboolean emitSignal);
VisuNode* visu_data_addNodeFromElementName(VisuData *data, gchar *name,
                                           float xyz[3], gboolean reduced,
                                           gboolean emitSignal);
VisuNode* visu_data_addNodeFromIndex(VisuData *data, unsigned int position,
                                     float xyz[3], gboolean reduced, gboolean emitSignal);

VisuNode* visu_data_getNodeFromNumber(VisuData *data, unsigned int number);

VisuNodeArray* visu_data_getNodeArray(VisuData *data);

void visu_data_setColorFunc(VisuData *data, VisuDataColorFunc func);
gboolean visu_data_hasUserColorFunc(VisuData *data);
gboolean visu_data_getUserColor(VisuData *data, VisuElement *ele,
				VisuNode *node, float rgba[4]);

float* visu_data_getXYZtranslation(VisuData* data);

gboolean visu_data_setXYZtranslation(VisuData* data, float xyz[3]);

void visu_data_convertXYZToReduced(VisuData *data, GArray *xyz,
				   float *u, float *v, float *w);
void visu_data_convertReducedToXYZ(VisuData *data, GArray *red,
				   float *x, float *y, float *z);
void visu_data_convertXYZtoBoxCoordinates(VisuData *data, float boxCoord[3], float xyz[3]);
void visu_data_convertBoxCoordinatestoXYZ(VisuData *data, float xyz[3], float boxCoord[3]);
void visu_data_convertFullCartesianToSpherical(VisuData *data, float sph[3], float xyz[3]);


void visu_data_getBoxMatrix(VisuData *data, float matrix[3][3]);
void visu_data_getBoxMatrixD0(VisuData *data, double matrix[3][3]);

void visu_data_setBoxGeometry(VisuData *data, double geometry[6],
                              VisuDataBoxBoundaries bc);
gboolean visu_data_setBoxGeometryFull(VisuData *data, double geometry[3][3],
                                      VisuDataBoxBoundaries bc);
void visu_data_applyBoxGeometry(VisuData *data);

float visu_data_getBoxGeometry(VisuData *data, int vector);

gboolean visu_data_getNodeBoxFromNumber(VisuData *data, guint nodeId, int nodeBox[3]);

gboolean visu_data_getNodeBoxFromCoord(VisuData *data, float xcart[3], int nodeBox[3]);

void visu_data_getBoxVertices(VisuData *data, float v[8][3], gboolean withExtension);

void visu_data_getBoxSpan(VisuData *data, float span[2]);
ToolUnits visu_data_getUnit(VisuData *data);
gboolean visu_data_setUnit(VisuData *data, ToolUnits unit);
VisuDataBoxBoundaries visu_data_getBoundaryConditions(VisuData *data);
void visu_data_setTightBox(VisuData *data);

gboolean visu_data_constrainedElementInTheBox(VisuData *data, VisuElement *element);

gboolean visu_data_constrainedInTheBox(VisuData *data);

gboolean visu_data_constrainedFree(VisuData *data);

gboolean visu_data_getTranslationStatus(VisuData *data);

gboolean visu_data_replicate(VisuData *data, float extension[3], gboolean *rebuild);

gboolean visu_data_restore(VisuData *data);

void visu_data_getExtension(VisuData *dataObj, float extension[3]);
float* visu_data_getBoxExtens(VisuData *dataObj);
void visu_data_getBoxLengths(VisuData *dataObj, float *ext1, float *ext2);
void visu_data_getBoxCentre(VisuData *dataObj, float centre[3]);


void visu_data_getNodeCoordinates(VisuData *data, VisuNode *node, float *x, float *y, float *z);
void visu_data_getNodePosition(VisuData *data, VisuNode *node, float coord[3]);


void visu_data_getReducedNodePosition(VisuData *data, VisuNode *node, float red[3]);
VisuNodeInfo* visu_data_getDistanceList(VisuData *data, guint nodeId, float *minVal);
gboolean visu_data_setNewBasisFromNodes(VisuData *data, guint nO, guint nA, guint nB, guint nC);
gboolean visu_data_setNewBasis(VisuData *data, float matA[3][3], float O[3]);
gboolean visu_data_reorder(VisuData *data, VisuData *dataRef);

/***************/
/* OpenGL Part */
/***************/
void visu_data_createNodes(VisuData *data, VisuElement *ele);
void visu_data_createNode(VisuData *data, VisuNode *node);
void visu_data_createAllNodes(VisuData *data);
void visu_data_createAllElements(VisuData *data);

void visu_data_emitAskForShowHideNodes(VisuData *data, gboolean *redraw);
void visu_data_emitNodePositionChanged(VisuData *data);
void visu_data_emitNodeRenderedChange(VisuData *data);

guint visu_data_addTimeout(VisuData *data, guint time, GSourceFunc func, gpointer user_data);
gboolean visu_data_removeTimeout(VisuData *data, guint timeoutId);


VisuOpenGLView* visu_data_getOpenGLView(VisuData *data);
gboolean visu_data_setAngleOfView(VisuData *data, float valueTheta,
				 float valuePhi, float valueOmega, int mask);

gboolean visu_data_setPositionOfView(VisuData *data,
				    float valueX, float valueY, int mask);

gboolean visu_data_setZoomOfView(VisuData *data, float value);

int visu_data_setPerspectiveOfView(VisuData *data, float value);

int visu_data_setSizeOfView(VisuData *data, guint width, guint height);

void visu_data_emitFacettesChanged(VisuData *data);

void visu_data_setNodeScalingFunc(VisuData *data, VisuDataScalingFunc scaling);

float visu_data_getNodeScalingFactor(VisuData *data, VisuNode *node);
GList* visu_data_getAllObjects(void);


void visu_data_removeNodes(VisuData *data, int *nodeNumbers);

/* Iterators. */
/**
 * VisuDataIterType:
 * @ITER_NODES_BY_TYPE: run on nodes, as V_Sim internal storage,
 * fastest.
 * @ITER_NODES_BY_NUMBER: run on nodes as entered in the input file.
 * @ITER_NODES_VISIBLE: run on visible nodes only (internal sort).
 * @ITER_NODES_ORIGINAL: run on original nodes only (internal sort).
 * @ITER_ELEMENTS: run on elements only.
 *
 * The kind of iterator to be used on #VisuData objects.
 *
 * Since: 3.6
 */
typedef enum
  {
    ITER_NODES_BY_TYPE,
    ITER_NODES_BY_NUMBER,
    ITER_NODES_VISIBLE,
    ITER_NODES_ORIGINAL,
    ITER_ELEMENTS
  } VisuDataIterType;

/**
 * VisuDataIter:
 * @data: a pointer the iterator is associated to ;
 * @idMax: current higher id used to identified nodes.
 * @nAllStoredNodes: the total number of stored nodes for the
 *                   associated #VisuData ;
 * @nElements: the number of #VisuElement for the associated #VisuData ;
 * @nStoredNodes: the number of stored nodes for each element ;
 * @iElement: the index corresponding to @element (or -1 if no set);
 * @node: a pointer on a current node ;
 * @element: a pointer on a current element.
 * @type: the kind of iterator, see #VisuDataIterType.
 * @init: an internal flag.
 *
 * This structure is an iterator over the nodes of a #VisuData object.
 * Create it with visu_data_iterNew(). Then the numbers are allocated and
 * correspond to the value of the #VisuData object. Use visu_data_iterStart()
 * to initialise the iterator for a run over the nodes, visu_data_iterNext()
 * to associate @node and @element to the next node, or NULL if there is no
 * more node to run over.
 */
typedef struct _VisuDataIter VisuDataIter;
struct _VisuDataIter
{
  VisuData *data;

  guint idMax;
  guint nAllStoredNodes;
  guint nElements;
  guint *nStoredNodes;

  guint iElement;
  VisuNode *node;
  VisuElement *element;

  VisuDataIterType type;
  gboolean init;
};

void visu_data_iterNew(VisuData *data, VisuDataIter *iter);

void visu_data_iterStart(VisuData *data, VisuDataIter *iter);
void visu_data_iterStartVisible(VisuData *data, VisuDataIter *iter);
void visu_data_iterStartNumber(VisuData *data, VisuDataIter *iter);
void visu_data_iterRestartNode(VisuData *data, VisuDataIter *iter);

void visu_data_iterNext(VisuData *data, VisuDataIter *iter);
void visu_data_iterNextNode(VisuData *data, VisuDataIter *iter);
void visu_data_iterNextNodeNumber(VisuData *data, VisuDataIter *iter);
void visu_data_iterNextNodeOriginal(VisuData *data, VisuDataIter *iter);
void visu_data_iterNextElement(VisuData *data, VisuDataIter *iter);
void visu_data_iterNextVisible(VisuData *data, VisuDataIter *iter);

gboolean visu_data_iter_next(VisuDataIter *dataIter);
gboolean visu_data_iter_next2(VisuDataIter *iter1, VisuDataIter *iter2);


G_END_DECLS

#endif
