/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#ifndef VISU_NODES_H
#define VISU_NODES_H

#include <glib.h>
#include <glib-object.h>


typedef struct _VisuNode VisuNode;

struct _VisuNode
{
  /* coordinates of the node in cartesian coordinates. */
  float xyz[3];

  /* translation */
  float translation[3];

  /* Number of this element in the input file. */
  unsigned int number;
  /* Position in the #VisuData structure. */
  unsigned int posElement, posNode;

  /* A boolean to specify if this node is rendered or not. */
  gboolean rendered;
};

/**
 * VisuNodeProperty:
 *
 * This structure defines a storage for one property for each node of a given
 * #VisuNodeArray. Use visu_node_property_newPointer() or
 * visu_node_property_newInteger() to create one property.
 */
typedef struct _VisuNodeProperty VisuNodeProperty;

typedef struct VisuNodeInfo_struct
{
  guint id;
  float dist;
} VisuNodeInfo;

typedef struct _VisuNodeArray VisuNodeArray;

/**
 * VisuNodeArray:
 * @ntype: number of #VisuElement used in this object.
 * @idCounter: an internal counter used to give an id to new nodes.
 * @nodeTable: give the #VisuNode knowing its element number and
 *             its own position.
 * @nodeTableSize: the size of the previous array.
 * @nNodes: the total of allocated VisuNodes.
 * @nbOfAllStoredNodes: number of registered #VisuNode in this object.
 * @numberOfNodes: give the number of allocated #VisuNode for each
 *                 #VisuElement.
 * @numberOfStoredNodes: give the number of stored #VisuNode for
 *                       each #VisuElement.
 * @nodes: array that stores all the nodes.
 * @properties: a list of properties (see #VisuNodeProperty).
 * @origProp: a property saying if the nodes are original or not.
 * @nOrigNodes: the number of original nodes.
 *
 * This structure describes a set of nodes of different #VisuElement types.
 * It is optimized for quick access, allocation and reallocation.
 */

struct _VisuNodeArray
{
  /* Number of VisuElements used by these data. */
  unsigned int ntype;
  /* A counter. */
  unsigned int idCounter;
  /* This array gives access to the good VisuNode
     when one has its number. This number is an integer ranging in
     [0;idCounter[. This value is readable in the #VisuNode structure
     as the number attribute. The current allocated size is stored in
     @nodeTableSize. */
  VisuNode   **nodeTable;
  unsigned int nodeTableSize;
  /* The total of allocated VisuNodes. */
  unsigned int nNodes;
  /* The total of stored VisuNodes. */
  unsigned int nbOfAllStoredNodes;

  /* Number of nodes allocated (size of the nodes array) per VisuElement. */
  unsigned int* numberOfNodes;
  /* Number of nodes physically present in the array per VisuElement. */
  unsigned int* numberOfStoredNodes;
  /* Coordinates of all these nodes. */
  VisuNode **nodes;

  /* This is a table to store data, reachable with string keys.
     It should be accessed via visu_node_setproperty()
     and visu_node_array_getProperty(). */
  GHashTable *properties;

  /* A convenient pointer on the original node property. */
  VisuNodeProperty *origProp;
  guint nOrigNodes;
};


void visu_node_newValues(VisuNode *node, float xyz[3]);

void visu_node_copy(VisuNode *nodeTo, VisuNode *nodeFrom);

int visu_node_setVisibility(VisuNode* node, gboolean visibility);

gboolean visu_node_getVisibility(VisuNode* node);


VisuNodeArray* visu_node_array_newNodes(unsigned int nTypes, unsigned int *nNodes);

void visu_node_array_removeNodes(VisuNodeArray *nodeArray, int *nodeNumbers);

gboolean visu_node_array_removeAllDuplicateNodes(VisuNodeArray *nodeArray,
					  int **nodeNumbers);

gint visu_node_array_getOriginal(VisuNodeArray *nodeArray,
			guint nodeId);

gboolean visu_node_array_setOriginal(VisuNodeArray *nodeArray,
			guint nodeId);

void visu_node_array_freeNodes(VisuNodeArray *nodeArray);

void visu_node_array_allocateNewNodes(VisuNodeArray *nodeArray, unsigned int iEle,
			       unsigned int step);


VisuNode* visu_node_array_getNewNode(VisuNodeArray *nodeArray, unsigned int iEle);
gboolean visu_node_array_switchNumber(VisuNodeArray *nodeArray, guint from, guint to);

VisuNode* visu_node_array_getCopyNode(VisuNodeArray *nodeArray, VisuNode *node);


void visu_node_array_traceProperty(VisuNodeArray *array, const gchar *id);


/*************************/
/* The property methods. */
/*************************/


VisuNodeProperty* visu_node_property_newPointer(VisuNodeArray* nodeArray,
					      const char* key, 
					      GFunc freeFunc,
					      GCopyFunc newAndCopyFunc,
					      gpointer user_data);

VisuNodeProperty* visu_node_property_newInteger(VisuNodeArray* nodeArray,
					  const char* key);

void visu_node_array_freeProperty(VisuNodeArray* nodeArray, const char* key);


VisuNodeProperty* visu_node_array_getProperty(VisuNodeArray* nodeArray, const char* key);

/**
 * visu_node_setpropertyValue:
 * @nodeArray: a #VisuNodeArray object ;
 * @node: a #VisuNode object ;
 * @key: a string ;
 * @value: A GValue pointer this the value to be stored.
 *
 * This method is used to store some values associated with
 * the given @node of the given @nodeArray. These values can be pointers to
 * anything allocated (will be free automatically when the property is deleted) or
 * they can be static values. This depends on the construction of the node property.
 * These values are described by the @key, and can be retrieved with the
 * visu_node_array_getPropertyValue() method.
 *
 * See visu_node_property_setValue() to directly set a value associated to a node.
 */
#define visu_node_setpropertyValue(nodeArray, node, key, value)	\
  visu_node_property_setValue(visu_node_array_getProperty(nodeArray, key), node, value)

void visu_node_property_setValue(VisuNodeProperty* nodeProp, VisuNode* node,
			       GValue *value);
/**
 * visu_node_array_getPropertyValue:
 * @nodeArray: a #VisuNodeArray object ;
 * @node: a #VisuNode object ;
 * @key: a string ;
 * @value: an initialise GValue location.
 *
 * This method is used to retrieve some data associated to
 * the specified @node, stored in the given @data. These return data
 * should not be freed after used. The read value is stored in the given
 * GValue pointer. This GValue must be of the right type, depending on the
 * creation of the #VisuNodeProperty.
 *
 * Returns: some data associated to the key, stored the given GValue location.
 */
#define visu_node_array_getPropertyValue(nodeArray, node, key, value)		\
  visu_node_property_getValue(visu_node_array_getProperty(nodeArray, key), node, value)

GValue* visu_node_property_getValue(VisuNodeProperty* nodeProp, VisuNode* node,
				  GValue *value);

#endif
