#include <stdlib.h>
#include <string.h>

#include <gtk/gtk.h>
#include <gdk/gdkx.h>

#include "guiutils.h"
#include "splash.h"
#include "config.h"

#include "vertex.xpm"


static gint SplashWinCloseCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
static void SplashWinDestroyCB(GtkObject *object, gpointer data);

splash_win_struct *SplashNew(gint total_pbars, const gchar *splash_path);
gint SplashUpdate(
	splash_win_struct *sw,
	gint pbar_num,		/* Progress bar number */
	gdouble pos,		/* Position from 0.0 to 1.0 */
	const gchar *mesg,	/* Message, can be NULL */
	gbool allow_iteration	/* Allow call to gtk_main_iteration */
);
void SplashWinMap(splash_win_struct *sw);
void SplashWinUnmap(splash_win_struct *sw);
void SplashDelete(splash_win_struct *sw);


#define PROGRESS_BAR_HEIGHT	(20 + (2 * 2))


/*
 *      Close callback.
 */
static gint SplashWinCloseCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	SplashWinUnmap((splash_win_struct *)data);
	return(TRUE);
}

/*
 *      Destroy callback.
 */
static void SplashWinDestroyCB(GtkObject *object, gpointer data)
{
	return;
}


/*
 *	Create a new splash window.
 */
splash_win_struct *SplashNew(gint total_pbars, const gchar *splash_path)
{
	gint i;
	GdkPixmap *pixmap;
	GdkBitmap *mask;
	GdkWindow *window;
	GtkStyle *style;
	GtkRcStyle *rcstyle;
	GtkWidget *w, *parent;
	GtkWidget *main_vbox;
	splash_win_struct *sw = (splash_win_struct *)g_malloc0(
	    sizeof(splash_win_struct)
	);
	if(sw == NULL)
	    return(NULL);


	/* Reset values */
	sw->initialized = TRUE;
	sw->map_state = FALSE;
	sw->width = -1;
	sw->height = -1;
	sw->font_title = NULL;


	/* Toplevel */
	sw->toplevel = w = gtk_window_new(GTK_WINDOW_DIALOG);
	gtk_widget_realize(w);
	window = w->window;
	if(window != NULL)
	{
	    gdk_window_set_decorations(window, 0);
	    gdk_window_set_functions(window, 0);
	    GUISetWMIcon(window, (guint8 **)vertex_xpm);
	}
	style = gtk_widget_get_style(w);
	gtk_window_set_title(GTK_WINDOW(w), PROG_NAME);
	gtk_window_set_policy(GTK_WINDOW(w), TRUE, TRUE, TRUE);
	gtk_signal_connect(
	    GTK_OBJECT(w), "delete_event",
	    GTK_SIGNAL_FUNC(SplashWinCloseCB),
	    (gpointer)sw
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "destroy",
	    GTK_SIGNAL_FUNC(SplashWinDestroyCB),
	    (gpointer)sw
	);
	gtk_container_border_width(GTK_CONTAINER(w), 0);
	parent = w;

	/* Frame */
	w = gtk_frame_new(NULL);
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_container_border_width(GTK_CONTAINER(w), 0);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_OUT);
	gtk_widget_show(w);
	parent = w;

	/* Main vbox */
	main_vbox = w = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_widget_show(w);
	parent = w;


	/* Load splash image pixmap and mask pair if we have enough
	 * information.
	 */
	pixmap = NULL;
	if((style != NULL) && (splash_path != NULL))
	{
	    pixmap = gdk_pixmap_create_from_xpm(
		window, &mask, &style->bg[GTK_STATE_NORMAL], splash_path
	    );
	}
	/* Pixmap loaded? */
	if(pixmap != NULL)
	{
	    /* Successfully loaded splash image pixmap and possibly the
	     * the mask
	     */
	    GtkWidget *fixed;
	    gint width, height;

	    /* Create a GtkFixed widget */
	    fixed = w = gtk_fixed_new();
	    gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);

	    /* Create a GtkPixmap from the loaded pixmap */
	    w = gtk_pixmap_new(pixmap, mask);
	    /* Get size of pixmap and set the new GtkFixed widget's
	     * size to match
	     */
	    gdk_window_get_size((GdkWindow *)pixmap, &width, &height);
	    gtk_widget_set_usize(fixed, width, height);
	    /* Put the new GtkPixmap into the GtkFixed widget */
	    if(w != NULL)
	    {
		gtk_fixed_put(GTK_FIXED(fixed), w, 0, 0);
		gtk_widget_show(w);
	    }

	    /* Update toplevel size */
	    sw->width = width;
	    sw->height = height;

	    /* Unref the GdkPixmap pixmap */
	    gdk_pixmap_unref(pixmap);
	    pixmap = NULL;

	    /* If mask is available then use it to shape the GtkFixed
	     * and then unref the mask
	     */
	    if(mask != NULL)
	    {
		gtk_widget_shape_combine_mask(fixed, mask, 0, 0);
		gdk_bitmap_unref(mask);
		mask = NULL;
	    }

	    /* Loaded pixmap and mask are now invalid */
	}
	else
	{
	    /* Failed to load pixmap, not able to create a splash image
	     * So instead create a placeholding label
	     */
	    gchar *buf = g_strdup_printf(
		"(Cannot load splash image \"%s\")",
		splash_path
	    );
            rcstyle = gtk_rc_style_new();
            rcstyle->font_name = g_strdup(VMA_GTK_FONT_NAME_TEXT_HEADING5);

	    w = gtk_label_new(buf);
	    gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	    gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
            gtk_widget_modify_style(w, rcstyle);
	    gtk_widget_show(w);

            gtk_rc_style_unref(rcstyle);                
	    g_free(buf);


	    w = gtk_label_new(PROG_NAME " Version " PROG_VERSION);
	    gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_CENTER);
	    rcstyle = gtk_rc_style_new();
	    g_free(rcstyle->font_name);
	    rcstyle->font_name = g_strdup(VMA_GTK_FONT_NAME_TEXT_HEADING1);
	    gtk_widget_modify_style(w, rcstyle);
	    gtk_rc_style_unref(rcstyle); rcstyle = NULL;
	    gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	}


	/* Begin creating progress bars */
	sw->total_pbars = total_pbars;
	if(sw->total_pbars > 0)
	{
	    GtkAdjustment *adj;

	    sw->pbar = (GtkWidget **)g_malloc0(
		sw->total_pbars * sizeof(GtkWidget *)
	    );
	    if(sw->pbar == NULL)
		sw->total_pbars = 0;

	    /* Iterate through and create each new progress bar */
	    for(i = 0; i < sw->total_pbars; i++)
	    {
		/* Progress bar */
		adj = (GtkAdjustment *)gtk_adjustment_new(0, 1, 100, 0, 0, 0);
		sw->pbar[i] = w = gtk_progress_bar_new_with_adjustment(adj);
		gtk_widget_set_usize(w, -1, PROGRESS_BAR_HEIGHT);
		gtk_progress_bar_set_orientation(
		    GTK_PROGRESS_BAR(w), GTK_PROGRESS_LEFT_TO_RIGHT
		);
		gtk_progress_bar_set_bar_style(
		    GTK_PROGRESS_BAR(w), GTK_PROGRESS_CONTINUOUS
		);
		gtk_progress_set_activity_mode(
		    GTK_PROGRESS(w), FALSE
		);
		gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
		gtk_widget_show(w);
	    }
	}

	return(sw);
}

/*
 *	Update splash window, returns one of SPLASH_*.
 */
gint SplashUpdate(
	splash_win_struct *sw,
	gint pbar_num,          /* Progress bar number */
	gdouble pos,            /* Position from 0.0 to 1.0 */
	const gchar *mesg,      /* Message, can be NULL */
	gbool allow_iteration   /* Allow call to gtk_main_iteration */
)
{
	GtkWidget *w;


	if((sw != NULL) ? !sw->initialized : 1)
	    return(SPLASH_ERROR);

	/* Get pointer to progress bar widget specified by pbar_num if
	 * it is valid.
	 */
	if((pbar_num >= 0) && (pbar_num < sw->total_pbars))
	    w = sw->pbar[pbar_num];
	else
	    w = NULL;
	/* Got valid progress bar? */
	if(w != NULL)
	{
	    GtkProgress *pr = GTK_PROGRESS(w);
	    GtkProgressBar *pb = GTK_PROGRESS_BAR(w);


	    /* Sanitize position */
	    if(pos > 1.0)
		pos = 1.0;
	    else if(pos < 0.0)
		pos = 0.0;

	    /* Progress bar message given for display? */
	    if(mesg != NULL)
	    {
		gtk_progress_set_format_string(pr, mesg);
		gtk_progress_set_show_text(pr, TRUE);
	    }
	    else
	    {
		gtk_progress_set_show_text(pr, FALSE);
	    }

/*
	    gtk_progress_bar_set_bar_style(
		pb, GTK_PROGRESS_DISCRETE
	    );
	    gtk_progress_bar_set_discrete_blocks(pb, nblocks);
 */
	    gtk_progress_bar_set_bar_style(
		pb, GTK_PROGRESS_CONTINUOUS
	    );
	    gtk_progress_bar_update(GTK_PROGRESS_BAR(pb), pos);
	}

	/* Allow us to call GTK+ main iterations to flush GTK+ events
	 * so progress bar gets updated?
	 */
	if(allow_iteration)
	{
	    while(gtk_events_pending() > 0)
		gtk_main_iteration();
	}

	return(SPLASH_SUCCESS);
}

/*
 *	Maps the splash window.
 */
void SplashWinMap(splash_win_struct *sw)
{
	GtkWidget *w;

	if((sw != NULL) ? !sw->initialized : 1)
	    return;

	if(!sw->map_state)
	{
	    w = sw->toplevel;
	    if((w != NULL) ? !GTK_WIDGET_NO_WINDOW(w) : 0)
	    {
		GdkWindow *window = w->window;
		GdkWindow *root = (GdkWindow *)GDK_ROOT_PARENT();

		if((window != NULL) && (root != NULL))
		{
		    gint rx, ry, rw, rh, rd;
		    gdk_window_get_geometry(
			root, &rx, &ry, &rw, &rh, &rd
		    );
		    gtk_widget_set_uposition(
			w,
			(sw->width > 0) ?
			    (rw / 2) - (sw->width / 2) :
			    (rw / 2) - (w->allocation.width / 2),
			(sw->height > 0) ?
			    (rh / 2) - (sw->height / 2) :
			    (rh / 2) - (w->allocation.height / 2)
		    );
		}

		gtk_widget_show(w);
	    }

	    sw->map_state = TRUE;
	}
}

/*
 *	Unmaps the splash window.
 */
void SplashWinUnmap(splash_win_struct *sw)
{
	GtkWidget *w;

	if((sw != NULL) ? !sw->initialized : 1)
	    return;

	if(sw->map_state)
	{
	    w = sw->toplevel;
	    if(w != NULL)
		gtk_widget_hide(w);
	    sw->map_state = FALSE;
	}
}

/*
 *	Destroys the given splash window.
 */
void SplashDelete(splash_win_struct *sw)
{
	gint i;
	GdkFont **font;
	GtkWidget **w;


	if(sw == NULL)
	    return;

	if(sw->initialized)
	{
#define DO_UNREF_FONT		\
{ \
 if((*font) != NULL) \
 { \
  GdkFont *tf = *font; \
  (*font) = NULL; \
  gdk_font_unref(tf); \
 } \
}

#define DO_DESTROY_WIDGET       \
{ \
 if((*w) != NULL) \
 { \
  GtkWidget *tmp_w = *w; \
  (*w) = NULL; \
  gtk_widget_destroy(tmp_w); \
 } \
}

	    for(i = sw->total_pbars - 1; i >= 0; i--)
	    {
		w = &sw->pbar[i];
		DO_DESTROY_WIDGET
	    }

	    w = &sw->toplevel;
	    DO_DESTROY_WIDGET

	    font = &sw->font_title;
	    DO_UNREF_FONT

#undef DO_UNREF_FONT
#undef DO_DESTROY_WIDGET
	}

	g_free(sw->pbar);
	sw->pbar = NULL;
	sw->total_pbars = 0;

	/* Deallocate structure itself */
	g_free(sw);
}
