//=======================================================================
//  V/VIDE helper class - vPipedProcess
//  Copyright (C) 2000  Bruce E. Wampler
//
//  This program is part of the VIDE and the V C++ GUI Framework
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  (see COPYING) along with this program; if not, write to the Free
//  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//=======================================================================
/*
 *	HOW TO USE THIS CLASS WITH V
 *
 * This class is designed to work with a V timer (or other timer
 * based directly on the OS API).  The idea is to
 * start the desired process running as a separate, detached process.
 * For example, VIDE uses this to run make and gdb. 
 * Once the child process has been started, it will be sending its
 * output to its stdout and stderr, and perhaps getting input from
 * its stdin. This class routes these to the parent via a pipe.
 * The parent will poll the child for the child's output
 * using a timer to periodically call readChar. The parent can send
 * commands to the child as needed with the writeChars method.
 * 
 * To use this class, first start the process. Then start a timer to
 * poll the child's output as often as needed (3 or 4 times a second
 * seems to work well). The vPipedProcess::readChar method will
 * return 0 if no output was available from the child, otherwise
 * one character is read. The parent should read all chars that
 * are available and display or process them as appropriate.
 * 
 * This class also provides a method to check if the child is
 * still running -- useful to see if the make is completed, for
 * example, also from the timer. You can kill the child as well.
 */

#include "vpipedproc.h"

#ifdef V_VersionX
#include <sys/wait.h>		// for wait
#include <signal.h>
#endif

//=====================>>> vPipedProcess::vPipedProcess <<<========================
  vPipedProcess::vPipedProcess()
  {
    _running = false;

    _errorCode = 0;
    _exitCode = 0;
  }

//=====================>>> vPipedProcess::vPipedProcess <<<========================
  vPipedProcess::~vPipedProcess()
  {
    killProcess();		// be sure anything we started is killed
  }

//====================>>> vPipedProcess::readChar <<<====================
  int vPipedProcess::readChar(char* chrin)
  {
    // read a char from process if there, otherwise return 0

#ifdef V_VersionWindows
    DWORD bytesRead;
    DWORD bytesAvail;

    if (!PeekNamedPipe(hChildStdoutRd,
	chrin, (DWORD)1, &bytesRead, &bytesAvail, 0))
      {
	return 0;
      }
    if (bytesAvail == 0)	// no output avail, so return
	return 0;

    if (!ReadFile(hChildStdoutRd,	// now really read it
	chrin, (DWORD)1, &bytesRead, 0) )
    	return 0;
    return bytesRead;
#endif
#ifdef V_VersionX
    ssize_t bytesRead;

    // set to non-blocking, which will return 0 if  no bytes read
    bytesRead = read(to_parent[pREAD],chrin,(size_t)1);
    if (bytesRead <= 0)
	return 0;
    else
	return bytesRead;
#endif
  }

//====================>>> vPipedProcess::writeChars <<<====================
  void vPipedProcess::writeChars(char* cmd)
  {
#ifdef V_VersionWindows
    DWORD written;

    if (!WriteFile(hChildStdinWrDup, cmd, strlen(cmd), &written, 0))
        return;
#endif
#ifdef V_VersionX
    (void) write(to_child[pWRITE],cmd,(size_t)strlen(cmd));
#endif
  }

//=====================>>> vPipedProcess::stillRunning <<<========================
  bool vPipedProcess::stillRunning()
  {
    // return true if the process is still running

    if (!_running)			// allow multiple calls
	return false;

#ifdef V_VersionWindows
    DWORD status = ::WaitForSingleObject(_processHandle,(DWORD)0);
    if (status == WAIT_OBJECT_0)	// signaled, means done
      {
	_running = false;
	_exitCode = 0;
	::GetExitCodeProcess(_processHandle,&_exitCode);

	::CloseHandle(_processThread);
	::CloseHandle(_processHandle);
	return false;
      }
    else
	return true;
#endif
#ifdef V_VersionX
    int status;

    if (waitpid(_pid, &status, WNOHANG) == 0)	// still running or error
      {
	return true;
      }
    if (WIFEXITED(status))			// normal exit
      {
	_running = false;
	_exitCode = WEXITSTATUS(status);
	close(to_parent_error[pREAD]);
	close(to_parent[pREAD]);
	close(to_child[pWRITE]);
	return false;
      }
    _running = false;				// some error, just try this
    close(to_parent_error[pREAD]);
    close(to_parent[pREAD]);
    close(to_child[pWRITE]);
    return false;
#endif
  }

//=====================>>> vPipedProcess::killProcess <<<========================
  void vPipedProcess::killProcess(int exitCode)
  {
    // kill the process

    if (!_running)		// don't do anything if already stopped
	return;
    _running = false;
    _exitCode = exitCode;

#ifdef V_VersionWindows
    ::TerminateProcess(_processHandle,exitCode);
    ::CloseHandle(_processThread);
    ::CloseHandle(_processHandle);
#endif
#ifdef V_VersionX
    kill(_pid,SIGKILL);
    _pid = 0;
    close(to_parent_error[pREAD]);
    close(to_parent[pREAD]);
    close(to_child[pWRITE]);
#endif
  }

//=====================>>> vPipedProcess::startPipedProcess <<<========================
  bool vPipedProcess::startPipedProcess(char* cmd)
  {
    /*
     *	This method will run the specified command (with args) as
     *  a detached process. The object will keep track of whatever
     *  is needed to interact with the running processs. Communication
     *  with the process is via pipes to the child's stdin and from
     *  the child's stdout and stderr (which can be read as a single
     *  stream by the parent).
     *
     *  Return true if process started, false if it didn't
     */

#ifdef V_VersionWindows			// Ms-Windows version
    SECURITY_ATTRIBUTES saAttr; 
 
    /* Set the bInheritHandle flag so pipe handles are inherited. */ 

    saAttr.nLength = sizeof(SECURITY_ATTRIBUTES);
    saAttr.bInheritHandle = TRUE;
    saAttr.lpSecurityDescriptor = NULL;
 
    /* 
     * The steps for redirecting child's STDOUT: 
     *     1.  Save current STDOUT, to be restored later. 
     *     2.  Create anonymous pipe to be STDOUT for child. 
     *     3.  Set STDOUT of parent to be write handle of pipe, so 
     *         it is inherited by child.
     */ 
 
    // Save the handle to the current STDOUT

    hSaveStdout = ::GetStdHandle(STD_OUTPUT_HANDLE);
 
    // Create a pipe for the child's STDOUT
 
    if (! ::CreatePipe(&hChildStdoutRd, &hChildStdoutWr, &saAttr, 0))
        return false;	// Stdout pipe creation failed
 
    // Set a write handle to the pipe to be STDOUT

    if (! ::SetStdHandle(STD_OUTPUT_HANDLE, hChildStdoutWr))
        return false;	// Redirecting STDOUT failed

   /* 
    * The steps for redirecting child's STDIN: 
    *     1.  Save current STDIN, to be restored later. 
    *     2.  Create anonymous pipe to be STDIN for child. 
    *     3.  Set STDIN of parent to be read handle of pipe, so 
    *         it is inherited by child. 
    *     4.  Create a noninheritable duplicate of write handle, 
    *         and close the inheritable write handle. 
    */ 
 
   // Save the handle to the current STDIN

    hSaveStdin = GetStdHandle(STD_INPUT_HANDLE);
 
    // Create a pipe for the child's STDIN
 
    if (! CreatePipe(&hChildStdinRd, &hChildStdinWr, &saAttr, 0)) 
        return 0;	// Stdin pipe creation failed

    // Set a read handle to the pipe to be STDIN
 
    if (! SetStdHandle(STD_INPUT_HANDLE, hChildStdinRd)) 
        return 0;	// Redirecting Stdin failed
 
    // Duplicate the write handle to the pipe so it is not inherited.
 
    BOOL fSuccess = DuplicateHandle(GetCurrentProcess(), hChildStdinWr, 
        GetCurrentProcess(), &hChildStdinWrDup, 0, 
        TRUE,       // is inherited
        DUPLICATE_SAME_ACCESS); 

    if (! fSuccess) 
        return 0;	// DuplicateHandle failed
 
    CloseHandle(hChildStdinWr);  
 
    // Now create the child process

    static PROCESS_INFORMATION piProcInfo;
    STARTUPINFO siStartInfo; 

    // Set up members of STARTUPINFO structure

    ::ZeroMemory(&siStartInfo, sizeof(siStartInfo));
    siStartInfo.cb = sizeof(STARTUPINFO); 
    siStartInfo.lpReserved = NULL; 
    siStartInfo.lpReserved2 = NULL; 
    siStartInfo.cbReserved2 = 0; 
    siStartInfo.lpDesktop = NULL;
    siStartInfo.wShowWindow = SW_MINIMIZE;
    siStartInfo.dwFlags = STARTF_USESHOWWINDOW | STARTF_USESTDHANDLES;
    siStartInfo.hStdOutput = hChildStdoutWr;
    siStartInfo.hStdError = hChildStdoutWr;
    siStartInfo.hStdInput = hChildStdinRd;

    // Create the child process
    OSVERSIONINFO osver;
    osver.dwOSVersionInfoSize = sizeof(OSVERSIONINFO);
    int createResult;

    ::GetVersionEx(&osver);
    if (osver.dwPlatformId == VER_PLATFORM_WIN32_NT)
      {
	createResult = ::CreateProcess(NULL,
		cmd,
		NULL,          // process security attributes
		NULL,          // primary thread security attributes
		TRUE,          // handles are inherited
		CREATE_NEW_PROCESS_GROUP,   // creation flags
//	 CREATE_NEW_CONSOLE, //| CREATE_NEW_PROCESS_GROUP,
		NULL,          // use parent's environment
		NULL,          // use parent's current directory

		&siStartInfo,  // STARTUPINFO pointer
		&piProcInfo);  // receives PROCESS_INFORMATION
      }
    else		// Win9x version
      {
	createResult = ::CreateProcess(NULL,
		cmd,
		NULL,          // process security attributes
		NULL,          // primary thread security attributes
		TRUE,          // handles are inherited
		0,             // creation flags
		NULL,          // use parent's environment
		NULL,          // use parent's current directory

		&siStartInfo,  // STARTUPINFO pointer
		&piProcInfo);  // receives PROCESS_INFORMATION
      }

    if (!createResult)
      {
	_errorCode = ::GetLastError();		// give some hope of finding out
	return false;	// create fails
      }

    _processId = piProcInfo.dwProcessId;	// save these for termination
    _processHandle = piProcInfo.hProcess;
    _processThread = piProcInfo.hThread;

    // After process creation, restore the saved STDIN and STDOUT

    if (! SetStdHandle(STD_INPUT_HANDLE, hSaveStdin))
        return false;	// Re-redirecting Stdout failed
    if (! SetStdHandle(STD_OUTPUT_HANDLE, hSaveStdout))
        return false;	// Re-redirecting Stdout failed

#endif			// end of Windows code

//  X  X  X  X  X  X  X  X  X  X  X  X  X  X  X  X  X  X  X  X  X  X  X

#ifdef V_VersionX

    // 1. setup parent side communications
    //    open pipes for stdin, stdout, and stderr
     if (pipe(to_child) < 0)
       {
	 return 0;
       }
    if (pipe(to_parent) < 0)
       {
	 return 0;
       }
    if (pipe(to_parent_error) < 0)
       {
	 return 0;
       }
    // 2. start child process

    if ((_pid = fork()) == 0)		// in child
      {
	// 2a. set up child communications
	//    close unused pipe ends
	close(to_child[pWRITE]);
	close(to_parent[pREAD]);
	close(to_parent_error[pREAD]);

	// assign stdin to to_child, stdout to to_parent, and
	// stderr to to_parent_error
	dup2(to_child[pREAD], fileno(stdin));
	close(to_child[pREAD]);

	dup2(to_parent[pWRITE], fileno(stdout));
	close(to_parent[pWRITE]);

	dup2(to_parent_error[pWRITE], fileno(stderr));
	close(to_parent_error[pWRITE]);

	// 2b. Now, execute child (we are in child fork...)
	execl("/bin/sh", "sh", "-c", cmd, (char *)0);

	// could not find child: send message to parent via stderr
	_errorCode = -1;
	return 0;
      }

    if (_pid == -1)		// failed
      {
	_pid = 0;
	return 0;
      }

    // OK, fork worked, and we are the parent
    // 3. setup parent communication

    // I am the parent: close unused pipe ends
    close(to_child[pREAD]);
    close(to_parent[pWRITE]);
    close(to_parent_error[pWRITE]);

    fcntl(to_parent[pREAD],F_SETFL,O_NONBLOCK);	// non-block on input from child
#endif

    _running = true;	// remember we are running
    return true;
  }
